/**
 * Task Manager — FocusFlow - Service Worker
 * 
 * Provides offline functionality, caching strategies, and PWA support
 * for the Task Manager application.
 * 
 * @author Jack Ewers / BloodWeb
 * @version 2.1.0
 * @date 2025-12-27
 */

const CACHE_VERSION = '2.1.0';
const CACHE_NAME = `tasklist-v2-cache-${CACHE_VERSION.replace(/\./g, '-')}`;
const API_CACHE_NAME = `tasklist-v2-api-cache-${CACHE_VERSION.replace(/\./g, '-')}`;

// Static assets to cache immediately
// Note: Only cache files that definitely exist to avoid cache errors
const STATIC_ASSETS = [
    './',
    './index.php',
    './main.js',
    './styles.css',
    './manifest.json',
    'https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css'
];

// API endpoints to cache with different strategies
const API_ENDPOINTS = [
    './api/tasks',
    './api/projects',
    './api/categories',
    './api/analytics'
];

// Network-first endpoints (always try network first)
const NETWORK_FIRST_ENDPOINTS = [
    './api/xp'
];

// Cache-first endpoints (try cache first, fallback to network)
const CACHE_FIRST_ENDPOINTS = [
    './api/projects',
    './api/categories'
];

// =============================================================================
// SERVICE WORKER INSTALLATION
// =============================================================================

self.addEventListener('install', (event) => {
    console.log('Service Worker: Installing...');
    
    event.waitUntil(
        caches.open(CACHE_NAME)
            .then((cache) => {
                console.log('Service Worker: Caching static assets');
                return cache.addAll(STATIC_ASSETS);
            })
            .then(() => {
                console.log('Service Worker: Installation complete');
                // Don't call skipWaiting() - let it activate on next page load naturally
            })
            .catch((error) => {
                console.error('Service Worker: Installation failed', error);
            })
    );
});

// =============================================================================
// MESSAGE HANDLING
// =============================================================================

self.addEventListener('message', (event) => {
    if (event.data && event.data.type === 'SKIP_WAITING') {
        console.log('Service Worker: Received SKIP_WAITING message');
        self.skipWaiting();
    }
});

// =============================================================================
// SERVICE WORKER ACTIVATION
// =============================================================================

self.addEventListener('activate', (event) => {
    console.log('Service Worker: Activating...');
    
    event.waitUntil(
        caches.keys()
            .then((cacheNames) => {
                return Promise.all(
                    cacheNames.map((cacheName) => {
                        if (cacheName !== CACHE_NAME && cacheName !== API_CACHE_NAME) {
                            console.log('Service Worker: Deleting old cache', cacheName);
                            return caches.delete(cacheName);
                        }
                    })
                );
            })
            .then(() => {
                console.log('Service Worker: Activation complete');
                return self.clients.claim();
            })
    );
});

// =============================================================================
// FETCH EVENT HANDLING
// =============================================================================

self.addEventListener('fetch', (event) => {
    const { request } = event;
    const url = new URL(request.url);
    
    // CRITICAL: Only handle requests within the TaskList application scope
    // Ignore all requests outside of /app/tasklist/ or /TaskList/
    const isTaskListScope = url.pathname.includes('/app/tasklist/') || 
                           url.pathname.includes('/TaskList/') ||
                           url.pathname === '/app/tasklist' ||
                           url.pathname === '/TaskList';
    
    if (!isTaskListScope && !url.href.includes('cdnjs.cloudflare.com')) {
        // Don't intercept requests outside our scope
        return;
    }
    
    // For reloads, don't intercept at all - let browser handle it normally
    if (request.cache === 'reload' || request.mode === 'reload') {
        console.log('Service Worker: Reload detected, not intercepting');
        return; // Don't call event.respondWith(), let browser handle it
    }
    
    // Handle different types of requests with appropriate strategies
    if (isStaticAsset(request)) {
        event.respondWith(cacheFirst(request));
    } else if (isAPIRequest(request)) {
        event.respondWith(handleAPIRequest(request));
    } else if (isNavigationRequest(request)) {
        event.respondWith(handleNavigationRequest(request));
    } else {
        // For other requests, try network first
        event.respondWith(networkFirst(request));
    }
});

// =============================================================================
// CACHING STRATEGIES
// =============================================================================

/**
 * Cache First Strategy
 * Try cache first, fallback to network, then cache the response
 */
async function cacheFirst(request) {
    try {
        const cachedResponse = await caches.match(request);
        if (cachedResponse) {
            return cachedResponse;
        }
        
        const networkResponse = await fetch(request);
        if (networkResponse.ok) {
            const cache = await caches.open(CACHE_NAME);
            cache.put(request, networkResponse.clone());
        }
        
        return networkResponse;
    } catch (error) {
        console.error('Cache First Strategy failed:', error);
        return new Response('Network error', { status: 503 });
    }
}

/**
 * Network First Strategy
 * Try network first, fallback to cache
 */
async function networkFirst(request) {
    try {
        const networkResponse = await fetch(request);
        
        if (networkResponse.ok && request.method === 'GET') {
            const cache = await caches.open(CACHE_NAME);
            cache.put(request, networkResponse.clone());
        }
        
        return networkResponse;
    } catch (error) {
        console.log('Network failed, trying cache:', error);
        const cachedResponse = await caches.match(request);
        
        if (cachedResponse) {
            return cachedResponse;
        }
        
        return new Response('Offline', { status: 503 });
    }
}

/**
 * Stale While Revalidate Strategy
 * Return cached version immediately, update cache in background
 */
async function staleWhileRevalidate(request) {
    const cache = await caches.open(API_CACHE_NAME);
    const cachedResponse = await cache.match(request);
    
    const fetchPromise = fetch(request).then((networkResponse) => {
        if (networkResponse.ok) {
            cache.put(request, networkResponse.clone());
        }
        return networkResponse;
    });
    
    return cachedResponse || fetchPromise;
}

// =============================================================================
// REQUEST HANDLERS
// =============================================================================

/**
 * Handle API requests with appropriate caching strategy
 */
async function handleAPIRequest(request) {
    const url = request.url;
    
    // Network first for dynamic data
    if (NETWORK_FIRST_ENDPOINTS.some(endpoint => url.includes(endpoint))) {
        return networkFirst(request);
    }
    
    // Cache first for relatively static data
    if (CACHE_FIRST_ENDPOINTS.some(endpoint => url.includes(endpoint))) {
        return staleWhileRevalidate(request);
    }
    
    // Handle write operations (POST, PUT, DELETE)
    if (request.method !== 'GET') {
        try {
            const response = await fetch(request);
            
            // If write operation succeeds, invalidate related caches
            if (response.ok) {
                await invalidateRelatedCaches(request);
            }
            
            return response;
        } catch (error) {
            // Store write operation for when network is available
            await storeFailedRequest(request);
            return new Response(
                JSON.stringify({ 
                    status: 'error', 
                    message: 'Operation queued for when online',
                    offline: true 
                }),
                { 
                    status: 202,
                    headers: { 'Content-Type': 'application/json' }
                }
            );
        }
    }
    
    // Default to stale while revalidate for other API requests
    return staleWhileRevalidate(request);
}

/**
 * Handle navigation requests (page loads)
 */
async function handleNavigationRequest(request) {
    // For normal navigation, try network first
    try {
        const networkResponse = await fetch(request);
        return networkResponse;
    } catch (error) {
        // Return cached index.html for offline navigation
        const cachedResponse = await caches.match('./index.php') || 
                              await caches.match('./');
        return cachedResponse || new Response('Offline', { status: 503 });
    }
}

// =============================================================================
// UTILITY FUNCTIONS
// =============================================================================

/**
 * Check if request is for a static asset
 */
function isStaticAsset(request) {
    const url = new URL(request.url);
    return url.pathname.match(/\.(css|js|png|jpg|jpeg|gif|svg|ico|webp|woff|woff2|ttf|eot)$/) ||
           STATIC_ASSETS.some(asset => request.url.includes(asset));
}

/**
 * Check if request is for an API endpoint
 */
function isAPIRequest(request) {
    return request.url.includes('/bloodweb/app/tasklist/v2/api/');
}

/**
 * Check if request is a navigation request
 */
function isNavigationRequest(request) {
    return request.mode === 'navigate' || 
           (request.method === 'GET' && request.headers.get('accept').includes('text/html'));
}

/**
 * Invalidate related caches after write operations
 */
async function invalidateRelatedCaches(request) {
    const cache = await caches.open(API_CACHE_NAME);
    const url = new URL(request.url);
    
    // Define which caches to invalidate based on the operation
    const invalidationMap = {
        'tasks': ['tasks', 'analytics'],
        'projects': ['projects', 'tasks', 'analytics'],
        'categories': ['categories', 'tasks'],
        'xp': ['xp', 'analytics']
    };
    
    // Determine what was modified based on URL or body
    let modifiedResource = 'tasks'; // default
    for (const [resource, _] of Object.entries(invalidationMap)) {
        if (url.searchParams.get('endpoint') === resource) {
            modifiedResource = resource;
            break;
        }
    }
    
    // Invalidate related caches
    const toInvalidate = invalidationMap[modifiedResource] || [modifiedResource];
    
    for (const resource of toInvalidate) {
        const keys = await cache.keys();
        for (const key of keys) {
            if (key.url.includes(`endpoint=${resource}`)) {
                await cache.delete(key);
                console.log(`Invalidated cache for: ${key.url}`);
            }
        }
    }
}

/**
 * Store failed requests for retry when online
 */
async function storeFailedRequest(request) {
    try {
        const requestData = {
            url: request.url,
            method: request.method,
            headers: Object.fromEntries(request.headers.entries()),
            body: request.method !== 'GET' ? await request.text() : null,
            timestamp: Date.now()
        };
        
        // Store in IndexedDB or localStorage for retry later
        const stored = localStorage.getItem('failed_requests') || '[]';
        const failedRequests = JSON.parse(stored);
        failedRequests.push(requestData);
        localStorage.setItem('failed_requests', JSON.stringify(failedRequests));
        
        console.log('Stored failed request for retry:', requestData);
    } catch (error) {
        console.error('Failed to store failed request:', error);
    }
}

// =============================================================================
// BACKGROUND SYNC (when available)
// =============================================================================

self.addEventListener('sync', (event) => {
    if (event.tag === 'background-sync') {
        event.waitUntil(retryFailedRequests());
    }
});

/**
 * Retry failed requests when back online
 */
async function retryFailedRequests() {
    try {
        const stored = localStorage.getItem('failed_requests');
        if (!stored) return;
        
        const failedRequests = JSON.parse(stored);
        const successful = [];
        
        for (let i = 0; i < failedRequests.length; i++) {
            const requestData = failedRequests[i];
            
            try {
                const request = new Request(requestData.url, {
                    method: requestData.method,
                    headers: requestData.headers,
                    body: requestData.body
                });
                
                const response = await fetch(request);
                
                if (response.ok) {
                    successful.push(i);
                    console.log('Successfully retried request:', requestData.url);
                    
                    // Notify the client about successful retry
                    self.clients.matchAll().then(clients => {
                        clients.forEach(client => {
                            client.postMessage({
                                type: 'sync-success',
                                data: requestData
                            });
                        });
                    });
                }
            } catch (error) {
                console.log('Failed to retry request:', requestData.url, error);
            }
        }
        
        // Remove successful requests from storage
        const remaining = failedRequests.filter((_, index) => !successful.includes(index));
        localStorage.setItem('failed_requests', JSON.stringify(remaining));
        
    } catch (error) {
        console.error('Error retrying failed requests:', error);
    }
}

// =============================================================================
// MESSAGE HANDLING
// =============================================================================

self.addEventListener('message', (event) => {
    const { data } = event;
    
    switch (data.type) {
        case 'skip-waiting':
            self.skipWaiting();
            break;
            
        case 'get-version':
            event.ports[0].postMessage({ version: CACHE_NAME });
            break;
            
        case 'clear-cache':
            caches.delete(CACHE_NAME).then(() => {
                event.ports[0].postMessage({ success: true });
            });
            break;
            
        case 'force-update':
            self.registration.update();
            break;
    }
});

// =============================================================================
// PUSH NOTIFICATIONS (future enhancement)
// =============================================================================

self.addEventListener('push', (event) => {
    if (!event.data) return;
    
    const data = event.data.json();
    const options = {
        body: data.body,
        icon: '/i/BloodWeb/logo192.png',
        badge: '/i/BloodWeb/badge.png',
        tag: data.tag || 'default',
        requireInteraction: true,
        actions: [
            {
                action: 'view',
                title: 'View',
                icon: '/i/BloodWeb/action-view.png'
            },
            {
                action: 'dismiss',
                title: 'Dismiss',
                icon: '/i/BloodWeb/action-dismiss.png'
            }
        ]
    };
    
    event.waitUntil(
        self.registration.showNotification(data.title || 'TaskList v2', options)
    );
});

self.addEventListener('notificationclick', (event) => {
    event.notification.close();
    
    if (event.action === 'view') {
        event.waitUntil(
            clients.openWindow('/bloodweb/app/tasklist/v2/frontend/')
        );
    }
});

console.log('TaskList v2 Service Worker loaded successfully');