
import { App } from './index.js';
import { HTTP_modify } from './http.js'; // Assuming we'll put these network functions in a http.js module
import { generateSuccessMessage,getFormTitle,removeActiveElementFocus, captiliseFirstLetter } from './helpers.js';
import { renderCategoryBindings,renderCategoriesView, getCategory } from './categories.js';
import { renderProjectsView, getProject } from './projects.js';
import { renderTasksView,deleteTasks,toggleTaskEditMode } from './tasks.js'; // Assuming we'll put these network functions in a http.js module



// =========================================================================
//    OPEN / CLOSE / MODIFY EXISTING ITEMS
// =========================================================================

export function cleanState() {
    App.state.editMode = {
        task: false,
        project: false,
        category: false
    };
    
}

// =========================================================================
//  Deprecated notification system, replaced with bwSuccess
//  kept here for reference in case we want to re-implement it later
// =========================================================================
// export function showNotification(message) {    
//     toggleDialog(dQ('#notification'), true); // Open the notification dialog
//     App.state.notificationIsOpen = true;
//     const notification = document.getElementById('notification');
//     const notificationMessage = document.getElementById('notificationMessage');
//     notificationMessage.innerHTML = message;
//     notification.showModal();
// }



export function toggleDisableAED(disabled=true, Button) {

    if (!Button) { debug('button not found', { importance: 1 }); }
    //disable the button
    App.state.readyToDelete = !disabled;
    if (disabled) {
        Button.setAttribute('aria-disabled', 'true');
        Button.setAttribute('disabled', 'true');
    }
    else {
        Button.removeAttribute('aria-disabled');
        Button.removeAttribute('disabled');
    }
    //toggle the class
    Button.classList.toggle('disabled', disabled);
}

export function toggleSideMenu(isOpen = null) {
    const menu = App.element.projectSideMenu;
    if (!menu) {
        debug('App.element.projectSideMenu not found', { importance: 1 });
        return;
    }

    const currentlyOpen = menu.style.width !== '0px' && menu.style.width !== '';
    const willOpen = isOpen !== null ? isOpen : !currentlyOpen;

    if (willOpen) {
        menu.style.width = `${App.settings.sideMenuWidth}px`;
        document.body.classList.add('menu-open');
    }
    else {
        menu.style.width = '0px';
        document.body.classList.remove('menu-open');
    }

    App.settings.sideMenuIsOpen = willOpen;
    localStorage.setItem('projectSideMenuOpen', willOpen);
}



export function toggleDialog(dialog, forceOpen = false, displayStyle='flex') {

    if (!dialog) {debug(`Dialog '${dialog}' not found`, { importance: 1 }); }

    if (forceOpen === true || dialog.style.display === 'none' ) {
            console.log('Opening dialog: because: forceOpen?',forceOpen, 'or dialog style is none:', dialog.style.display === 'none');
        dialog.showModal(); dialog.style.display = displayStyle;
    }
    else {
        dialog.close(); dialog.style.display = 'none';
    }
}

export function closeMainForm() {
    toggleDialog(App.element.MainForm.form, false);
}

export function openMainForm(method = App.state.activeView.toUpperCase()) {
    const form = App.element.MainForm;
    if (!form) { debug('MainForm not found in App.element', { importance: 1, category: 'ui' }); }

    // Set method and clear hidden ID field
    form.method.value = method;
    form.id.value = '';
    form.title.innerHTML = getFormTitle(method);
    
    // Always hide all optional sections first
    form.allOptionalFields.forEach(wrapper => wrapper.style.display = 'none');
    
    const methodUpper = method.toUpperCase();
     switch (true) {
         case methodUpper.includes('TASK'):
             form.taskComponents.wrapper.style.display = '';
             break;
         case methodUpper.includes('CATEGORY')||methodUpper.includes('CATEGORIES'):
             form.categoryComponents.wrapper.style.display = '';
             break;
         case methodUpper.includes('PROJECT'):
            form.projectComponents.wrapper.style.display = '';
            break;
                
         // Break on invalid method
         default: debug(`Unknown form method: ${method}`, { importance: 1, category: 'ui' });
     }
    toggleDialog(App.element.MainForm.form, true);
}


export function setExpandedItem(item=false,Name,view=App.state.activeView) {
        
    const noun = App.settings.singularViews[App.settings.possibleViews.indexOf(view)];
    let stateToSet = null;
    switch (noun) {
        case 'task': stateToSet = App.state.task.expandedTask; break;
        case 'category': stateToSet = App.state.category.expandedCategory; break;
        case 'project': stateToSet = App.state.project.expandedProject; break;
        default: debug(`Unknown noun: ${noun}`, { importance: 1, category: 'ui' });
    }
    // always remove expanded class from all items
    document.querySelectorAll('.expansion-list-item.expanded').forEach(el => el.classList.remove('expanded'));
    
    console.log(`Setting expanded item for ${noun}:`, { item, Name, stateToSet });

    if (item === false) {
        stateToSet = null;
        localStorage.removeItem(`expanded${captiliseFirstLetter(noun)}`);
        return;
    }
    item.classList.add('expanded');
    stateToSet = Name;
    localStorage.setItem(`expanded${captiliseFirstLetter(noun)}`, Name);
}

 
  
// =========================================================================
//    FORM SUBMISSION ( or buttons that open the form )
// =========================================================================

// export function handleMainInputClick (method,view=App.state.activeView) {
//     removeActiveElementFocus(); // Remove focus from any active element
//     event.preventDefault(); // Prevent default button behavior
//     const form = App.element.MainForm;

//     if (!method) { debug('No method found for MainForm submission', { importance: 1, category: 'ui' }); return; }
//     switch (view.toLocaleUpperCase()) {
//         case 'TASKS':
//             openMainForm(`${method.toLocaleUpperCase()}`)
//             switch (method.toUpperCase()) {

//                 case 'ADD':
//             openMainForm('ADD_TASKS');
//             form.text.value = '';
//             }
//         }
//         }

export function handleAED(event) {
    removeActiveElementFocus(); // Remove focus from any active element
    event.preventDefault(); // Prevent default button behavior
    const form = App.element.MainForm;
    
    const method = event.target.getAttribute('data-action')? event.target.getAttribute('data-action') : event.target.parentElement.getAttribute('data-action');
          
    if (!method) { debug('No method found for MainForm submission', { importance: 1, category: 'ui' }); return; }

    const view = App.state.activeView.toLowerCase();

    const  handlers = {
        // Add handlers for each view
        add_tasks: handleAddTaskClick,
        add_projects: handleAddProjectClick,
        add_category: handleAddCategoryClick,
        add_categories: handleAddCategoryClick,
        // Edit handlers
        edit_tasks: () => { triggerEditMode(App.state.task.expandedTask); },
        edit_projects: () => { debug('Edit button clicked, button not yet implemented', {
            importance: 1, category: 'ui' }); },
        edit_categories: () => { debug('Edit button clicked, button not yet implemented', {
            importance: 1, category: 'ui' }); },
        // Delete handlers
        delete_tasks: deleteTasks,
        delete_projects: () => { debug('Delete button clicked, button not yet implemented', {
            importance: 1, category: 'ui' }); },
        delete_categories: () => { debug('Delete button clicked, button not yet implemented', {
            importance: 1, category: 'ui' }); },

        // ABNORMAL HANDLERS (edit-save, revert, cancel)
        // These only usable after modify AES has been called
        // edit_save_tasks: () => { modifyAES('edit', 'save'); },
        
        // EDIT FINALIZING HANDLERS
        'edit-save_tasks': () => { console.log('Edit save button clicked, button not yet implemented'); },
        'edit-save_projects': () => { debug('Edit save button clicked, button not yet implemented', { importance: 1, category: 'ui' }); },
        'edit-save_categories': () => { debug('Edit save button clicked, button not yet implemented', { importance: 1, category: 'ui' }); },


        'cancel-edit_tasks': () => { triggerEditMode(); },
    };


    
    const AED_to_handle = `${method}_${view}`;

    // Check if the method and view combination has a handler
    if (handlers[AED_to_handle]) {
        handlers[AED_to_handle]();
    } else {
    debug(`AED function Unknown method: ${AED_to_handle}`, { importance: 1, category: 'ui' });
    }
}


export function handleAddTaskClick() {
    const form = App.element.MainForm;

    openMainForm('ADD_TASKS');
    

    form.text.value = '';
    form.taskComponents.daysToComplete.value = App.settings.task.default_days;

    if (App.state.category.currentCategoryID) { form.taskComponents.categorySelect.value = App.state.category.currentCategoryID;}
}

export function handleAddCategoryClick() {
    const form = App.element.MainForm;

    openMainForm('ADD_CATEGORIES');

    form.text.value = '';
    form.categoryComponents.emoji.value = '';
    form.categoryComponents.color.value = '#000000';
    form.categoryComponents.bindList.innerHTML = '';
    form.categoryComponents.unbindList.innerHTML = '';

    renderCategoryBindings();
}

export function handleAddProjectClick() {
    const form = App.element.MainForm;  
    openMainForm('ADD_PROJECTS');
    form.text.value = '';
}

function handleDeleteCategoryClick() {
  const selected = App.state.categorySettings.selectedCategory;
  if (!selected) return bwSuccess({heading:'Error!',body:'No category selected.'});

  if (!confirm(`Are you sure you want to delete category "${selected}"?`)) return;

  HTTP_modify('DELETE_CATEGORIES', { ids: [selected] }, `Category "${selected}" deleted.`)
    .then(() => {
      App.state.categorySettings.selectedCategory = null;
      rehydrateApp();
    });
}


export function handleMainFormSubmit(event) {
    event.preventDefault(); // Prevent page reload

    const form = App.element.MainForm;

    const method = form.method.value;
    const text = form.text.value.trim();

    if (!method || !text) { debug('Method or Text missing in form submission', { importance: 1, category: 'data' }); }

    let data = {};
    const upperMethod = method.toUpperCase();

    switch (true) {
        case upperMethod.includes('TASK'):
            data = {
                name: text,
                description: form.taskComponents.description.value || '',
                category: form.taskComponents.categorySelect.value || App.state.category.currentCategoryID,
                project: App.state.project.currentProjectID,
                days_to_complete: form.taskComponents.daysToComplete.value || null,
                upload_date: new Date().toISOString().split('T')[0] // Format YYYY-MM-DD
            };
            break;

        case upperMethod.includes('CATEGOR'):
            const bindlist = Array.from(App.element.MainForm.categoryComponents.bindList.children)
                .map(el => el.getAttribute('data-project'))
                .filter(Boolean)
                .join(',');

            data = {
                name: text,
                emoji: form.categoryComponents.emoji.value || '📚',
                color: form.categoryComponents.color.value || null,
                projectIDs: bindlist,
            };
            console.log('Category data:', data);
            break;

        case upperMethod.includes('PROJECT'):
            data = {
                name: text
            };
            break;

        default:
            debug(`Unknown method submitted: ${method}`, { importance: 3, category: 'data' });
            return;
    }

    // Include ID for edit operations (if present)
    if (form.id.value) { data.id = form.id.value; }

    data.method = method;

    const splitMethod = method.toLowerCase().split('_');
    const successMessage = generateSuccessMessage(splitMethod[0], splitMethod[1], data.name);
 

    HTTP_modify(method, data, successMessage)
        .then(() => { // Rehydrate app with affected entities
            bwSuccess({heading:'Notification:',body:successMessage}); // Show success message
            closeMainForm(); /// Close the form
        })
        .catch(err => {
            bwSuccess({heading:'Error!',body:`Error: ${err.message}`});
        });
}


// =========================================================================
//    RENDERING VIEWS / MAKING ELEMENTS
// =========================================================================



export function renderView(view,moveIndicator = true) {

    cleanState(); // Reset state for the new view
    App.element.ViewContent.innerHTML = ''; // Clear current view

    function updateView(newView){
        App.state.activeView = newView;
        localStorage.setItem('activeView', newView);
        App.element.viewtabs.forEach(tab => {
            tab.classList.remove('active');
            if (tab.getAttribute('data-view') === newView) {
                tab.classList.add('active');
                if (moveIndicator) { moveIndicatorTo(tab); }
            }
        });
    }

    debug(`Rendering view: ${view}`, { importance: 2, category: 'ui' });
    updateView(view); // update the view in storage and move the indicator

    switch (view) {
      case 'tasks':
        renderTasksView();
        break;
      case 'projects':
        renderProjectsView();
        break;
      case 'categories':
        renderCategoriesView();
        break;
    }

    // toggleDisableAED(true); // reset delete button to disabled
}

  export function moveIndicatorTo(button) {
    const indicator = document.querySelector('.tab-indicator');
    const rect = button.getBoundingClientRect();
    const parentRect = button.parentElement.getBoundingClientRect();
    const offsetX = rect.left - parentRect.left;
    indicator.style.transform = `translateX(${offsetX}px) scaleY(1.2)`;
    setTimeout(() => {
    indicator.style.transform = `translateX(${offsetX}px) scaleY(1)`;
    }, 150);

    indicator.style.width = `${rect.width}px`;
  }
  

export function makeMainInputs(locationToAppend = null, type = App.state.activeView ) {
    if (!['tasks', 'projects', 'categories'].includes(type)) { debug(`Invalid type '${type}' for makeMainInputs`, { importance: 1, category: 'ui' }); }

    const doesButtonContainerExist = dQ('.button-container') ?? false;
    if (doesButtonContainerExist) { dQ('.button-container').remove();}
    
    const addButton = createElement('button', { id: 'Aed-Button', 'data-action': 'add', className: 'AED-Button', innerHTML: '<i class="fas fa-plus"></i>' });
        addButton.setAttribute('data-action', 'add'); 
    const editButton = createElement('button', { id: 'aEd-Button', 'data-action': 'edit', className: 'AED-Button', innerHTML: '<i class="fas fa-pen"></i>' });
        editButton.setAttribute('data-action', 'edit');
    const deleteButton = createElement('button', {id: 'aeD-Button', 'data-action': 'delete', className: 'AED-Button disabled', innerHTML: '<i class="fas fa-trash"></i>',disabled: true, 'aria-disabled': true });
        deleteButton.setAttribute('data-action', 'delete');
    const AEDbuttons= [addButton, editButton, deleteButton]; // Loop through these buttons to add event listeners later

    const activeView = App.state.activeView;
    console.warn('Setting up buttons for view:', activeView);
    
    // Call handleAED for each button
    AEDbuttons.forEach(button => { button.addEventListener('click', event => { return handleAED(event); }); });
    
    const buttonContainer = createElement('div', { className: 'button-container' });
    buttonContainer.append( addButton, editButton, deleteButton);
    // Append the button container to the specified location
    // If no location is specified, append to TaskView after h2
    if (!locationToAppend) { return dQ(App.element.TaskView).insertBefore(buttonContainer,dQ('h2').nextSibling); }

    locationToAppend.appendChild(buttonContainer);
   
}


export function triggerEditMode(mode='tasks') {
  const view = App.state.activeView;

    if (view === 'projects') {
        
        console.log('Triggering edit mode for projects');
        const expanded = document.querySelector('.expansion-list-item.expanded');
        if (!expanded) debug('No expanded project found', { importance: 1, category: 'ui' });

        const nameSpan = expanded.querySelector('.expansion-title');
        const nameInput = expanded.querySelector('.edit-project-input');

        if (!nameSpan || !nameInput) debug('Name elements not found in expanded project', { importance: 1, category: 'ui' });

        const isActive = App.state.project.expandedProject !== null;
        if (!isActive) {
        nameSpan.style.display = 'none';
        nameInput.style.display = 'inline-block';
        nameInput.focus();
        App.state.editMode.projects = expanded.dataset.id;
        highlightEditButton(true);
        }
        else {
        debug('Exiting edit mode for projects', { importance: 1, category: 'ui' });
        }
    }

    if (view === 'categories') {
        const expanded = document.querySelector('.expansion-list-item.expanded');
        if (!expanded) return;

        const nameSpan = expanded.querySelector('.expansion-title');
        const nameInput = expanded.querySelector('.edit-category-name');
        const emojiSpan = expanded.querySelector('.category-emoji');
        const emojiInput = expanded.querySelector('.edit-category-emoji');

        nameSpan.style.display = 'none';
        emojiSpan.style.display = 'none';
        nameInput.style.display = 'inline-block';
        emojiInput.style.display = 'inline-block';

        nameInput.focus();
        App.state.editMode.categories = expanded.dataset.id;
        highlightEditButton(true);
    }


    if (view === 'tasks') {
        toggleTaskEditMode();
    }
}

export function modifyAEDButtons(editing = false) {
    const container = document.querySelector('.button-container');
    if (!container) return debug('Button container not found', { importance: 1 });

    // Clear current buttons
    container.innerHTML = '';

    if (editing) {
        // Create Save button
        const saveBtn = createElement('button', { id: 'Task-Save-Button', className: 'AED-Button', innerHTML: '<i class="fas fa-save"></i>' });
        saveBtn.setAttribute('data-action', 'edit-save');
        // Create Cancel button
        const cancelBtn = createElement('button', { id: 'Task-Cancel-Button', className: 'AED-Button', innerHTML: '<i class="fas fa-times"></i>'});
        cancelBtn.setAttribute('data-action', 'cancel-edit');

        // Hook into handleAED
        [saveBtn, cancelBtn].forEach(btn => {
            btn.addEventListener('click', event => handleAED(event));
        });

        container.append(cancelBtn, saveBtn);
        toggleDisableAED(true,saveBtn); // Save is disabled by default
    } else {
        // Re-render normal AED buttons
        makeMainInputs();
    }
}

export function blockAESConfirm(AES) {
    

}

export function highlightEditButton(isActive) {
  const editBtn = dQ(App.element.taskActionButtons.edit);
  if (!editBtn) debug('Edit button not found in App.element.taskActionButtons', { importance: 1, category: 'ui' });
  editBtn.classList.toggle('edit-mode-active', isActive);
}

export function createNotificationDialog() {
    const dialog = createElement('dialog', { id: 'notification', className: 'notification',style:'display:none;' });
    const notificationMessage = createElement('p', { id: 'notificationMessage' });
    
    // Make the close button and add event listeners
    const closeButton = createElement('button', { id: 'closenotification',type:'button', className: 'close-button', innerHTML: '&times;' });
    closeButton.addEventListener('click', () => { toggleDialog(dialog,false); });
    closeButton.addEventListener('keydown', (event) => {if (event.key === 'Enter' || event.key === ' ') {toggleDialog(dialog,false);}});

    dialog.appendChild(notificationMessage);
    dialog.appendChild(closeButton);
    document.body.appendChild(dialog);    
}
