#!/usr/bin/env php
<?php
/**
 * TaskList v2 API Test Script
 * 
 * This script tests the new API endpoints to ensure they're working correctly.
 */

// Configuration
$BASE_URL = 'https://bloodweb.net/app/tasklist/v2/api';
$API_KEY = 'test=1'; // Use test mode parameter

// Colors for output
define('GREEN', "\033[32m");
define('RED', "\033[31m");
define('YELLOW', "\033[33m");
define('BLUE', "\033[34m");
define('RESET', "\033[0m");

class APITester {
    private $baseUrl;
    private $apiKey;
    private $testResults = [];
    
    public function __construct($baseUrl, $apiKey) {
        $this->baseUrl = rtrim($baseUrl, '/');
        $this->apiKey = $apiKey;
    }
    
    private function log($message, $color = RESET) {
        echo $color . "[" . date('H:i:s') . "] " . $message . RESET . "\n";
    }
    
    private function success($message) {
        $this->log("✓ " . $message, GREEN);
    }
    
    private function error($message) {
        $this->log("✗ " . $message, RED);
    }
    
    private function info($message) {
        $this->log("ℹ " . $message, BLUE);
    }
    
    private function warning($message) {
        $this->log("⚠ " . $message, YELLOW);
    }
    
    private function makeRequest($endpoint, $method = 'GET', $data = null) {
        $url = $this->baseUrl . '/' . ltrim($endpoint, '/');
        
        // Add test parameter to URL for testing mode
        if ($this->apiKey === 'test=1') {
            $separator = strpos($url, '?') !== false ? '&' : '?';
            $url .= $separator . $this->apiKey;
        }
        
        $context = [
            'http' => [
                'method' => $method,
                'header' => [
                    'Content-Type: application/json'
                ],
                'ignore_errors' => true
            ]
        ];
        
        // Only add API key header if it's not the test parameter
        if ($this->apiKey !== 'test=1') {
            $context['http']['header'][] = 'X-API-Key: ' . $this->apiKey;
        }
        
        if ($data !== null) {
            $context['http']['content'] = json_encode($data);
        }
        
        $context = stream_context_create($context);
        $response = file_get_contents($url, false, $context);
        
        // Get response headers
        $responseHeaders = $http_response_header ?? [];
        $statusLine = $responseHeaders[0] ?? 'HTTP/1.1 500 Internal Server Error';
        preg_match('/HTTP\/\d\.\d (\d{3})/', $statusLine, $matches);
        $statusCode = (int)($matches[1] ?? 500);
        
        return [
            'status_code' => $statusCode,
            'body' => $response,
            'data' => json_decode($response, true)
        ];
    }
    
    private function testEndpoint($name, $endpoint, $method = 'GET', $data = null, $expectedStatus = 200) {
        $this->info("Testing: $name");
        
        try {
            $response = $this->makeRequest($endpoint, $method, $data);
            
            $passed = $response['status_code'] === $expectedStatus;
            
            if ($passed) {
                $this->success("$name - Status: {$response['status_code']}");
                if ($response['data'] && isset($response['data']['success'])) {
                    if ($response['data']['success']) {
                        $this->success("Response indicates success");
                    } else {
                        $this->warning("Response indicates failure: " . ($response['data']['error']['message'] ?? 'Unknown error'));
                    }
                }
            } else {
                $this->error("$name - Expected: $expectedStatus, Got: {$response['status_code']}");
                if ($response['data'] && isset($response['data']['error'])) {
                    $this->error("Error: " . $response['data']['error']['message']);
                }
            }
            
            $this->testResults[] = [
                'name' => $name,
                'passed' => $passed,
                'status_code' => $response['status_code'],
                'expected_status' => $expectedStatus
            ];
            
            return $response;
            
        } catch (Exception $e) {
            $this->error("$name - Exception: " . $e->getMessage());
            $this->testResults[] = [
                'name' => $name,
                'passed' => false,
                'error' => $e->getMessage()
            ];
            return null;
        }
    }
    
    public function runTests() {
        $this->log("Starting TaskList v2 API Tests", BLUE);
        $this->log("==============================", BLUE);
        
        // Test 1: API Root
        $this->testEndpoint('API Root', '/', 'GET', null, 200);
        
        // Test 2: Get Projects
        $this->testEndpoint('Get Projects', '/projects', 'GET', null, 200);
        
        // Test 3: Get Categories
        $this->testEndpoint('Get Categories', '/categories', 'GET', null, 200);
        
        // Test 4: Get Tasks
        $this->testEndpoint('Get Tasks', '/tasks', 'GET', null, 200);
        
        // Test 5: Create Project
        $projectData = [
            'name' => 'Test Project API',
            'description' => 'Created via API test',
            'color' => '#FF5733'
        ];
        $projectResponse = $this->testEndpoint('Create Project', '/projects', 'POST', $projectData, 201);
        
        $createdProjectId = null;
        if ($projectResponse && $projectResponse['data'] && isset($projectResponse['data']['data']['id'])) {
            $createdProjectId = $projectResponse['data']['data']['id'];
            $this->success("Created project with ID: $createdProjectId");
        }
        
        // Test 6: Create Category
        $categoryData = [
            'name' => 'Test Category API',
            'emoji' => '🧪',
            'color' => '#9B59B6'
        ];
        $categoryResponse = $this->testEndpoint('Create Category', '/categories', 'POST', $categoryData, 201);
        
        $createdCategoryId = null;
        if ($categoryResponse && $categoryResponse['data'] && isset($categoryResponse['data']['data']['id'])) {
            $createdCategoryId = $categoryResponse['data']['data']['id'];
            $this->success("Created category with ID: $createdCategoryId");
        }
        
        // Test 7: Create Task (if we have project and category)
        if ($createdProjectId && $createdCategoryId) {
            $taskData = [
                'title' => 'Test Task API',
                'description' => 'Created via API test',
                'project_id' => $createdProjectId,
                'category_id' => $createdCategoryId,
                'priority' => 'high',
                'due_date' => date('Y-m-d', strtotime('+7 days'))
            ];
            $taskResponse = $this->testEndpoint('Create Task', '/tasks', 'POST', $taskData, 201);
            
            $createdTaskId = null;
            if ($taskResponse && $taskResponse['data'] && isset($taskResponse['data']['data']['id'])) {
                $createdTaskId = $taskResponse['data']['data']['id'];
                $this->success("Created task with ID: $createdTaskId");
                
                // Test 8: Mark Task as In Progress
                $this->testEndpoint('Start Task', "/tasks/$createdTaskId/start", 'POST', null, 200);
                
                // Test 9: Mark Task as Complete
                $this->testEndpoint('Complete Task', "/tasks/$createdTaskId/complete", 'POST', null, 200);
            }
        }
        
        // Test 10: Get XP Information
        $this->testEndpoint('Get XP Info', '/xp', 'GET', null, 200);
        
        // Test 11: Get Analytics
        $this->testEndpoint('Get Analytics', '/analytics/overview', 'GET', null, 200);
        
        // Test 12: Search
        $this->testEndpoint('Search', '/search?q=test&type=all', 'GET', null, 200);
        
        // Test 13: Invalid Endpoint (should return 404)
        $this->testEndpoint('Invalid Endpoint', '/invalid', 'GET', null, 404);
        
        // Test 14: Missing Authentication (simulate by using invalid key)
        $originalKey = $this->apiKey;
        $this->apiKey = 'invalid_key';
        $this->testEndpoint('Invalid Auth', '/tasks', 'GET', null, 401);
        $this->apiKey = $originalKey;
        
        $this->printSummary();
    }
    
    private function printSummary() {
        $this->log("", RESET);
        $this->log("Test Summary", BLUE);
        $this->log("============", BLUE);
        
        $passed = 0;
        $failed = 0;
        
        foreach ($this->testResults as $result) {
            if ($result['passed']) {
                $passed++;
                $this->success($result['name']);
            } else {
                $failed++;
                $this->error($result['name'] . (isset($result['error']) ? " - " . $result['error'] : ""));
            }
        }
        
        $total = $passed + $failed;
        $this->log("", RESET);
        $this->log("Total Tests: $total", BLUE);
        $this->log("Passed: $passed", GREEN);
        $this->log("Failed: $failed", $failed > 0 ? RED : GREEN);
        
        if ($failed === 0) {
            $this->success("All tests passed! 🎉");
        } else {
            $this->warning("Some tests failed. Check the logs above for details.");
        }
    }
}

// Run tests if script is executed directly
if (php_sapi_name() === 'cli') {
    $tester = new APITester($BASE_URL, $API_KEY);
    $tester->runTests();
} else {
    echo "<pre>";
    echo "TaskList v2 API Test Script\n";
    echo "===========================\n\n";
    echo "This script should be run from the command line:\n";
    echo "php test_api.php\n\n";
    echo "Or make it executable and run directly:\n";
    echo "chmod +x test_api.php\n";
    echo "./test_api.php\n";
    echo "</pre>";
}
?>