// =========================================================================
// FILE: tasks.js
// PURPOSE: Global application state and constants
// AUTHOR: Jack Ewers / bloodweb.net
// LAST UPDATED: 2025-05-07
// =========================================================================

import { App } from './index.js';
import { HTTP_modify } from './http.js';
import { makeMainInputs, toggleDisableAED, modifyAEDButtons } from './ui.js';
import { daysSince, generateSuccessMessage,getDaysTillDue,getPriority,removeActiveElementFocus,validateStateData } from './helpers.js';
import { getCategory, getCategoryNameById, setCurrentCategoryID } from './categories.js';
import { getProject, setCurrentProject } from './projects.js';


// =========================================================================
//  FUNCTIONS:
//  
//   Function Name - shorthand(fn)
//     : Desription (CRITICAL?: Failure stops program) 
//  
//      clearSelectedTasks - CST
//          : unselect tasks 
//      getTaskById - GTBID (CRITICAL)
//          : get the task object with a matching id
//      validateSelectedTasks - VST
//          : validate the selected tasks against the current project and category (strips any empty tasks from the selected tasks array)
//      selectTask - ST
//          : select or unselect a task by ID and update the UI accordingly
//      deleteTasks - DT
//          : delete selected tasks and update the UI
//      renderTasksView - RTV
//          : render the task list view, including project and category selectors
//      createTaskElement - CTE
//          : create a DOM element for a task item
//      addSelectOptionstoCategorySelect - ASOCS
//          : add options to the category select element based on the current project
//      addCategoriesToMainForm - ACTMF
//          : add categories to the main form's category select element based on the current project  

// =========================================================================



export function clearSelectedTasks() {
    App.state.task.selectedTasks = [];
    localStorage.removeItem('selectedTasks');
}

export function getTask(value,by){
  const Task = App.state.tasks.find(task => task[by] === value);
  if (!Task) { debug(`Task with ${by} '${value}' not found`, { importance: 1, category: 'data' }); }
  return Task;
}

const getTaskById = function(taskId){
    const task = App.state.tasks.find(task => task.id === taskId);
    if (!task) { debug(`Task with ID ${taskId} not found`, { importance: 1, category: 'data' }); }
    return task;
}

function selectTask(taskId) {
  if (App.state.editMode.task) {return debug('Cannot select task while in edit mode', { importance: 2, category: 'ui' });}
  const selected = App.state.task.selectedTasks;
  const index = selected.indexOf(taskId);

  if (index === -1) { selected.push(taskId); } 
  else { selected.splice(index, 1);}

  localStorage.setItem('selectedTasks', JSON.stringify(selected));

  // Update only the checkbox, no full render
  const item = dQ(`#task_${taskId}`);
  if (item) {
    const checkbox = item.querySelector('.task-checkbox');
    checkbox.checked = index === -1;
    item.classList.toggle('selected', index === -1);
    
    toggleDisableAED(selected.length < 1,dQ(App.element.taskActionButtons.delete)); // Enable delete button if selected tasks exist
  }
}



export async function deleteTasks() { // Delete a task by ID
    removeActiveElementFocus(); // Remove focus from any active element

    const selectedTasks = App.state.task.selectedTasks;
    if (selectedTasks.length === 0) return bwSuccess({heading :'Error!',body:'No tasks selected for deletion.'});
    let IdandTasks = selectedTasks.map(id => `•${id}: ${getTaskById(id).name}`).join(' <br> ');
    const SuccessMessage = generateSuccessMessage( 'delete', 'task',`${IdandTasks}`, selectedTasks.length > 1);
    const payload = { ids: selectedTasks };

      await HTTP_modify('DELETE_TASKS', payload, SuccessMessage).then(() => { 
        clearSelectedTasks();
        // bwSuccess({heading:'Notification:',body:SuccessMessage}); 
        renderTasksView();
    }); 
    //catch (error) { debug(`Failed to delete tasks: ${error}`, { importance: 1, category: 'data' }); }

}   

export function renderTasksView() {
  validateStateData('selectedTasks'); // Strip any empty tasks from the selected tasks array

  const container = App.element.ViewContent;
  container.innerHTML = '';

  const wrapper = createElement('div', { className: 'task-view' });

  const header = createElement('h2')//,{innerText:'Task List'});

  const controls = createElement('div', { className: 'task-filters' });

  // === Project Selector ===
  const projectSelectWrapper = createElement('div', { className: 'task-select-wrapper' });
  const projectSelectLabel = createElement('label', { for: 'task-project-select', innerText: 'Project: ', });
  const projectSelect = createElement('select', { id: 'task-project-select' });
  App.state.projects.forEach(p => {
    const opt = createElement('option', { value: p.id, innerText: p.name, });
    if (p.id == App.state.project.currentProjectID) opt.selected = true;
    projectSelect.appendChild(opt);
  });
  projectSelect.addEventListener('change', (e) => {
    debug(`Project changed to: ${e.target.value}`, { importance: 2, category: 'ui' });
    setCurrentProject(e.target.value, true);
    validateStateData('projects'); // Validate the current category against the new project
    renderTasksView();
  });
  projectSelectWrapper.append(projectSelectLabel, projectSelect);

  // === Category Selector ===
  const categorySelectWrapper = createElement('div', { className: 'task-select-wrapper' });
  const categorySelectLabel = createElement('label', { for: 'task-category-select', innerText: 'Category: ', });
  const categorySelect = createElement('select', { id: 'task-category-select' });
  categorySelect.addEventListener('change', (e) => {
    setCurrentCategoryID( e.target.value );
    renderTasksView();
  });
  populateSelectOptions({
    data: App.state.categories,
    selectElement: categorySelect,
    prependOptions: [{ value: '*', label: 'Show All' }],
    currentValue: App.state.category.currentCategoryID,
    groupByTaskPresence: true,
  });
  populateSelectOptions({
    data: App.state.categories,
    selectElement: App.element.MainForm.taskComponents.categorySelect,
    useOptGroups: false,
    optionValueKey: 'id',
    optionLabelKey: 'name',
  }); 
  categorySelectWrapper.append(categorySelectLabel, categorySelect);



  controls.append(projectSelectWrapper, categorySelectWrapper);
  header.append(controls);

  wrapper.append(header);
  makeMainInputs(header, 'tasks');
  // === Task List ===
  const list = createElement('ul', { className: 'task-list-view' });
  const tasks = App.state.tasks.filter(task => {
      // Return tasks with that match current project and category
      const matchesProject = 
        task.project == getProject(App.state.project.currentProjectID,'id') ||
        task.project == App.state.project.currentProjectID;

      const matchesCategory = ['*', task.category].includes(App.state.category.currentCategoryID);
      return matchesProject && matchesCategory;
  });

  if (tasks.length === 0) {
    list.appendChild(createElement('li', { className: 'empty-state', innerText: `No tasks found.`,}));
  } else {
    tasks.forEach(task => {
      list.appendChild(createTaskElement(task));
    });
  }


  wrapper.append(list);
  container.appendChild(wrapper);

  if (App.state.task.selectedTasks.length > 0) { toggleDisableAED(false,dQ(App.element.taskActionButtons.delete)); } // Enable delete button if any task is selected
}


function createTaskElement(task) {
  const isSelected = App.state.task.selectedTasks.includes(task.id);
  const isExpanded = App.state.task.expandedTask == task.id;

  const category = App.state.categories.find(cat => cat.id === task.category) 
  if (!category){ debug(`Category '${task.category}' not found for task ${task.id}`, { importance: 1, category: 'data' });}
  const color = category.color || '#000000'; // Default color if not found

  const item = createElement('li', { className: `task-item ${isExpanded ? '' : 'un'}expanded`
    , id: `task_${task.id}`, style: `border-color:${color};`, });

  // HIDDEN ID
  // This is used to store the task ID for later reference, such as when selecting or deleting tasks
  const ID = createElement('span', { className: 'TL_id', innerText: task.id, style: 'display:none;' });
  // VISIBLE ITEMS
  const unExpanded = createElement('div', { className: 'task-unexpanded-section', });
  
  const checkbox = createElement('input', { type: 'checkbox', className: 'task-checkbox', });
  const checkboxWrapper = createElement('div', { className: 'task-checkbox-wrapper' });
  checkboxWrapper.append(checkbox);

  const label = createElement('span', { className: 'task-label', innerText: task.name, });
  const labelWrapper = createElement('div', { className: 'task-label-wrapper' });
  labelWrapper.append(label);

  const TASKURGENCY = getPriority('task_days_left', task.days_left_to_complete) // Get the priority based on days left
  const daysLeftWrapper = createElement('div', { className: 'task-days-wrapper' });
  const daysLeft = createElement('span', { className:`task-days ${TASKURGENCY}-urgency`, innerHTML: `${task.days_left_to_complete}` });
  const daysText = createElement('p',{className:'task-days-text',innerText:'days'});
  daysLeftWrapper.append(daysLeft,daysText);

  // EXPANSION ITEMS
  const expandedTaskDetails= createElement( 'div', { className: 'expanded-tasks', });
  
  const descriptionWrapper = createElement('div', { className: 'task-description-wrapper' });
  const description = createElement('p', { className: 'task-description', innerText: task.description || 'No description provided.', });
  descriptionWrapper.append(description);
  
  const hr = createElement('hr', { className: 'task-separator' });

  const selectedProject = createElement( 'div', { innerHTML: `Project: <span class='task-project'>${getProject(task.project, 'id').name || 'No Project'}</span>`, });
  const selectedCategory = createElement('div', { innerHTML:`Category: <span style="color:${category.color};">&#9679;</span> <span class="task-category">${category.name}</span>`, });

  const metaContainer = createElement('div', { className: 'task-meta' });
  const uploadDate = createElement('p', { className: 'task-upload-date', innerText: `Created: ${new Date(task.upload_date).toLocaleDateString()} (${daysSince(task.upload_date)} days ago)`, });
  
  metaContainer.append(uploadDate);
  unExpanded.append(ID, checkboxWrapper, labelWrapper, daysLeftWrapper);
  expandedTaskDetails.append(descriptionWrapper, hr, selectedCategory,selectedProject,metaContainer);  

  item.append(unExpanded, expandedTaskDetails);
  
  checkboxWrapper.addEventListener('click', (e) => {selectTask(task.id); e.stopPropagation();}); // Prevent checkbox click from expanding the task
  item.addEventListener('click', (e) => { 
    validateAndTriggerTaskExpansion(e.target.closest('.task-item')); // Toggle task expansion on item click
    e.stopPropagation(); // Prevent the click from selecting the task
  });

  if (isSelected) {item.classList.add('selected'); checkbox.checked = isSelected;}
  return item;
}


export function validateAndTriggerTaskExpansion(element,newState=null) {
  if (App.state.editMode.task) return debug('Cannot expand task while in edit mode', { importance: 2, category: 'ui' });
  if (!element) return bwSuccess({heading:'Error!',body:'No task element provided for expansion.'});
  // Always collapse the currently expanded task
  const taskItem = element.closest('.task-item');
  const taskID = taskItem.id.split('_')[1];
  const isElementExpanded=taskItem.classList.contains('expanded');
  if (App.state.task.expandedTask.includes(taskID)) { 
    toggleTaskExpansion(taskItem, false); 
  }
  // if the current task is already expanded then the call was to collapse it, so return
  if (isElementExpanded) { return; }  

  toggleTaskExpansion(taskItem, newState??!isElementExpanded);
  console.log('Current expanded task:', App.state.task.expandedTask, 'Clicked task ID:', taskID); 
}

export function toggleTaskExpansion(element,open=!element.classList.contains('expanded')) {
  let expandedTasks = App.state.task.expandedTask;
  const taskId = element.id.split('_')[1];
  
  if (open){ 
    element.classList.replace('unexpanded','expanded')  // Add expanded class to the task item
    if (!expandedTasks.includes(taskId)) {
      expandedTasks.push(taskId);
    }
  }
  else { 
    element.classList.replace('expanded','unexpanded'); // Remove expanded class to the task item
    const index = expandedTasks.indexOf(taskId);
    if (index > -1) {
      expandedTasks.splice(index, 1);
    }
  }

  App.updateStateValue('task.expandedTask', 
   expandedTasks
  ); // Update the state with the expanded task ID or null
}


export function toggleTaskEditMode() {
  if (App.state.task.expandedTask === null){ bwSuccess({heading:'Error!',body:'Please expand task to enter edit mode'}); }

  const selected_tasks = document.querySelectorAll('.task-item.expanded');
  if (!selected_tasks) { debug('No editable tasks found is a task expanded', { importance: 1, category: 'ui' }); return; }
  
  const newMode = !App.state.editMode.task;

  selected_tasks.forEach(el => { 
    
    const taskId = el.querySelector('.TL_id');
    const TASK = getTask(taskId.innerText, 'id');
    
    // Replace the task text 
    const taskText = el.querySelector('.task-label');
    const task_input = createElement( `${newMode?'textarea':'span'}`, { type: 'text', className: taskText.className, });
    taskText.replaceWith(task_input);

    // Replace the days left 
    const daysLeft = el.querySelector('.task-days');
    const days_input = createElement( `${newMode?'input':'span'}`, { type: 'number', className: daysLeft.className , value: daysLeft.innerText, });
    daysLeft.replaceWith(days_input);
    
    // Replace the category text
    const categorySelect = el.querySelector('.task-category');
    const category_select = createElement( `${newMode?'select':'span'}`, {className: categorySelect.className, });
    categorySelect.replaceWith(category_select);

    const projectSelect = el.querySelector('.task-project');
    const project_select = createElement(`${newMode?'select':'span'}`, { className:projectSelect.className, });
    projectSelect.replaceWith(project_select);

    if (newMode){ // change to Edit mode 
      task_input.value = taskText.innerText;
      days_input.value = daysLeft.innerText.replace('d', ''); // Remove the 'd' suffix
      
      // Set up the category and project select elements 
      populateSelectOptions({ data:App.state.categories, selectElement:category_select,  useOptGroup:false, currentValue:TASK.category, });
      populateSelectOptions({ data:App.state.projects, selectElement:project_select, useOptGroup:false, currentValue:TASK.project, });
      
      //after each input, or select change: check if the task has been changed
      ['keydown', 'change'].forEach(eventType => {
          el.addEventListener(eventType, (event) => { validateTaskEdit(event); });
      });
    }
    else {
        task_input.innerText = taskText.value ;
        days_input.innerText = daysLeft.value + 'd'; // Add the 'd' suffix
        category_select.innerText = getCategory(TASK.category, 'id').name || 'No Category';
        project_select.innerText = getProject(TASK.project, 'id').name || 'No Project';
   
        ['keydown', 'change'].forEach(eventType => {
          el.removeEventListener(eventType, (event) => { validateTaskEdit(event); });
        });
    }

    App.state.editMode.task = newMode;
    // highlightEditButton(newMode);
    modifyAEDButtons(newMode);
  });

  validateTaskEdit(); //  (Useful for initial state , if was being edited before)
}


//Check if the current task being edited has been changed
export function validateTaskEdit(){
  const task = getTask(App.state.task.expandedTask,'id');
  const current_task = dQ(`#task_${task.id}`);
  const saveBtn = dQ(App.element.taskActionButtons.save);
  if (!saveBtn) return;

  const current_task_cache = {
    id: task.id,
    name: task.name,
    project: task.project,
    category: task.category,
  }
  const edit_task = {
    id: current_task.querySelector(App.element.taskElements.taskidCLASS).innerText,
    name: current_task.querySelector(App.element.taskElements.taskheaderCLASS).value,
    project: current_task.querySelector(App.element.taskElements.taskProjectCLASS).value,
    category: current_task.querySelector(App.element.taskElements.taskCategoryCLASS).value,
  }

  const taskHasChanges = JSON.stringify(current_task_cache) != JSON.stringify(edit_task);
  //Compare the current task with the one in edit mode

  debug(`Task edit has ${taskHasChanges ? '' : 'no '}unsaved changes`, { importance: 2, category: 'ui' });
  if (taskHasChanges) {
    toggleDisableAED(!taskHasChanges,saveBtn); // Enable save button if there are unsaved changes
    // console.log('Current Task:', current_task_cache);
    // console.log('Edit Task:', edit_task);
    return false;
  }
  else {
    debug('Task edit has no unsaved changes', { importance: 2, category: 'ui' });
    toggleDisableAED(true,saveBtn); // Disable save button if there are no unsaved changes
  }

  return true;
}




export function populateSelectOptions({
  data = [],
  selectElement,
  optionValueKey = 'id',
  optionLabelKey = 'name',
  prependOptions = [],
  currentValue = null,
  groupByTaskPresence = false, // NEW
}) {
  if (!selectElement) return;

  selectElement.innerHTML = '';

  // Prepend static options like "Show All"
  prependOptions.forEach(opt => {
    const option = document.createElement('option');
    option.value = opt.value;
    option.textContent = opt.label;
    selectElement.appendChild(option);
  });

  if (groupByTaskPresence) {
    const hasTasksGroup = document.createElement('optgroup');
    hasTasksGroup.label = 'has_tasks';

    const emptyGroup = document.createElement('optgroup');
    emptyGroup.label = 'empty';

    data.forEach(cat => {
      const hasTasks = App.state.tasks.some(
        t => t.project === App.state.project.currentProjectID && t.category === cat.id
      );

      const option = document.createElement('option');
      option.value = cat[optionValueKey];
      option.textContent = `${cat.emoji || ''} ${cat[optionLabelKey]}`;

      if (cat.id === currentValue) option.selected = true;

      (hasTasks ? hasTasksGroup : emptyGroup).appendChild(option);
    });

    if (hasTasksGroup.children.length > 0) selectElement.appendChild(hasTasksGroup);
    if (emptyGroup.children.length > 0) selectElement.appendChild(emptyGroup);

  } else {
    // Flat list — no groups
    data.forEach(item => {
      const option = document.createElement('option');
      option.value = item[optionValueKey];
      option.textContent = item[optionLabelKey];
      if (item[optionValueKey] === currentValue) option.selected = true;
      selectElement.appendChild(option);
    });
  }
}
