<?php
/**
 * Public Share Access Page
 * 
 * Allows anyone with a valid share token to view/edit a project
 * 
 * @author Jack Ewers / BloodWeb
 * @version 2.0.0
 * @date 2025-11-18
 */

// Start session using the same bootstrap as the main app
require_once $_SERVER['DOCUMENT_ROOT'] . '/auth/session_bootstrap.php';

// Get token from URL
$token = $_GET['token'] ?? null;

if (!$token) {
    header('Location: /TaskList/?error=invalid_token');
    exit;
}

// Validate token and get project data via API
$api_url = $_SERVER['REQUEST_SCHEME'] . '://' . $_SERVER['HTTP_HOST'] . '/app/tasklist/api/sharing.php?action=public_project&token=' . urlencode($token);
$ch = curl_init($api_url);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
$response = curl_exec($ch);
$http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
curl_close($ch);

if ($http_code !== 200) {
    // Debug: log the error
    error_log("Shared link validation failed. Token: $token, HTTP Code: $http_code, Response: $response");
    header('Location: /TaskList/?error=invalid_or_expired_token&debug=' . urlencode(substr($response, 0, 100)));
    exit;
}

$project_data = json_decode($response, true);

if (!$project_data || !isset($project_data['id'])) {
    header('Location: /TaskList/?error=project_not_found');
    exit;
}

// Store token info in session for API access
$_SESSION['share_token'] = $token;
$_SESSION['share_permission'] = $project_data['permission_level'];
$_SESSION['share_project_id'] = $project_data['id'];
$_SESSION['share_project_name'] = $project_data['name'];

// Create a temporary "guest" marker in session if user not logged in
$is_logged_in = isset($_SESSION['user_id']);
if (!$is_logged_in) {
    $_SESSION['guest_user'] = true;
    $_SESSION['guest_project_id'] = $project_data['id'];
}

// Redirect to main app with share context
// The main app will detect share token and filter to this project only
header('Location: /TaskList/?share_mode=1&project_id=' . $project_data['id']);
exit;

// OLD CODE BELOW - keeping for reference but never reached due to redirect above
$permission_level = $project_data['permission_level'];
$can_edit = $permission_level === 'edit';
$project_name = htmlspecialchars($project_data['name']);
$owner_name = htmlspecialchars($project_data['owner_username'] ?? 'Unknown');
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $project_name ?> - Shared Project</title>
    
    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <!-- TaskList Styles -->
    <link rel="stylesheet" href="styles.css">
    
    <style>
        /* Shared View Specific Styles */
        .shared-header {
            background: linear-gradient(135deg, var(--primary-color), var(--primary-dark));
            color: white;
            padding: var(--spacing-xl);
            margin-bottom: var(--spacing-xxl);
            border-radius: var(--border-radius);
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
        }
        
        .shared-header h1 {
            margin: 0 0 var(--spacing-sm) 0;
            display: flex;
            align-items: center;
            gap: var(--spacing-md);
        }
        
        .shared-header h1 i {
            font-size: 1.8rem;
        }
        
        .shared-meta {
            display: flex;
            align-items: center;
            gap: var(--spacing-xl);
            margin-top: var(--spacing-md);
            opacity: 0.95;
        }
        
        .shared-meta-item {
            display: flex;
            align-items: center;
            gap: var(--spacing-xs);
        }
        
        .permission-badge-large {
            padding: var(--spacing-sm) var(--spacing-md);
            background: rgba(255, 255, 255, 0.2);
            border-radius: 20px;
            font-size: 0.9rem;
            font-weight: 600;
        }
        
        .login-prompt {
            background: var(--bg-secondary);
            border: 2px solid var(--border-color);
            border-radius: var(--border-radius);
            padding: var(--spacing-xl);
            margin-bottom: var(--spacing-xxl);
            text-align: center;
        }
        
        .login-prompt h3 {
            margin: 0 0 var(--spacing-sm) 0;
            color: var(--text-primary);
        }
        
        .login-prompt p {
            margin: 0 0 var(--spacing-md) 0;
            color: var(--text-secondary);
        }
        
        .read-only-notice {
            background: rgba(251, 191, 36, 0.1);
            border-left: 4px solid #f59e0b;
            padding: var(--spacing-md);
            margin-bottom: var(--spacing-xl);
            border-radius: var(--border-radius);
        }
        
        .read-only-notice i {
            color: #f59e0b;
            margin-right: var(--spacing-sm);
        }
        
        .shared-content {
            max-width: 1400px;
            margin: 0 auto;
            padding: var(--spacing-xl);
        }
        
        /* Hide edit buttons in view-only mode */
        body.view-only-mode .btn-edit,
        body.view-only-mode .btn-delete,
        body.view-only-mode .task-actions .btn-icon[title="Edit"],
        body.view-only-mode .task-actions .btn-icon[title="Delete"] {
            display: none !important;
        }
    </style>
</head>
<body class="<?= !$can_edit ? 'view-only-mode' : '' ?>" data-share-token="<?= htmlspecialchars($token) ?>" data-share-permission="<?= htmlspecialchars($permission_level) ?>">
    
    <div class="shared-content">
        <!-- Shared Project Header -->
        <div class="shared-header">
            <h1>
                <i class="fas fa-share-alt"></i>
                <?= $project_name ?>
            </h1>
            <div class="shared-meta">
                <div class="shared-meta-item">
                    <i class="fas fa-user"></i>
                    <span>Shared by <strong><?= $owner_name ?></strong></span>
                </div>
                <div class="shared-meta-item">
                    <span class="permission-badge-large">
                        <i class="fas fa-<?= $can_edit ? 'edit' : 'eye' ?>"></i>
                        <?= $can_edit ? 'Can Edit' : 'View Only' ?>
                    </span>
                </div>
            </div>
        </div>
        
        <?php if (!$is_logged_in): ?>
        <!-- Login Prompt for Non-Authenticated Users -->
        <div class="login-prompt">
            <h3><i class="fas fa-sign-in-alt"></i> Want to collaborate?</h3>
            <p>Login to your BloodWeb account to save changes and collaborate with the project owner.</p>
            <a href="/auth/login.php?redirect=<?= urlencode($_SERVER['REQUEST_URI']) ?>" class="btn btn-primary">
                <i class="fas fa-sign-in-alt"></i> Login to Collaborate
            </a>
        </div>
        <?php endif; ?>
        
        <?php if (!$can_edit): ?>
        <!-- Read-Only Notice -->
        <div class="read-only-notice">
            <i class="fas fa-info-circle"></i>
            <strong>Read-Only Access:</strong> You can view this project but cannot make changes. Contact the project owner for edit access.
        </div>
        <?php endif; ?>
        
        <!-- Project Content -->
        <div id="shared-project-view">
            <div class="loading-spinner" style="text-align: center; padding: var(--spacing-xxl);">
                <i class="fas fa-spinner fa-spin" style="font-size: 3rem; color: var(--primary-color);"></i>
                <p style="margin-top: var(--spacing-md); color: var(--text-secondary);">Loading project...</p>
            </div>
        </div>
    </div>
    
    <script>
        // Shared View Configuration
        const SHARED_VIEW_CONFIG = {
            token: '<?= htmlspecialchars($token) ?>',
            projectId: <?= $project_data['id'] ?>,
            permission: '<?= htmlspecialchars($permission_level) ?>',
            isLoggedIn: <?= $is_logged_in ? 'true' : 'false' ?>,
            canEdit: <?= $can_edit ? 'true' : 'false' ?>
        };
        
        // Initialize shared view when DOM is ready
        document.addEventListener('DOMContentLoaded', async () => {
            try {
                // Load project data
                const project = await loadSharedProject();
                
                // Render project view
                renderSharedProject(project);
                
                // Load tasks for this project
                await loadSharedTasks(project.id);
                
            } catch (error) {
                console.error('Error loading shared project:', error);
                document.getElementById('shared-project-view').innerHTML = `
                    <div class="error-state" style="text-align: center; padding: var(--spacing-xxl);">
                        <i class="fas fa-exclamation-triangle" style="font-size: 3rem; color: var(--danger-color);"></i>
                        <h3 style="margin-top: var(--spacing-md);">Failed to Load Project</h3>
                        <p style="color: var(--text-secondary);">${error.message || 'An error occurred while loading the shared project.'}</p>
                    </div>
                `;
            }
        });
        
        async function loadSharedProject() {
            console.log('Loading shared project with token:', SHARED_VIEW_CONFIG.token);
            const response = await fetch(`/app/tasklist/api/sharing.php?action=public_project&token=${SHARED_VIEW_CONFIG.token}`);
            console.log('Project response:', response.status);
            if (!response.ok) {
                const error = await response.text();
                console.error('Project load error:', error);
                throw new Error('Failed to load project');
            }
            const data = await response.json();
            console.log('Project data:', data);
            return data;
        }
        
        async function loadSharedTasks(projectId) {
            console.log('Loading tasks for project:', projectId);
            const response = await fetch(`/TaskList/api/tasks?project_id=${projectId}`);
            console.log('Tasks response:', response.status);
            if (!response.ok) {
                const error = await response.text();
                console.error('Tasks load error:', error);
                throw new Error('Failed to load tasks');
            }
            const data = await response.json();
            console.log('Tasks data:', data);
            const tasks = data.data || data.tasks || data;
            
            renderSharedTasks(tasks);
        }
        
        function renderSharedProject(project) {
            const container = document.getElementById('shared-project-view');
            
            let html = `
                <div class="project-details" style="margin-bottom: var(--spacing-xxl);">
                    ${project.description ? `
                        <div class="project-description" style="padding: var(--spacing-lg); background: var(--bg-secondary); border-radius: var(--border-radius); margin-bottom: var(--spacing-xl);">
                            <h3 style="margin: 0 0 var(--spacing-sm) 0;"><i class="fas fa-align-left"></i> Description</h3>
                            <p style="margin: 0; color: var(--text-secondary);">${escapeHtml(project.description)}</p>
                        </div>
                    ` : ''}
                    
                    <div class="project-stats" style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: var(--spacing-md); margin-bottom: var(--spacing-xl);">
                        <div class="stat-card" style="padding: var(--spacing-lg); background: var(--bg-secondary); border-radius: var(--border-radius); text-align: center;">
                            <i class="fas fa-tasks" style="font-size: 2rem; color: var(--primary-color); margin-bottom: var(--spacing-sm);"></i>
                            <div style="font-size: 1.5rem; font-weight: 600; color: var(--text-primary);" id="total-tasks">-</div>
                            <div style="color: var(--text-secondary); font-size: 0.9rem;">Total Tasks</div>
                        </div>
                        <div class="stat-card" style="padding: var(--spacing-lg); background: var(--bg-secondary); border-radius: var(--border-radius); text-align: center;">
                            <i class="fas fa-check-circle" style="font-size: 2rem; color: var(--success-color); margin-bottom: var(--spacing-sm);"></i>
                            <div style="font-size: 1.5rem; font-weight: 600; color: var(--text-primary);" id="completed-tasks">-</div>
                            <div style="color: var(--text-secondary); font-size: 0.9rem;">Completed</div>
                        </div>
                        <div class="stat-card" style="padding: var(--spacing-lg); background: var(--bg-secondary); border-radius: var(--border-radius); text-align: center;">
                            <i class="fas fa-hourglass-half" style="font-size: 2rem; color: var(--warning-color); margin-bottom: var(--spacing-sm);"></i>
                            <div style="font-size: 1.5rem; font-weight: 600; color: var(--text-primary);" id="pending-tasks">-</div>
                            <div style="color: var(--text-secondary); font-size: 0.9rem;">In Progress</div>
                        </div>
                    </div>
                    
                    <h2 style="margin: var(--spacing-xl) 0 var(--spacing-md) 0;">
                        <i class="fas fa-list"></i> Tasks
                    </h2>
                </div>
                
                <div id="shared-tasks-container" class="tasks-grid">
                    <!-- Tasks will be rendered here -->
                </div>
            `;
            
            container.innerHTML = html;
        }
        
        function renderSharedTasks(tasks) {
            const container = document.getElementById('shared-tasks-container');
            
            if (!tasks || tasks.length === 0) {
                container.innerHTML = `
                    <div class="empty-state" style="text-align: center; padding: var(--spacing-xxl); color: var(--text-tertiary);">
                        <i class="fas fa-inbox" style="font-size: 3rem; opacity: 0.5; margin-bottom: var(--spacing-md);"></i>
                        <p>No tasks in this project yet.</p>
                    </div>
                `;
                return;
            }
            
            // Update stats
            const completed = tasks.filter(t => t.status === 'completed' || t.is_completed === 1).length;
            const pending = tasks.filter(t => t.status === 'pending' || (t.status !== 'completed' && t.is_completed === 0)).length;
            
            document.getElementById('total-tasks').textContent = tasks.length;
            document.getElementById('completed-tasks').textContent = completed;
            document.getElementById('pending-tasks').textContent = pending;
            
            // Render tasks using the app's renderer if available
            if (window.app && window.app.ui && window.app.ui.renderTaskCard) {
                container.innerHTML = tasks.map(task => window.app.ui.renderTaskCard(task)).join('');
            } else {
                // Fallback simple rendering
                container.innerHTML = tasks.map(task => renderSimpleTask(task)).join('');
            }
        }
        
        function renderSimpleTask(task) {
            const isCompleted = task.status === 'completed' || task.is_completed === 1;
            return `
                <div class="task-item ${isCompleted ? 'completed' : ''}" style="padding: var(--spacing-md); background: var(--bg-secondary); border-radius: var(--border-radius); margin-bottom: var(--spacing-sm);">
                    <div style="display: flex; align-items: center; gap: var(--spacing-md);">
                        ${isCompleted ? '<i class="fas fa-check-circle" style="color: var(--success-color);"></i>' : '<i class="far fa-circle" style="color: var(--text-tertiary);"></i>'}
                        <div style="flex: 1;">
                            <h4 style="margin: 0; ${isCompleted ? 'text-decoration: line-through; opacity: 0.7;' : ''}">${escapeHtml(task.title)}</h4>
                            ${task.description ? `<p style="margin: var(--spacing-xs) 0 0 0; color: var(--text-secondary); font-size: 0.9rem;">${escapeHtml(task.description)}</p>` : ''}
                        </div>
                        ${task.priority ? `<span class="priority-badge priority-${task.priority}" style="padding: 0.25rem 0.5rem; border-radius: 12px; font-size: 0.8rem;">P${task.priority}</span>` : ''}
                    </div>
                </div>
            `;
        }
        
        function escapeHtml(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }
    </script>
</body>
</html>
