// =========================================================================
// FILE: projects.js
// PURPOSE: Project rendering and selection logic
// AUTHOR: Jack Ewers / bloodweb.net
// LAST UPDATED: 2025-05-07
// =========================================================================


// =========================================================================
// IMPORTS
// =========================================================================

import { App,stackTrace } from './index.js'; // AppManager instance must be exported from index.js
import { makeMainInputs, setExpandedItem, toggleDisableAED, highlightEditButton } from './ui.js';
import { HTTP_modify, rehydrateApp } from './http.js';
import { generateSuccessMessage,getItem } from './helpers.js';
import { getCategoryNameById } from './categories.js';

// =========================================================================
//    Project   FUNCTIONS
// 
// . renderProjectsView()    - Renders the project management view
// . renderProjects()        - Deprecated: Renders the project list
// . setCurrentProject()     - Sets the current project and updates the UI
// =========================================================================


function moveCategoryBindButton(projectName, categoryID, toBound) {
  const listItem = [...document.querySelectorAll('.expansion-list-item')].find(el => el.querySelector('.expansion-title')?.innerText.includes(projectName) );
  if (!listItem) debug(`Project list item for ${projectName} not found`, { importance: 1 });
  
  const oldContainer = toBound ? '.unbound-categories' : '.bound-categories';
  const categoryName = getCategoryNameById(categoryID);
  
  const fromBtn = [...listItem.querySelectorAll(`${oldContainer} button`)]
    .find(btn => parseInt(btn.dataset.category) === parseInt(categoryID));
  if (!fromBtn) debug(`Button for ${categoryID} not found in ${oldContainer} section`, { importance: 1 });

  const boundContainer = listItem.querySelector('.bound-categories');
  const unboundContainer = listItem.querySelector('.unbound-categories');
  const toContainer = toBound ? boundContainer : unboundContainer;

  const newBtn = fromBtn.cloneNode(true);
  newBtn.className = toBound ? 'project-bind fade-in' : 'project-unbind fade-in';
  newBtn.innerHTML = `<i class="fas fa-${toBound?'':'un'}link"></i> ${categoryName}`

  newBtn.addEventListener('click', (e) => {
    e.stopPropagation();
    handleBindCategory(projectName, categoryID, !toBound);
  });

  fromBtn.classList.add('fade-out');
  setTimeout(() => {
    fromBtn.remove();
    toContainer.appendChild(newBtn);
  }, 180);

  debug(`Moved category ${categoryName} to ${toBound ? '' : 'un'}bound in project ${projectName}`, { importance: 3 });
}

function handleBindCategory(projectName, categoryID, toBind) {
  const project = getItem('projects',projectName, 'name');
  if (!project) debug(`Project '${projectName}' not found`, { importance: 1 });

  let categoryIDs = new Set(Array.isArray(project.categoryIDs)
    ? project.categoryIDs
    : (project.categoryIDs || ''));

  toBind
    ? categoryIDs.add(parseInt(categoryID))
    : categoryIDs.delete(parseInt(categoryID));

  const updatedArray = Array.from(categoryIDs);
  project.categoryIDs = updatedArray; // 🔥 Update live project object

  console.log(`Updated categoryIDs for ${projectName}: ${updatedArray.join(',')}`);

  moveCategoryBindButton(projectName, categoryID, toBind);

  HTTP_modify('EDIT_PROJECTS', {
    ...project,
    categoryIDs: updatedArray.join(','),
  }, false, true);
}


export function renderProjectsView() {
  
    const container = App.element.ViewContent;
    container.innerHTML = ''; // Clear current view
    const wrapper = createElement('div',{className:'project-view'});
    const header = createElement('h2')//,{innerText:'Project Management'});
    const projectList = createElement('ul',{ className: 'expansion-list'});
    
    App.state.projects.forEach(project => {
      projectList.appendChild(createProjectElement(project));
    });
  
    makeMainInputs(header,'projects');
   
    wrapper.append(header, projectList);
    container.appendChild(wrapper);
    toggleDisableAED(false,dQ(App.element.taskActionButtons.delete)); // Show delete button (delete Selected Project)
}

function createProjectElement(project) {
  const item = createElement('li', { className: `expansion-list-item ${App.state.project.expandedProject == project.id?'expanded':''}`, });
  item.dataset.id = project.id; // Set the project ID for easy access

  const titleRow = createElement('div', { className: 'expansion-header' });
  const namespan = createElement('span', { className: 'expansion-title', innerText: project.name });

  const nameInput = createElement('input', { className: 'edit-project-input', value: project.name, style: 'display: none;',});
  attachProjectEditHandlers(project, namespan, nameInput);

  const toggleBtn = createElement('button', { className: 'toggle-expand', innerHTML: '<i class="fas fa-chevron-down"></i>', });

  titleRow.append(toggleBtn,namespan,nameInput);
  item.appendChild(titleRow);

  const expand = createElement('div', { className: 'expandable-section' });
  const boundList = createElement('div', { className: 'bound-categories' ,innerHTML: '<strong>Bound Categories:</strong>'});
  const unboundList = createElement('div', { className: 'unbound-categories', innerHTML: '<strong>Available Categories:</strong>'});

  App.state.categories.forEach(cat => {
    const isBound = project.categoryIDs.includes(parseInt(cat.id));
    const btn = createElement('button', { innerHTML: `<i class="fas fa-${isBound?'':'un'}link"></i> ${cat.name}`, 
      className: isBound ? 'project-bind' : 'project-unbind', });
    btn.setAttribute('data-category', cat.id),

    btn.addEventListener('click', (e) => {
      e.stopPropagation();
      handleBindCategory(project.name, cat.id, !isBound); // ← flip the state correctly
    });    // Initially append to the correct list
    (isBound? boundList : unboundList).appendChild(btn);

    if (project.name.toLocaleLowerCase() === 'bloodweb') debug(`Category ${cat.name} (${cat.id}) is ${isBound ? 'bound' : 'unbound'} to project ${project.name}`, { importance: 3 });
  });
  expand.append(boundList, unboundList);
  item.appendChild(expand);

 item.addEventListener('click', (e) => {
    const isExpanded = item.classList.contains('expanded');
    const clickedToggle = e.target.closest('.toggle-expand');
    const clickedTitle = e.target.closest('.expansion-title');

    if (!isExpanded) {
      setExpandedItem(item,project.id);
      toggleDisableAED(false,dQ(App.element.taskActionButtons.delete))
    } 
    else if (clickedToggle || clickedTitle) { // only close if arrow or title was clicked
      setExpandedItem(false);
      item.classList.remove('expanded');
      toggleDisableAED(true,dQ(App.element.taskActionButtons.delete));
    }
  });
  return item;
}

function attachProjectEditHandlers(project, nameSpan, nameInput) {
  const commit = () => {
    const newName = nameInput.value.trim();
    if (newName && newName !== project.name) {
      project.name = newName;
      nameSpan.innerText = newName;
      HTTP_modify('EDIT_PROJECTS', { ...project }, 'Project updated');
    }
    nameSpan.style.display = 'inline-block';
    nameInput.style.display = 'none';
    App.state.editMode.projects = null;
    highlightEditButton(false);
  };

  nameInput.addEventListener('blur', commit);
  nameInput.addEventListener('keydown', (e) => {
    if (e.key === 'Enter') {
      e.preventDefault();
      commit();
    }
  });
}


export function deleteProject(id) {
  const project = App.state.projects.find(project => project.id === id);
  if (!project) { debug(`Project '${name}' not found`, { importance: 1 }); }
  debug(`Attempting to delete project  ${project.id}:'${project.name}'`, {category:'data', importance: 3 });
  
    const data = { id:project.id, }; 
    const SuccessMessage = generateSuccessMessage('delete', 'project', `${project.id}:${project.name}`);

    // Remove project from local state
    App.state.projects = App.state.projects.filter(project => project.name !== name);
    localStorage.setItem('projects', JSON.stringify(App.state.projects));
    HTTP_modify('DELETE_PROJECTS', data, SuccessMessage);
    
    debug(`Project '${name}' deleted`, { importance: 3, category: 'data' });
    rehydrateApp();
}


export function setCurrentProject(projectID=null, noRender = false) {
    // Catch if no project name is provided or if the project is not found (CRITICAL)
    if (!projectID) { debug('No project name provided', { importance: 1 });}
    if (!App.state.projects.some(project => project.id === projectID)) { debug(`Project '${projectID}' not found`, { importance: 1 }); }
    
    App.state.project.currentProjectID = projectID;
    localStorage.setItem('currentProjectID', projectID);

//    stackTrace(`Project '${projectID}' set as current project`);
    debug(`Current project has been set: ${projectID}`, { importance: 3 });
    if (noRender) return;
    
    renderProjectsView(); // Render the project view
}




export function getProject(value, variable) {
    const project = App.state.projects.find(project => project[variable] == value);
    if (!project) { debug(`Project with ${variable} '${value}' not found`, { importance: 1, category: 'data' }); }
    return project;
}