<?php
/**
 * TaskList v2 Data Migration Script
 * 
 * This script migrates data from the old TaskList schema to the new v2 schema
 * 
 * IMPORTANT: Always backup your database before running this migration!
 * 
 * Usage: php migrate_data.php [--dry-run] [--verbose]
 */

// Configuration
$OLD_DB_CONFIG = [
    'host' => 'localhost',
    'username' => 'taskmanager_user',
    'password' => 'BLOODWEBMYSQLX.X01',
    'database' => 'bloodweb'
];

$NEW_DB_CONFIG = [
    'host' => 'localhost',
    'username' => 'taskmanager_user',
    'password' => 'BLOODWEBMYSQLX.X01',
    'database' => 'tasklist_v2'
];

class TaskListMigration {
    private $oldDb;
    private $newDb;
    private $dryRun;
    private $verbose;
    private $migrationLog = [];

    public function __construct($oldConfig, $newConfig, $options = []) {
        $this->dryRun = $options['dry_run'] ?? false;
        $this->verbose = $options['verbose'] ?? false;
        
        try {
            $this->oldDb = new PDO(
                "mysql:host={$oldConfig['host']};dbname={$oldConfig['database']};charset=utf8mb4",
                $oldConfig['username'],
                $oldConfig['password'],
                [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
            );
            
            $this->newDb = new PDO(
                "mysql:host={$newConfig['host']};dbname={$newConfig['database']};charset=utf8mb4",
                $newConfig['username'],
                $newConfig['password'],
                [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
            );
            
            $this->log("Database connections established successfully");
        } catch (PDOException $e) {
            throw new Exception("Database connection failed: " . $e->getMessage());
        }
    }

    private function log($message, $level = 'INFO') {
        $timestamp = date('Y-m-d H:i:s');
        $logEntry = "[$timestamp] [$level] $message";
        $this->migrationLog[] = $logEntry;
        
        if ($this->verbose || $level === 'ERROR') {
            echo $logEntry . "\n";
        }
    }

    public function migrate() {
        $this->log("Starting TaskList v2 migration...");
        
        if ($this->dryRun) {
            $this->log("DRY RUN MODE - No actual data will be modified");
        }

        try {
            // Step 1: Create default admin user if none exists
            $this->migrateUsers();
            
            // Step 2: Migrate projects
            $this->migrateProjects();
            
            // Step 3: Migrate categories and create project-category relationships
            $this->migrateCategories();
            
            // Step 4: Migrate tasks
            $this->migrateTasks();
            
            // Step 5: Set up initial XP and achievements
            $this->setupInitialGameification();
            
            $this->log("Migration completed successfully!");
            
        } catch (Exception $e) {
            $this->log("Migration failed: " . $e->getMessage(), 'ERROR');
            throw $e;
        }
    }

    private function migrateUsers() {
        $this->log("Migrating users...");
        
        // Check if we need to create a default user
        $stmt = $this->newDb->query("SELECT COUNT(*) FROM users");
        $userCount = $stmt->fetchColumn();
        
        if ($userCount == 0) {
            if (!$this->dryRun) {
                $stmt = $this->newDb->prepare("
                    INSERT INTO users (username, email, password_hash, first_name, last_name, xp_total) 
                    VALUES (?, ?, ?, ?, ?, ?)
                ");
                $stmt->execute([
                    'bloodweb_user',
                    'user@bloodweb.net',
                    password_hash('default_password_change_me', PASSWORD_DEFAULT),
                    'BloodWeb',
                    'User',
                    0
                ]);
            }
            $this->log("Created default user account");
        }
    }

    private function migrateProjects() {
        $this->log("Migrating projects...");
        
        try {
            // Get projects from old database
            $stmt = $this->oldDb->query("SELECT * FROM projects ORDER BY id");
            $projects = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $this->log("Found " . count($projects) . " projects to migrate");
            
            foreach ($projects as $project) {
                if (!$this->dryRun) {
                    $stmt = $this->newDb->prepare("
                        INSERT INTO projects (id, user_id, name, description, color, created_at) 
                        VALUES (?, ?, ?, ?, ?, NOW())
                    ");
                    
                    $stmt->execute([
                        $project['id'],
                        1, // Default user ID
                        $project['name'],
                        $project['description'] ?? '',
                        '#3498db', // Default color
                    ]);
                }
                
                $this->log("Migrated project: " . $project['name']);
            }
            
        } catch (PDOException $e) {
            if (strpos($e->getMessage(), "doesn't exist") !== false) {
                $this->log("Projects table doesn't exist in old database, creating default project");
                if (!$this->dryRun) {
                    $stmt = $this->newDb->prepare("
                        INSERT INTO projects (user_id, name, description, color) 
                        VALUES (?, ?, ?, ?)
                    ");
                    $stmt->execute([1, 'Default Project', 'Migrated tasks project', '#3498db']);
                }
            } else {
                throw $e;
            }
        }
    }

    private function migrateCategories() {
        $this->log("Migrating categories...");
        
        try {
            // Get categories from old database
            $stmt = $this->oldDb->query("SELECT * FROM categories ORDER BY id");
            $categories = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $this->log("Found " . count($categories) . " categories to migrate");
            
            foreach ($categories as $category) {
                if (!$this->dryRun) {
                    // Insert category
                    $stmt = $this->newDb->prepare("
                        INSERT INTO categories (id, user_id, name, emoji, color, created_at) 
                        VALUES (?, ?, ?, ?, ?, NOW())
                    ");
                    
                    $stmt->execute([
                        $category['id'],
                        1, // Default user ID
                        $category['name'],
                        $category['emoji'] ?? '📝',
                        $category['color'] ?? '#95a5a6'
                    ]);
                    
                    // Create project-category relationships
                    if (!empty($category['projectIDs'])) {
                        $projectIds = array_filter(explode(',', $category['projectIDs']));
                        foreach ($projectIds as $projectId) {
                            $projectId = trim($projectId);
                            if (is_numeric($projectId)) {
                                $stmt = $this->newDb->prepare("
                                    INSERT IGNORE INTO project_categories (project_id, category_id) 
                                    VALUES (?, ?)
                                ");
                                $stmt->execute([$projectId, $category['id']]);
                            }
                        }
                    }
                }
                
                $this->log("Migrated category: " . $category['name']);
            }
            
        } catch (PDOException $e) {
            if (strpos($e->getMessage(), "doesn't exist") !== false) {
                $this->log("Categories table doesn't exist in old database, creating default categories");
                if (!$this->dryRun) {
                    $defaultCategories = [
                        ['General', '📝', '#95a5a6'],
                        ['Work', '💼', '#2c3e50'],
                        ['Personal', '🏠', '#27ae60']
                    ];
                    
                    foreach ($defaultCategories as $index => $cat) {
                        $stmt = $this->newDb->prepare("
                            INSERT INTO categories (user_id, name, emoji, color) 
                            VALUES (?, ?, ?, ?)
                        ");
                        $stmt->execute([1, $cat[0], $cat[1], $cat[2]]);
                        
                        $categoryId = $this->newDb->lastInsertId();
                        
                        // Link to default project
                        $stmt = $this->newDb->prepare("
                            INSERT INTO project_categories (project_id, category_id) 
                            VALUES (?, ?)
                        ");
                        $stmt->execute([1, $categoryId]);
                    }
                }
            } else {
                throw $e;
            }
        }
    }

    private function migrateTasks() {
        $this->log("Migrating tasks...");
        
        try {
            // Get tasks from old database
            $stmt = $this->oldDb->query("SELECT * FROM tasks ORDER BY id");
            $tasks = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $this->log("Found " . count($tasks) . " tasks to migrate");
            
            foreach ($tasks as $task) {
                if (!$this->dryRun) {
                    // Calculate due date from upload_date and days_to_complete
                    $dueDate = null;
                    if (!empty($task['upload_date']) && !empty($task['days_to_complete'])) {
                        $uploadDate = new DateTime($task['upload_date']);
                        $uploadDate->add(new DateInterval('P' . $task['days_to_complete'] . 'D'));
                        $dueDate = $uploadDate->format('Y-m-d');
                    }
                    
                    // Map old category names to new category IDs
                    $categoryId = $this->getCategoryIdByName($task['category'] ?? 'General');
                    $projectId = $task['project'] ?? 1;
                    
                    $stmt = $this->newDb->prepare("
                        INSERT INTO tasks (
                            id, user_id, project_id, category_id, title, description, 
                            priority, status, due_date, created_at
                        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                    ");
                    
                    $stmt->execute([
                        $task['id'],
                        1, // Default user ID
                        $projectId,
                        $categoryId,
                        $task['name'],
                        $task['description'] ?? '',
                        'medium', // Default priority
                        'pending', // Default status
                        $dueDate,
                        $task['upload_date'] ?? date('Y-m-d H:i:s')
                    ]);
                }
                
                $this->log("Migrated task: " . $task['name']);
            }
            
        } catch (PDOException $e) {
            if (strpos($e->getMessage(), "doesn't exist") !== false) {
                $this->log("Tasks table doesn't exist in old database, skipping task migration");
            } else {
                throw $e;
            }
        }
    }

    private function getCategoryIdByName($categoryName) {
        $stmt = $this->newDb->prepare("SELECT id FROM categories WHERE name = ? LIMIT 1");
        $stmt->execute([$categoryName]);
        $result = $stmt->fetchColumn();
        
        if ($result) {
            return $result;
        }
        
        // If category doesn't exist, return first available category
        $stmt = $this->newDb->query("SELECT id FROM categories ORDER BY id LIMIT 1");
        return $stmt->fetchColumn() ?: 1;
    }

    private function setupInitialGameification() {
        $this->log("Setting up initial gamification...");
        
        if (!$this->dryRun) {
            // Award initial XP for existing completed tasks
            $stmt = $this->newDb->query("
                SELECT COUNT(*) FROM tasks 
                WHERE status = 'completed' AND user_id = 1
            ");
            $completedTasks = $stmt->fetchColumn();
            
            if ($completedTasks > 0) {
                $initialXp = $completedTasks * 10;
                $stmt = $this->newDb->prepare("
                    UPDATE users SET xp_total = ? WHERE id = 1
                ");
                $stmt->execute([$initialXp]);
                
                $this->log("Awarded $initialXp XP for $completedTasks completed tasks");
            }
        }
    }

    public function getLog() {
        return $this->migrationLog;
    }
}

// Command line execution
if (php_sapi_name() === 'cli') {
    $options = [
        'dry_run' => in_array('--dry-run', $argv),
        'verbose' => in_array('--verbose', $argv) || in_array('-v', $argv)
    ];
    
    try {
        $migration = new TaskListMigration($OLD_DB_CONFIG, $NEW_DB_CONFIG, $options);
        $migration->migrate();
        
        echo "\nMigration completed successfully!\n";
        
    } catch (Exception $e) {
        echo "Migration failed: " . $e->getMessage() . "\n";
        exit(1);
    }
} else {
    // Web execution (for manual testing)
    echo "<pre>";
    echo "TaskList v2 Migration Script\n";
    echo "===========================\n\n";
    echo "This script should be run from the command line:\n";
    echo "php migrate_data.php [--dry-run] [--verbose]\n\n";
    echo "Options:\n";
    echo "  --dry-run    Show what would be migrated without making changes\n";
    echo "  --verbose    Show detailed migration progress\n";
    echo "</pre>";
}
?>