// =========================================================================
// FILE: http.js
// PURPOSE: Handles all HTTP communication with the server
// AUTHOR: Jack Ewers / bloodweb.net
// LAST UPDATED: 2025-05-07
// =========================================================================

import { AppManager } from './AppManager.js';
import { App } from './index.js'; // Assuming AppManager instance is exported from index.js
import { moveIndicatorTo, renderView} from './ui.js';
import { validateStateData } from './helpers.js';




// http.js
export async function FETCH(method) {
    try {
        const response = await fetch(`${AppManager.bloodweb_tasklist}manipulate_tasks.php`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ method })
        });
        const data = await response.json();
        return data;
    } catch (error) {
        debug(`FETCH failed for ${method}: ${error}`, { importance: 1,category: 'network' });
        throw error;
    }
}

export async function HTTP_modify(method, payload = {}, successMessage = `${method} succeeded`,noRender = false) {
    try {
        const response = await fetch(AppManager.bloodweb_tasklist + 'manipulate_tasks.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ method, ...payload })
        });

        const data = await response.json();

        if ( data.error ) { throw new Error(data.error); }
        if ( successMessage != false ) { bwSuccess({heading:'Notification:',body:successMessage}); }
        if ( noRender ) return;
        await rehydrateApp();

   
        return data;
    } catch (error) {
        debug(`HTTP_modify failed (${method}): ${error}`, { importance: 1 });
        throw error;
    }
}



export async function rehydrateApp(firstLoad = false) {
    const fetchMap = {
        projects: () => FETCH('GET_PROJECTS'),
        categories: () => FETCH('GET_CATEGORIES'),
        tasks: () => FETCH('GET_TASKS'),
    };

    const entities = Object.keys(fetchMap);
    const results = await Promise.all(
        entities.map(entity => fetchMap[entity]())
    );

    entities.forEach((entity, index) => {
        const raw = results[index];
        const data = raw[entity] || raw; // Flexible in case server returns either
        App.state[entity] = data;
    });

    if (App.state.isFirstRun === true) {
        App.state.project.currentProjectID = App.state.projects[0]?.name;
        App.state.category.currentCategoryID = App.state.categories[0]?.name;
        
        let initialTab = dQ(`.view-tab[data-view="${App.state.activeView}"]`);
        App.element.viewtabs.forEach(tab => { tab.classList.remove('active'); }); // Remove active class from all tabs
        moveIndicatorTo(initialTab) // Handle the initial view
        initialTab.classList.add('active');
        
        validateStateData(['projects','tasks','categories']); // Ensure current project is valid \\ removes null project
        renderView(App.state.activeView ?? App.settings.defaultView); // Render the initial view
        App.state.isFirstRun = false;
        localStorage.setItem('isFirstRun', 'false'); // Set first run to false
    }
    validateStateData(['projects','tasks','categories','expandedTasks']); // Ensure current project is valid \\ removes null project

    if (App.settings.possibleViews.includes(App.state.activeView)) { renderView(App.state.activeView); }
    else { debug(`Invalid view: ${App.state.activeView}`, { importance: 1, category: 'ui' }); }

}