import { App } from './index.js';
import { getProject } from './projects.js';

export function captiliseFirstLetter(string = '') {
    return string.charAt(0).toUpperCase() + string.slice(1);
}
 
export function removeActiveElementFocus() {
    if (document.activeElement && typeof document.activeElement.blur === 'function') {
        document.activeElement.blur();
    }
}

export function getPriority(medium, count) {
    if (medium === 'task_days_left') {
        // check if state and urgency is defined
        const urgency = App?.state?.task?.daysUrgency;

        if (!urgency) return 'low'; // fallback if not defined
        if (count <= urgency.alert) return 'alert'; // ON FIRE // DO NOW
        if (count <= urgency.high) return 'high';
        if (count <= urgency.medium) return 'medium';
        return 'low';
    }
}

export function daysSince(dateString) {
    // Ensure dateString is in YYYY-MM-DD format
    if (!dateString || !/^\d{4}-\d{2}-\d{2}$/.test(dateString)) return 0;
    const date = new Date(dateString);
    const today = new Date();
    today.setHours(0, 0, 0, 0); // Zero out the time for accurate day difference
    date.setHours(0, 0, 0, 0);
    const diffTime = today - date;
    const days = Math.floor(diffTime / (1000 * 60 * 60 * 24));
    return days > 0 ? days : 0; // Return 0 if the date is today or in the future
}

export function getDaysTillDue(totalDays, upload_date) {

    // Ensure upload_date is in YYYY-MM-DD format
    if (!totalDays || !upload_date || !/^\d{4}-\d{2}-\d{2}$/.test(upload_date)) return 0;
    const uploadDate = new Date(upload_date);
    const dueDate = new Date(uploadDate);
    dueDate.setDate(uploadDate.getDate() + Number(totalDays));
      
    // Zero out the time for accurate day difference
    const today = new Date();
    today.setHours(0,0,0,0);
    dueDate.setHours(0,0,0,0);

    const diffTime = dueDate - today;
    const days = Math.ceil(diffTime / (1000 * 60 * 60 * 24));
    return days > 0 ? days : 0;

}



export function getItem(item, value, variable = 'id') {

    let result = null;

    switch (item) {
        case 'projects':
            result = App.state.projects.find(project => project[variable] === value); break;
        case 'categories':
            result = App.state.categories.find(category => category[variable] === value); break;
        case 'tasks':       
            result = App.state.tasks.find(task => task[variable] === value); break;
        default:
            debug(`No item type found for ${item}. please use: projects,categories or tasks`, { importance: 1, category: 'data' });
    }
    if (!result) { debug(`Item with ${variable} '${value}' not found in ${item}`, { importance: 1, category: 'data' }); }
    return result;

}


export function validateStateData(state=App.state.activeView, data) {
    if (Array.isArray(state)) {
        debug(`Validating multiple state data on the following items: ${state}`, { importance: 8, category: 'data' });
        return state.every(item => validateStateData(item, data)); 
    }   
    debug(`Validating state data for: ${state}`, { importance: 8, category: 'data' });
   
    switch (state) {
        case 'projects':
            // if project is not set or invalid, set to first project
            if (
                !App.state.project.currentProjectID 
                || isNaN(App.state.project.currentProjectID)
                || App.state.projects.findIndex(p => p.id === App.state.project.currentProjectID) === -1
            ) {
                const firstProject = App.state.projects[0];
                if( !firstProject ) { debug('No first project to set. has the app initialised properly/', { importance: 1 }); return; }
                debug(`No current project set, defaulting to first project: is${firstProject.id}:${firstProject.name}`, { importance: 4, category: 'ui' });
                App.state.project.currentProjectID = App.state.projects[0].id;
                localStorage.setItem('currentProject', App.state.project.currentProjectID);
            }
            return true; // Validation passed (valid project is set)

        case 'categories':
            const currentProject = App.state.project.currentProjectID;
            if (!currentProject) { debug('No currentProjectID, cannot validate category', { importance: 1, category: 'data' }); return false; }
            const categoriesForProject = getItem('projects',currentProject)?.categoryIDs || [];
            if (categoriesForProject.length === 0) { debug(`Categories for project(${currentProject}) is empty, Projects must have at least 1 bound category`,{importance:1,category:'data'} );} 
        
        
            const currentCategoryID = App.state.category.currentCategoryID;
            const isValid = currentCategoryID!==null && categoriesForProject.includes(Number(currentCategoryID));
            if (!isValid) {
                App.state.category.currentCategoryID = categoriesForProject[0] || null;
                localStorage.setItem('currentCategoryID', App.state.category.currentCategoryID);
                debug(`Current categoryID:${currentCategoryID} is invalid. Set to item 1 in ${categoriesForProject}`, { importance: 4, category: 'data' });
            }
            return isValid;
    
        case 'tasks': 
            // make sure tasks days to complete is accurate (upload date (YYYY-MM-DD) - days to complete = due date)
            App.state.tasks.forEach(task => {
                if (task.days_to_complete && task.upload_date) {
                    const daysLeft = getDaysTillDue(task.days_to_complete,task.upload_date); // e.g., returns number of days left
                    task.days_left_to_complete = daysLeft;

                    debug(`Task ${task.id} due date updated to ${task.days_left_to_complete}`, { importance: 10, category: 'data' });
                }
                else{
                    debug(`Task ${task.id} is missing days_to_complete or upload_date. Cannot update due date.`, { importance: 2, category: 'data' });
                }

            });

            //return data && data.name && data.name.trim() !== '' && data.project && data.category;
            return true;

        case 'selectedTasks':
            const selectedTasks = App.state.task.selectedTasks;
            const validTasks = selectedTasks.filter(taskId => {
            const task = App.state.tasks.find(t => t.id === taskId);
            return task && task.project === App.state.project.currentProjectID && task.category === App.state.category.currentCategoryID;
            });
            if (validTasks.length !== selectedTasks.length) {
                App.state.task.selectedTasks = validTasks;
                localStorage.setItem('selectedTasks', JSON.stringify(validTasks));
                debug('Some selected tasks are no longer valid and have been removed from the selection.',{ importance: 3, category: 'data' });
            }
            return validTasks;

        case 'expandedTasks':
            const expandedTask = App.state.task.expandedTask;
            debug(`Expanded task value: ${expandedTask} (type: ${typeof expandedTask})`, { importance: 10, category: 'data' });

            if (expandedTask == '' || expandedTask === null || expandedTask === undefined || expandedTask === 'null') {
                return true; // No expanded task to validate
            }

            const validExpandedTask = App.state.tasks.find(t => t.id === expandedTask && t.project === App.state.project.currentProjectID && t.category === App.state.category.currentCategoryID);
            if (!validExpandedTask) {
                debug(`Expanded task (${expandedTask}) is no longer valid.`,{ importance: 3, category: 'data' });
                App.state.task.expandedTask = null;
                localStorage.removeItem('task.expandedTask');
            }
            return true;

        default:
            console.warn(`No validation for state: ${state}`);
            return false;


    }
}


export function getFormTitle(method,pastTense=false,colorCoded = true) {
    const splitMethod = method.toLocaleLowerCase().split('_');
    const actionType = splitMethod[0]; // Action
    const entityType = splitMethod[1]; // Entity
    const noun = actionType === 'add' ? 'new' : 'selected';

    const actionColors = { add: 'green', delete: 'red', edit: 'blue' };
    const typeColors = { project: 'purple', category: 'orange', task: 'teal' };

    const actionColor = actionColors[actionType] || 'black';
    const entityColor = typeColors[entityType] || 'gray';
    
    const ActionWord = captiliseFirstLetter(actionType) + (pastTense ? (actionType.endsWith('e') ? 'd' : 'ed'):'');

    return `
    <h4>
        <span style="color: ${actionColor}; font-weight: bold;">
            ${ActionWord}
        </span> 
        <span> ${noun} </span> 
        <span style="border-bottom:2px solid ${entityColor}">
        ${captiliseFirstLetter(entityType)}</span>:
    </h4>
    `;

}

export function generateSuccessMessage(actionType, entityType, message, multipleChanges = false, asArray = false) {
    
    const actionColors = { add: 'green', delete: 'red', edit: 'blue' };
    const typeColors = { project: 'purple', category: 'orange', task: 'teal' };

    const actionColor = actionColors[actionType] || 'black';
    const entityColor = typeColors[entityType] || 'gray';
    
    const plural = multipleChanges ? 's' : '';
    const ActionWord = captiliseFirstLetter(actionType) + (actionType.endsWith('e') ? 'd' : 'ed') + plural;

    return `
        <div>
            <h4>
                <span style="color: ${actionColor}; font-weight: bold;">
                    ${ActionWord}
                </span> 
                <span style="border-bottom:2px solid ${entityColor}">
                ${captiliseFirstLetter(entityType)}</span>:
            </h4>
            <b>
                <span>
                    ${message}
                </span>
            </p>
            <p style="color:li" ><u>Successfully</u></p>
        </div>
    `;
}
