#!/usr/bin/env python3
"""
CSS Deduplication Script
Analyzes styles.css and removes duplicate selector blocks while preserving
media queries and unique styles.
"""

import re
from collections import defaultdict
from typing import Dict, List, Tuple

def parse_css_blocks(content: str) -> List[Tuple[int, str, str, str]]:
    """
    Parse CSS into blocks with their line numbers.
    Returns: [(line_num, selector, properties, full_block), ...]
    """
    blocks = []
    lines = content.split('\n')
    i = 0
    
    while i < len(lines):
        line = lines[i].strip()
        
        # Skip empty lines and comments
        if not line or line.startswith('/*'):
            i += 1
            continue
            
        # Check if this is a selector (ends with { or is followed by {)
        if '{' in line or (i + 1 < len(lines) and lines[i + 1].strip() == '{'):
            selector_line_num = i + 1  # 1-indexed
            selector = line.replace('{', '').strip()
            
            # Find the opening brace
            if '{' not in line:
                i += 1
                if i >= len(lines):
                    break
            
            # Collect the properties until closing brace
            properties = []
            i += 1
            brace_count = 1
            
            while i < len(lines) and brace_count > 0:
                prop_line = lines[i]
                
                if '{' in prop_line:
                    brace_count += 1
                if '}' in prop_line:
                    brace_count -= 1
                    
                if brace_count > 0:
                    properties.append(prop_line)
                    
                i += 1
            
            # Normalize properties for comparison
            props_text = '\n'.join(properties).strip()
            full_block = f"{selector} {{\n{props_text}\n}}"
            
            blocks.append((selector_line_num, selector, props_text, full_block))
        else:
            i += 1
    
    return blocks

def find_duplicates(blocks: List[Tuple[int, str, str, str]]) -> Dict[str, List[Tuple[int, str]]]:
    """
    Find duplicate selector blocks.
    Returns: {selector: [(line_num, properties), ...]}
    """
    selector_map = defaultdict(list)
    
    for line_num, selector, properties, full_block in blocks:
        # Normalize selector for comparison
        norm_selector = selector.strip()
        selector_map[norm_selector].append((line_num, properties, full_block))
    
    # Filter to only duplicates
    duplicates = {k: v for k, v in selector_map.items() if len(v) > 1}
    
    return duplicates

def analyze_css(filepath: str):
    """Analyze CSS file for duplicates and report findings."""
    
    with open(filepath, 'r') as f:
        content = f.read()
    
    print(f"Analyzing {filepath}...")
    print(f"Total size: {len(content)} characters, {len(content.splitlines())} lines\n")
    
    blocks = parse_css_blocks(content)
    print(f"Found {len(blocks)} CSS blocks\n")
    
    duplicates = find_duplicates(blocks)
    print(f"Found {len(duplicates)} selectors with duplicates:\n")
    
    # Sort by number of occurrences
    sorted_dups = sorted(duplicates.items(), key=lambda x: len(x[1]), reverse=True)
    
    for selector, occurrences in sorted_dups[:20]:  # Top 20
        print(f"  {selector}")
        print(f"    Appears {len(occurrences)} times at lines: {', '.join(str(o[0]) for o in occurrences)}")
        
        # Check if properties are identical
        props_set = set(o[1].strip() for o in occurrences)
        if len(props_set) == 1:
            print(f"    ✓ All instances are identical (can be safely merged)")
        else:
            print(f"    ⚠ Different properties (requires manual review)")
        print()
    
    return duplicates, blocks

def generate_report(filepath: str, output_file: str):
    """Generate detailed duplication report."""
    
    duplicates, blocks = analyze_css(filepath)
    
    with open(output_file, 'w') as f:
        f.write("CSS Duplication Report\n")
        f.write("=" * 80 + "\n\n")
        
        sorted_dups = sorted(duplicates.items(), key=lambda x: len(x[1]), reverse=True)
        
        for selector, occurrences in sorted_dups:
            f.write(f"\n{selector}\n")
            f.write(f"{'-' * len(selector)}\n")
            f.write(f"Occurrences: {len(occurrences)}\n")
            f.write(f"Line numbers: {', '.join(str(o[0]) for o in occurrences)}\n\n")
            
            # Check if identical
            props_set = set(o[1].strip() for o in occurrences)
            if len(props_set) == 1:
                f.write("Status: IDENTICAL - Safe to merge\n")
                f.write("\nProperties:\n")
                f.write(occurrences[0][1] + "\n")
            else:
                f.write("Status: DIFFERENT - Requires review\n")
                for i, (line_num, props, full) in enumerate(occurrences, 1):
                    f.write(f"\n  Occurrence {i} (line {line_num}):\n")
                    for line in props.split('\n'):
                        if line.strip():
                            f.write(f"    {line}\n")
            
            f.write("\n" + "=" * 80 + "\n")
    
    print(f"\nDetailed report written to {output_file}")

if __name__ == "__main__":
    css_file = "styles.css"
    report_file = "css_duplication_report.txt"
    
    print("CSS Deduplication Analysis")
    print("=" * 80 + "\n")
    
    analyze_css(css_file)
    generate_report(css_file, report_file)
    
    print("\nTo proceed with deduplication:")
    print("1. Review the report: css_duplication_report.txt")
    print("2. Backup created at: styles.css.backup")
    print("3. Manually merge duplicates or create an automated cleanup script")
