// =========================================================================
// FILE: categories.js
// PURPOSE: Category filtering and selection logic
// AUTHOR: Jack Ewers / bloodweb.net
// LAST UPDATED: 2025-05-07
// =========================================================================


import { getItem } from './helpers.js';
import { HTTP_modify } from './http.js';
import { App } from './index.js';
import { getProject } from './projects.js';
import { makeMainInputs, setExpandedItem, toggleDisableAED, } from './ui.js';

// =========================================================================
//    CATEGORY   FUNCTIONS
//
// . deleteCategories()        - Deletes a category from the list
// . renderCategoriesView()    - Renders the category management view
// . setcurrentCategoryID()      - Sets the current category and updates the UI
// . renderCategoryBindings()  - Renders the category binding UI
// =========================================================================

export function getCategory(category,by){
  return App.state.categories.find(c => c[by] === category) || null;
}
export function getCategoryNameById(id) {
  return App.state.categories.find(c => c.id == id)?.name || '[Unknown]';
}

export function setExpandedCategory(item=false,categoryName) {
  // if no item(false), remove the expanded category from local storage
  // always remove expanded class from all items
  document.querySelectorAll('.category-list-item.expanded').forEach(el => el.classList.remove('expanded'));
      
  if (item === false) {
    App.state.category.expandedCategory = null;
    localStorage.removeItem('expandedCategory');
    return;
  }

  item.classList.add('expanded');
  App.state.category.expandedCategory = categoryName;
  localStorage.setItem('expandedCategory', categoryName);
}

export function deleteCategories(categoryId) {  
  // if no category IDs are provided, 
  if (!categoryIs || categoryId.length === 0) { debug('No category IDs provided for deletion', { importance: 1, category: 'data' }); }
  // Find the category to delete ( confirm it exists )
  const category = App.state.categories.find(c => c.id === categoryId);
  if (!category) { debug(`Category with ID ${categoryId} not found`, { importance: 1, category: 'data' }); }
  // Check if the category is bound to any projects

  const successMessage = generateSuccessMessage('Category deleted', `Category '${category.name}' has been deleted successfully.`. false);

  // Send delete request to server
  HTTP_modify('DELETE_CATEGORIES', { id: categoryId }, successMessage);
}

export function renderCategoriesView() {

  const wrapper = createElement('div', { className: 'category-view' });
  const header = createElement('h2', { innerText: 'Category Management' });
  const list = createElement('ul', { className: 'expansion-list' });

  
  App.state.categories.forEach(cat => {
    /// Create items
    list.append(createCategoryElements(cat))
  });

  wrapper.append(header, list );
  makeMainInputs(header,'categories');
  App.element.ViewContent.appendChild(wrapper);
}

function createCategoryElements(cat) {
  // Create the list item
  const item = createElement('li', { className: 'expansion-list-item', });
  // Base viewable section
  const headerRow = createElement('div', { className: 'expansion-header' });
  const nameSpan = createElement('span', {className: 'expansion-title', innerText: `${cat.name}`, });
  const toggleBtn = createElement('button', { className: 'toggle-expand', innerHTML: '<i class="fas fa-chevron-down"></i>', });
  const colorDot = createElement('span', { className: 'category-color-dot', style: `background-color:${cat.color || '#aaa'}`,});
  // Expandable section
  const expand = createElement('div', { className: 'expandable-section' });
  const deleteInline = createElement('button', { className: 'inline-delete', innerHTML: '<i class="fas fa-trash"></i> Delete',});
  const nameInput = createElement('input', { className: 'edit-category-name', value: cat.name,style: 'display: none;', });
  const emojiSpan = createElement('span', { className: 'category-emoji', innerText: cat.emoji || '❓', style: 'margin-right: 0.5rem;', });
  const emojiInput = createElement('input', { className: 'edit-category-emoji', value: cat.emoji || '', style: 'display:none; width: 3ch; margin-right: 0.5rem;',});

  headerRow.append(toggleBtn, nameSpan, colorDot);
  item.appendChild(headerRow);

  // Toggle logic
  item.addEventListener('click', (e) => {
    const isExpanded = item.classList.contains('expanded');
    const clickedToggle = e.target.closest('.toggle-expand');
    const clickedTitle = e.target.closest('.expansion-title');

    if (!isExpanded) {
      setExpandedItem(item,);
      toggleDisableAED(false,dQ(App.element.taskActionButtons.delete))
    } 
    else if (clickedToggle || clickedTitle) { // only close if arrow or title was clicked
      setExpandedItem(false);
      toggleDisableAED(true,dQ(App.element.taskActionButtons.delete));
    }
  });

  deleteInline.addEventListener('click', (e) => {
    if (!confirm(`Delete category "${cat.name}"?`)) return;
    HTTP_modify('DELETE_CATEGORIES', { ids: [cat.id] }, `Category "${cat.name}" deleted`)
  });

  expand.append( nameInput,emojiSpan,emojiInput,deleteInline);
  item.appendChild(expand);
  attachCategoryEditHandlers(cat, item, nameSpan, nameInput, emojiSpan, emojiInput);
  return item;
}

function attachCategoryEditHandlers(cat, item, nameSpan, nameInput, emojiSpan, emojiInput) {
  const commit = () => {
    const newName = nameInput.value.trim();
    const newEmoji = emojiInput.value.trim();
    const changed = newName !== cat.name || newEmoji !== cat.emoji;

    if (changed) {
      cat.name = newName;
      cat.emoji = newEmoji;

      nameSpan.innerText = newName;
      emojiSpan.innerText = newEmoji;

      HTTP_modify('EDIT_CATEGORIES', { ...cat }, `Updated category "${newName}"`);
    }

    nameSpan.style.display = 'inline-block';
    emojiSpan.style.display = 'inline-block';
    nameInput.style.display = 'none';
    emojiInput.style.display = 'none';

    App.state.editMode.categories = null;
    highlightEditButton(false);
  };

  nameInput.addEventListener('blur', commit);
  emojiInput.addEventListener('blur', commit);
  nameInput.addEventListener('keydown', e => { if (e.key === 'Enter') commit(); });
  emojiInput.addEventListener('keydown', e => { if (e.key === 'Enter') commit(); });
}


export function setCurrentCategoryID(categoryName, noRender = false) {
    if (!categoryName || typeof categoryName !== 'string') {
       debug(`Invalid category name provided to setcurrentCategoryID(): ${categoryName}`, { importance: 1, category: 'data' });}

    App.state.category.currentCategoryID = categoryName;
    localStorage.setItem('currentCategoryID', categoryName);

    debug(`Set current category to '${categoryName}'`, { importance: 4, category: 'data' });
    
    if (noRender) return;
   
}


//  CATEGORY INSERT FORM
export function renderCategoryBindings() {
    const { bindList, unbindList } = App.element.MainForm.categoryComponents;

    bindList.innerHTML = '';
    unbindList.innerHTML = '';

    App.state.projects.forEach(project => {
        const el = createElement('button',{type:'button',
          'data-project':project.name,
            innerHTML: `<span class="icon">${project.name}</span> <i class="bind-icon fas fa-link"></i>`, 
            className: 'bind-project-item',});

        project.name === getItem('projects',App.state.project.currentProjectID) ? bindList.appendChild(el) : unbindList.appendChild(el);

        el.addEventListener('click', () => {
            const isBound = el.parentElement === bindList;
            const targetList = isBound ? unbindList : bindList;
        
            // Update icon
            const icon = el.querySelector('.bind-icon');
            icon.className = `bind-icon fas fa-${isBound ? 'link' : 'unlink'}`;
        
            // Move + animate
            el.classList.remove('project-binding-animated'); // reset if re-clicked
            requestAnimationFrame(() => {
                el.classList.add('project-binding-animated');
                targetList.appendChild(el);
            });
        });
    })
};




function handleBindProject(categoryName, projectID, toBind) {
  const category = App.state.categories.find(c => c.name === categoryName);
  if (!category) return;

  let projectIDs = new Set(Array.isArray(category.projectIDs)
    ? category.projectIDs
    : (category.projectIDs || ''))//.split(',').map(k => parseInt(k)));

  toBind ? projectIDs.add(parseInt(projectID)) : projectIDs.delete(parseInt(projectID));

  const updatedArray = Array.from(projectIDs);
  category.projectIDs = updatedArray; // 🔥 Update the live category object

  console.log(`Updated IDs for ${categoryName}: ${updatedArray.join(',')}`);

  moveProjectBindButton(categoryName, projectID, toBind);

  HTTP_modify('EDIT_CATEGORIES', {
    ...category,
    projectIDs: updatedArray.join(','),
  }, false, true);
}


function moveProjectBindButton(categoryName, projectID, toBound) {
  const listItem = [...document.querySelectorAll('.category-list-item')].find(el =>
    el.querySelector('.category-title')?.innerText.includes(categoryName)
  );
  if (!listItem) debug(`Category list item for ${categoryName} not found`, { importance: 1 });

  const oldContainer = toBound ? '.project-unbind' : '.project-bind'
  const project = getItem('projects',projectID);

  const fromBtn = [...listItem.querySelectorAll(oldContainer)]
    .find(btn => btn.dataset.project === projectID);
  if (!fromBtn) debug(`Button for ${projectID} not found in ${oldContainer} section`, { importance: 1 });

  const boundContainer = listItem.querySelector('.bind-section div:nth-child(1)');
  const unboundContainer = listItem.querySelector('.bind-section div:nth-child(2)');
  const toContainer = toBound ? boundContainer : unboundContainer;

  const newBtn = fromBtn.cloneNode(true);
  newBtn.className = toBound ? 'project-bind fade-in' : 'project-unbind fade-in';
  newBtn.innerHTML = toBound
    ? `<i class="fas fa-link"></i> ${project.name}`
    : `<i class="fas fa-unlink"></i> ${project.name}`;

  newBtn.addEventListener('click', (e) => {
    e.stopPropagation();
    handleBindProject(categoryName, projectID, !toBound);
  });

  fromBtn.classList.add('fade-out');
  setTimeout(() => {
    fromBtn.remove();
    toContainer.appendChild(newBtn);
  }, 180);

  debug(`Moved button for ${project.name} to ${toBound?'':'un'}bound`, { importance: 3 });
}

