#!/bin/bash
# TaskList v2 Backup and Rollback Script
# 
# This script provides backup and rollback functionality for safe migration

set -e  # Exit on any error

# Configuration
MYSQL_USER="taskmanager_user"
MYSQL_PASS="BLOODWEBMYSQLX.X01"
MYSQL_HOST="localhost"
OLD_DB="bloodweb"
NEW_DB="tasklist_v2"
BACKUP_DIR="/var/www/html/bloodweb/app/tasklist/backups"
TIMESTAMP=$(date +"%Y%m%d_%H%M%S")

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# Functions
log() {
    echo -e "${GREEN}[$(date '+%Y-%m-%d %H:%M:%S')] $1${NC}"
}

warn() {
    echo -e "${YELLOW}[$(date '+%Y-%m-%d %H:%M:%S')] WARNING: $1${NC}"
}

error() {
    echo -e "${RED}[$(date '+%Y-%m-%d %H:%M:%S')] ERROR: $1${NC}"
}

create_backup_dir() {
    if [ ! -d "$BACKUP_DIR" ]; then
        mkdir -p "$BACKUP_DIR"
        log "Created backup directory: $BACKUP_DIR"
    fi
}

backup_old_schema() {
    log "Creating backup of current TaskList data..."
    
    local backup_file="$BACKUP_DIR/tasklist_backup_$TIMESTAMP.sql"
    
    # Backup the relevant tables from the old database
    mysqldump -h "$MYSQL_HOST" -u "$MYSQL_USER" -p"$MYSQL_PASS" \
        "$OLD_DB" \
        tasks projects categories \
        --single-transaction \
        --routines \
        --triggers \
        --complete-insert \
        --add-drop-table > "$backup_file"
    
    if [ $? -eq 0 ]; then
        log "Backup created successfully: $backup_file"
        
        # Compress the backup
        gzip "$backup_file"
        log "Backup compressed: ${backup_file}.gz"
        
        # Create a metadata file
        cat > "$BACKUP_DIR/backup_${TIMESTAMP}_info.txt" << EOF
Backup Information
==================
Date: $(date)
Original Database: $OLD_DB
Tables: tasks, projects, categories
Backup File: ${backup_file}.gz
MySQL Version: $(mysql --version)
EOF
        
        return 0
    else
        error "Backup failed!"
        return 1
    fi
}

create_new_database() {
    log "Creating new TaskList v2 database..."
    
    # Create the new database
    mysql -h "$MYSQL_HOST" -u "$MYSQL_USER" -p"$MYSQL_PASS" << EOF
CREATE DATABASE IF NOT EXISTS $NEW_DB;
USE $NEW_DB;
SOURCE /var/www/html/bloodweb/app/tasklist/v2_schema.sql;
EOF
    
    if [ $? -eq 0 ]; then
        log "New database created successfully"
        return 0
    else
        error "Failed to create new database"
        return 1
    fi
}

run_migration() {
    log "Running data migration..."
    
    cd /var/www/html/bloodweb/app/tasklist/
    
    # Run migration script
    php migrate_data.php --verbose
    
    if [ $? -eq 0 ]; then
        log "Migration completed successfully"
        return 0
    else
        error "Migration failed"
        return 1
    fi
}

test_new_schema() {
    log "Testing new database schema..."
    
    # Run basic tests
    local test_result=$(mysql -h "$MYSQL_HOST" -u "$MYSQL_USER" -p"$MYSQL_PASS" "$NEW_DB" << EOF
SELECT 
    (SELECT COUNT(*) FROM users) as user_count,
    (SELECT COUNT(*) FROM projects) as project_count,
    (SELECT COUNT(*) FROM categories) as category_count,
    (SELECT COUNT(*) FROM tasks) as task_count;
EOF
)
    
    if [ $? -eq 0 ]; then
        log "Schema test passed"
        echo "$test_result"
        return 0
    else
        error "Schema test failed"
        return 1
    fi
}

rollback_migration() {
    warn "Rolling back migration..."
    
    # Find the most recent backup
    local latest_backup=$(ls -t "$BACKUP_DIR"/tasklist_backup_*.sql.gz 2>/dev/null | head -n1)
    
    if [ -z "$latest_backup" ]; then
        error "No backup found to rollback to!"
        return 1
    fi
    
    log "Rolling back to: $latest_backup"
    
    # Decompress and restore
    gunzip -c "$latest_backup" | mysql -h "$MYSQL_HOST" -u "$MYSQL_USER" -p"$MYSQL_PASS" "$OLD_DB"
    
    if [ $? -eq 0 ]; then
        log "Rollback completed successfully"
        
        # Optionally drop the new database
        read -p "Drop the new database? (y/N): " -n 1 -r
        echo
        if [[ $REPLY =~ ^[Yy]$ ]]; then
            mysql -h "$MYSQL_HOST" -u "$MYSQL_USER" -p"$MYSQL_PASS" -e "DROP DATABASE IF EXISTS $NEW_DB;"
            log "New database dropped"
        fi
        
        return 0
    else
        error "Rollback failed!"
        return 1
    fi
}

list_backups() {
    log "Available backups:"
    
    if [ -d "$BACKUP_DIR" ]; then
        ls -la "$BACKUP_DIR"/tasklist_backup_*.sql.gz 2>/dev/null || echo "No backups found"
        echo
        ls -la "$BACKUP_DIR"/backup_*_info.txt 2>/dev/null || echo "No backup info files found"
    else
        warn "Backup directory does not exist"
    fi
}

cleanup_old_backups() {
    log "Cleaning up old backups (keeping last 5)..."
    
    cd "$BACKUP_DIR"
    
    # Keep only the 5 most recent backups
    ls -t tasklist_backup_*.sql.gz 2>/dev/null | tail -n +6 | xargs rm -f
    ls -t backup_*_info.txt 2>/dev/null | tail -n +6 | xargs rm -f
    
    log "Cleanup completed"
}

# Main command handling
case "${1:-help}" in
    "backup")
        create_backup_dir
        backup_old_schema
        ;;
    
    "migrate")
        create_backup_dir
        backup_old_schema || { error "Backup failed, aborting migration"; exit 1; }
        create_new_database || { error "Database creation failed"; exit 1; }
        run_migration || { error "Migration failed"; exit 1; }
        test_new_schema || { warn "Schema test failed, but migration may still be valid"; }
        log "Migration completed successfully!"
        ;;
    
    "rollback")
        rollback_migration
        ;;
    
    "test")
        test_new_schema
        ;;
    
    "list")
        list_backups
        ;;
    
    "cleanup")
        cleanup_old_backups
        ;;
    
    "help"|*)
        echo "TaskList v2 Backup and Migration Tool"
        echo "====================================="
        echo
        echo "Usage: $0 <command>"
        echo
        echo "Commands:"
        echo "  backup    - Create backup of current TaskList data"
        echo "  migrate   - Full migration process (backup + migrate + test)"
        echo "  rollback  - Rollback to the most recent backup"
        echo "  test      - Test the new database schema"
        echo "  list      - List available backups"
        echo "  cleanup   - Remove old backups (keep last 5)"
        echo "  help      - Show this help message"
        echo
        echo "Examples:"
        echo "  $0 backup           # Create a backup"
        echo "  $0 migrate          # Run full migration"
        echo "  $0 rollback         # Rollback if something went wrong"
        echo
        ;;
esac