/**
 * EventModal - Modular event creation/editing modal
 * 
 * Features:
 * - Add new events or edit existing ones
 * - Form fields: title, date, time, end time, description, location, color
 * - All-day event toggle
 * - Reminder settings
 * - Reusable across the application
 * 
 * @author Jack Ewers / BloodWeb
 * @version 1.0.0
 * @date 2025-11-18
 */

class EventModal {
    constructor(app) {
        this.app = app;
        this.modalId = 'event-modal';
        this.currentEvent = null;
        this.onSaveCallback = null;
    }

    /**
     * Show modal to create new event
     * @param {Date} defaultDate - Optional default date
     * @param {Function} onSave - Callback after successful save
     */
    showCreate(defaultDate = null, onSave = null) {
        this.currentEvent = null;
        this.onSaveCallback = onSave;
        const date = defaultDate || new Date();
        
        this.renderModal({
            title: '',
            event_date: this.formatDateForInput(date),
            event_time: '',
            end_time: '',
            is_all_day: false,
            description: '',
            location: '',
            color: '#3b82f6',
            reminder_minutes: 15
        });
        
        this.show();
    }

    /**
     * Show modal to edit existing event
     * @param {Object} event - Event object to edit
     * @param {Function} onSave - Callback after successful save
     */
    showEdit(event, onSave = null) {
        this.currentEvent = event;
        this.onSaveCallback = onSave;
        this.renderModal(event);
        this.show();
    }

    /**
     * Render modal HTML
     */
    renderModal(data) {
        const isEdit = this.currentEvent !== null;
        const modalTitle = isEdit ? 'Edit Event' : 'Create New Event';
        
        const html = `
            <div id="${this.modalId}" class="modal-overlay" style="display: none;">
                <div class="modal-container event-modal-container">
                    <div class="modal-header">
                        <h2 class="modal-title">
                            <i class="fas fa-calendar-${isEdit ? 'edit' : 'plus'}"></i>
                            ${modalTitle}
                        </h2>
                        <button class="btn-close-modal" id="close-event-modal">
                            <i class="fas fa-times"></i>
                        </button>
                    </div>
                    
                    <div class="modal-body">
                        <form id="event-form" class="event-form">
                            <!-- Title -->
                            <div class="form-group">
                                <label for="event-title" class="form-label required">
                                    <i class="fas fa-heading"></i>
                                    Event Title
                                </label>
                                <input 
                                    type="text" 
                                    id="event-title" 
                                    name="title"
                                    class="form-input" 
                                    placeholder="Enter event title"
                                    value="${this.escapeHtml(data.title || '')}"
                                    required
                                />
                            </div>

                            <!-- Date and Time Row -->
                            <div class="form-row">
                                <div class="form-group">
                                    <label for="event-date" class="form-label required">
                                        <i class="fas fa-calendar"></i>
                                        Date
                                    </label>
                                    <input 
                                        type="date" 
                                        id="event-date" 
                                        name="event_date"
                                        class="form-input" 
                                        value="${data.event_date || ''}"
                                        required
                                    />
                                </div>

                                <div class="form-group">
                                    <label for="all-day" class="form-label checkbox-label">
                                        <input 
                                            type="checkbox" 
                                            id="all-day" 
                                            name="is_all_day"
                                            ${data.is_all_day ? 'checked' : ''}
                                        />
                                        All Day Event
                                    </label>
                                </div>
                            </div>

                            <!-- Time Inputs -->
                            <div id="time-inputs" class="form-row" ${data.is_all_day ? 'style="display:none;"' : ''}>
                                <div class="form-group">
                                    <label for="event-time" class="form-label">
                                        <i class="fas fa-clock"></i>
                                        Start Time
                                    </label>
                                    <input 
                                        type="time" 
                                        id="event-time" 
                                        name="event_time"
                                        class="form-input" 
                                        value="${data.event_time || ''}"
                                    />
                                </div>

                                <div class="form-group">
                                    <label for="end-time" class="form-label">
                                        <i class="fas fa-clock"></i>
                                        End Time
                                    </label>
                                    <input 
                                        type="time" 
                                        id="end-time" 
                                        name="end_time"
                                        class="form-input" 
                                        value="${data.end_time || ''}"
                                    />
                                </div>
                            </div>

                            <!-- Description -->
                            <div class="form-group">
                                <label for="event-description" class="form-label">
                                    <i class="fas fa-align-left"></i>
                                    Description
                                </label>
                                <textarea 
                                    id="event-description" 
                                    name="description"
                                    class="form-textarea" 
                                    rows="3"
                                    placeholder="Add event details..."
                                >${this.escapeHtml(data.description || '')}</textarea>
                            </div>

                            <!-- Location -->
                            <div class="form-group">
                                <label for="event-location" class="form-label">
                                    <i class="fas fa-map-marker-alt"></i>
                                    Location
                                </label>
                                <input 
                                    type="text" 
                                    id="event-location" 
                                    name="location"
                                    class="form-input" 
                                    placeholder="Add location"
                                    value="${this.escapeHtml(data.location || '')}"
                                />
                            </div>

                            <!-- Color and Reminder Row -->
                            <div class="form-row">
                                <div class="form-group">
                                    <label for="event-color" class="form-label">
                                        <i class="fas fa-palette"></i>
                                        Color
                                    </label>
                                    <div class="color-picker-wrapper">
                                        <input 
                                            type="color" 
                                            id="event-color" 
                                            name="color"
                                            class="form-color-input" 
                                            value="${data.color || '#3b82f6'}"
                                        />
                                        <div class="color-presets">
                                            <button type="button" class="color-preset" data-color="#3b82f6" style="background: #3b82f6;" title="Blue"></button>
                                            <button type="button" class="color-preset" data-color="#22c55e" style="background: #22c55e;" title="Green"></button>
                                            <button type="button" class="color-preset" data-color="#f59e0b" style="background: #f59e0b;" title="Orange"></button>
                                            <button type="button" class="color-preset" data-color="#ef4444" style="background: #ef4444;" title="Red"></button>
                                            <button type="button" class="color-preset" data-color="#a855f7" style="background: #a855f7;" title="Purple"></button>
                                            <button type="button" class="color-preset" data-color="#ec4899" style="background: #ec4899;" title="Pink"></button>
                                        </div>
                                    </div>
                                </div>

                                <div class="form-group">
                                    <label for="event-reminder" class="form-label">
                                        <i class="fas fa-bell"></i>
                                        Reminder
                                    </label>
                                    <select id="event-reminder" name="reminder_minutes" class="form-select">
                                        <option value="0" ${data.reminder_minutes === 0 ? 'selected' : ''}>No reminder</option>
                                        <option value="5" ${data.reminder_minutes === 5 ? 'selected' : ''}>5 minutes before</option>
                                        <option value="15" ${data.reminder_minutes === 15 ? 'selected' : ''}>15 minutes before</option>
                                        <option value="30" ${data.reminder_minutes === 30 ? 'selected' : ''}>30 minutes before</option>
                                        <option value="60" ${data.reminder_minutes === 60 ? 'selected' : ''}>1 hour before</option>
                                        <option value="1440" ${data.reminder_minutes === 1440 ? 'selected' : ''}>1 day before</option>
                                    </select>
                                </div>
                            </div>
                        </form>
                    </div>
                    
                    <div class="modal-footer">
                        <button type="button" class="btn-secondary" id="cancel-event-btn">
                            <i class="fas fa-times"></i>
                            Cancel
                        </button>
                        <button type="submit" form="event-form" class="btn-primary" id="save-event-btn">
                            <i class="fas fa-save"></i>
                            ${isEdit ? 'Update Event' : 'Create Event'}
                        </button>
                    </div>
                </div>
            </div>
        `;

        // Remove existing modal if present
        const existingModal = document.getElementById(this.modalId);
        if (existingModal) {
            existingModal.remove();
        }

        // Add to body
        document.body.insertAdjacentHTML('beforeend', html);
        
        // Attach event listeners
        this.attachEventListeners();
    }

    /**
     * Attach event listeners to modal elements
     */
    attachEventListeners() {
        const modal = document.getElementById(this.modalId);
        const form = document.getElementById('event-form');
        const closeBtn = document.getElementById('close-event-modal');
        const cancelBtn = document.getElementById('cancel-event-btn');
        const allDayCheckbox = document.getElementById('all-day');
        const timeInputs = document.getElementById('time-inputs');
        
        // Close modal
        closeBtn?.addEventListener('click', () => this.hide());
        cancelBtn?.addEventListener('click', () => this.hide());
        modal?.addEventListener('click', (e) => {
            if (e.target === modal) this.hide();
        });

        // All-day toggle
        allDayCheckbox?.addEventListener('change', (e) => {
            if (e.target.checked) {
                timeInputs.style.display = 'none';
            } else {
                timeInputs.style.display = 'flex';
            }
        });

        // Color presets
        document.querySelectorAll('.color-preset').forEach(btn => {
            btn.addEventListener('click', (e) => {
                const color = e.currentTarget.dataset.color;
                document.getElementById('event-color').value = color;
            });
        });

        // Form submission
        form?.addEventListener('submit', (e) => {
            e.preventDefault();
            this.handleSubmit(form);
        });
    }

    /**
     * Handle form submission
     */
    async handleSubmit(form) {
        const formData = new FormData(form);
        const data = {
            title: formData.get('title'),
            event_date: formData.get('event_date'),
            event_time: formData.get('event_time'),
            end_time: formData.get('end_time'),
            is_all_day: formData.get('is_all_day') ? 1 : 0,
            description: formData.get('description'),
            location: formData.get('location'),
            color: formData.get('color'),
            reminder_minutes: parseInt(formData.get('reminder_minutes')) || 0
        };

        try {
            let response;
            
            if (this.currentEvent) {
                // Update existing event
                response = await this.app.api.put(`calendar/${this.currentEvent.id}`, data);
            } else {
                // Create new event
                response = await this.app.api.post('calendar/events', data);
            }

            if (response.success) {
                this.app.modals.showToast(
                    this.currentEvent ? 'Event updated successfully' : 'Event created successfully',
                    'success'
                );
                
                // Call callback if provided
                if (this.onSaveCallback) {
                    this.onSaveCallback(response.data);
                }
                
                this.hide();
            } else {
                this.app.modals.showToast('Failed to save event', 'error');
            }
        } catch (error) {
            logger.error('[EventModal] Failed to save event:', error);
            this.app.modals.showToast('An error occurred while saving', 'error');
        }
    }

    /**
     * Show modal
     */
    show() {
        const modal = document.getElementById(this.modalId);
        if (modal) {
            modal.style.display = 'flex';
            // Focus first input
            setTimeout(() => {
                document.getElementById('event-title')?.focus();
            }, 100);
        }
    }

    /**
     * Hide modal
     */
    hide() {
        const modal = document.getElementById(this.modalId);
        if (modal) {
            modal.style.display = 'none';
        }
    }

    /**
     * Helper: Format date for input (YYYY-MM-DD)
     */
    formatDateForInput(date) {
        const year = date.getFullYear();
        const month = String(date.getMonth() + 1).padStart(2, '0');
        const day = String(date.getDate()).padStart(2, '0');
        return `${year}-${month}-${day}`;
    }

    /**
     * Helper: Escape HTML to prevent XSS
     */
    escapeHtml(text) {
        if (!text) return '';
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
}

// Export for use in main app
if (typeof module !== 'undefined' && module.exports) {
    module.exports = EventModal;
}
