// =========================================================================
// FILE: AppManager.js
// PURPOSE: Global application state and constants
// AUTHOR: Jack Ewers / bloodweb.net
// LAST UPDATED: 2025-05-07
// =========================================================================



/* 
╔═══════════════════════════════════════════════════════════════════════════╗
║                               DESCRIPTION                                 ║
║ - Manages global state: tasks, projects, current selections, app flags    ║
║ - Each project contains its own categories                                ║
║ - Tasks reference projectId and categoryId                                ║
╚═══════════════════════════════════════════════════════════════════════════╝
*/

export class AppManager {
    static bloodweb_tasklist = 'https://bloodweb.net/TaskList/';
    constructor() {
        this.version = '0.0.1';
        this.isLive = false;

        this.formMethods = [
            'GET_TASKS', 'GET_PROJECTS', 'GET_CATEGORIES',
            'ADD_TASKS', 'ADD_PROJECTS', 'ADD_CATEGORIES',
            'EDIT_TASKS', 'EDIT_PROJECTS', 'EDIT_CATEGORYS',
            'DELETE_TASKS', 'DELETE_PROJECTS', 'DELETE_CATEGORIES',
            'USER_LOGIN', 'USER_LOGOUT',
            'REGISTER_USER', 'LOG_XP',
        ];
      
        

        this.element = {
            // Page Level Elements
            ViewContent: document.getElementById('view-content'),
            H1: document.querySelector('h1'),
            TaskList: document.getElementById('task-list'),
            TaskView: '.task-view',

            viewtabs:document.querySelectorAll('.view-tab'),
            // Category dropdowns
            CategorySelect: [
                
                document.getElementById('categorySwap'),
                document.getElementById('categoryInsert')
            ],

            taskActionButtons:{
                add:'#Aed-Button',
                edit:'#aEd-Button',
                delete:'#aeD-Button',
                save:'#Task-Save-Button',
                cancel:'#Task-Cancel-Button',
            },

            // TaskElements in TaskVIEW
            taskElements:{
                wrapperID:'',

                containerCLASS:'task-item',
                taskidCLASS:'.TL_id',
                taskheaderCLASS:'.task-label',
                taskCategoryCLASS:'.task-category',
                taskProjectCLASS:'.task-project',
                taskDaysCLASS:'.task-days',
            },

            quickAddCategory: document.getElementById('QuickAddCategory'),
            // Main Form (Task/Category)
            MainForm: {
                form: document.getElementById('MainForm'),
                id: document.querySelector('#MainForm #mainId'),
                method: document.querySelector('#MainForm #mainMethod'),
                title: document.querySelector('#MainForm #MainFormTitle'),
                text: document.querySelector('#MainForm #MainText-Input'),
                label: document.querySelector('#MainForm #MainText-Label'),

                formCloseBtn: document.querySelector('#MainForm #closeMainForm'),
                formSubmitBtn: document.querySelector('#MainForm #submitMainForm'),

                taskComponents: {
                    wrapper: document.querySelector('#MainForm #taskSpecificComponents'),
                    description: document.querySelector('#MainForm #MainText-Description'),
                    daysToComplete: document.querySelector('#MainForm #daysToComplete'),
                    categorySelect: document.querySelector('#MainForm #categoryInsert')
                },

                categoryComponents: {
                    wrapper: document.querySelector('#MainForm #categorySpecificComponents'),
                    emoji: document.querySelector('#MainForm #selectedEmoji'),
                    color: document.querySelector('#MainForm #categoryColor'),
                    categoryBind: document.querySelector('#MainForm #categoryBind'),
                    bindList: document.querySelector('#MainForm #projectBind'),
                    unbindList: document.querySelector('#MainForm #projectUnbind')
                },

                projectComponents: {
                    wrapper: document.querySelector('#MainForm #projectSpecificComponents'),
               },

                allOptionalFields: [
                    document.querySelector('#MainForm #taskSpecificComponents'),
                    document.querySelector('#MainForm #categorySpecificComponents')
                ]
            },

            categorySelectElements: document.querySelectorAll('.category-select'),

            // Sidebar
            projectSideMenu: document.getElementById('projectSideMenu'),
            ProjectList: document.getElementById('projectList'),
            ProjectListToggle: document.getElementById('Left-SideMenu-Toggle'),
            projectideMenuToggle: document.getElementById('close-side-menu'),
            projectSideMenuToggles: document.querySelectorAll('.side-menu-toggle'),

        };

        this.state = {
          
            // Tasks, Categories, Projects ( Main Data )
            tasks: [],
            categories: {},  
            projects: [],

            activeView: localStorage.getItem('activeView')||'tasks', // Default view
            // Item is expanded or selected
            
            
            currentProject: localStorage.getItem('currentProject')||null,
            selectedCategoryId: null,

            notificationIsOpen: false,


            task:{
                selectedTasks:JSON.parse(localStorage.getItem('selectedTasks'))||[],    // Array of selected task IDs
                expandedTask: JSON.parse(localStorage.getItem('task.expandedTask')) || [], // ID of the expanded task
                changesToEdit:false,
                
                // Counters
                counter: 0,         // count the total number of tasks
                daysUrgencyCount: { // Count of tasks in each urgency category
                    alert: 0, 
                    high: 0, 
                    medium: 0, 
                    low: 0, 
                },
                // Settings for task management
                default_days: 30,    // Default days to complete
                daysUrgency:{
                    alert: 0, // ON FIRE // DO NOW
                    high: 10, // Running out of time
                    medium: 30, // Moderate urgency
                    low: null, // Low urgency
                },

                single_expansion: localStorage.getItem('single_expansion') === 'true' || true, // Collapse task on expand
                multi_edit: localStorage.getItem('multi_edit') === 'true' || false, // Is multi-edit mode active?
            },
            
            category:{
                counter: 0,         // count the total number of categories
                currentCategoryID: localStorage.getItem('currentCategoryID')||null,
                expandedCategory:null||localStorage.getItem('expandedCategory'), // ID of the expanded category
                
                // Last expanded category
                selectedCategories: [], // Array of selected category IDs
            },
            project:{
                counter: 0,         // count the total number of projects
                currentProjectID: localStorage.getItem('currentProjectID')||1,
                expandedProject: localStorage.getItem('expandedProject')||null, // ID of the expanded project
                selectedProjects: [], // Array of selected project IDs
            },

            

            readyToDelete: false, // Are we ready to delete items? ( Tasks || Projects selected...Category Expanaded?)

            isFirstRun: localStorage.getItem('isFirstRun')||true, // Is this the first run of the app? 

            categoryCounter: 0,
            
            
            editMode: {
                task: false, // Is the task edit mode active?
                project: false, // Is the project edit mode active?
                category: false // Is the category edit mode active?
            },
            filterActive: false,
            savePending: false,
            loadPending: false,
            timeoutHandle: null,
        
       
        };


        // Deprecated settings, kept for compatibility ( do not use in new code )
        // Use state.settings instead
        this.settings = {
            possibleViews: ['tasks', 'projects', 'categories'],
            singularViews: ['task', 'project', 'category'],

            defaultView: 'tasks', // Default view
            defaultCategory: 'General', // Default category
            defaultProject: 'Default', // Default project
            defaultCategoryColor: '#ffffff', // Default category color
            defaultCategoryEmoji: '📁', // Default category emoji

            colorIcon://ball emoji:
            '🔵', // Default color icon for categories

            // These variables override the default values if set
            single_expansion: null, // (tasks,projects,categories)
            multi_edit: null, // (tasks,projects,categories)

            task:{
                default_days:30,    // default days to complete
                urgency:{           // Urgency of the tasks
                    high:5,
                    medium:20,
                    low:30,      
                },
                single_expansion: true, // Collapse task on expand
                multi_edit: false, // Is multi-edit mode active?
            },

            sideMenuIsOpen:false, // Is the side menu open?
            sideMenuWidth:Math.min(document.body.clientWidth / 3 * 2, 400), // Default width of the side menu


            taskEditModeActive: false,
            showAllCategories: localStorage.getItem('showAllCategories') === 'true',
            projectSideMenuOpen: localStorage.getItem('projectSideMenuOpen') === 'true',
            closeNavOnProjectChange: true,
            projectEditModeActive: false,
            darkModeActive: false,
            editModeActive: false,
            anyKeyPressClosesNotification: true,
           
            debugging: {
                tasksInit: false,
                categoriesInit: false,
                projectsInit: false
            }
        };
    }

    updateStateValue(path, newValue) {
        const keys = path.split('.');
        let target = this.state;

        // Walk down to the final object
        for (let i = 0; i < keys.length - 1; i++) {
            if (!(keys[i] in target)) {
                console.warn(`[updateStateValue] Invalid path: '${keys[i]}' in '${path}'`);
                return;
            }
            target = target[keys[i]];
        }

        const finalKey = keys[keys.length - 1];
        if (!(finalKey in target)) {
            console.warn(`[updateStateValue] Final key '${finalKey}' not found in '${path}'`);
            return;
        }

        // Update in-memory state
        target[finalKey] = newValue;

        // Save using exact path as localStorage key
        const serialized = typeof newValue === 'string' ? newValue : JSON.stringify(newValue);
        if (newValue === null) { localStorage.removeItem(path); }
        else {  
            localStorage.setItem(path, serialized);
        }
        debug(`Updated App.state.${path} and saved to localStorage`, { value: newValue, category: 'data', importance: 4 });
    }




    dumpElements() {
        console.log('----- Stored DOM Elements -----');
        console.log(this.element);
        console.log('-------------------------------');
    }
    dumpState() {
        console.log('----- App.state -----');
        console.log(this.state);
        console.log('----------------------');
    }
    dump(values='all',type='log') {
        const VALIDARGS = ['elements','state','all'];
        console.log(`!----- App.state Dump (${type})-----!`);
        switch (values) {
            case 'elements':
                this.dumpElements();
                break;
            case 'state':
                this.dumpState();
                break;
            case 'all':
                this.dumpElements();
                this.dumpState();
                break;
            default:
                console.warn(`Invalid dump type: ${values}. Valid types are: ${VALIDARGS.join(', ')}`);
        }
        console.log('!--------------------------!');
    }
}  