/**
 * Task Manager — FocusFlow - Main Application Module
 * 
 * A modern, consolidated JavaScript application for task management
 * with gamification, responsive design, and efficient API integration.
 * 
 * @author Jack Ewers / BloodWeb
 * @version 2.1.0
 * @date 2025-12-27
 */

// =============================================================================
// VERSION MANAGEMENT
// =============================================================================

let APP_VERSION = null;
let CHANGELOG = null;

/**
 * Load version data from version.json
 */
async function loadVersionData() {
    try {
        const response = await fetch('./version.json?v=' + Date.now());
        const data = await response.json();
        
        APP_VERSION = {
            version: data.version,
            releaseDate: data.releaseDate,
            codename: data.codename,
            buildNumber: data.buildNumber,
            name: data.name,
            fullName: data.fullName
        };
        
        CHANGELOG = data.changelog;
        
        return data;
    } catch (error) {
        logger.error('Failed to load version data:', error);
        // Fallback version
        APP_VERSION = {
            version: '2.1.0',
            releaseDate: '2025-12-27',
            codename: 'FocusFlow',
            buildNumber: 'b2025.12.27',
            name: 'Task Manager',
            fullName: 'Task Manager — FocusFlow'
        };
        CHANGELOG = [];
        return APP_VERSION;
    }
}

/**
 * Get current version info
 */
function getVersion() {
    return APP_VERSION;
}

/**
 * Get full changelog
 */
function getChangelog() {
    return CHANGELOG;
}

/**
 * Get latest version entry
 */
function getLatestVersion() {
    return CHANGELOG ? CHANGELOG[0] : null;
}

/**
 * Get version by number
 */
function getVersionInfo(version) {
    return CHANGELOG ? CHANGELOG.find(v => v.version === version) : null;
}

// =============================================================================
// LOGGING CONFIGURATION
// =============================================================================

const LOG_LEVEL = {
    NONE: 0,
    ERROR: 1,
    WARN: 2,
    INFO: 3,
    DEBUG: 4
};

// Set to INFO for production, DEBUG for development
const CURRENT_LOG_LEVEL = LOG_LEVEL.ERROR;

const logger = {
    error: (...args) => {
        if (CURRENT_LOG_LEVEL >= LOG_LEVEL.ERROR) console.error(...args);
    },

    warn: (...args) => {
        if (CURRENT_LOG_LEVEL >= LOG_LEVEL.WARN) console.warn(...args);
    },

    info: (...args) => {
        if (CURRENT_LOG_LEVEL >= LOG_LEVEL.INFO) console.log(...args);
    },
    debug: (...args) => {
        if (CURRENT_LOG_LEVEL >= LOG_LEVEL.DEBUG) console.log(...args);
    },
    log: (...args) => {
        if (CURRENT_LOG_LEVEL >= LOG_LEVEL.DEBUG) console.log(...args);
    }
};

// Make logger globally available for all modules
window.logger = logger;

// =============================================================================
// CONFIGURATION & CONSTANTS
// =============================================================================

const CONFIG = {
    API_BASE_URL: window.location.origin + '/TaskList/api/',
    VERSION: '2.1.0', // Will be updated after version loads
    STORAGE_PREFIX: 'tasklist_v2_',
    ROUTES: {
        tasks: 'tasks',
        projects: 'projects', 
        categories: 'categories',
        users: 'users',
        xp: 'xp',
        analytics: 'analytics',
        search: 'search'
    },
    PRIORITY_COLORS: {
        low: '#10b981',
        medium: '#f59e0b', 
        high: '#ef4444',
        urgent: '#dc2626'
    },
    XP_VALUES: {
        task_complete: 10,
        task_create: 5,
        project_create: 15,
        daily_login: 5
    },
    // Default filter options
    FILTER_OPTIONS: {
        status: [
            { value: 'all', label: 'All Status' },
            { value: 'pending', label: 'Pending' },
            { value: 'in_progress', label: 'In Progress' },
            { value: 'completed', label: 'Completed' }
        ],
        priority: [
            { value: 'all', label: 'All Priorities' },
            { value: 'urgent', label: 'Urgent' },
            { value: 'high', label: 'High' },
            { value: 'medium', label: 'Medium' },
            { value: 'low', label: 'Low' }
        ]
    },
    // Default values for forms and filters
    DEFAULTS: {
        filters: {
            projects: [],
            categories: [],
            statuses: ['pending'],
            priorities: []
        },
        task: {
            priority: 'medium',
            complexity: 'none',
            dueDays: 1
        },
        project: {
            color: '#2563eb',
            icon: 'fas fa-folder'
        },
        category: {
            color: '#2563eb'
        },
        pagination: {
            tasksPerPage: 25
        },
        virtualScrolling: {
            enabled: true,
            threshold: 50, // Enable for lists with 50+ items
            batchSize: 20
        },
        notifications: {
            reminderTime: 60, // minutes
            browserNotifications: true,
            taskReminders: true,
            soundNotifications: false
        }
    },
    // UI Configuration
    UI: {
        sidebar: {
            autoHideOnMobile: true,
            defaultOpen: true
        },
        theme: {
            defaultFontSize: 'medium',
            reducedMotion: false
        },
        toasts: {
            duration: 5000,
            autoRemove: true
        },
        keyboard: {
            enabled: true,
            shortcuts: {
                'n': 'newTask',
                't': 'viewTasks',
                'd': 'viewDashboard',
                'p': 'newProject',
                'c': 'newCategory',
                '/': 'focusSearch',
                'Escape': 'closeModal',
                'ctrl+s': 'save',
                'ctrl+k': 'commandPalette',
                '?': 'showHelp'
            }
        }
    }
};

// Make CONFIG globally available
window.CONFIG = CONFIG;

// Get current user from PHP session
// CURRENT_USER is now defined in index.php
// const CURRENT_USER = window.BLOODWEB_USER || { id: 3 };

// =============================================================================
// APPLICATION STATE MANAGEMENT
// =============================================================================

class AppState {
    constructor() {
        this.data = {
            user: null,
            tasks: [],
            projects: [],
            categories: [],
            currentView: 'dashboard',
            selectedProject: null,
            selectedTasks: [],
            filters: {
                projects: CONFIG.DEFAULTS.filters.projects || [],
                categories: CONFIG.DEFAULTS.filters.categories || [],
                statuses: CONFIG.DEFAULTS.filters.statuses || [],
                priorities: CONFIG.DEFAULTS.filters.priorities || []
            },
            ui: {
                sidebarOpen: false,
                loading: false,
                error: null
            }
        };
        
        this.loadFromStorage();
    }

    get(key) {
        return key.split('.').reduce((obj, k) => obj?.[k], this.data);
    }

    set(key, value) {
        const keys = key.split('.');
        const lastKey = keys.pop();
        const target = keys.reduce((obj, k) => obj[k] = obj[k] || {}, this.data);
        target[lastKey] = value;
        this.saveToStorage();
        this.notifySubscribers(key, value);
    }

    update(key, updater) {
        const current = this.get(key);
        const updated = typeof updater === 'function' ? updater(current) : updater;
        this.set(key, updated);
    }

    subscribe(key, callback) {
        this.subscribers = this.subscribers || {};
        this.subscribers[key] = this.subscribers[key] || [];
        this.subscribers[key].push(callback);
    }

    notifySubscribers(key, value) {
        if (!this.subscribers?.[key]) return;
        this.subscribers[key].forEach(callback => callback(value, key));
    }

    loadFromStorage() {
        try {
            const stored = localStorage.getItem(`${CONFIG.STORAGE_PREFIX}state`);
            if (stored) {
                const parsed = JSON.parse(stored);
                this.data = { ...this.data, ...parsed };
            }
        } catch (error) {
            logger.warn('Failed to load state from storage:', error);
        }
    }

    saveToStorage() {
        try {
            const toStore = {
                selectedProject: this.data.selectedProject,
                filters: this.data.filters,
                ui: { sidebarOpen: this.data.ui.sidebarOpen }
            };
            localStorage.setItem(`${CONFIG.STORAGE_PREFIX}state`, JSON.stringify(toStore));
        } catch (error) {
            logger.warn('Failed to save state to storage:', error);
        }
    }
}

// =============================================================================
// API CLIENT
// =============================================================================

class APIClient {
    constructor(baseUrl) {
        this.baseUrl = baseUrl;
        this.cache = new Map();
        this.pending = new Map();
        
        // Initialize API cache with 60s TTL and localStorage persistence
        this.apiCache = new APICache({
            ttl: 60000, // 60 seconds
            maxSize: 100,
            storage: 'localStorage',
            prefix: 'TL_api_'
        });
        
        // Define cacheable endpoints
        this.cacheableEndpoints = [
            'tasks',
            'projects',
            'categories',
            'analytics',
            'achievements',
            'settings'
        ];
    }

    async request(endpoint, options = {}) {
        // Build URL based on API's path-based routing
        let url = this.baseUrl;
        
        // Handle query parameter style endpoint to path style
        if (options.params && options.params.endpoint) {
            url += options.params.endpoint;
            delete options.params.endpoint;
            
            // Add test parameter if needed
            if (options.params.test || options.testMode) {
                url += '?test=1';
                delete options.params.test;
            }
            
            // Add other parameters
            if (Object.keys(options.params).length > 0) {
                const separator = url.includes('?') ? '&' : '?';
                url += separator + new URLSearchParams(options.params).toString();
            }
        } else {
            // Direct endpoint name provided
            url += endpoint;
            
            // Only add test parameter if explicitly requested or if user is development user
            if (options.testMode || (CURRENT_USER.id === 3 && options.testMode !== false)) {
                url += '?test=1';
            }
            
            if (options.params && Object.keys(options.params).length > 0) {
                const separator = url.includes('?') ? '&' : '?';
                url += separator + new URLSearchParams(options.params).toString();
            }
        }

        const config = {
            method: options.method || 'GET',
            credentials: 'include', // Include cookies for session authentication
            headers: {
                'Content-Type': 'application/json',
                ...options.headers
            },
            ...options
        };

        if (options.body && config.method !== 'GET') {
            config.body = JSON.stringify(options.body);
        }

        // Check cache for GET requests (unless bypassCache is set)
        if (config.method === 'GET' && !options.bypassCache) {
            const cacheKey = endpoint + (options.params ? JSON.stringify(options.params) : '');
            const isCacheable = this.cacheableEndpoints.some(e => endpoint.includes(e));
            
            if (isCacheable) {
                const cachedData = this.apiCache.get(endpoint, options.params || {});
                if (cachedData) {
                    logger.debug(`[APICache] Cache HIT: ${endpoint}`);
                    return cachedData;
                }
                logger.debug(`[APICache] Cache MISS: ${endpoint}`);
            }
        }

        // Debug logging
        logger.debug(`API Request: ${config.method} ${url}`);

        let response;
        try {
            response = await fetch(url, config);
            
            // Handle 204 No Content responses (like DELETE operations)
            if (response.status === 204) {
                return { success: true };
            }

            const data = await response.json();
            
            if (!response.ok) {
                // Extract detailed error message from API response
                const errorMessage = data.error?.message || data.message || `HTTP ${response.status}: ${response.statusText}`;
                throw new Error(errorMessage);
            }
            
            if (data.success === false) {
                throw new Error(data.error?.message || data.message || 'API request failed');
            }

            // Cache successful GET responses
            if (config.method === 'GET') {
                const isCacheable = this.cacheableEndpoints.some(e => endpoint.includes(e));
                if (isCacheable) {
                    this.apiCache.set(endpoint, options.params || {}, data);
                }
            }

            // Invalidate cache on mutations
            if (['POST', 'PUT', 'DELETE'].includes(config.method)) {
                this.invalidateRelatedCache(endpoint);
            }

            return data;
        } catch (error) {
            // If it's a JSON parsing error and response wasn't ok, provide fallback error
            if (error instanceof SyntaxError && response && !response.ok) {
                logger.error(`API Error (${endpoint}):`, error);
                throw new Error(`HTTP ${response.status || 'Unknown'}: ${response.statusText || 'Request failed'}`);
            }
            
            logger.error(`API Error (${endpoint}):`, error);
            throw error;
        }
    }

    /**
     * Invalidate related cache entries on mutations
     * @param {string} endpoint - Endpoint that was mutated
     */
    invalidateRelatedCache(endpoint) {
        const invalidationMap = {
            'tasks': ['tasks', 'analytics', 'projects', 'categories'],
            'projects': ['projects', 'tasks', 'analytics'],
            'categories': ['categories', 'tasks', 'analytics'],
            'achievements': ['achievements', 'analytics'],
            'settings': ['settings']
        };

        // Find related endpoints to invalidate
        for (const [key, related] of Object.entries(invalidationMap)) {
            if (endpoint.includes(key)) {
                related.forEach(relatedEndpoint => {
                    this.apiCache.invalidate(relatedEndpoint);
                    logger.debug(`[APICache] Invalidated: ${relatedEndpoint}`);
                });
                break;
            }
        }
    }

    /**
     * Clear entire API cache
     */
    clearCache() {
        this.apiCache.clear();
    }

    /**
     * Get cache metrics
     */
    getCacheMetrics() {
        return this.apiCache.getMetrics();
    }

    // CRUD Operations
    async get(endpoint, params = {}) {
        return this.request(endpoint, { params });
    }

    async post(endpoint, body, params = {}) {
        return this.request(endpoint, { 
            method: 'POST', 
            body, 
            params
        });
    }

    async put(endpoint, body, params = {}) {
        return this.request(endpoint, { 
            method: 'PUT', 
            body, 
            params
        });
    }

    async delete(endpoint, params = {}) {
        return this.request(endpoint, { 
            method: 'DELETE', 
            params
        });
    }

    // Task operations
    async getTasks(filters = {}) {
        // Request enough tasks to account for filtering
        // We request more than the display limit so that after filtering
        // we still have a good number to show
        if (!filters.per_page) {
            filters.per_page = 50; // Request 50 tasks to ensure we have ~20 after filters
        }
        return this.get('tasks', filters);
    }

    async getTask(id) {
        return this.get(`tasks/${id}`);
    }

    async createTask(task) {
        return this.post('tasks', task);
    }

    async updateTask(id, updates) {
        return this.put(`tasks/${id}`, updates);
    }

    async deleteTask(id) {
        return this.delete(`tasks/${id}`);
    }

    async completeTask(id) {
        return this.post(`tasks/${id}/complete`, {});
    }

    async getSubtasks(taskId) {
        return this.get(`tasks/${taskId}/subtasks`);
    }

    async createSubtask(taskId, subtask) {
        return this.post(`tasks/${taskId}/subtasks`, subtask);
    }

    async updateSubtask(taskId, subtaskId, updates) {
        return this.put(`tasks/${taskId}/subtasks/${subtaskId}`, updates);
    }

    async deleteSubtask(taskId, subtaskId) {
        return this.delete(`tasks/${taskId}/subtasks/${subtaskId}`);
    }

    async toggleSubtask(taskId, subtaskId) {
        return this.post(`tasks/${taskId}/subtasks/${subtaskId}/toggle`, {});
    }

    // Project operations
    async getProjects() {
        return this.get('projects');
    }

    async createProject(project) {
        return this.post('projects', project);
    }

    async updateProject(id, updates) {
        return this.put(`projects/${id}`, updates);
    }

    async deleteProject(id) {
        return this.delete(`projects/${id}`);
    }

    // Category operations
    async getCategories() {
        return this.get('categories');
    }

    async createCategory(category) {
        return this.post('categories', category);
    }

    async updateCategory(id, updates) {
        return this.put(`categories/${id}`, updates);
    }

    async deleteCategory(categoryId) {
        return this.delete(`categories/${categoryId}`);
    }

    // Project-Category operations
    async getProjectCategories(projectId) {
        return this.get(`projects/${projectId}/categories`);
    }

    async updateProjectCategories(projectId, categoryIds) {
        return this.put(`projects/${projectId}/categories`, { category_ids: categoryIds });
    }

    async updateCategoryProjects(categoryId, projectIds) {
        return this.put(`categories/${categoryId}/projects`, { project_ids: projectIds });
    }

    // User settings operations
    async getUserSettings() {
        return this.get('user/settings');
    }

    async getUserSettings() {
        const response = await this.get('user/settings');
        // API returns {success: true, data: {...settings...}}
        return response.data || response;
    }

    async updateUserSettings(settings) {
        return this.post('user/settings', settings);
    }

    // Account overview operations
    async getUserProfile() {
        return this.get('user/profile');
    }

    async getUserStats() {
        return this.get('user/stats');
    }

    async getXPHistory() {
        return this.get('user/xp-history');
    }

    // XP and gamification
    async getUserXP(userId) {
        return this.get(`xp/${userId}`);
    }

    async getAnalytics() {
        return this.get('analytics/overview');
    }

    async search(query) {
        return this.get('search', { q: query });
    }
}

// =============================================================================
// UI COMPONENTS & RENDERING
// =============================================================================

class UIRenderer {
    constructor(state, api, app = null) {
        this.state = state;
        this.api = api;
        this.app = app; // Reference to main app for accessing achievements, etc.
        this.views = new Map();
        this.components = new Map();
        
        this.initializeViews();
    }

    initializeViews() {
        this.views.set('dashboard', this.renderDashboard.bind(this));
        this.views.set('focus', this.renderFocusMode.bind(this));
        this.views.set('tasks', this.renderTasks.bind(this));
        this.views.set('projects', this.renderProjects.bind(this));
        this.views.set('categories', this.renderCategories.bind(this));
        this.views.set('calendar', this.renderCalendar.bind(this));
        this.views.set('achievements', this.renderAchievements.bind(this));
        this.views.set('analytics', this.renderAnalyticsView.bind(this));
        this.views.set('changelog', this.renderChangelog.bind(this));
        this.views.set('accountsettings', this.renderAccountSettings.bind(this));
        // Legacy routes for backwards compatibility
        this.views.set('account', this.renderAccountSettings.bind(this));
        this.views.set('settings', this.renderAccountSettings.bind(this));
    }

    getViewTitle(viewName) {
        const titles = {
            'dashboard': '<i class="fas fa-home"></i> Dashboard',
            'focus': '<i class="fas fa-bullseye"></i> Focus Mode',
            'tasks': '<i class="fas fa-tasks"></i> Tasks',
            'projects': '<i class="fas fa-folder"></i> Projects',
            'categories': '<i class="fas fa-tags"></i> Categories',
            'calendar': '<i class="fas fa-calendar"></i> Calendar',
            'achievements': '<i class="fas fa-trophy"></i> Achievements',
            'analytics': '<i class="fas fa-chart-line"></i> Analytics',
            'changelog': '<i class="fas fa-list"></i> Changelog',
            'accountsettings': '<i class="fas fa-user-cog"></i> Account Settings',
            'account': '<i class="fas fa-user-cog"></i> Account Settings',
            'settings': '<i class="fas fa-cog"></i> Settings'
        };
        return titles[viewName] || '<i class="fas fa-circle"></i> ' + viewName.charAt(0).toUpperCase() + viewName.slice(1);
    }

    getViewActions(viewName) {
        // Dashboard quick actions
        if (viewName === 'dashboard') {
            return `
                <div class="dashboard-quick-actions">
                    <button id="dashboard-new-task" class="btn btn-primary">
                        <i class="fas fa-plus"></i> New Task
                    </button>
                    <button id="dashboard-new-project" class="btn btn-outline">
                        <i class="fas fa-folder-plus"></i> New Project
                    </button>
                    <button id="dashboard-new-category" class="btn btn-outline">
                        <i class="fas fa-tag"></i> New Category
                    </button>
                    <button id="dashboard-add-event" class="btn btn-outline">
                        <i class="fas fa-calendar-plus"></i> Add Event
                    </button>
                </div>
            `;
        }
        
        // Achievements stats display
        if (viewName === 'achievements') {
            return `
                <div class="achievements-stats-compact">
                    <span class="stat-item">
                        <i class="fas fa-trophy"></i>
                        <span id="header-unlocked-count">0</span>/<span id="header-total-count">0</span>
                    </span>
                    <span class="stat-item">
                        <i class="fas fa-lock-open"></i>
                        <span id="header-completion-pct">0%</span>
                    </span>
                </div>
            `;
        }
        
        // No actions for changelog
        if (viewName === 'changelog') {
            return '';
        }
        
        // Calendar filters
        if (viewName === 'calendar') {
            const calendar = this.app?.calendar || this.calendar;
            const startDate = calendar?.startDate ? this.formatDateForInput(calendar.startDate) : this.formatDateForInput(new Date());
            const daysToShow = calendar?.daysToShow || 30;
            
            return `
                <div class="calendar-filters-inline">
                    <div class="filter-group">
                        <label for="start-date-picker">
                            <i class="fas fa-calendar-day"></i>
                            <span class="filter-label">Start</span>
                        </label>
                        <input 
                            type="date" 
                            id="start-date-picker" 
                            class="date-input"
                            value="${startDate}"
                        />
                    </div>
                    <div class="filter-group">
                        <label>
                            <i class="fas fa-clock"></i>
                            <span class="filter-label">Range</span>
                        </label>
                        <div class="range-buttons">
                            <button class="btn-range ${daysToShow === 7 ? 'active' : ''}" data-days="7">7d</button>
                            <button class="btn-range ${daysToShow === 14 ? 'active' : ''}" data-days="14">14d</button>
                            <button class="btn-range ${daysToShow === 30 ? 'active' : ''}" data-days="30">1mo</button>
                            <button class="btn-range ${daysToShow === 90 ? 'active' : ''}" data-days="90">3mo</button>
                            <button class="btn-range ${daysToShow === 180 ? 'active' : ''}" data-days="180">6mo</button>
                        </div>
                    </div>
                </div>
            `;
        }
        
        const taskSort = localStorage.getItem('taskSort') || '';
        const projectSort = localStorage.getItem('projectSort') || '';
        const categorySort = localStorage.getItem('categorySort') || '';
        
        const actions = {
            tasks: `
                <div class="tasks-actions">
                    <button id="bulk-select-btn" class="btn btn-outline" title="Bulk Select">
                        <i class="fas fa-check-square"></i>
                        <span>Bulk Select</span>
                    </button>
                    <select id="task-sort" class="sort-select">
                        <option value="">Sort by...</option>
                        <option value="date_desc" ${taskSort === 'date_desc' ? 'selected' : ''}>Newest First</option>
                        <option value="date_asc" ${taskSort === 'date_asc' ? 'selected' : ''}>Oldest First</option>
                        <option value="priority" ${taskSort === 'priority' ? 'selected' : ''}>Priority</option>
                        <option value="title" ${taskSort === 'title' ? 'selected' : ''}>Title A-Z</option>
                    </select>
                </div>
            `,
            projects: `
                <select id="project-sort" class="sort-select">
                    <option value="">Sort by...</option>
                    <option value="name" ${projectSort === 'name' ? 'selected' : ''}>Name A-Z</option>
                    <option value="date" ${projectSort === 'date' ? 'selected' : ''}>Newest First</option>
                    <option value="tasks" ${projectSort === 'tasks' ? 'selected' : ''}>Most Tasks</option>
                </select>
            `,
            categories: `
                <select id="category-sort" class="sort-select">
                    <option value="">Sort by...</option>
                    <option value="alpha" ${categorySort === 'alpha' ? 'selected' : ''}>Alphabetical</option>
                    <option value="newest" ${categorySort === 'newest' ? 'selected' : ''}>Newest</option>
                    <option value="projects" ${categorySort === 'projects' ? 'selected' : ''}>Most Projects</option>
                    <option value="tasks" ${categorySort === 'tasks' ? 'selected' : ''}>Most Tasks</option>
                </select>
            `
        };
        return actions[viewName] || '';
    }

    async renderView(viewName) {
        logger.debug('[renderView] Called with viewName:', viewName);
        
        const contentBody = document.getElementById('content-body');
        const contentTitle = document.getElementById('content-title');
        const contentActions = document.getElementById('content-actions');
        const contentPrimaryAction = document.getElementById('content-primary-action');
        const contentFilters = document.getElementById('content-filters');
        const contentHeader = document.getElementById('content-header');
        
        if (!contentBody || !contentTitle || !contentActions) {
            logger.error('Required DOM elements not found: content-body, content-title, or content-actions');
            return;
        }
        
        if (!this.views.has(viewName)) {
            logger.error(`View '${viewName}' not found`);
            return;
        }

        try {
            this.state.set('ui.loading', true);
            
            // Hide content header for pages that have their own header
            if (contentHeader) {
                contentHeader.style.display = (viewName === 'settings' || viewName === 'account') ? 'none' : 'flex';
            }
            
            contentTitle.innerHTML = this.getViewTitle(viewName);
            
            logger.debug('[renderView] Set title for view:', viewName);
            
            // Update action buttons based on view
            contentActions.innerHTML = this.getViewActions(viewName);
            
            const content = await this.views.get(viewName)();
            contentBody.innerHTML = content;
            
            logger.debug('[renderView] Rendered content for view:', viewName);
            
            this.bindViewEvents(viewName);
            this.state.set('currentView', viewName);
            
            logger.debug('[renderView] Set currentView state to:', viewName);
            
            // Check for onboarding task completion based on view
            if (window.app && window.app.checkOnboardingTaskCompletion) {
                if (viewName === 'dashboard') {
                    await window.app.checkOnboardingTaskCompletion('visit_dashboard');
                } else if (viewName === 'account') {
                    await window.app.checkOnboardingTaskCompletion('visit_account');
                } else if (viewName === 'settings') {
                    await window.app.checkOnboardingTaskCompletion('visit_settings');
                } else if (viewName === 'tasks') {
                    await window.app.checkOnboardingTaskCompletion('visit_tasks');
                }
            }
        } catch (error) {
            logger.error(`Error rendering view '${viewName}':`, error);
            this.renderError(error.message);
        } finally {
            this.state.set('ui.loading', false);
        }
    }
    
    formatDateForInput(date) {
        const year = date.getFullYear();
        const month = String(date.getMonth() + 1).padStart(2, '0');
        const day = String(date.getDate()).padStart(2, '0');
        return `${year}-${month}-${day}`;
    }

    async renderDashboard() {
        const analytics = await this.api.getAnalytics();
        const recentTasks = this.state.get('tasks').slice(0, 5);
        
        // Check if mobile view (768px breakpoint)
        const isMobile = window.innerWidth <= 768;
        
        // Get featured tracker from localStorage or default to first
        const featuredTracker = localStorage.getItem('TL_featuredTracker') || 'total_tasks';
        
        // Define all trackers
        const trackers = {
            total_tasks: {
                icon: 'fas fa-tasks',
                label: 'Total Tasks',
                value: analytics.data?.total_tasks || 0,
                color: '#2563eb'
            },
            completed_tasks: {
                icon: 'fas fa-check',
                label: 'Completed',
                value: analytics.data?.completed_tasks || 0,
                color: '#10b981'
            },
            pending_tasks: {
                icon: 'fas fa-clock',
                label: 'Pending',
                value: analytics.data?.pending_tasks || 0,
                color: '#f59e0b'
            },
            in_progress_tasks: {
                icon: 'fas fa-spinner',
                label: 'In Progress',
                value: analytics.data?.in_progress_tasks || 0,
                color: '#06b6d4'
            },
            xp_total: {
                icon: 'fas fa-star',
                label: 'Total XP',
                value: analytics.data?.xp_total || 0,
                color: '#8b5cf6'
            }
        };
        
        let statsHTML = '';
        
        if (isMobile) {
            // Mobile: Featured stat with cycling
            const featured = trackers[featuredTracker] || trackers.total_tasks;
            const otherTrackers = Object.entries(trackers).filter(([key]) => key !== featuredTracker);
            
            statsHTML = `
                <div class="stats-featured">
                    <div class="featured-stat">
                        <div class="featured-main">
                            <div class="featured-icon" style="color: ${featured.color}">
                                <i class="${featured.icon}"></i>
                            </div>
                            <div class="featured-content">
                                <h2>${featured.value}</h2>
                                <p>${featured.label}</p>
                            </div>
                        </div>
                        <button class="cycle-tracker-btn" onclick="app.ui.cycleFeaturedTracker()" title="Cycle featured stat">
                            <i class="fas fa-sync-alt"></i>
                        </button>
                    </div>
                    <div class="other-stats">
                        ${otherTrackers.map(([key, tracker]) => `
                            <div class="mini-stat">
                                <span class="mini-stat-label">${tracker.label}:</span>
                                <span class="mini-stat-value">${tracker.value}</span>
                            </div>
                        `).join('')}
                    </div>
                </div>
            `;
        } else {
            // Desktop: Full stats grid
            statsHTML = `
                <div class="stats-grid">
                    <div class="stat-card">
                        <div class="stat-icon"><i class="fas fa-tasks"></i></div>
                        <div class="stat-content">
                            <h3>${analytics.data?.total_tasks || 0}</h3>
                            <p>Total Tasks</p>
                        </div>
                    </div>
                    <div class="stat-card">
                        <div class="stat-icon"><i class="fas fa-check"></i></div>
                        <div class="stat-content">
                            <h3>${analytics.data?.completed_tasks || 0}</h3>
                            <p>Completed</p>
                        </div>
                    </div>
                    <div class="stat-card">
                        <div class="stat-icon"><i class="fas fa-clock"></i></div>
                        <div class="stat-content">
                            <h3>${analytics.data?.pending_tasks || 0}</h3>
                            <p>Pending</p>
                        </div>
                    </div>
                    <div class="stat-card">
                        <div class="stat-icon"><i class="fas fa-spinner"></i></div>
                        <div class="stat-content">
                            <h3>${analytics.data?.in_progress_tasks || 0}</h3>
                            <p>In Progress</p>
                        </div>
                    </div>
                    <div class="stat-card">
                        <div class="stat-icon"><i class="fas fa-star"></i></div>
                        <div class="stat-content">
                            <h3>${analytics.data?.xp_total || 0}</h3>
                            <p>Total XP</p>
                        </div>
                    </div>
                </div>
            `;
        }
        
        return `
            <div class="dashboard">
                ${statsHTML}
                
                <div class="dashboard-sections">
                    <div class="recent-tasks">
                        <h3>Recent Tasks</h3>
                        <div class="task-list-simple">
                            ${recentTasks.map(task => this.renderTaskCard(task)).join('')}
                        </div>
                    </div>
                </div>
            </div>
        `;
    }

    async renderTasks() {
        let tasks = this.state.get('tasks');
        const projects = this.state.get('projects');
        const categories = this.state.get('categories');
        const filters = this.state.get('filters');
        
        // DEBUG: Log all tasks before filtering
        console.log('[renderTasks] All tasks before filtering:', tasks.map(t => ({id: t.id, title: t.title, status: t.status, project_id: t.project_id})));
        console.log('[renderTasks] Active filters:', filters);
        console.log('[renderTasks] Task filter:', this.state.get('taskFilter'));
        
        // Apply category filter if set (from category navigation)
        const taskFilter = this.state.get('taskFilter');
        if (taskFilter && taskFilter.categoryId) {
            tasks = tasks.filter(task => task.category_id === taskFilter.categoryId);
            console.log('[renderTasks] After category filter:', tasks.length);
        }
        
        // Apply multi-select filters
        if (filters.projects && filters.projects.length > 0) {
            console.log('[renderTasks] Filtering by projects:', filters.projects);
            tasks = tasks.filter(task => filters.projects.includes(task.project_id));
            console.log('[renderTasks] After projects filter:', tasks.length);
        }
        
        if (filters.categories && filters.categories.length > 0) {
            console.log('[renderTasks] Filtering by categories:', filters.categories);
            tasks = tasks.filter(task => filters.categories.includes(task.category_id));
            console.log('[renderTasks] After categories filter:', tasks.length);
        }
        
        if (filters.statuses && filters.statuses.length > 0) {
            console.log('[renderTasks] Filtering by statuses:', filters.statuses);
            tasks = tasks.filter(task => filters.statuses.includes(task.status));
            console.log('[renderTasks] After status filter:', tasks.length);
        }
        
        if (filters.priorities && filters.priorities.length > 0) {
            console.log('[renderTasks] Filtering by priorities:', filters.priorities);
            tasks = tasks.filter(task => filters.priorities.includes(task.priority));
            console.log('[renderTasks] After priority filter:', tasks.length);
        }
        
        console.log('[renderTasks] Final task count:', tasks.length);
        
        logger.debug('[renderTasks] Final filtered tasks:', tasks.map(t => ({id: t.id, title: t.title})));
        
        // Check if we need more tasks after filtering
        const tasksMeta = this.state.get('tasksMeta') || {};
        const hasMoreTasks = tasksMeta.has_next || false;
        const showLoadMore = hasMoreTasks && tasks.length < 20;
        
        // Check if any filters are active
        const hasFilters = (taskFilter && taskFilter.categoryId) || 
                          (filters.projects && filters.projects.length > 0) ||
                          (filters.categories && filters.categories.length > 0) ||
                          (filters.statuses && filters.statuses.length > 0) ||
                          (filters.priorities && filters.priorities.length > 0);

        // Note: Filter banner is now handled by updateFilterBanner() method

        return `
            <div class="tasks-view">
                
                <div class="tasks-list" id="tasks-list-container">
                    ${tasks.length === 0 ? `
                        <div class="empty-state">
                            <i class="fas fa-check-circle" style="font-size: 3rem; color: var(--color-success); opacity: 0.5; margin-bottom: 1rem;"></i>
                            <p style="font-size: 1.1rem; font-weight: 500;">No tasks found</p>
                            <p style="opacity: 0.7;">${hasFilters ? 'Try adjusting your filters' : 'Create your first task!'}</p>
                        </div>
                    ` : tasks.map(task => this.renderTaskItem(task)).join('')}
                </div>
                
                ${showLoadMore ? `
                    <div class="load-more-container">
                        <button class="btn btn-secondary" onclick="app.loadMoreTasks()">
                            <i class="fas fa-chevron-down"></i> Load More Tasks
                        </button>
                    </div>
                ` : ''}
            </div>
        `;
    }

    renderTaskItem(task) {
        const project = this.state.get('projects').find(p => p.id === task.project_id);
        const category = this.state.get('categories').find(c => c.id === task.category_id);
        const isSelected = this.state.get('selectedTasks').includes(task.id);
        const hasSubtasks = Array.isArray(task.subtasks) && task.subtasks.length > 0;
        const subtasksCompleted = hasSubtasks ? task.subtasks.filter(s => s.is_completed).length : 0;
        
        if (hasSubtasks) {
            logger.debug(`Rendering task ${task.id} with ${task.subtasks.length} subtasks (${subtasksCompleted} completed)`);
        }
        
        return `
            <div class="task-item ${isSelected ? 'selected' : ''} priority-${task.priority}" data-task-id="${task.id}">
                <div class="task-checkbox-container">
                    <input type="checkbox" class="task-checkbox" ${isSelected ? 'checked' : ''}>
                </div>
                
                <div class="task-content">
                    <div class="task-header">
                        <h4 class="task-title">${this.escapeHtml(task.title)}</h4>
                        <div class="task-actions">
                            ${hasSubtasks ? `
                                <button class="btn-icon subtasks-toggle" onclick="app.ui.toggleTaskSubtasks(${task.id})" title="Show steps">
                                    <i class="fas fa-chevron-down"></i>
                                    <span class="subtasks-badge">${subtasksCompleted}/${task.subtasks.length}</span>
                                </button>
                            ` : ''}
                            <button class="btn-icon" onclick="app.editTask(${task.id}, 'edit')" title="Edit">
                                <i class="fas fa-edit"></i>
                            </button>
                            <button class="btn-icon" onclick="app.deleteTask(${task.id})" title="Delete">
                                <i class="fas fa-trash"></i>
                            </button>
                        </div>
                    </div>
                    
                     ${task.description ? `<p class="task-description">${task.description}</p>` : ''}
                    
                    ${hasSubtasks ? `
                        <div class="task-subtasks-preview" id="task-subtasks-${task.id}" style="display: none;">
                            <div class="subtasks-progress-mini">
                                <div class="progress-bar-mini">
                                    <div class="progress-fill-mini" style="width: ${(subtasksCompleted / task.subtasks.length * 100).toFixed(0)}%"></div>
                                </div>
                                <span class="progress-text-mini">${subtasksCompleted} / ${task.subtasks.length} completed</span>
                            </div>
                            <ul class="subtasks-list-mini">
                                ${task.subtasks.map(subtask => `
                                    <li class="subtask-item-mini ${subtask.is_completed ? 'completed' : ''}" onclick="app.ui.toggleTaskSubtaskStatus(${task.id}, ${subtask.id})">
                                        <div class="subtask-checkbox-mini ${subtask.is_completed ? 'checked' : ''}"></div>
                                        <span class="subtask-text-mini">${this.escapeHtml(subtask.title)}</span>
                                    </li>
                                `).join('')}
                            </ul>
                        </div>
                    ` : ''}
                    
                    <div class="task-meta">
                        <div class="task-meta-row priority-complexity">
                            <span class="task-priority priority-${task.priority}">
                                <i class="fas fa-flag"></i> ${task.priority}
                            </span>
                            
                            ${task.complexity && task.complexity !== 'none' ? `<span class="task-complexity complexity-${task.complexity}">
                                <i class="fas fa-puzzle-piece"></i> ${task.complexity}
                            </span>` : ''}
                        </div>
                        
                        <div class="task-meta-row project-category">
                            ${project ? `<span class="task-project">
                                <i class="fas fa-folder"></i> ${project.name}
                            </span>` : ''}
                            
                            ${category ? `<span class="task-category">
                                <i class="fas fa-tag"></i> ${category.name}
                            </span>` : ''}
                        </div>
                        
                        ${task.due_date ? `<div class="task-meta-row due-date-row">
                            <span class="task-due-date">
                                <i class="fas fa-calendar"></i> ${this.formatDate(task.due_date)}
                            </span>
                        </div>` : ''}
                    </div>
                </div>
                
                <div class="task-status">
                    <button class="btn btn-sm ${task.status === 'completed' ? 'btn-success' : 'btn-outline'}" 
                            onclick="app.toggleTaskStatus(${task.id})">
                        <i class="fas ${task.status === 'completed' ? 'fa-check' : 'fa-circle'}"></i>
                        ${task.status === 'completed' ? 'Completed' : 'Complete'}
                    </button>
                </div>
            </div>
        `;
    }

    renderTaskCard(task) {
        return `
            <div class="task-card" data-task-id="${task.id}" onclick="app.editTask(${task.id}, 'view')" style="cursor: pointer;">
                <div class="task-card-content">
                    <h4>${this.escapeHtml(task.title)}</h4>
                    <p class="task-card-meta">
                        <span class="priority priority-${task.priority}">${task.priority}</span>
                        ${task.due_date ? `<span class="due-date">${this.formatDate(task.due_date)}</span>` : ''}
                    </p>
                </div>
                <div class="task-card-status">
                    <i class="fas ${task.status === 'completed' ? 'fa-check-circle text-success' : 'fa-circle text-muted'}"></i>
                </div>
            </div>
        `;
    }

    renderGuestShareInfo() {
        const shareProjectId = window.app?.shareMode?.projectId;
        const sharePermission = window.app?.shareMode?.permission || 'view';
        const project = this.state.get('projects').find(p => p.id === shareProjectId);
        const projectName = project ? project.name : 'Shared Project';
        
        return `
            <div class="guest-share-info-view">
                <div class="share-info-hero">
                    <i class="fas fa-share-nodes"></i>
                    <h2>Shared Project Access</h2>
                    <p>You're viewing a project that has been shared with you</p>
                </div>
                
                <div class="share-info-cards">
                    <div class="share-info-card">
                        <div class="share-info-icon">
                            <i class="fas fa-folder"></i>
                        </div>
                        <div class="share-info-content">
                            <h3>Project Name</h3>
                            <p class="share-info-value">${this.escapeHtml(projectName)}</p>
                        </div>
                    </div>
                    
                    <div class="share-info-card">
                        <div class="share-info-icon">
                            <i class="fas fa-shield-alt"></i>
                        </div>
                        <div class="share-info-content">
                            <h3>Your Permission</h3>
                            <p class="share-info-value">
                                <span class="permission-badge ${sharePermission}">
                                    <i class="fas fa-${sharePermission === 'edit' ? 'edit' : 'eye'}"></i>
                                    ${sharePermission === 'edit' ? 'Can Edit Tasks' : 'View Only'}
                                </span>
                            </p>
                        </div>
                    </div>
                    
                    <div class="share-info-card">
                        <div class="share-info-icon">
                            <i class="fas fa-calendar-alt"></i>
                        </div>
                        <div class="share-info-content">
                            <h3>Access Date</h3>
                            <p class="share-info-value">${new Date().toLocaleDateString('en-US', { month: 'long', day: 'numeric', year: 'numeric' })}</p>
                        </div>
                    </div>
                </div>
                
                <div class="share-info-cta">
                    <div class="cta-content">
                        <i class="fas fa-star"></i>
                        <h3>Want Full Access to TaskList?</h3>
                        <p>Create a free account to unlock all features including your own projects, achievements, analytics, and more!</p>
                        <a href="/auth/login.php?redirect=${encodeURIComponent(window.location.pathname)}" class="btn btn-primary btn-lg">
                            <i class="fas fa-user-circle"></i> Create Account or Sign In
                        </a>
                    </div>
                </div>
            </div>
        `;
    }

    async renderProjects() {
        // Check if user is a guest - show share info instead
        const isGuestUser = window.app?.shareMode?.isGuest || false;
        if (isGuestUser) {
            return this.renderGuestShareInfo();
        }
        
        let projects = this.state.get('projects');
        
        // Apply category filter if set
        const projectFilter = this.state.get('projectFilter');
        if (projectFilter && projectFilter.categoryId) {
            projects = projects.filter(project => 
                project.category_ids && project.category_ids.includes(projectFilter.categoryId)
            );
        }
        
        // Separate active and completed projects
        const activeProjects = projects.filter(p => !p.is_completed && !p.is_archived);
        const completedProjects = projects.filter(p => p.is_completed && !p.is_archived);
        
        return `
            <div class="projects-view">
                ${projectFilter && projectFilter.categoryId ? `
                    <div class="filter-banner">
                        <span class="filter-info">
                            <i class="fas fa-filter"></i>
                            Showing projects in "${this.state.get('categories').find(c => c.id === projectFilter.categoryId)?.name || 'Unknown Category'}"
                        </span>
                        <button class="clear-filter-btn" onclick="app.clearProjectFilter()">
                            <i class="fas fa-times"></i>
                            Show All Projects
                        </button>
                    </div>
                ` : ''}
                
                ${activeProjects.length > 0 ? `
                    <div class="projects-section active-projects">
                        <h3 class="section-title">Active Projects</h3>
                        <div class="projects-grid">
                            ${activeProjects.map(project => this.renderProjectCard(project)).join('')}
                        </div>
                    </div>
                ` : ''}
                
                ${completedProjects.length > 0 ? `
                    <div class="projects-section completed-projects">
                        <h3 class="section-title">
                            <i class="fas fa-check-circle"></i>
                            Completed Projects
                        </h3>
                        <div class="projects-grid">
                            ${completedProjects.map(project => this.renderProjectCard(project, true)).join('')}
                        </div>
                    </div>
                ` : ''}
                
                ${projects.length === 0 ? `<div class="empty-state">
                    <i class="fas fa-lightbulb" style="font-size: 3rem; color: var(--color-warning); opacity: 0.5; margin-bottom: 1rem;"></i>
                    <p style="font-size: 1.1rem; font-weight: 500;">Let's create your first project! 💡</p>
                    <p style="opacity: 0.7;">Projects help you organize related tasks together</p>
                </div>` : ''}
            </div>
        `;
    }

    renderProjectCard(project, isCompleted = false) {
        const taskCount = this.state.get('tasks').filter(t => t.project_id === project.id).length;
        const completedTaskCount = this.state.get('tasks').filter(t => t.project_id === project.id && t.status === 'completed').length;
        
        // Check if user is in share mode (guest user)
        const isGuestUser = window.app?.shareMode?.isGuest || false;
        
        return `
            <div class="project-card ${isCompleted ? 'project-completed' : ''}" data-project-id="${project.id}" style="border-left-color: ${project.color}">
                <div class="project-header">
                    <div class="project-icon">
                        <i class="${project.icon || 'fas fa-folder'}"></i>
                        ${isCompleted ? '<span class="completion-badge" title="Completed"><i class="fas fa-check"></i></span>' : ''}
                    </div>
                    ${!isGuestUser ? `<div class="project-actions">
                        <button class="btn-icon share-project-btn ${project.is_shared ? 'is-shared' : ''}" 
                                data-project-id="${project.id}" 
                                title="${project.is_shared ? 'Manage Sharing (Currently Shared)' : 'Share Project'}">
                            <i class="fas fa-share-alt"></i>
                            ${project.is_shared ? '<span class="share-badge"></span>' : ''}
                        </button>
                        <button class="btn-icon toggle-project-completion-btn" data-project-id="${project.id}" title="${isCompleted ? 'Mark as Active' : 'Mark as Complete'}">
                            <i class="fas fa-${isCompleted ? 'undo' : 'check-circle'}"></i>
                        </button>
                        <button class="btn-icon edit-project-btn" data-project-id="${project.id}" title="Edit">
                            <i class="fas fa-edit"></i>
                        </button>
                        <button class="btn-icon delete-project-btn" data-project-id="${project.id}" title="Permanently Delete">
                            <i class="fas fa-trash"></i>
                        </button>
                    </div>` : ''}
                </div>
                
                <div class="project-content">
                    <h3 class="project-name">${this.escapeHtml(project.name)}</h3>
                    ${project.description ? `<p class="project-description">${this.escapeHtml(project.description)}</p>` : ''}
                    
                    <div class="project-stats">
                        <span class="stat">
                            <i class="fas fa-tasks"></i>
                            ${completedTaskCount}/${taskCount} tasks
                        </span>
                    </div>
                </div>
            </div>
        `;
    }

    async renderCategories() {
        const categories = this.state.get('categories');
        
        return `
            <div class="categories-view">
                <div class="categories-grid">
                    ${categories.map(category => this.renderCategoryCard(category)).join('')}
                </div>
                
                ${categories.length === 0 ? '<div class="empty-state"><p>No categories found.</p></div>' : ''}
            </div>
        `;
    }

    renderCategoryCard(category) {
        const taskCount = this.state.get('tasks').filter(t => t.category_id === category.id).length;
        const projects = this.state.get('projects').filter(p => p.category_ids && p.category_ids.includes(category.id));
        
        return `
            <div class="category-card" data-category-id="${category.id}" style="border-left-color: ${category.color || '#2563eb'}">
                <div class="category-header">
                    <div class="category-color-indicator" style="background-color: ${category.color || '#2563eb'}"></div>
                    <div class="category-actions">
                        <button class="btn-icon" onclick="app.editCategory(${category.id})" title="Edit Category">
                            <i class="fas fa-edit"></i>
                        </button>
                        <button class="btn-icon" onclick="app.manageCategoryProjects(${category.id})" title="Manage Projects">
                            <i class="fas fa-link"></i>
                        </button>
                        <button class="btn-icon" onclick="app.deleteCategory(${category.id})" title="Delete Category">
                            <i class="fas fa-trash"></i>
                        </button>
                    </div>
                </div>
                
                <div class="category-content">
                    <h3 class="category-name">${this.escapeHtml(category.name)}</h3>
                    ${category.description ? `<p class="category-description">${this.escapeHtml(category.description)}</p>` : ''}
                    
                    <div class="category-stats">
                        <span class="stat clickable" onclick="app.navigateToTasks(${category.id})" title="View tasks in this category">
                            <i class="fas fa-tasks"></i>
                            ${taskCount} tasks
                        </span>
                        <span class="stat clickable" onclick="app.navigateToProjects(${category.id})" title="View projects in this category">
                            <i class="fas fa-folder"></i>
                            ${projects.length} projects
                        </span>
                    </div>
                    
                    ${projects.length > 0 ? `
                        <div class="category-projects">
                            <h4>Projects:</h4>
                            <div class="project-tags">
                                ${projects.slice(0, 3).map(p => `<span class="project-tag" style="background-color: ${p.color || '#6b7280'}">${this.escapeHtml(p.name)}</span>`).join('')}
                                ${projects.length > 3 ? `<span class="project-tag more">+${projects.length - 3} more</span>` : ''}
                            </div>
                        </div>
                    ` : ''}
                </div>
            </div>
        `;
    }

    async renderCalendar() {
        // Calendar manager will handle its own rendering
        return '<div id="calendar-placeholder">Loading calendar...</div>';
    }

    async renderAchievements() {
        // Check if app exists
        if (!this.app) {
            logger.error('App not initialized');
            return '<div class="error-message">Application is loading...</div>';
        }
        
        // Check if achievements system exists
        if (!this.app.achievements) {
            logger.error('AchievementSystem not initialized');
            return '<div class="error-message">Achievement system is loading...</div>';
        }
        
        // Achievements are already initialized in app.init(), just render the view
        try {
            // Return achievements view from system
            return this.app.achievements.renderAchievementsView();
        } catch (error) {
            logger.error('Error rendering achievements:', error);
            return '<div class="error-message">Failed to load achievements. Please try again.</div>';
        }
    }

    async renderAnalyticsView() {
        const tasks = this.state.get('tasks') || [];
        const projects = this.state.get('projects') || [];
        const categories = this.state.get('categories') || [];
        const user = this.state.get('user') || {};
        
        // Task Analytics
        const completedTasks = tasks.filter(t => t.status === 'completed');
        const inProgressTasks = tasks.filter(t => t.status === 'in-progress');
        const pendingTasks = tasks.filter(t => t.status === 'pending');
        const highPriorityTasks = tasks.filter(t => t.priority === 'high');
        const mediumPriorityTasks = tasks.filter(t => t.priority === 'medium');
        const lowPriorityTasks = tasks.filter(t => t.priority === 'low');
        
        const tasksWithDueDate = tasks.filter(t => t.due_date);
        const overdueTasks = tasks.filter(t => {
            if (!t.due_date || t.status === 'completed') return false;
            return new Date(t.due_date) < new Date();
        });
        
        const tasksWithSubtasks = tasks.filter(t => t.subtasks && t.subtasks.length > 0);
        const totalSubtasks = tasks.reduce((sum, t) => sum + (t.subtasks?.length || 0), 0);
        const completedSubtasks = tasks.reduce((sum, t) => 
            sum + (t.subtasks?.filter(s => s.is_completed || s.completed).length || 0), 0);
        
        logger.debug('[Analytics] Subtasks:', {tasksWithSubtasks: tasksWithSubtasks.length, totalSubtasks, completedSubtasks});
        if (tasksWithSubtasks.length > 0) {
            logger.debug('[Analytics] Sample task with subtasks:', tasksWithSubtasks[0]);
        }
        
        const completionRate = tasks.length > 0 ? ((completedTasks.length / tasks.length) * 100).toFixed(1) : 0;
        const subtaskCompletionRate = totalSubtasks > 0 ? ((completedSubtasks / totalSubtasks) * 100).toFixed(1) : 0;
        
        // Project Analytics
        logger.debug('[Analytics] All projects:', projects.map(p => ({id: p.id, name: p.name, status: p.status})));
        const activeProjects = projects.filter(p => p.status === 'active');
        const completedProjects = projects.filter(p => p.status === 'completed');
        const archivedProjects = projects.filter(p => p.status === 'archived');
        logger.debug('[Analytics] Project counts:', {active: activeProjects.length, completed: completedProjects.length, archived: archivedProjects.length});
        
        const projectCompletionRate = projects.length > 0 ? ((completedProjects.length / projects.length) * 100).toFixed(1) : 0;
        
        // Calculate tasks per project
        const tasksPerProject = projects.map(p => ({
            name: p.name,
            color: p.color,
            taskCount: tasks.filter(t => t.project_id === p.id).length,
            completedCount: tasks.filter(t => t.project_id === p.id && t.status === 'completed').length
        })).sort((a, b) => b.taskCount - a.taskCount).slice(0, 5);
        
        // Category Analytics
        const tasksPerCategory = categories.map(c => ({
            name: c.name,
            color: c.color,
            taskCount: tasks.filter(t => t.category_id === c.id).length,
            completedCount: tasks.filter(t => t.category_id === c.id && t.status === 'completed').length
        })).sort((a, b) => b.taskCount - a.taskCount).slice(0, 5);
        
        const uncategorizedTasks = tasks.filter(t => !t.category_id).length;
        
        // User Analytics
        const achievements = this.app.achievements?.getStats() || {};
        const currentLevel = this.app.achievements?.currentLevel || 1;
        const totalXP = this.app.achievements?.currentXP || 0;
        const currentStreak = this.app.achievements?.streak?.current || 0;
        const longestStreak = this.app.achievements?.streak?.longest || 0;
        
        // Time-based analytics (last 7 days, 30 days)
        const now = new Date();
        const last7Days = new Date(now - 7 * 24 * 60 * 60 * 1000);
        const last30Days = new Date(now - 30 * 24 * 60 * 60 * 1000);
        
        const tasksLast7Days = tasks.filter(t => new Date(t.created_at) >= last7Days).length;
        const tasksLast30Days = tasks.filter(t => new Date(t.created_at) >= last30Days).length;
        const completedLast7Days = completedTasks.filter(t => t.completed_at && new Date(t.completed_at) >= last7Days).length;
        const completedLast30Days = completedTasks.filter(t => t.completed_at && new Date(t.completed_at) >= last30Days).length;
        
        return `
            <div class="analytics-view">
                <!-- Overview Tab -->
                <div class="analytics-panel active" data-analytics-panel="overview">
                    <h3 style="margin-bottom: var(--spacing-lg);">Quick Overview</h3>
                    <div class="stats-grid">
                        <div class="stat-card">
                            <i class="fas fa-tasks"></i>
                            <div class="stat-value">${tasks.length}</div>
                            <div class="stat-label">Total Tasks</div>
                        </div>
                        <div class="stat-card">
                            <i class="fas fa-check-circle"></i>
                            <div class="stat-value">${completedTasks.length}</div>
                            <div class="stat-label">Completed</div>
                        </div>
                        <div class="stat-card">
                            <i class="fas fa-percentage"></i>
                            <div class="stat-value">${completionRate}%</div>
                            <div class="stat-label">Completion Rate</div>
                        </div>
                        <div class="stat-card">
                            <i class="fas fa-folder"></i>
                            <div class="stat-value">${projects.length}</div>
                            <div class="stat-label">Total Projects</div>
                        </div>
                        <div class="stat-card">
                            <i class="fas fa-star"></i>
                            <div class="stat-value">${totalXP}</div>
                            <div class="stat-label">Total XP</div>
                        </div>
                        <div class="stat-card">
                            <i class="fas fa-fire"></i>
                            <div class="stat-value">${currentStreak}</div>
                            <div class="stat-label">Current Streak</div>
                        </div>
                    </div>

                    <h3 style="margin: var(--spacing-2xl) 0 var(--spacing-lg);">Recent Activity</h3>
                    <div class="stats-grid">
                        <div class="stat-card">
                            <i class="fas fa-calendar-week"></i>
                            <div class="stat-value">${tasksLast7Days}</div>
                            <div class="stat-label">Tasks (Last 7 Days)</div>
                        </div>
                        <div class="stat-card">
                            <i class="fas fa-check"></i>
                            <div class="stat-value">${completedLast7Days}</div>
                            <div class="stat-label">Completed (Last 7 Days)</div>
                        </div>
                        <div class="stat-card">
                            <i class="fas fa-calendar-alt"></i>
                            <div class="stat-value">${tasksLast30Days}</div>
                            <div class="stat-label">Tasks (Last 30 Days)</div>
                        </div>
                        <div class="stat-card">
                            <i class="fas fa-check-double"></i>
                            <div class="stat-value">${completedLast30Days}</div>
                            <div class="stat-label">Completed (Last 30 Days)</div>
                        </div>
                    </div>

                    <h3 style="margin: var(--spacing-2xl) 0 var(--spacing-lg);">Activity Heatmap</h3>
                    ${this.generateActivityCalendar(tasks)}
                </div>

                <!-- Tasks Tab -->
                <div class="analytics-panel" data-analytics-panel="tasks">
                    <h3 style="margin-bottom: var(--spacing-lg);">Task Statistics</h3>
                    <div class="stats-grid">
                        <div class="stat-card">
                            <i class="fas fa-list"></i>
                            <div class="stat-value">${tasks.length}</div>
                            <div class="stat-label">Total Tasks</div>
                        </div>
                        <div class="stat-card">
                            <i class="fas fa-check-circle"></i>
                            <div class="stat-value">${completedTasks.length}</div>
                            <div class="stat-label">Completed</div>
                        </div>
                        <div class="stat-card">
                            <i class="fas fa-spinner"></i>
                            <div class="stat-value">${inProgressTasks.length}</div>
                            <div class="stat-label">In Progress</div>
                        </div>
                        <div class="stat-card">
                            <i class="fas fa-clock"></i>
                            <div class="stat-value">${pendingTasks.length}</div>
                            <div class="stat-label">Pending</div>
                        </div>
                        <div class="stat-card">
                            <i class="fas fa-percentage"></i>
                            <div class="stat-value">${completionRate}%</div>
                            <div class="stat-label">Completion Rate</div>
                        </div>
                        <div class="stat-card">
                            <i class="fas fa-exclamation-triangle"></i>
                            <div class="stat-value">${overdueTasks.length}</div>
                            <div class="stat-label">Overdue</div>
                        </div>
                    </div>

                    <h3 style="margin: var(--spacing-2xl) 0 var(--spacing-lg);">Priority Distribution</h3>
                    <div class="stats-grid">
                        <div class="stat-card">
                            <i class="fas fa-exclamation-circle" style="color: #ef4444;"></i>
                            <div class="stat-value">${highPriorityTasks.length}</div>
                            <div class="stat-label">High Priority</div>
                        </div>
                        <div class="stat-card">
                            <i class="fas fa-minus-circle" style="color: #f59e0b;"></i>
                            <div class="stat-value">${mediumPriorityTasks.length}</div>
                            <div class="stat-label">Medium Priority</div>
                        </div>
                        <div class="stat-card">
                            <i class="fas fa-circle" style="color: #10b981;"></i>
                            <div class="stat-value">${lowPriorityTasks.length}</div>
                            <div class="stat-label">Low Priority</div>
                        </div>
                    </div>

                    <h3 style="margin: var(--spacing-2xl) 0 var(--spacing-lg);">Subtasks</h3>
                    <div class="stats-grid">
                        <div class="stat-card">
                            <i class="fas fa-list-ul"></i>
                            <div class="stat-value">${tasksWithSubtasks.length}</div>
                            <div class="stat-label">Tasks with Subtasks</div>
                        </div>
                        <div class="stat-card">
                            <i class="fas fa-tasks"></i>
                            <div class="stat-value">${totalSubtasks}</div>
                            <div class="stat-label">Total Subtasks</div>
                        </div>
                        <div class="stat-card">
                            <i class="fas fa-check"></i>
                            <div class="stat-value">${completedSubtasks}</div>
                            <div class="stat-label">Completed Subtasks</div>
                        </div>
                        <div class="stat-card">
                            <i class="fas fa-percentage"></i>
                            <div class="stat-value">${subtaskCompletionRate}%</div>
                            <div class="stat-label">Subtask Completion</div>
                        </div>
                    </div>

                    <h3 style="margin: var(--spacing-2xl) 0 var(--spacing-lg);">Due Dates</h3>
                    <div class="stats-grid">
                        <div class="stat-card">
                            <i class="fas fa-calendar"></i>
                            <div class="stat-value">${tasksWithDueDate.length}</div>
                            <div class="stat-label">Tasks with Due Date</div>
                        </div>
                        <div class="stat-card">
                            <i class="fas fa-calendar-times"></i>
                            <div class="stat-value">${overdueTasks.length}</div>
                            <div class="stat-label">Overdue Tasks</div>
                        </div>
                    </div>
                </div>

                <!-- Projects Tab -->
                <div class="analytics-panel" data-analytics-panel="projects">
                    <h3 style="margin-bottom: var(--spacing-lg);">Project Statistics</h3>
                    <div class="stats-grid">
                        <div class="stat-card">
                            <i class="fas fa-folder-open"></i>
                            <div class="stat-value">${projects.length}</div>
                            <div class="stat-label">Total Projects</div>
                        </div>
                        <div class="stat-card">
                            <i class="fas fa-play-circle"></i>
                            <div class="stat-value">${activeProjects.length}</div>
                            <div class="stat-label">Active</div>
                        </div>
                        <div class="stat-card">
                            <i class="fas fa-check-circle"></i>
                            <div class="stat-value">${completedProjects.length}</div>
                            <div class="stat-label">Completed</div>
                        </div>
                        <div class="stat-card">
                            <i class="fas fa-archive"></i>
                            <div class="stat-value">${archivedProjects.length}</div>
                            <div class="stat-label">Archived</div>
                        </div>
                        <div class="stat-card">
                            <i class="fas fa-percentage"></i>
                            <div class="stat-value">${projectCompletionRate}%</div>
                            <div class="stat-label">Completion Rate</div>
                        </div>
                    </div>

                    ${tasksPerProject.length > 0 ? `
                        <h3 style="margin: var(--spacing-2xl) 0 var(--spacing-lg);">Top 5 Projects by Task Count</h3>
                        <div class="analytics-list">
                            ${tasksPerProject.map(p => `
                                <div class="analytics-list-item">
                                    <div class="analytics-list-label">
                                        <span class="color-indicator" style="background-color: ${p.color || '#6366f1'};"></span>
                                        <span>${p.name}</span>
                                    </div>
                                    <div class="analytics-list-value">
                                        ${p.taskCount} tasks (${p.completedCount} completed)
                                    </div>
                                </div>
                            `).join('')}
                        </div>
                    ` : ''}
                </div>

                <!-- Categories Tab -->
                <div class="analytics-panel" data-analytics-panel="categories">
                    <h3 style="margin-bottom: var(--spacing-lg);">Category Statistics</h3>
                    <div class="stats-grid">
                        <div class="stat-card">
                            <i class="fas fa-tags"></i>
                            <div class="stat-value">${categories.length}</div>
                            <div class="stat-label">Total Categories</div>
                        </div>
                        <div class="stat-card">
                            <i class="fas fa-question-circle"></i>
                            <div class="stat-value">${uncategorizedTasks}</div>
                            <div class="stat-label">Uncategorized Tasks</div>
                        </div>
                    </div>

                    ${tasksPerCategory.length > 0 ? `
                        <h3 style="margin: var(--spacing-2xl) 0 var(--spacing-lg);">Top 5 Categories by Task Count</h3>
                        <div class="analytics-list">
                            ${tasksPerCategory.map(c => `
                                <div class="analytics-list-item">
                                    <div class="analytics-list-label">
                                        <span class="color-indicator" style="background-color: ${c.color || '#6366f1'};"></span>
                                        <span>${c.name}</span>
                                    </div>
                                    <div class="analytics-list-value">
                                        ${c.taskCount} tasks (${c.completedCount} completed)
                                    </div>
                                </div>
                            `).join('')}
                        </div>
                    ` : ''}
                </div>

                <!-- User Stats Tab -->
                <div class="analytics-panel" data-analytics-panel="user">
                    <h3 style="margin-bottom: var(--spacing-lg);">User Progress</h3>
                    <div class="stats-grid">
                        <div class="stat-card">
                            <i class="fas fa-trophy"></i>
                            <div class="stat-value">${achievements.unlocked || 0}/${achievements.total || 0}</div>
                            <div class="stat-label">Achievements</div>
                        </div>
                        <div class="stat-card">
                            <i class="fas fa-level-up-alt"></i>
                            <div class="stat-value">${currentLevel}</div>
                            <div class="stat-label">Current Level</div>
                        </div>
                        <div class="stat-card">
                            <i class="fas fa-star"></i>
                            <div class="stat-value">${totalXP}</div>
                            <div class="stat-label">Total XP</div>
                        </div>
                        <div class="stat-card">
                            <i class="fas fa-fire"></i>
                            <div class="stat-value">${currentStreak}</div>
                            <div class="stat-label">Current Streak</div>
                        </div>
                        <div class="stat-card">
                            <i class="fas fa-chart-line"></i>
                            <div class="stat-value">${longestStreak}</div>
                            <div class="stat-label">Longest Streak</div>
                        </div>
                        <div class="stat-card">
                            <i class="fas fa-percentage"></i>
                            <div class="stat-value">${achievements.completionRate || 0}%</div>
                            <div class="stat-label">Achievement Completion</div>
                        </div>
                    </div>

                    <h3 style="margin: var(--spacing-2xl) 0 var(--spacing-lg);">Productivity Metrics</h3>
                    <div class="stats-grid">
                        <div class="stat-card">
                            <i class="fas fa-calendar-week"></i>
                            <div class="stat-value">${completedLast7Days}</div>
                            <div class="stat-label">Tasks Completed (7 Days)</div>
                        </div>
                        <div class="stat-card">
                            <i class="fas fa-calendar-alt"></i>
                            <div class="stat-value">${completedLast30Days}</div>
                            <div class="stat-label">Tasks Completed (30 Days)</div>
                        </div>
                        <div class="stat-card">
                            <i class="fas fa-chart-bar"></i>
                            <div class="stat-value">${(completedLast7Days / 7).toFixed(1)}</div>
                            <div class="stat-label">Avg Tasks/Day (7 Days)</div>
                        </div>
                        <div class="stat-card">
                            <i class="fas fa-chart-line"></i>
                            <div class="stat-value">${(completedLast30Days / 30).toFixed(1)}</div>
                            <div class="stat-label">Avg Tasks/Day (30 Days)</div>
                        </div>
                    </div>
                </div>
            </div>
        `;
    }

    async renderFocusMode() {
        // Access focusMode from app instance
        const focusMode = this.app?.focusMode;
        
        if (!focusMode) {
            return '<div class="error-message">Focus mode not initialized</div>';
        }
        
        const html = await focusMode.render();
        
        // Bind events after render
        setTimeout(() => focusMode.bindEvents(), 100);
        
        return html;
    }

    async renderChangelog() {
        const version = APP_VERSION;
        const changelog = CHANGELOG;
        
        const categoryIcons = {
            added: 'fa-plus-circle',
            changed: 'fa-edit',
            fixed: 'fa-wrench',
            removed: 'fa-minus-circle',
            security: 'fa-shield-alt'
        };
        
        const categoryLabels = {
            added: 'Added',
            changed: 'Changed',
            fixed: 'Fixed',
            removed: 'Removed',
            security: 'Security'
        };
        
        return `
            <div class="changelog-container">
                <div class="changelog-timeline">
                    ${changelog.map((release, index) => `
                        <div class="version-card ${index === 0 ? 'latest' : ''}">
                            <div class="version-header">
                                <div class="version-info">
                                    <h3 class="version-number">
                                        ${index === 0 ? '<span class="latest-badge"><i class="fas fa-star"></i> Latest</span>' : ''}
                                        Version ${release.version}
                                        ${release.codename ? `<span class="version-codename">"${release.codename}"</span>` : ''}
                                    </h3>
                                    <div class="version-meta">
                                        <span class="version-date">
                                            <i class="fas fa-calendar"></i>
                                            ${new Date(release.releaseDate).toLocaleDateString('en-US', { 
                                                year: 'numeric', 
                                                month: 'long', 
                                                day: 'numeric' 
                                            })}
                                        </span>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="version-changes">
                                ${Object.entries(release.changes).map(([category, items]) => items.length > 0 ? `
                                    <div class="change-category">
                                        <h4 class="category-title ${category}">
                                            <i class="fas ${categoryIcons[category]}"></i>
                                            ${categoryLabels[category]}
                                        </h4>
                                        <ul class="change-list">
                                            ${items.map(item => `
                                                <li class="change-item">${item}</li>
                                            `).join('')}
                                        </ul>
                                    </div>
                                ` : '').join('')}
                            </div>
                        </div>
                    `).join('')}
                </div>
                
                <div class="changelog-footer">
                    <p class="footer-text">
                        <i class="fas fa-info-circle"></i>
                        For technical support or feature requests, contact the development team.
                    </p>
                </div>
            </div>
        `;
    }

    async renderSettings() {
        const template = document.getElementById('settings-page-template');
        if (!template) {
            return '<div class="error-message">Settings template not found</div>';
        }
        
        const settingsHTML = template.innerHTML;
        
        // Return the settings page HTML with the full structure from template
        return `
            <div class="settings-page">
                <div class="settings-header">
                    <h2><i class="fas fa-cog"></i> Settings</h2>
                    <p class="settings-subtitle">Customize your TaskList experience</p>
                </div>
                <div class="settings-container">
                    <div class="settings-tabs">
                        <button class="settings-tab active" data-tab="general">
                            <i class="fas fa-cog"></i> General
                        </button>
                        <button class="settings-tab" data-tab="appearance">
                            <i class="fas fa-palette"></i> Display
                        </button>
                        <button class="settings-tab" data-tab="notifications">
                            <i class="fas fa-bell"></i> Notifications
                        </button>
                    </div>
                    
                    <div class="settings-content">
                        <!-- General Settings -->
                        <div class="settings-panel active" id="general-settings">
                            <h3 class="panel-title">General Settings</h3>
                            
                            <!-- Tasks Subsection -->
                            <div class="settings-subsection">
                                <h4 class="subsection-title"><i class="fas fa-tasks"></i> Tasks</h4>
                                
                                <div class="setting-group">
                                    <label for="default-due-days">
                                        <span class="setting-label">Default Due Date</span>
                                        <small class="setting-description">Default number of days until a new task is due</small>
                                    </label>
                                    <select id="default-due-days" class="setting-input">
                                        <option value="1">Tomorrow (1 day)</option>
                                        <option value="7">1 Week</option>
                                        <option value="14">2 Weeks</option>
                                        <option value="30">1 Month</option>
                                    </select>
                                </div>
                                
                                <div class="setting-group">
                                    <label for="tasks-per-page">
                                        <span class="setting-label">Tasks Per Page</span>
                                        <small class="setting-description">Number of tasks to display per page</small>
                                    </label>
                                    <select id="tasks-per-page" class="setting-input">
                                        <option value="10">10</option>
                                        <option value="25" selected>25</option>
                                        <option value="50">50</option>
                                        <option value="100">100</option>
                                    </select>
                                </div>
                                
                                <div class="setting-group">
                                    <label for="default-task-priority">
                                        <span class="setting-label">Default Task Priority</span>
                                        <small class="setting-description">Priority pre-selected when creating new tasks</small>
                                    </label>
                                    <select id="default-task-priority" class="setting-input">
                                        ${CONFIG.FILTER_OPTIONS.priority.slice(1).map(option => 
                                            `<option value="${option.value}" ${option.value === CONFIG.DEFAULTS.task.priority ? 'selected' : ''}>${option.label}</option>`
                                        ).join('')}
                                    </select>
                                </div>
                                
                                <div class="setting-group checkbox-group">
                                    <label class="checkbox-label">
                                        <input type="checkbox" id="use-last-project" checked>
                                        <span class="checkmark"></span>
                                        <span class="checkbox-text">
                                            <span class="setting-label">Remember Last Project</span>
                                            <small class="setting-description">Use last selected project for new tasks</small>
                                        </span>
                                    </label>
                                </div>
                                
                                <div class="setting-group checkbox-group">
                                    <label class="checkbox-label">
                                        <input type="checkbox" id="use-last-category" checked>
                                        <span class="checkmark"></span>
                                        <span class="checkbox-text">
                                            <span class="setting-label">Remember Last Category</span>
                                            <small class="setting-description">Use last selected category for new tasks</small>
                                        </span>
                                    </label>
                                </div>
                            </div>
                            
                            <!-- Filters Subsection -->
                            <div class="settings-subsection">
                                <h4 class="subsection-title"><i class="fas fa-filter"></i> Default Filters</h4>
                                
                                <div class="setting-group">
                                    <label for="default-status-filter">
                                        <span class="setting-label">Default Status Filter</span>
                                        <small class="setting-description">Status filter selected by default when viewing tasks</small>
                                    </label>
                                    <select id="default-status-filter" class="setting-input">
                                        ${CONFIG.FILTER_OPTIONS.status.map(option => 
                                            `<option value="${option.value}" ${option.value === CONFIG.DEFAULTS.filters.status ? 'selected' : ''}>${option.label}</option>`
                                        ).join('')}
                                    </select>
                                </div>
                                
                                <div class="setting-group">
                                    <label for="default-priority-filter">
                                        <span class="setting-label">Default Priority Filter</span>
                                        <small class="setting-description">Priority filter selected by default when viewing tasks</small>
                                    </label>
                                    <select id="default-priority-filter" class="setting-input">
                                        ${CONFIG.FILTER_OPTIONS.priority.map(option => 
                                            `<option value="${option.value}" ${option.value === CONFIG.DEFAULTS.filters.priority ? 'selected' : ''}>${option.label}</option>`
                                        ).join('')}
                                    </select>
                                </div>
                            </div>
                            
                            <!-- System Subsection -->
                            <div class="settings-subsection">
                                <h4 class="subsection-title"><i class="fas fa-cogs"></i> System</h4>
                                
                                <div class="setting-group checkbox-group">
                                    <label class="checkbox-label">
                                        <input type="checkbox" id="auto-save">
                                        <span class="checkmark"></span>
                                        <span class="checkbox-text">
                                            <span class="setting-label">Auto-save Changes</span>
                                            <small class="setting-description">Automatically save changes without confirmation</small>
                                        </span>
                                    </label>
                                </div>
                                
                                <div class="setting-group checkbox-group">
                                    <label class="checkbox-label">
                                        <input type="checkbox" id="auto-complete-projects">
                                        <span class="checkmark"></span>
                                        <span class="checkbox-text">
                                            <span class="setting-label">Auto-complete Projects</span>
                                            <small class="setting-description">Automatically mark projects as complete when all tasks are done</small>
                                        </span>
                                    </label>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Appearance Settings -->
                        <div class="settings-panel" id="appearance-settings">
                            <h3 class="panel-title">Appearance</h3>
                            
                            <div class="settings-subsection">
                                <h4 class="subsection-title"><i class="fas fa-text-height"></i> Display</h4>
                                
                                <div class="setting-group">
                                    <label for="font-size">
                                        <span class="setting-label">Font Size</span>
                                        <small class="setting-description">Adjust the text size throughout the application</small>
                                    </label>
                                    <select id="font-size" class="setting-input">
                                        <option value="small">Small</option>
                                        <option value="medium" selected>Medium</option>
                                        <option value="large">Large</option>
                                    </select>
                                </div>
                                
                                <div class="setting-group checkbox-group">
                                    <label class="checkbox-label">
                                        <input type="checkbox" id="reduced-motion">
                                        <span class="checkmark"></span>
                                        <span class="checkbox-text">
                                            <span class="setting-label">Reduce Animations</span>
                                            <small class="setting-description">Minimize motion effects for better accessibility</small>
                                        </span>
                                    </label>
                                </div>
                            </div>
                            
                            <div class="setting-info">
                                <i class="fas fa-info-circle"></i>
                                <div class="info-content">
                                    <p><strong>Theme Settings</strong></p>
                                    <p>Theme and color preferences can be changed from your <a href="/account/#tab=preferences">Account Preferences</a>.</p>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Notification Settings -->
                        <div class="settings-panel" id="notifications-settings">
                            <h3 class="panel-title">Notifications</h3>
                            
                            <div class="settings-subsection">
                                <h4 class="subsection-title"><i class="fas fa-bell"></i> Notification Preferences</h4>
                                
                                <div class="setting-group checkbox-group">
                                    <label class="checkbox-label">
                                        <input type="checkbox" id="browser-notifications" checked>
                                        <span class="checkmark"></span>
                                        <span class="checkbox-text">
                                            <span class="setting-label">Browser Notifications</span>
                                            <small class="setting-description">Show desktop notifications for important events</small>
                                        </span>
                                    </label>
                                </div>
                                
                                <div class="setting-group checkbox-group">
                                    <label class="checkbox-label">
                                        <input type="checkbox" id="task-reminders" checked>
                                        <span class="checkmark"></span>
                                        <span class="checkbox-text">
                                            <span class="setting-label">Task Due Date Reminders</span>
                                            <small class="setting-description">Receive notifications when tasks are due soon</small>
                                        </span>
                                    </label>
                                </div>
                                
                                <div class="setting-group">
                                    <label for="reminder-time">
                                        <span class="setting-label">Reminder Time</span>
                                        <small class="setting-description">How far in advance to receive due date reminders</small>
                                    </label>
                                    <select id="reminder-time" class="setting-input">
                                        <option value="15">15 minutes before</option>
                                        <option value="30">30 minutes before</option>
                                        <option value="60" selected>1 hour before</option>
                                        <option value="1440">1 day before</option>
                                    </select>
                                </div>
                                
                                <div class="setting-group checkbox-group">
                                    <label class="checkbox-label">
                                        <input type="checkbox" id="sound-notifications">
                                        <span class="checkmark"></span>
                                        <span class="checkbox-text">
                                            <span class="setting-label">Sound Notifications</span>
                                            <small class="setting-description">Play sound effects for notifications</small>
                                        </span>
                                    </label>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="settings-actions">
                    <button type="button" class="btn btn-outline" id="reset-settings">Reset to Defaults</button>
                </div>
            </div>
        `;
    }

    async renderAccountSettings() {
        return `
            <div class="accountsettings-container">
                <!-- Tab Panels -->
                <div class="accountsettings-panels">
                    <!-- Account Panel -->
                    <div class="accountsettings-panel active" data-accountsettings-panel="account">
                        ${await this.renderAccountTab()}
                    </div>
                    
                    <!-- Progress Panel -->
                    <div class="accountsettings-panel" data-accountsettings-panel="progress">
                        ${await this.renderProgressTab()}
                    </div>
                    
                    <!-- History Panel -->
                    <div class="accountsettings-panel" data-accountsettings-panel="history">
                        ${await this.renderHistoryTab()}
                    </div>
                    
                    <!-- Settings Panel -->
                    <div class="accountsettings-panel" data-accountsettings-panel="settings">
                        ${await this.renderSettingsTab()}
                    </div>
                </div>
            </div>
        `;
    }
    
    async renderAccountTab() {
        const user = this.state.get('user') || {};
        
        // Debug logging
        logger.debug('[Account Tab] User data:', user);
        
        // Get name - check various property names
        const firstName = user.first_name || user.firstname || user.name || '';
        const lastName = user.last_name || user.lastname || '';
        const fullName = firstName || lastName ? `${firstName} ${lastName}`.trim() : 'User';
        
        // Get username
        const username = user.username || user.user_name || user.login || 'N/A';
        
        // Get email
        const email = user.email || user.user_email || 'N/A';
        
        // Get member since date
        const memberSince = user.created_at || user.created || user.registration_date;
        const memberSinceFormatted = memberSince ? new Date(memberSince).toLocaleDateString() : 'N/A';
        
        // Get role display
        const roleDisplay = {
            'admin': '<span class="role-badge admin"><i class="fas fa-crown"></i> Administrator</span>',
            'moderator': '<span class="role-badge moderator"><i class="fas fa-shield-alt"></i> Moderator</span>',
            'user': '<span class="role-badge user"><i class="fas fa-user"></i> User</span>'
        }[user.role || 'user'] || '<span class="role-badge user"><i class="fas fa-user"></i> User</span>';
        
        // Calculate account age in days
        let accountAgeDays = 0;
        if (memberSince) {
            const createdDate = new Date(memberSince);
            const now = new Date();
            accountAgeDays = Math.floor((now - createdDate) / (1000 * 60 * 60 * 24));
        }
        
        return `
            <div class="account-page">
                <div class="account-content">
                    <div class="profile-header-card">
                        <div class="profile-avatar-large">
                            ${firstName ? firstName.charAt(0).toUpperCase() : 'U'}
                        </div>
                        <div class="profile-header-info">
                            <h2>${fullName}</h2>
                            <p class="profile-username">@${username}</p>
                            ${roleDisplay}
                        </div>
                    </div>
                    
                    <div class="profile-info">
                        <h4><i class="fas fa-id-card"></i> Account Information</h4>
                        <div class="info-grid">
                            <div class="info-item">
                                <span class="info-label">Full Name:</span>
                                <span class="info-value">${fullName}</span>
                            </div>
                            <div class="info-item">
                                <span class="info-label">Username:</span>
                                <span class="info-value">${username}</span>
                            </div>
                            <div class="info-item">
                                <span class="info-label">Email Address:</span>
                                <span class="info-value">${email}</span>
                            </div>
                            <div class="info-item">
                                <span class="info-label">Account Role:</span>
                                <span class="info-value">${user.role || 'user'}</span>
                            </div>
                            <div class="info-item">
                                <span class="info-label">Member Since:</span>
                                <span class="info-value">${memberSinceFormatted}</span>
                            </div>
                            <div class="info-item">
                                <span class="info-label">Account Age:</span>
                                <span class="info-value">${accountAgeDays} days</span>
                            </div>
                        </div>
                    </div>
                    
                    <div class="profile-actions">
                        <h4><i class="fas fa-tools"></i> Account Actions</h4>
                        <div class="action-buttons">
                            <button class="btn btn-outline" onclick="app.ui.navigateToView('accountsettings'); app.ui.switchAccountSettingsTab('settings');">
                                <i class="fas fa-cog"></i> Edit Preferences
                            </button>
                            <button class="btn btn-outline" onclick="window.location.href = '/account/';">
                                <i class="fas fa-user-edit"></i> Edit Profile
                            </button>
                            <button class="btn btn-outline" onclick="window.location.href = '/account/';">
                                <i class="fas fa-key"></i> Change Password
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        `;
    }
    
    async renderProgressTab() {
        return `
            <div class="account-page">
                <div class="account-content">
                    <div class="xp-overview">
                        <div class="xp-stats">
                            <div class="stat-item">
                                <div class="stat-value" id="current-xp">0</div>
                                <div class="stat-label">Current XP</div>
                            </div>
                            <div class="stat-item">
                                <div class="stat-value" id="current-level">1</div>
                                <div class="stat-label">Level</div>
                            </div>
                            <div class="stat-item">
                                <div class="stat-value" id="xp-to-next">40</div>
                                <div class="stat-label">XP to Next Level</div>
                            </div>
                        </div>
                        
                        <div class="level-progress">
                            <div class="progress-info">
                                <span>Level Progress</span>
                                <span id="level-progress-text">60%</span>
                            </div>
                            <div class="progress-bar">
                                <div class="progress-fill" id="level-progress-fill" style="width: 60%;"></div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="recent-activity">
                        <h4>Recent XP Activity</h4>
                        <div class="activity-list" id="recent-xp-activity">
                            <div class="activity-item">
                                <div class="activity-icon"><i class="fas fa-check"></i></div>
                                <div class="activity-text">Task completed</div>
                                <div class="activity-xp">+10 XP</div>
                                <div class="activity-time">2 hours ago</div>
                            </div>
                            <div class="activity-item">
                                <div class="activity-icon"><i class="fas fa-project-diagram"></i></div>
                                <div class="activity-text">Project completed</div>
                                <div class="activity-xp">+50 XP</div>
                                <div class="activity-time">1 day ago</div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="progress-stats">
                        <h4>Productivity Metrics</h4>
                        <div class="stats-grid">
                            <div class="stat-card">
                                <div class="stat-number" id="total-xp-earned">0</div>
                                <div class="stat-description">Total XP Earned</div>
                            </div>
                            <div class="stat-card">
                                <div class="stat-number" id="avg-xp-daily">0</div>
                                <div class="stat-description">Avg XP per Day</div>
                            </div>
                            <div class="stat-card">
                                <div class="stat-number" id="longest-streak">0</div>
                                <div class="stat-description">Longest Streak</div>
                            </div>
                            <div class="stat-card">
                                <div class="stat-number" id="total-achievements">0</div>
                                <div class="stat-description">Achievements Unlocked</div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        `;
    }
    
    async renderHistoryTab() {
        const tasks = this.state.get('tasks') || [];
        const completedTasks = tasks.filter(t => t.status === 'completed').slice(0, 50);
        
        return `
            <div class="history-page">
                <div class="history-content">
                    <div class="history-stats">
                        <div class="stat-card">
                            <div class="stat-icon"><i class="fas fa-check-circle"></i></div>
                            <div class="stat-info">
                                <div class="stat-value">${completedTasks.length}</div>
                                <div class="stat-label">Completed Tasks</div>
                            </div>
                        </div>
                        <div class="stat-card">
                            <div class="stat-icon"><i class="fas fa-calendar-check"></i></div>
                            <div class="stat-info">
                                <div class="stat-value">${completedTasks.filter(t => {
                                    const weekAgo = new Date();
                                    weekAgo.setDate(weekAgo.getDate() - 7);
                                    return new Date(t.completed_at || t.updated_at) >= weekAgo;
                                }).length}</div>
                                <div class="stat-label">Last 7 Days</div>
                            </div>
                        </div>
                        <div class="stat-card">
                            <div class="stat-icon"><i class="fas fa-calendar-alt"></i></div>
                            <div class="stat-info">
                                <div class="stat-value">${completedTasks.filter(t => {
                                    const monthAgo = new Date();
                                    monthAgo.setDate(monthAgo.getDate() - 30);
                                    return new Date(t.completed_at || t.updated_at) >= monthAgo;
                                }).length}</div>
                                <div class="stat-label">Last 30 Days</div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="completed-tasks-list">
                        <h4>Recently Completed Tasks</h4>
                        ${completedTasks.length > 0 ? `
                            <div class="history-tasks">
                                ${completedTasks.map(task => `
                                    <div class="history-task-item">
                                        <div class="task-icon">
                                            <i class="fas fa-check"></i>
                                        </div>
                                        <div class="task-details">
                                            <div class="task-title">${task.title}</div>
                                            <div class="task-meta">
                                                <span class="task-date">
                                                    <i class="fas fa-calendar"></i>
                                                    ${new Date(task.completed_at || task.updated_at).toLocaleDateString()}
                                                </span>
                                                ${task.priority ? `
                                                    <span class="task-priority priority-${task.priority}">
                                                        <i class="fas fa-flag"></i>
                                                        ${task.priority.charAt(0).toUpperCase() + task.priority.slice(1)}
                                                    </span>
                                                ` : ''}
                                            </div>
                                        </div>
                                    </div>
                                `).join('')}
                            </div>
                        ` : `
                            <div class="empty-state">
                                <i class="fas fa-inbox"></i>
                                <p>No completed tasks yet</p>
                            </div>
                        `}
                    </div>
                </div>
            </div>
        `;
    }
    
    async renderSettingsTab() {
        const template = document.getElementById('settings-page-template');
        if (!template) {
            return '<div class="error-message">Settings template not found</div>';
        }
        
        return template.innerHTML;
    }

    async bindViewEvents(viewName) {
        if (viewName === 'tasks') {
            this.bindTaskEvents();
        } else if (viewName === 'projects') {
            this.bindProjectEvents();
        } else if (viewName === 'categories') {
            // Category sort dropdown
            document.getElementById('category-sort')?.addEventListener('change', async (e) => {
                await window.app.sortCategories(e.target.value);
            });
        } else if (viewName === 'calendar') {
            // Calendar manager handles its own events
            if (this.app && this.app.calendar) {
                await this.app.calendar.init();
            }
        } else if (viewName === 'accountsettings' || viewName === 'account' || viewName === 'settings') {
            this.bindAccountSettingsEvents();
        } else if (viewName === 'achievements') {
            // Achievements stats are updated by the render method
            // No additional events needed
        } else if (viewName === 'analytics') {
            this.app.bindAnalyticsEvents();
        }
    }

    bindTaskEvents() {
        // Task selection
        document.querySelectorAll('.task-checkbox').forEach(checkbox => {
            checkbox.addEventListener('change', (e) => {
                const taskId = parseInt(e.target.closest('.task-item').dataset.taskId);
                window.app.toggleTaskSelection(taskId);
            });
        });

        // Filter toggle button
        document.getElementById('filter-toggle')?.addEventListener('click', () => {
            window.app.toggleFilterPanel();
        });

        // Filter pill buttons
        document.querySelectorAll('.filter-pill').forEach(pill => {
            pill.addEventListener('click', async () => {
                await window.app.toggleFilterPill(pill);
                await window.app.checkOnboardingTaskCompletion('use_filter');
            });
        });

        // Clear filters button
        document.getElementById('clear-filters-btn')?.addEventListener('click', async () => {
            await window.app.clearAllFilters();
        });
        
        // Bulk select button
        const bulkSelectBtn = document.getElementById('bulk-select-btn');
        if (bulkSelectBtn) {
            bulkSelectBtn.addEventListener('click', (e) => {
                e.stopPropagation();
                window.app.bulkActions.toggleSelectionMode();
            });
        }
        
        // Task sort dropdown
        const taskSort = document.getElementById('task-sort');
        if (taskSort) {
            taskSort.addEventListener('change', async (e) => {
                e.stopPropagation();
                await window.app.sortTasks(e.target.value);
            });
            taskSort.addEventListener('click', (e) => {
                e.stopPropagation();
            });
        }
        
        // Restore filter panel visibility from cache
        const panelVisible = localStorage.getItem('filterPanelVisible') === 'true';
        const panel = document.getElementById('filter-panel');
        const toggle = document.getElementById('filter-toggle');
        if (panelVisible && panel && toggle) {
            panel.style.display = 'block';
            toggle.classList.add('active');
        }
        
        // Update filter banner on load
        window.app.updateFilterBanner();
    }

    bindProjectEvents() {
        // Project sort dropdown
        document.getElementById('project-sort')?.addEventListener('change', async (e) => {
            await window.app.sortProjects(e.target.value);
        });
        
        // Project stat clicks for tasks view navigation
        document.querySelectorAll('.project-card .stat').forEach(stat => {
            stat.style.cursor = 'pointer'; // Make it clear it's clickable
            stat.addEventListener('click', async (e) => {
                e.preventDefault();
                e.stopPropagation();
                const card = stat.closest('.project-card');
                const projectId = parseInt(card.dataset.projectId);
                
                // Clear category filter and set project filter
                window.app.state.set('taskFilter', null);
                const filters = window.app.state.get('filters');
                filters.projects = [projectId];
                filters.categories = [];
                filters.statuses = [];
                filters.priorities = [];
                window.app.state.set('filters', filters);
                
                // Navigate to tasks view
                await window.app.navigateToView('tasks', true);
                
                // Show toast to inform user
                const projectName = window.app.state.get('projects').find(p => p.id === projectId)?.name || 'Project';
                window.app.modals.showToast(`Showing tasks in "${projectName}"`, 'info');
            });
        });

        // Project toggle completion button clicks
        document.querySelectorAll('.toggle-project-completion-btn').forEach(btn => {
            btn.addEventListener('click', async (e) => {
                e.preventDefault();
                e.stopPropagation();
                const projectId = parseInt(btn.dataset.projectId);
                await window.app.toggleProjectCompletion(projectId);
            });
        });

        // Project edit button clicks
        document.querySelectorAll('.edit-project-btn').forEach(btn => {
            btn.addEventListener('click', (e) => {
                e.preventDefault();
                e.stopPropagation();
                const projectId = parseInt(btn.dataset.projectId);
                window.app.editProject(projectId);
            });
        });

        // Project delete button clicks
        document.querySelectorAll('.delete-project-btn').forEach(btn => {
            btn.addEventListener('click', (e) => {
                e.preventDefault();
                e.stopPropagation();
                const projectId = parseInt(btn.dataset.projectId);
                window.app.deleteProject(projectId);
            });
        });

        // Project share button clicks
        document.querySelectorAll('.share-project-btn').forEach(btn => {
            btn.addEventListener('click', (e) => {
                e.preventDefault();
                e.stopPropagation();
                const projectId = parseInt(btn.dataset.projectId);
                window.app.sharing.openShareModal(projectId);
            });
        });
    }

    async bindAccountSettingsEvents() {
        // Bind tab switching
        document.querySelectorAll('.accountsettings-tab').forEach(tab => {
            tab.addEventListener('click', async (e) => {
                const tabName = e.currentTarget.dataset.accountsettingsTab;
                await this.switchAccountSettingsTab(tabName);
            });
        });
        
        // Load initial tab content based on URL or default
        const urlParams = new URLSearchParams(window.location.search);
        const tab = urlParams.get('tab') || 'account';
        await this.switchAccountSettingsTab(tab);
    }
    
    async switchAccountSettingsTab(tabName) {
        // Update tab buttons
        document.querySelectorAll('.accountsettings-tab').forEach(tab => {
            tab.classList.toggle('active', tab.dataset.accountsettingsTab === tabName);
        });
        
        // Update tab panels
        document.querySelectorAll('.accountsettings-panel').forEach(panel => {
            panel.classList.toggle('active', panel.dataset.accountsettingsPanel === tabName);
        });
        
        // Load data for specific tabs
        if (tabName === 'account') {
            await this.loadAccountOverview();
        } else if (tabName === 'progress') {
            await window.app.updateUserXP();
            await this.loadAccountOverview();
        } else if (tabName === 'settings') {
            await this.loadCurrentSettings();
            await this.bindSettingsEvents();
        }
        
        // Update URL without reload
        const url = new URL(window.location);
        url.searchParams.set('tab', tabName);
        window.history.replaceState({}, '', url);
    }

    async bindAccountEvents() {
        // Ensure we have the latest XP data, then load account overview
        logger.debug('Account Events - Starting XP update...');
        await window.app.updateUserXP();
        logger.debug('Account Events - XP update complete, loading account overview...');
        await this.loadAccountOverview();
        logger.debug('Account Events - Account overview loaded');
    }

    switchSettingsTab(tabName) {
        // Update active tab
        document.querySelectorAll('.settings-tab').forEach(t => t.classList.remove('active'));
        const activeTab = document.querySelector(`.settings-tab[data-tab="${tabName}"]`);
        if (activeTab) activeTab.classList.add('active');
        
        // Update active panel
        document.querySelectorAll('.settings-panel').forEach(panel => {
            panel.classList.toggle('active', panel.id === `${tabName}-settings`);
        });
    }

    async bindSettingsEvents() {
        // Load current settings when settings page is rendered
        await this.loadCurrentSettings();
        
        // Restore last active settings tab (default to 'general')
        const lastSettingsTab = localStorage.getItem('TL_lastSettingsTab') || 'general';
        this.switchSettingsTab(lastSettingsTab);
        
        // Settings tabs
        document.querySelectorAll('.settings-tab').forEach(tab => {
            tab.addEventListener('click', () => {
                const tabName = tab.dataset.tab;
                this.switchSettingsTab(tabName);
                
                // Save the active tab to localStorage
                localStorage.setItem('TL_lastSettingsTab', tabName);
            });
        });
        
        // Auto-save on any setting change
        const settingInputs = [
            'default-due-days', 'tasks-per-page', 'default-status-filter',
            'default-priority-filter', 'default-task-priority', 'auto-save',
            'use-last-project', 'use-last-category', 'auto-complete-projects',
            'font-size', 'reduced-motion', 'browser-notifications',
            'task-reminders', 'reminder-time', 'sound-notifications'
        ];
        
        settingInputs.forEach(inputId => {
            const element = document.getElementById(inputId);
            if (element) {
                element.addEventListener('change', () => {
                    this.saveSettings();
                });
            }
        });
        
        // Reset settings button
        document.getElementById('reset-settings')?.addEventListener('click', () => {
            this.resetSettings();
        });
    }

    initializeVirtualScrolling() {
        // Destroy existing scroller if any
        if (this.virtualScroller) {
            this.virtualScroller.destroy();
            this.virtualScroller = null;
        }

        const container = document.getElementById('tasks-list-container');
        if (!container) return;

        let tasks = this.state.get('tasks');
        const filters = this.state.get('filters');
        const taskFilter = this.state.get('taskFilter');

        // Apply filters
        if (taskFilter && taskFilter.categoryId) {
            tasks = tasks.filter(task => task.category_id === taskFilter.categoryId);
        }
        if (filters.project !== 'all') {
            tasks = tasks.filter(task => task.project_id === parseInt(filters.project));
        }
        if (filters.status !== 'all') {
            tasks = tasks.filter(task => task.status === filters.status);
        }
        if (filters.priority !== 'all') {
            tasks = tasks.filter(task => task.priority === filters.priority);
        }

        // Only use virtual scrolling for large lists
        const virtualScrollingConfig = window.CONFIG?.virtualScrolling || { enabled: true, threshold: 50, batchSize: 20 };
        const useVirtualScrolling = virtualScrollingConfig.enabled && 
                                    tasks.length >= virtualScrollingConfig.threshold;

        if (!useVirtualScrolling) {
            // Render all tasks normally
            container.innerHTML = tasks.length === 0 
                ? '<div class="empty-state"><p>No tasks found. Create your first task!</p></div>'
                : tasks.map(task => this.renderTaskItem(task)).join('');
            return;
        }

        // Initialize virtual scroller
        this.virtualScroller = new VirtualScroller({
            container: container,
            items: tasks,
            renderItem: (task) => this.renderTaskItem(task),
            itemHeight: 100,
            bufferSize: 5,
            batchSize: virtualScrollingConfig.batchSize
        });

        logger.debug(`[VirtualScrolling] Enabled for ${tasks.length} tasks`);
    }

    toggleTaskSelection(taskId) {
        const selected = this.state.get('selectedTasks');
        const index = selected.indexOf(taskId);
        
        if (index === -1) {
            selected.push(taskId);
        } else {
            selected.splice(index, 1);
        }
        
        this.state.set('selectedTasks', selected);
        
        // Update UI
        const taskItem = document.querySelector(`[data-task-id="${taskId}"]`);
        if (taskItem) {
            taskItem.classList.toggle('selected', index === -1);
        }
    }

    applyTaskFilters() {
        const filters = this.state.get('filters');
        const tasks = document.querySelectorAll('.task-item');
        
        tasks.forEach(taskEl => {
            const taskId = parseInt(taskEl.dataset.taskId);
            const task = this.state.get('tasks').find(t => t.id === taskId);
            
            if (!task) return;
            
            let visible = true;
            
            if (filters.project !== 'all' && task.project_id !== parseInt(filters.project)) {
                visible = false;
            }
            
            if (filters.status !== 'all' && task.status !== filters.status) {
                visible = false;
            }
            
            if (filters.priority !== 'all' && task.priority !== filters.priority) {
                visible = false;
            }
            
            taskEl.style.display = visible ? 'flex' : 'none';
        });
    }

    renderError(message) {
        const contentBody = document.getElementById('content-body');
        if (contentBody) {
            contentBody.innerHTML = `
                <div class="error-state">
                    <div class="error-icon">
                        <i class="fas fa-exclamation-triangle"></i>
                    </div>
                    <h3>Something went wrong</h3>
                    <p>${this.escapeHtml(message)}</p>
                    <button class="btn btn-primary" onclick="window.location.reload()">
                        <i class="fas fa-refresh"></i> Reload
                    </button>
                </div>
            `;
        } else {
            logger.error('Content body element not found, error message:', message);
        }
    }

    showSkeletonLoader(viewName, container) {
        if (!container || typeof SkeletonLoader === 'undefined') {
            return;
        }

        const skeletonMap = {
            'dashboard': () => {
                const isMobile = window.innerWidth <= 768;
                return isMobile ? SkeletonLoader.featuredStat() : SkeletonLoader.statsGrid(5);
            },
            'tasks': () => SkeletonLoader.tasks(8),
            'projects': () => SkeletonLoader.projects(4),
            'categories': () => SkeletonLoader.categories(6),
            'achievements': () => SkeletonLoader.tasks(5),
            'account': () => '<div class="skeleton-container"><div class="skeleton skeleton-text"></div></div>',
            'settings': () => '<div class="skeleton-container"><div class="skeleton skeleton-text"></div></div>'
        };

        const skeletonHTML = skeletonMap[viewName] 
            ? skeletonMap[viewName]() 
            : SkeletonLoader.tasks(5);

        container.innerHTML = skeletonHTML;
    }

    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    formatDate(dateString) {
        return new Date(dateString).toLocaleDateString();
    }

    // Cycle featured tracker on mobile dashboard
    async cycleFeaturedTracker() {
        const trackerOrder = ['total_tasks', 'completed_tasks', 'pending_tasks', 'in_progress_tasks', 'xp_total'];
        const currentTracker = localStorage.getItem('TL_featuredTracker') || 'total_tasks';
        const currentIndex = trackerOrder.indexOf(currentTracker);
        const nextIndex = (currentIndex + 1) % trackerOrder.length;
        const nextTracker = trackerOrder[nextIndex];
        
        localStorage.setItem('TL_featuredTracker', nextTracker);
        
        // Re-render dashboard to show new featured tracker
        await this.renderView('dashboard');
    }

    // Subtasks in task list view
    toggleTaskSubtasks(taskId) {
        const subtasksDiv = document.getElementById(`task-subtasks-${taskId}`);
        const taskItem = document.querySelector(`[data-task-id="${taskId}"]`);
        const toggleBtn = taskItem ? taskItem.querySelector('.subtasks-toggle i') : null;
        
        if (subtasksDiv) {
            const isVisible = subtasksDiv.style.display === 'block';
            subtasksDiv.style.display = isVisible ? 'none' : 'block';
            
            if (toggleBtn) {
                toggleBtn.className = isVisible ? 'fas fa-chevron-down' : 'fas fa-chevron-up';
            }
        }
    }

    async toggleTaskSubtaskStatus(taskId, subtaskId) {
        try {
            // Check if subtasks are currently expanded before reload
            const subtasksDiv = document.getElementById(`task-subtasks-${taskId}`);
            const wasExpanded = subtasksDiv && subtasksDiv.style.display === 'block';
            
            const response = await fetch(`${CONFIG.API_BASE_URL}subtasks/${subtaskId}/toggle`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                credentials: 'include'
            });
            
            const result = await response.json();
            
            if (result.success) {
                const subtask = result.data.subtask;
                
                // Award XP if subtask was just completed
                if (subtask.is_completed) {
                    const wordCount = subtask.title.trim().split(/\s+/).length;
                    let xpAmount = 2; // Base XP
                    
                    // Bonus XP for descriptive subtasks (4-8 words)
                    if (wordCount >= 4 && wordCount <= 8) {
                        xpAmount += 3; // Total 5 XP
                    }
                    
                    // Award XP via API
                    try {
                        await window.app.api.post('xp/log', {
                            source_type: 'subtask_completed',
                            xp_amount: xpAmount,
                            description: subtask.title
                        });
                        
                        // Update user XP display
                        await window.app.updateUserXP();
                        
                        // Show XP notification
                        const bonus = xpAmount > 2 ? ' (with description bonus!)' : '';
                        this.showToast(`+${xpAmount} XP${bonus}`, 'success');
                    } catch (error) {
                        logger.error('Failed to award XP:', error);
                    }
                }
                
                // Reload tasks to update the view
                await window.app.loadTasks();
                await this.renderView('tasks');
                
                // Re-expand the subtasks if they were visible before
                if (wasExpanded) {
                    // Use setTimeout to ensure DOM is fully rendered
                    setTimeout(() => {
                        const newSubtasksDiv = document.getElementById(`task-subtasks-${taskId}`);
                        const taskItem = document.querySelector(`[data-task-id="${taskId}"]`);
                        const toggleBtn = taskItem ? taskItem.querySelector('.subtasks-toggle i') : null;
                        
                        if (newSubtasksDiv) {
                            newSubtasksDiv.style.display = 'block';
                        }
                        if (toggleBtn) {
                            toggleBtn.className = 'fas fa-chevron-up';
                        }
                    }, 0);
                }
            }
        } catch (error) {
            logger.error('Failed to toggle subtask:', error);
        }
    }

    // Settings management methods
    async loadCurrentSettings() {
        try {
            // Load current settings from user database
            const userSettings = await this.api.getUserSettings();
            
            // Wait for DOM to be ready
            await new Promise(resolve => setTimeout(resolve, 50));
            
            // NO CONVERSION - use values EXACTLY as they come from database
            // The database returns proper booleans, so use them directly
            
            // Ensure settings elements exist before trying to set values
            // For checkboxes: use the EXACT value from DB, or sensible defaults for undefined
            const elements = {
                'default-due-days': userSettings.TL_defaultDueDays || CONFIG.DEFAULTS.task.dueDays.toString(),
                'tasks-per-page': userSettings.TL_tasksPerPage || CONFIG.DEFAULTS.pagination.tasksPerPage.toString(),
                'default-status-filter': userSettings.TL_defaultStatusFilter || CONFIG.DEFAULTS.filters.status,
                'default-priority-filter': userSettings.TL_defaultPriorityFilter || CONFIG.DEFAULTS.filters.priority,
                'default-task-priority': userSettings.TL_defaultTaskPriority || CONFIG.DEFAULTS.task.priority,
                // Checkboxes: Use exact DB value, default to true if undefined
                'auto-save': userSettings.TL_autoSave === undefined ? true : userSettings.TL_autoSave,
                'use-last-project': userSettings.TL_useLastProject === undefined ? true : userSettings.TL_useLastProject,
                'use-last-category': userSettings.TL_useLastCategory === undefined ? true : userSettings.TL_useLastCategory,
                'auto-complete-projects': userSettings.TL_autoCompleteProjects === undefined ? false : userSettings.TL_autoCompleteProjects,
                'font-size': userSettings.TL_fontSize || CONFIG.UI.theme.defaultFontSize,
                'reduced-motion': userSettings.TL_reducedMotion === undefined ? false : userSettings.TL_reducedMotion,
                'browser-notifications': userSettings.TL_browserNotifications === undefined ? true : userSettings.TL_browserNotifications,
                'task-reminders': userSettings.TL_taskReminders === undefined ? true : userSettings.TL_taskReminders,
                'reminder-time': userSettings.TL_reminderTime || CONFIG.DEFAULTS.notifications.reminderTime.toString(),
                'sound-notifications': userSettings.TL_soundNotifications === undefined ? false : userSettings.TL_soundNotifications,
                'show-completed-in-search': userSettings.TL_showCompletedInSearch === undefined ? false : userSettings.TL_showCompletedInSearch
            };
            
            // Set values for existing elements
            Object.entries(elements).forEach(([elementId, value]) => {
                const element = document.getElementById(elementId);
                if (element) {
                    if (element.type === 'checkbox') {
                        element.checked = Boolean(value);
                    } else {
                        element.value = value;
                    }
                }
            });
            
        } catch (error) {
            logger.error('Error loading user settings:', error);
            // Fall back to default values if loading fails
            this.setDefaultSettings();
        }
    }

    setDefaultSettings() {
        // Set default values if settings can't be loaded
        document.getElementById('default-due-days').value = CONFIG.DEFAULTS.task.dueDays.toString();
        document.getElementById('tasks-per-page').value = CONFIG.DEFAULTS.pagination.tasksPerPage.toString();
        document.getElementById('default-status-filter').value = CONFIG.DEFAULTS.filters.status;
        document.getElementById('default-priority-filter').value = CONFIG.DEFAULTS.filters.priority;
        document.getElementById('default-task-priority').value = CONFIG.DEFAULTS.task.priority;
        document.getElementById('auto-save').checked = true;
        document.getElementById('use-last-project').checked = true;
        document.getElementById('use-last-category').checked = true;
        document.getElementById('auto-complete-projects').checked = false; // Default OFF as requested
        document.getElementById('font-size').value = CONFIG.UI.theme.defaultFontSize;
        document.getElementById('reduced-motion').checked = CONFIG.UI.theme.reducedMotion;
        document.getElementById('browser-notifications').checked = CONFIG.DEFAULTS.notifications.browserNotifications;
        document.getElementById('task-reminders').checked = CONFIG.DEFAULTS.notifications.taskReminders;
        document.getElementById('reminder-time').value = CONFIG.DEFAULTS.notifications.reminderTime.toString();
        document.getElementById('sound-notifications').checked = CONFIG.DEFAULTS.notifications.soundNotifications;
    }

    async loadAccountOverview() {
        try {
            // Get current user XP data from multiple sources
            const currentUser = this.state.get('user') || {};
            let currentXP = currentUser.xp_total || 0;
            
            // If no XP in state, try to get from the header element
            if (currentXP === 0) {
                const xpElement = document.getElementById('xp-amount');
                if (xpElement) {
                    currentXP = parseInt(xpElement.textContent) || 0;
                }
            }
            
            // If still no XP, set a test value to see if elements are working
            if (currentXP === 0) {
                currentXP = 60; // Test value to match what we expect
            }
            
            logger.debug('Account Overview Debug - Current XP:', currentXP, 'User State:', currentUser);
            
            const currentLevel = this.calculateLevel(currentXP);
            const xpForNextLevel = this.getXPForLevel(currentLevel + 1);
            const xpInCurrentLevel = currentXP - this.getXPForLevel(currentLevel);
            const xpNeededForNext = xpForNextLevel - this.getXPForLevel(currentLevel);
            const progressPercent = xpNeededForNext > 0 ? (xpInCurrentLevel / xpNeededForNext) * 100 : 0;
            
            // Update XP and level information
            const currentXpEl = document.getElementById('current-xp');
            const currentLevelEl = document.getElementById('current-level');
            const xpToNextEl = document.getElementById('xp-to-next');
            const progressTextEl = document.getElementById('level-progress-text');
            const progressFillEl = document.getElementById('level-progress-fill');
            
            logger.debug('Account Overview Debug - Elements found:', {
                currentXpEl: !!currentXpEl,
                currentLevelEl: !!currentLevelEl,
                xpToNextEl: !!xpToNextEl,
                progressTextEl: !!progressTextEl,
                progressFillEl: !!progressFillEl
            });
            
            if (currentXpEl) currentXpEl.textContent = currentXP;
            if (currentLevelEl) currentLevelEl.textContent = currentLevel;
            if (xpToNextEl) xpToNextEl.textContent = Math.max(0, xpForNextLevel - currentXP);
            if (progressTextEl) progressTextEl.textContent = `${Math.round(progressPercent)}%`;
            if (progressFillEl) progressFillEl.style.width = `${progressPercent}%`;
            
            // Calculate statistics from actual data
            let totalTasks = 0;
            let totalProjects = 0;
            let currentStreak = 0;
            let xpHistory = [];
            
            try {
                // Get real task and project data
                const tasks = await this.api.getTasks();
                const projects = await this.api.getProjects();
                
                logger.debug('Account Overview Debug - API Data:', { tasks, projects });
                
                // Count completed tasks (tasks use status === 'completed')
                if (tasks && tasks.data) {
                    totalTasks = tasks.data.filter(task => task.status === 'completed').length;
                    logger.debug('Account Overview Debug - Total completed tasks:', totalTasks, 'out of', tasks.data.length);
                } else if (tasks && Array.isArray(tasks)) {
                    totalTasks = tasks.filter(task => task.status === 'completed').length;
                    logger.debug('Account Overview Debug - Total completed tasks (array):', totalTasks, 'out of', tasks.length);
                }
                
                // Count completed projects (assuming inactive projects are completed)
                if (projects && projects.data) {
                    totalProjects = projects.data.filter(project => !project.is_active).length;
                    logger.debug('Account Overview Debug - Total completed projects:', totalProjects, 'out of', projects.data.length);
                } else if (projects && Array.isArray(projects)) {
                    totalProjects = projects.filter(project => !project.is_active).length;
                    logger.debug('Account Overview Debug - Total completed projects (array):', totalProjects, 'out of', projects.length);
                }
                
                // Try to get XP history (fallback to example data if not available)
                try {
                    const xpResponse = await this.api.getXPHistory();
                    // Handle both direct array and wrapped response
                    xpHistory = Array.isArray(xpResponse) ? xpResponse : (xpResponse?.data || []);
                } catch (xpError) {
                    logger.warn('Could not load XP history:', xpError);
                    // Provide some example XP activity data
                    xpHistory = [
                        {
                            xp_amount: 10,
                            source: 'task_completion',
                            earned_at: new Date(Date.now() - 2 * 60 * 60 * 1000).toISOString(), // 2 hours ago
                            description: 'Complete project setup documentation'
                        },
                        {
                            xp_amount: 50,
                            source: 'project_completion',
                            earned_at: new Date(Date.now() - 24 * 60 * 60 * 1000).toISOString(), // 1 day ago
                            description: 'Task Manager Development'
                        },
                        {
                            xp_amount: 10,
                            source: 'task_completion',
                            earned_at: new Date(Date.now() - 3 * 24 * 60 * 60 * 1000).toISOString(), // 3 days ago
                            description: 'Review user feedback and create improvement list'
                        }
                    ];
                }
                
            } catch (apiError) {
                logger.warn('Could not load task/project data:', apiError);
                // Set some test values so we can see if the elements are working
                totalTasks = 5;  // Test value
                totalProjects = 2;  // Test value
            }
            
            // Calculate account age
            let accountAge = 0;
            if (currentUser.created_at) {
                const createdDate = new Date(currentUser.created_at);
                const now = new Date();
                accountAge = Math.floor((now - createdDate) / (1000 * 60 * 60 * 24));
            }
            
            // Update account statistics
            const totalTasksEl = document.getElementById('total-tasks');
            const totalProjectsEl = document.getElementById('total-projects');
            const currentStreakEl = document.getElementById('current-streak');
            const totalXpEarnedEl = document.getElementById('total-xp-earned');
            const accountAgeEl = document.getElementById('account-age');
            const avgXpDailyEl = document.getElementById('avg-xp-daily');
            const longestStreakEl = document.getElementById('longest-streak');
            const totalAchievementsEl = document.getElementById('total-achievements');
            
            logger.debug('Account Overview Debug - Statistics elements found:', {
                totalTasksEl: !!totalTasksEl,
                totalProjectsEl: !!totalProjectsEl,
                currentStreakEl: !!currentStreakEl,
                totalXpEarnedEl: !!totalXpEarnedEl,
                accountAgeEl: !!accountAgeEl
            });
            
            logger.debug('Account Overview Debug - Setting values:', { totalTasks, totalProjects, currentStreak, currentXP, accountAge });
            
            if (totalTasksEl) totalTasksEl.textContent = totalTasks;
            if (totalProjectsEl) totalProjectsEl.textContent = totalProjects;
            if (currentStreakEl) currentStreakEl.textContent = currentStreak;
            if (totalXpEarnedEl) totalXpEarnedEl.textContent = currentXP;
            if (accountAgeEl) accountAgeEl.textContent = accountAge;
            
            // Update progress tab stats if they exist
            if (avgXpDailyEl) {
                const avgXp = accountAge > 0 ? Math.round(currentXP / accountAge) : 0;
                avgXpDailyEl.textContent = avgXp;
            }
            if (longestStreakEl) longestStreakEl.textContent = this.app.achievements?.streak?.longest || 0;
            if (totalAchievementsEl) totalAchievementsEl.textContent = this.app.achievements?.getStats()?.unlocked || 0;
            
            // Update recent XP activity
            this.updateRecentXPActivity(xpHistory);
            
        } catch (error) {
            logger.error('Error loading account overview:', error);
            // Set fallback values for XP and stats
            document.getElementById('current-xp').textContent = '0';
            document.getElementById('current-level').textContent = '1';
            document.getElementById('xp-to-next').textContent = '100';
        }
    }

    // XP calculation helper methods
    calculateLevel(totalXP) {
        // Scaled level calculation with tier-based progression
        // Formula: 100 + level * (10 + tier_multiplier)
        // Tier 0 (L1-9):   multiplier = 0
        // Tier 1 (L10-19): multiplier = 10
        // Tier 2 (L20-29): multiplier = 30
        // Tier 3 (L30-39): multiplier = 70
        let currentLevel = 1;
        let xpAccumulator = 0;
        
        while (true) {
            const tier = Math.floor((currentLevel - 1) / 10);
            let tierMultiplier = 0;
            for (let i = 0; i < tier; i++) {
                tierMultiplier += (10 * Math.pow(2, i));
            }
            
            const xpForNextLevel = 100 + (currentLevel * (10 + tierMultiplier));
            
            if (xpAccumulator + xpForNextLevel > totalXP) {
                break;
            }
            xpAccumulator += xpForNextLevel;
            currentLevel++;
        }
        
        return currentLevel;
    }

    getXPForLevel(level) {
        // Calculate total XP needed to reach this level
        let totalXP = 0;
        
        for (let currentLevel = 1; currentLevel < level; currentLevel++) {
            const tier = Math.floor((currentLevel - 1) / 10);
            let tierMultiplier = 0;
            for (let i = 0; i < tier; i++) {
                tierMultiplier += (10 * Math.pow(2, i));
            }
            
            totalXP += 100 + (currentLevel * (10 + tierMultiplier));
        }
        
        return totalXP;
    }

    updateRecentXPActivity(xpHistory) {
        const activityList = document.getElementById('recent-xp-activity');
        if (!activityList) return;
        
        // Clear existing activity items
        activityList.innerHTML = '';
        
        // Ensure xpHistory is an array
        if (!Array.isArray(xpHistory)) {
            xpHistory = [];
        }
        
        // If no XP history, show placeholder message
        if (!xpHistory || xpHistory.length === 0) {
            activityList.innerHTML = '<div class="no-activity">No XP activity yet. Complete some tasks to see your progress!</div>';
            return;
        }
        
        // Show last 5 XP entries
        const recentEntries = xpHistory.slice(0, 5);
        
        recentEntries.forEach(entry => {
            const activityItem = document.createElement('div');
            activityItem.className = 'activity-item';
            
            // Determine icon and title based on action type
            let icon = 'fas fa-check';
            let activityTitle = 'Task Completed';
            
            if (entry.action_type === 'project_completed') {
                icon = 'fas fa-project-diagram';
                activityTitle = 'Project Completed';
            } else if (entry.action_type === 'streak_bonus') {
                icon = 'fas fa-fire';
                activityTitle = 'Streak Bonus';
            } else if (entry.action_type === 'achievement_unlocked') {
                icon = 'fas fa-trophy';
                activityTitle = 'Achievement Unlocked';
            }
            
            // Format time ago
            const timeAgo = this.formatTimeAgo(new Date(entry.earned_at));
            
            // Create breakdown display
            let breakdownText = '';
            if (entry.breakdown) {
                const breakdownParts = [];
                for (const [key, value] of Object.entries(entry.breakdown)) {
                    if (value > 0) {
                        const displayKey = key.replace('_', ' ').replace(/\b\w/g, l => l.toUpperCase());
                        breakdownParts.push(`${displayKey}: +${value}`);
                    }
                }
                if (breakdownParts.length > 0) {
                    breakdownText = `<div class="activity-breakdown">${breakdownParts.join(', ')}</div>`;
                }
            }
            
            activityItem.innerHTML = `
                <div class="activity-icon"><i class="${icon}"></i></div>
                <div class="activity-content">
                    <div class="activity-title">${activityTitle}</div>
                    <div class="activity-description">${entry.description || 'No description'}</div>
                    ${breakdownText}
                </div>
                <div class="activity-xp">+${entry.xp_amount} XP</div>
                <div class="activity-time">${timeAgo}</div>
            `;
            
            activityList.appendChild(activityItem);
        });
    }

    formatTimeAgo(date) {
        const now = new Date();
        const diffMs = now - date;
        const diffSecs = Math.floor(diffMs / 1000);
        const diffMins = Math.floor(diffSecs / 60);
        const diffHours = Math.floor(diffMins / 60);
        const diffDays = Math.floor(diffHours / 24);
        
        if (diffDays > 0) return `${diffDays} day${diffDays > 1 ? 's' : ''} ago`;
        if (diffHours > 0) return `${diffHours} hour${diffHours > 1 ? 's' : ''} ago`;
        if (diffMins > 0) return `${diffMins} minute${diffMins > 1 ? 's' : ''} ago`;
        return 'Just now';
    }

    async initializeUserSettings() {
        try {
            // Check if user already has TaskList settings
            const userSettings = await this.api.getUserSettings();
            
            // Check if any TL_ settings exist
            const hasTaskListSettings = Object.keys(userSettings).some(key => key.startsWith('TL_'));
            
            if (!hasTaskListSettings) {
                logger.info('First-time TaskList user detected. Initializing default settings...');
                
                // Create default TaskList settings
                const defaultSettings = {
                    TL_defaultDueDays: '1',
                    TL_tasksPerPage: '25',
                    TL_autoSave: true,
                    TL_useLastProject: true,
                    TL_useLastCategory: true,
                    TL_fontSize: 'medium',
                    TL_reducedMotion: false,
                    TL_browserNotifications: true,
                    TL_taskReminders: true,
                    TL_reminderTime: '60',
                    TL_soundNotifications: false,
                    TL_initialized: true, // Flag to indicate settings have been initialized
                    TL_version: APP_VERSION.version.split('.').slice(0, 2).join('.') // Version for future migrations
                };
                
                // Save default settings to user database
                await this.api.updateUserSettings(defaultSettings);
                logger.info('Default TaskList settings initialized successfully');
                
                return { ...defaultSettings, _isFirstTime: true };
            } else {
                logger.debug('Existing TaskList settings found');
                return { ...userSettings, _isFirstTime: false };
            }
        } catch (error) {
            logger.error('Error initializing user settings:', error);
            // Return default settings as fallback
            return {
                TL_defaultDueDays: '1',
                TL_tasksPerPage: '25',
                TL_autoSave: true,
                TL_useLastProject: true,
                TL_useLastCategory: true,
                TL_fontSize: 'medium',
                TL_reducedMotion: false,
                TL_browserNotifications: true,
                TL_taskReminders: true,
                TL_reminderTime: '60',
                TL_soundNotifications: false
            };
        }
    }

    async loadAndApplySettings() {
        try {
            // Load user settings from database
            const userSettings = await this.api.getUserSettings();
            
            // Store ALL settings in localStorage for app-wide access
            // Use JSON.stringify to preserve data types
            Object.entries(userSettings).forEach(([key, value]) => {
                localStorage.setItem(key, JSON.stringify(value));
            });
            
            // Apply theme setting (from old taskListSettings)
            const oldSettings = JSON.parse(localStorage.getItem('taskListSettings') || '{}');
            const theme = oldSettings.theme || 'auto';
            this.applyThemeSettings(theme);
            
            // Apply font size setting from database
            const fontSize = userSettings.TL_fontSize || 'medium';
            this.applyFontSizeSettings(fontSize);
            
            // Update UI controls to reflect current settings
            const themeSelect = document.getElementById('theme-select');
            if (themeSelect) themeSelect.value = theme;
            
            const fontSizeSelect = document.getElementById('font-size-select');
            if (fontSizeSelect) fontSizeSelect.value = fontSize;
            
            logger.debug('Settings loaded from database and applied:', userSettings);
        } catch (error) {
            logger.error('Error loading settings:', error);
            // Fall back to defaults on error
        }
    }

    applyThemeSettings(theme) {
        const root = document.documentElement;
        
        if (theme === 'light') {
            root.classList.add('light-theme');
            root.classList.remove('dark-theme');
        } else if (theme === 'dark') {
            root.classList.add('dark-theme');
            root.classList.remove('light-theme');
        } else {
            // Auto theme - remove manual classes, let CSS media query handle it
            root.classList.remove('light-theme', 'dark-theme');
        }
    }

    applyFontSizeSettings(fontSize) {
        const root = document.documentElement;
        
        // Remove existing font size classes
        root.classList.remove('font-small', 'font-medium', 'font-large');
        
        // Add new font size class
        if (fontSize === 'small') {
            root.classList.add('font-small');
        } else if (fontSize === 'large') {
            root.classList.add('font-large');
        } else {
            // Default medium doesn't need a class
            root.classList.add('font-medium');
        }
    }

    async saveSettings() {
        const settings = {
            TL_defaultDueDays: document.getElementById('default-due-days').value,
            TL_tasksPerPage: document.getElementById('tasks-per-page').value,
            TL_defaultStatusFilter: document.getElementById('default-status-filter').value,
            TL_defaultPriorityFilter: document.getElementById('default-priority-filter').value,
            TL_defaultTaskPriority: document.getElementById('default-task-priority').value,
            TL_autoSave: document.getElementById('auto-save').checked,
            TL_useLastProject: document.getElementById('use-last-project').checked,
            TL_useLastCategory: document.getElementById('use-last-category').checked,
            TL_autoCompleteProjects: document.getElementById('auto-complete-projects').checked,
            TL_fontSize: document.getElementById('font-size').value,
            TL_reducedMotion: document.getElementById('reduced-motion').checked,
            TL_browserNotifications: document.getElementById('browser-notifications').checked,
            TL_taskReminders: document.getElementById('task-reminders').checked,
            TL_reminderTime: document.getElementById('reminder-time').value,
            TL_soundNotifications: document.getElementById('sound-notifications').checked,
            TL_showCompletedInSearch: document.getElementById('show-completed-in-search').checked
        };
        
        try {
            await this.api.updateUserSettings(settings);
            
            // Update localStorage with the new settings so they take effect immediately
            Object.entries(settings).forEach(([key, value]) => {
                localStorage.setItem(key, String(value));
            });
        } catch (error) {
            logger.error('Error saving settings:', error);
            window.app.modals.showToast('Failed to save settings', 'error');
        }
    }

    async resetSettings() {
        if (confirm('Are you sure you want to reset all TaskList settings to defaults? This cannot be undone.')) {
            try {
                // Reset to default values
                const defaultSettings = {
                    TL_defaultDueDays: '1',
                    TL_tasksPerPage: '25',
                    TL_autoSave: true,
                    TL_useLastProject: true,
                    TL_useLastCategory: true,
                    TL_autoCompleteProjects: false,
                    TL_fontSize: 'medium',
                    TL_reducedMotion: false,
                    TL_browserNotifications: true,
                    TL_taskReminders: true,
                    TL_reminderTime: '60',
                    TL_soundNotifications: false,
                    TL_showCompletedInSearch: false
                };
                
                await this.api.updateUserSettings(defaultSettings);
                
                // Update localStorage
                Object.entries(defaultSettings).forEach(([key, value]) => {
                    localStorage.setItem(key, String(value));
                });
                
                // Reload settings to show defaults
                await this.loadCurrentSettings();
                window.app.modals.showToast('Settings reset to defaults', 'info');
            } catch (error) {
                logger.error('Error resetting settings:', error);
                window.app.modals.showToast('Failed to reset settings', 'error');
            }
        }
    }

    showToast(message, type = 'info') {
        const container = document.getElementById('notification-container');
        const toast = document.createElement('div');
        toast.className = `toast toast-${type}`;
        
        // Create elements programmatically for better event handling
        const toastContent = document.createElement('div');
        toastContent.className = 'toast-content';
        
        const toastMessage = document.createElement('span');
        toastMessage.className = 'toast-message';
        toastMessage.textContent = message;
        
        const closeButton = document.createElement('button');
        closeButton.className = 'toast-close';
        closeButton.innerHTML = '<i class="fas fa-times"></i>';
        
        toastContent.appendChild(toastMessage);
        toastContent.appendChild(closeButton);
        toast.appendChild(toastContent);
        
        // Make the entire toast clickable to close
        toast.style.cursor = 'pointer';
        toast.addEventListener('click', (e) => {
            toast.remove();
        });
        
        container.appendChild(toast);
        
        // Auto remove after 5 seconds
        setTimeout(() => {
            if (toast.parentElement) {
                toast.remove();
            }
        }, 5000);
    }
}

// =============================================================================
// MODAL MANAGEMENT
// =============================================================================

class ModalManager {
    constructor(state, api) {
        this.state = state;
        this.api = api;
        this.activeModal = null;
        
        this.initializeModals();
    }

    initializeModals() {
        // Remove old individual listeners since we'll use event delegation
        
        // Form submission events
        document.getElementById('task-form')?.addEventListener('submit', (e) => this.handleTaskSubmit(e));
        document.getElementById('project-form')?.addEventListener('submit', (e) => this.handleProjectSubmit(e));
        document.getElementById('category-form')?.addEventListener('submit', (e) => this.handleCategorySubmit(e));
        document.getElementById('event-form')?.addEventListener('submit', (e) => this.handleEventSubmit(e));
        
        // Getting Started button event
        document.getElementById('create-getting-started')?.addEventListener('click', () => {
            this.createGettingStartedProjectManually();
        });
        
        // Setup new modal event listeners
        this.setupSettingsEventListeners = () => {}; // Remove old modal settings listeners

        // Close modals on overlay click
        document.querySelectorAll('.modal').forEach(modal => {
            modal.addEventListener('click', (e) => {
                if (e.target === modal) {
                    this.closeModal(modal.id);
                }
            });
        });

        // ESC key to close modals
        document.addEventListener('keydown', (e) => {
            if (e.key === 'Escape' && this.activeModal) {
                this.closeModal(this.activeModal);
            }
        });

        // Event delegation for all modal buttons
        document.addEventListener('click', (e) => {
            // Check for modal close buttons
            const closeBtn = e.target.closest('.modal-close');
            if (closeBtn) {
                const modalId = closeBtn.getAttribute('data-modal');
                if (modalId) {
                    this.closeModal(modalId);
                }
                return;
            }
            
            // Check for cancel buttons
            const cancelBtn = e.target.closest('[data-modal]');
            if (cancelBtn && cancelBtn.textContent.includes('Cancel')) {
                const modalId = cancelBtn.getAttribute('data-modal');
                if (modalId) {
                    this.closeModal(modalId);
                }
                return;
            }
            
            // Check if the clicked element or its parent matches our selectors
            const target = e.target.closest('#dashboard-new-task, #new-task-btn, #fab');
            const projectTarget = e.target.closest('#dashboard-new-project, #new-project-btn');
            const categoryTarget = e.target.closest('#dashboard-new-category, #new-category-btn');
            const addEventTarget = e.target.closest('#dashboard-add-event');
            
            if (target) {
                e.preventDefault();
                this.openTaskModal();
            } else if (projectTarget) {
                e.preventDefault();
                this.openProjectModal();
            } else if (categoryTarget) {
                e.preventDefault();
                this.openCategoryModal();
            } else if (addEventTarget) {
                e.preventDefault();
                if (this.calendar) {
                    this.calendar.openEventModal();
                }
            }

            // Category-specific button handlers
            if (e.target.closest('#save-category-bindings')) {
                e.preventDefault();
                this.saveCategoryBindings();
            } else if (e.target.closest('#confirm-delete-category')) {
                e.preventDefault();
                this.confirmDeleteCategory();
            }

            // Color preset buttons
            const colorPreset = e.target.closest('.color-preset');
            if (colorPreset) {
                e.preventDefault();
                const color = colorPreset.getAttribute('data-color');
                
                // Check which modal is open and update the appropriate color input
                const activeModal = document.querySelector('.modal[style*="flex"]');
                if (activeModal) {
                    if (activeModal.id === 'category-modal') {
                        document.getElementById('category-color').value = color;
                    } else if (activeModal.id === 'project-modal') {
                        document.getElementById('project-color').value = color;
                    }
                }
                
                // Update visual selection
                const allPresets = colorPreset.closest('.color-presets').querySelectorAll('.color-preset');
                allPresets.forEach(preset => preset.classList.remove('selected'));
                colorPreset.classList.add('selected');
            }
        });
    }

    openTaskModal(task = null, mode = 'edit') {
        const modal = document.getElementById('task-modal');
        const form = document.getElementById('task-form');
        const title = document.getElementById('task-modal-title');
        const viewMode = document.getElementById('task-view-mode');
        const modeToggle = document.getElementById('task-mode-toggle');
        const viewActions = document.getElementById('task-view-actions');
        const editActions = document.getElementById('task-edit-actions');
        const completeBtn = document.getElementById('task-view-complete');
        const completeText = document.getElementById('task-complete-text');
        
        // Store current task and mode
        this.currentTaskModalTask = task;
        this.currentTaskModalMode = mode;
        
        if (mode === 'view' && task) {
            // VIEW MODE
            title.textContent = 'Task Details';
            viewMode.style.display = 'block';
            form.style.display = 'none';
            modeToggle.style.display = 'inline-flex';
            viewActions.style.display = 'flex';
            editActions.style.display = 'none';
            
            // Update complete button text based on task status
            if (task.is_completed) {
                completeText.textContent = 'Mark Incomplete';
                completeBtn.innerHTML = '<i class="fas fa-undo"></i> <span id="task-complete-text">Mark Incomplete</span>';
            } else {
                completeText.textContent = 'Complete Task';
                completeBtn.innerHTML = '<i class="fas fa-check"></i> <span id="task-complete-text">Complete Task</span>';
            }
            
            this.populateTaskView(task);
            
        } else {
            // EDIT MODE
            title.textContent = task ? 'Edit Task' : 'Add New Task';
            viewMode.style.display = 'none';
            form.style.display = 'block';
            modeToggle.style.display = 'none';
            viewActions.style.display = 'none';
            editActions.style.display = 'flex';
            
            // Populate dropdowns first
            this.populateProjectOptions();
            this.setupDurationButtons();
            this.setupProjectCategoryListener();
            this.setupSubtasksHandlers();
            
            if (task) {
                // For editing, populate categories based on task's project
                this.populateCategoryOptions(task.project_id);
                // Populate form after dropdowns are ready
                setTimeout(() => {
                    this.populateTaskForm(task);
                }, 0);
                
                // Show subtasks section and load subtasks
                document.getElementById('subtasks-section').style.display = 'block';
                this.loadSubtasks(task.id);
            } else {
                // For new tasks, show all categories initially
                this.populateCategoryOptions();
                form.reset();
                document.getElementById('task-id').value = '';
                
                // Set default values from config
                const defaultDueDays = parseInt(localStorage.getItem('TL_defaultDueDays') || CONFIG.DEFAULTS.task.dueDays);
                const defaultPriority = localStorage.getItem('TL_defaultTaskPriority') || CONFIG.DEFAULTS.task.priority;
                
                // Set default due date
                const futureDate = new Date();
                futureDate.setDate(futureDate.getDate() + defaultDueDays);
                const futureDateStr = futureDate.toISOString().split('T')[0];
                document.getElementById('task-due-date').value = futureDateStr;
                
                // Set default priority
                document.getElementById('task-priority').value = defaultPriority;
                
                // Set default complexity
                document.getElementById('task-complexity').value = CONFIG.DEFAULTS.task.complexity;
                
                // Apply last used settings if enabled
                this.applyLastUsedSettings();
                
                // Hide subtasks section for new tasks
                document.getElementById('subtasks-section').style.display = 'none';
            }
        }
        
        this.showModal('task-modal');
    }

    setupProjectCategoryListener() {
        const projectSelect = document.getElementById('task-project');
        
        // Remove existing listener to avoid duplicates
        if (projectSelect.taskProjectListener) {
            projectSelect.removeEventListener('change', projectSelect.taskProjectListener);
        }
        
        // Add new listener
        projectSelect.taskProjectListener = (e) => {
            const selectedProjectId = e.target.value;
            this.populateCategoryOptions(selectedProjectId);
            
            // Clear category selection when project changes
            const categorySelect = document.getElementById('task-category');
            categorySelect.value = '';
        };
        
        projectSelect.addEventListener('change', projectSelect.taskProjectListener);
        
        // Setup task suggestions listeners
        this.setupTaskSuggestionsListeners();
    }

    setupTaskSuggestionsListeners() {
        const titleInput = document.getElementById('task-title');
        const descInput = document.getElementById('task-description');
        const suggestionsContainer = document.getElementById('task-suggestions-container');
        const suggestionsToggle = document.getElementById('suggestions-toggle');
        
        if (!titleInput || !descInput || !suggestionsContainer || !suggestionsToggle) {
            return;
        }

        let suggestionsTimeout;
        let currentSuggestions = null;

        const generateSuggestions = () => {
            clearTimeout(suggestionsTimeout);
            
            suggestionsTimeout = setTimeout(() => {
                const title = titleInput.value.trim();
                const description = descInput.value.trim();
                
                // Only show suggestions if editing mode (task-id is empty)
                const taskIdInput = document.getElementById('task-id');
                const isEditMode = taskIdInput && taskIdInput.value;
                
                if (isEditMode || (!title && !description)) {
                    suggestionsToggle.style.display = 'none';
                    suggestionsContainer.style.display = 'none';
                    return;
                }
                
                if (title || description) {
                    currentSuggestions = window.app.taskAssistant.generateSuggestions(title, description);
                    
                    // Only show toggle if we have suggestions
                    const hasSuggestions = 
                        (currentSuggestions.categories && currentSuggestions.categories.length > 0) ||
                        currentSuggestions.dueDate ||
                        (currentSuggestions.relatedTasks && currentSuggestions.relatedTasks.length > 0);
                    
                    if (hasSuggestions) {
                        suggestionsToggle.style.display = 'flex';
                        window.app.taskAssistant.renderSuggestions(currentSuggestions, suggestionsContainer);
                    } else {
                        suggestionsToggle.style.display = 'none';
                        suggestionsContainer.style.display = 'none';
                    }
                }
            }, 500); // Debounce for 500ms
        };

        // Attach input event listeners
        titleInput.addEventListener('input', generateSuggestions);
        descInput.addEventListener('input', generateSuggestions);

        // Toggle suggestions visibility
        suggestionsToggle.addEventListener('click', () => {
            const isVisible = suggestionsContainer.style.display === 'block';
            suggestionsContainer.style.display = isVisible ? 'none' : 'block';
            suggestionsToggle.classList.toggle('active', !isVisible);
        });
    }

    async applyLastUsedSettings() {
        try {
            const userSettings = await this.api.getUserSettings();
            const lastUsed = JSON.parse(localStorage.getItem('TL_lastUsedTaskSettings') || '{}');
            
            // Apply last used project if setting is enabled (defaults to true for new users)
            if (userSettings.TL_useLastProject !== false && lastUsed.projectId) {
                // Will be set when options are populated
                setTimeout(() => {
                    const projectSelect = document.getElementById('task-project');
                    if (projectSelect && projectSelect.querySelector(`option[value="${lastUsed.projectId}"]`)) {
                        projectSelect.value = lastUsed.projectId;
                    }
                }, 100);
            }
            
            // Apply last used category if setting is enabled (defaults to true for new users)
            if (userSettings.TL_useLastCategory !== false && lastUsed.categoryId) {
                // Will be set when options are populated
                setTimeout(() => {
                    const categorySelect = document.getElementById('task-category');
                    if (categorySelect && categorySelect.querySelector(`option[value="${lastUsed.categoryId}"]`)) {
                        categorySelect.value = lastUsed.categoryId;
                    }
                }, 100);
            }
        } catch (error) {
            logger.error('Error loading user settings for last used:', error);
            // Graceful fallback - assume defaults are enabled
        }
    }

    saveLastUsedSettings(projectId, categoryId) {
        const lastUsed = {
            projectId: projectId || null,
            categoryId: categoryId || null,
            timestamp: Date.now()
        };
        
        localStorage.setItem('TL_lastUsedTaskSettings', JSON.stringify(lastUsed));
    }

    setupDurationButtons() {
        // Remove existing listeners
        document.querySelectorAll('.duration-btn').forEach(btn => {
            btn.replaceWith(btn.cloneNode(true));
        });
        
        // Add new listeners
        document.querySelectorAll('.duration-btn').forEach(btn => {
            btn.addEventListener('click', (e) => {
                e.preventDefault();
                
                // Remove active class from all buttons
                document.querySelectorAll('.duration-btn').forEach(b => b.classList.remove('active'));
                
                // Add active class to clicked button
                btn.classList.add('active');
                
                // Calculate new due date
                const days = parseInt(btn.getAttribute('data-days'));
                const today = new Date();
                today.setDate(today.getDate() + days);
                const dateStr = today.toISOString().split('T')[0];
                
                // Set the due date
                document.getElementById('task-due-date').value = dateStr;
            });
        });
    }

    openProjectModal(project = null) {
        const modal = document.getElementById('project-modal');
        const form = document.getElementById('project-form');
        const title = document.getElementById('project-modal-title');
        
        title.textContent = project ? 'Edit Project' : 'Add New Project';
        
        if (project && project.id) {
            this.populateProjectForm(project);
        } else {
            form.reset();
            const projectIdEl = document.getElementById('project-id');
            if (projectIdEl) {
                projectIdEl.value = '';
                projectIdEl.removeAttribute('value');
            }
            
            // Set default values from config
            const projectColorEl = document.getElementById('project-color');
            const projectIconEl = document.getElementById('project-icon');
            if (projectColorEl) projectColorEl.value = CONFIG.DEFAULTS.project.color;
            if (projectIconEl) projectIconEl.value = CONFIG.DEFAULTS.project.icon;
        }
        
        this.showModal('project-modal');
    }

    populateTaskForm(task) {
        document.getElementById('task-id').value = task.id;
        document.getElementById('task-title').value = task.title;
        document.getElementById('task-description').value = task.description || '';
        document.getElementById('task-project').value = task.project_id || '';
        document.getElementById('task-category').value = task.category_id || '';
        document.getElementById('task-priority').value = task.priority;
        document.getElementById('task-complexity').value = task.complexity || 'none';
        document.getElementById('task-due-date').value = task.due_date || '';
    }

    populateTaskView(task) {
        // Title and status
        document.getElementById('task-view-title').textContent = task.title;
        const statusBadge = document.getElementById('task-view-status');
        statusBadge.textContent = task.is_completed ? 'Completed' : 'Active';
        statusBadge.className = `task-status-badge ${task.is_completed ? 'completed' : 'active'}`;
        
        // Description - inline, can wrap
        const descEl = document.getElementById('task-view-description');
        descEl.textContent = task.description || 'No description';
        descEl.className = task.description ? 'task-view-value' : 'task-view-value empty';
        
        // Project with icon and color
        const projects = this.state.get('projects');
        const project = projects.find(p => p.id === task.project_id);
        const projectEl = document.getElementById('task-view-project');
        if (project) {
            projectEl.innerHTML = `
                <span class="project-badge" style="--project-color: ${project.color}">
                    <i class="${project.icon || 'fas fa-folder'}"></i>
                    ${this.escapeHtml(project.name)}
                </span>
            `;
        } else {
            projectEl.innerHTML = '<span class="empty">N/A</span>';
        }
        
        // Category with color indicator
        const categories = this.state.get('categories');
        const category = categories.find(c => c.id === task.category_id);
        const categoryEl = document.getElementById('task-view-category');
        if (category) {
            categoryEl.innerHTML = `
                <span class="category-badge">
                    <i class="fas fa-tag"></i>
                    ${this.escapeHtml(category.name)}
                </span>
            `;
        } else {
            categoryEl.innerHTML = '<span class="empty">N/A</span>';
        }
        
        // Priority
        const priorityBadge = document.getElementById('task-view-priority');
        priorityBadge.textContent = task.priority.charAt(0).toUpperCase() + task.priority.slice(1);
        priorityBadge.className = `priority-badge priority-${task.priority}`;
        
        // Complexity
        const complexityBadge = document.getElementById('task-view-complexity');
        const complexityMap = {
            'none': 'None',
            'simple': 'Simple',
            'moderate': 'Moderate',
            'complex': 'Complex',
            'very_complex': 'Very Complex'
        };
        complexityBadge.textContent = complexityMap[task.complexity] || 'None';
        complexityBadge.className = `complexity-badge complexity-${task.complexity}`;
        
        // Due Date - inline format
        const dueDateEl = document.getElementById('task-view-due-date');
        if (task.due_date) {
            const dueDate = new Date(task.due_date);
            const today = new Date();
            today.setHours(0, 0, 0, 0);
            const taskDate = new Date(dueDate);
            taskDate.setHours(0, 0, 0, 0);
            
            const diffTime = taskDate - today;
            const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));
            
            let dueDateText = dueDate.toLocaleDateString('en-US', { 
                month: 'short', 
                day: 'numeric',
                year: 'numeric'
            });
            
            if (diffDays < 0) {
                dueDateText += ` <span class="overdue">(${Math.abs(diffDays)} days overdue)</span>`;
            } else if (diffDays === 0) {
                dueDateText += ' <span class="due-today">(Today)</span>';
            } else if (diffDays === 1) {
                dueDateText += ' <span class="due-soon">(Tomorrow)</span>';
            } else if (diffDays <= 7) {
                dueDateText += ` <span class="due-soon">(${diffDays} days)</span>`;
            }
            
            dueDateEl.innerHTML = dueDateText;
            dueDateEl.className = 'task-view-value';
        } else {
            dueDateEl.textContent = 'Not set';
            dueDateEl.className = 'task-view-value empty';
        }
        
        // Load subtasks in view mode
        if (task.id) {
            this.loadSubtasksView(task.id);
        }
    }

    async loadSubtasksView(taskId) {
        try {
            const subtasks = await this.api.getSubtasks(taskId);
            const container = document.getElementById('task-view-subtasks');
            const list = document.getElementById('task-view-subtasks-list');
            const progressBar = document.getElementById('task-view-progress-bar');
            const progressText = document.getElementById('task-view-progress-text');
            
            if (subtasks.length === 0) {
                container.style.display = 'none';
                return;
            }
            
            container.style.display = 'block';
            
            // Calculate progress
            const completed = subtasks.filter(s => s.is_completed).length;
            const total = subtasks.length;
            const percentage = total > 0 ? (completed / total) * 100 : 0;
            
            progressBar.style.width = `${percentage}%`;
            progressText.textContent = `${completed} / ${total} completed`;
            
            // Render subtasks
            list.innerHTML = subtasks.map(subtask => `
                <li class="subtask-view-item ${subtask.is_completed ? 'completed' : ''}">
                    <i class="fas ${subtask.is_completed ? 'fa-check-circle' : 'fa-circle'}"></i>
                    <span>${this.escapeHtml(subtask.title)}</span>
                </li>
            `).join('');
            
        } catch (error) {
            logger.error('Failed to load subtasks for view:', error);
        }
    }

    populateProjectForm(project) {
        // Add null checks to prevent errors
        const projectIdEl = document.getElementById('project-id');
        const projectNameEl = document.getElementById('project-name');
        const projectDescEl = document.getElementById('project-description');
        const projectColorEl = document.getElementById('project-color');
        const projectIconEl = document.getElementById('project-icon');
        
        // Always clear the form first
        const form = document.getElementById('project-form');
        if (form) form.reset();
        
        if (projectIdEl) {
            projectIdEl.value = project.id;
        }
        if (projectNameEl) projectNameEl.value = project.name;
        if (projectDescEl) projectDescEl.value = project.description || '';
        if (projectColorEl) projectColorEl.value = project.color || '#2563eb';
        if (projectIconEl) projectIconEl.value = project.icon || 'fas fa-folder';
    }

    populateProjectOptions() {
        const select = document.getElementById('task-project');
        const projects = this.state.get('projects');
        
        select.innerHTML = '<option value="">Select a project</option>';
        projects.forEach(project => {
            select.innerHTML += `<option value="${project.id}">${project.name}</option>`;
        });
    }

    populateCategoryOptions(selectedProjectId = null) {
        const select = document.getElementById('task-category');
        const categories = this.state.get('categories');
        const projects = this.state.get('projects');
        
        // If no project selected, use the currently selected project from dropdown
        if (!selectedProjectId) {
            const projectSelect = document.getElementById('task-project');
            selectedProjectId = projectSelect?.value || null;
        }
        
        let availableCategories = categories;
        
        // Filter categories by project if a project is selected
        if (selectedProjectId) {
            const project = projects.find(p => p.id == selectedProjectId);
            if (project && project.category_ids) {
                availableCategories = categories.filter(cat => 
                    project.category_ids.includes(cat.id)
                );
            }
        }
        
        select.innerHTML = '<option value="">Select a category</option>';
        availableCategories.forEach(category => {
            select.innerHTML += `<option value="${category.id}">${category.name}</option>`;
        });
        
        // If no categories available for project, show helpful message
        if (selectedProjectId && availableCategories.length === 0) {
            select.innerHTML += '<option value="" disabled>No categories bound to this project</option>';
        }
    }

    async handleTaskSubmit(e) {
        e.preventDefault();
        
        const form = e.target;
        const formData = new FormData(form);
        const taskData = Object.fromEntries(formData.entries());
        
        // Clean up data
        Object.keys(taskData).forEach(key => {
            if (taskData[key] === '') taskData[key] = null;
        });

        const submitBtn = document.getElementById('task-submit');
        this.setButtonLoading(submitBtn, true);

        try {
            if (taskData.id) {
                await this.api.updateTask(taskData.id, taskData);
                this.showToast('Task updated successfully!', 'success');
            } else {
                delete taskData.id;
                const result = await this.api.createTask(taskData);
                this.showToast('Task created successfully!', 'success');
                
                // Save last used settings for new tasks
                this.saveLastUsedSettings(taskData.project_id, taskData.category_id);
                
                // Check for onboarding task completion
                await app.checkOnboardingTaskCompletion('create_task');
                
                // Check achievements
                if (app.achievements && app.achievements.initialized) {
                    try {
                        await app.achievements.checkAchievements('task_created', { taskData, result });
                    } catch (error) {
                        logger.error('[Task] Error checking achievements for task_created:', error);
                    }
                } else {
                    logger.warn('[Task] Achievements not initialized yet, skipping tracking');
                }
            }
            
            // Check for onboarding task completion when task properties are set
            if (taskData.due_date || taskData.priority || taskData.complexity) {
                await app.checkOnboardingTaskCompletion('set_task_properties');
            }
            
            this.closeModal('task-modal');
            await app.loadTasks();
            
            // Refresh the current view to update any task counts or data
            const currentView = this.state.get('currentView');
            if (currentView === 'tasks' || currentView === 'projects') {
                await app.ui.renderView(currentView);
            }
        } catch (error) {
            logger.error('Task save error:', error);
            this.showToast('Failed to save task: ' + error.message, 'error');
        } finally {
            this.setButtonLoading(submitBtn, false);
        }
    }

    /**
     * Open event modal for create or edit
     */
    openEventModal(event = null) {
        const modal = document.getElementById('event-modal');
        const form = document.getElementById('event-form');
        const title = document.getElementById('event-modal-title');
        
        title.textContent = event ? 'Edit Event' : 'Add New Event';
        
        if (event) {
            // Populate form with event data
            document.getElementById('event-id').value = event.id || '';
            document.getElementById('event-title').value = event.title || '';
            document.getElementById('event-date').value = event.event_date || '';
            document.getElementById('event-time').value = event.event_time || '';
            document.getElementById('event-end-time').value = event.end_time || '';
            document.getElementById('event-all-day').checked = event.is_all_day || false;
            document.getElementById('event-description').value = event.description || '';
            document.getElementById('event-location').value = event.location || '';
            document.getElementById('event-color').value = event.color || '#3b82f6';
            document.getElementById('event-reminder').value = event.reminder_minutes || '';
        } else {
            form.reset();
            document.getElementById('event-id').value = '';
            // Set default date to today
            document.getElementById('event-date').value = new Date().toISOString().split('T')[0];
            // Set default color
            document.getElementById('event-color').value = '#3b82f6';
        }
        
        // Toggle time inputs based on all-day checkbox
        const allDayCheckbox = document.getElementById('event-all-day');
        const timeInputs = [document.getElementById('event-time'), document.getElementById('event-end-time')];
        
        allDayCheckbox.onchange = () => {
            timeInputs.forEach(input => {
                input.disabled = allDayCheckbox.checked;
                if (allDayCheckbox.checked) input.value = '';
            });
        };
        
        // Show modal
        modal.style.display = 'flex';
    }

    /**
     * Handle event form submission
     */
    async handleEventSubmit(e) {
        e.preventDefault();
        
        const submitBtn = document.getElementById('event-submit-btn');
        this.setButtonLoading(submitBtn, true);
        
        const formData = new FormData(e.target);
        const eventData = {
            title: formData.get('title'),
            event_date: formData.get('event_date'),
            event_time: formData.get('event_time') || null,
            end_time: formData.get('end_time') || null,
            is_all_day: formData.get('is_all_day') ? 1 : 0,
            description: formData.get('description') || null,
            location: formData.get('location') || null,
            color: formData.get('color') || '#3b82f6',
            reminder_minutes: formData.get('reminder_minutes') || null
        };
        
        const eventId = formData.get('id');
        
        try {
            if (eventId && eventId.trim() !== '') {
                // Update existing event
                await this.app.api.put(`calendar/${eventId}`, eventData);
                this.showToast('Event updated successfully!', 'success');
            } else {
                // Create new event
                await this.app.api.post('calendar/events', eventData);
                this.showToast('Event created successfully!', 'success');
            }
            
            this.closeModal('event-modal');
            
            // Refresh calendar if we're on the calendar view
            const currentView = this.state.get('currentView');
            if (currentView === 'calendar' && this.app.calendar) {
                await this.app.calendar.refresh();
            }
        } catch (error) {
            logger.error('Event save error:', error);
            this.showToast('Failed to save event: ' + error.message, 'error');
        } finally {
            this.setButtonLoading(submitBtn, false);
        }
    }

    async handleProjectSubmit(e) {
        e.preventDefault();
        
        const form = e.target;
        
        // Get values directly from form elements
        const projectData = {
            id: document.getElementById('project-id')?.value || '',
            name: document.getElementById('project-name')?.value || '',
            description: document.getElementById('project-description')?.value || '',
            color: document.getElementById('project-color')?.value || '#2563eb',
            icon: document.getElementById('project-icon')?.value || 'fas fa-folder'
        };
        
        const submitBtn = document.getElementById('project-submit');
        this.setButtonLoading(submitBtn, true);

        try {
            let newProject;
            
            if (projectData.id && projectData.id.trim() !== '') {
                // Update existing project
                await this.api.updateProject(projectData.id, projectData);
                this.showToast('Project updated successfully!', 'success');
                
                // Check for onboarding task completion (editing colors)
                await app.checkOnboardingTaskCompletion('edit_colors');
            } else {
                // Create new project
                delete projectData.id;
                const response = await this.api.createProject(projectData);
                newProject = response.data || response;
                
                // Auto-bind categories that have auto_bind = true
                await this.autoBindCategoriesToProject(newProject.id);
                
                // Check for onboarding task completion
                await app.checkOnboardingTaskCompletion('create_project');
                
                // Check achievements for project creation
                if (app.achievements && app.achievements.initialized) {
                    await app.achievements.checkAchievements('project_created', { project_id: newProject.id });
                }
                
                this.showToast('Project created successfully!', 'success');
            }
            
            this.closeModal('project-modal');
            await app.loadProjects();
            
            if (this.state.get('currentView') === 'projects') {
                await app.ui.renderView('projects');
            }
        } catch (error) {
            logger.error('Project save error:', error);
            this.showToast('Failed to save project: ' + error.message, 'error');
        } finally {
            this.setButtonLoading(submitBtn, false);
        }
    }

    async handleCategorySubmit(e) {
        e.preventDefault();
        
        const form = e.target;
        const formData = new FormData(form);
        const categoryData = Object.fromEntries(formData.entries());
        
        const submitBtn = document.getElementById('save-category');
        this.setButtonLoading(submitBtn, true);

        try {
            if (this.currentCategory) {
                // Update existing category
                await this.api.updateCategory(this.currentCategory.id, categoryData);
                this.showToast('Category updated successfully!', 'success');
                
                // Check for onboarding task completion (editing colors)
                await app.checkOnboardingTaskCompletion('edit_colors');
            } else {
                // Create new category
                await this.api.createCategory(categoryData);
                
                // Check for onboarding task completion
                await app.checkOnboardingTaskCompletion('create_category');
                
                this.showToast('Category created successfully!', 'success');
            }
            
            this.closeModal('category-modal');
            await app.loadCategories();
            
            if (this.state.get('currentView') === 'categories') {
                await app.ui.renderView('categories');
            }
        } catch (error) {
            logger.error('Category save error:', error);
            this.showToast('Failed to save category: ' + error.message, 'error');
        } finally {
            this.setButtonLoading(submitBtn, false);
        }
    }

    showModal(modalId) {
        const modal = document.getElementById(modalId);
        if (modal) {
            modal.style.display = 'flex';
            document.body.classList.add('modal-open');
            this.activeModal = modalId;
        }
    }

    closeModal(modalId) {
        const modal = document.getElementById(modalId);
        if (modal) {
            modal.style.display = 'none';
            document.body.classList.remove('modal-open');
            this.activeModal = null;
        }
    }

    // =============================================================================
    // SUBTASKS METHODS
    // =============================================================================

    setupSubtasksHandlers() {
        // Add subtask button
        const addBtn = document.getElementById('add-subtask-btn');
        const newInput = document.getElementById('new-subtask-input');
        
        if (addBtn && !addBtn.hasSubtaskListener) {
            addBtn.hasSubtaskListener = true;
            addBtn.addEventListener('click', () => this.handleAddSubtask());
        }
        
        if (newInput && !newInput.hasSubtaskListener) {
            newInput.hasSubtaskListener = true;
            newInput.addEventListener('keypress', (e) => {
                if (e.key === 'Enter') {
                    e.preventDefault();
                    this.handleAddSubtask();
                }
            });
        }
    }

    async loadSubtasks(taskId) {
        try {
            const response = await fetch(`${CONFIG.API_BASE_URL}subtasks?task_id=${taskId}`, {
                method: 'GET',
                headers: { 'Content-Type': 'application/json' },
                credentials: 'include'
            });
            
            const result = await response.json();
            
            if (result.success) {
                this.renderSubtasks(result.data.subtasks || []);
                this.updateSubtasksProgress(result.data.stats || {total: 0, completed: 0});
            }
        } catch (error) {
            logger.error('Failed to load subtasks:', error);
        }
    }

    renderSubtasks(subtasks) {
        const list = document.getElementById('subtasks-list');
        if (!list) return;
        
        if (subtasks.length === 0) {
            list.innerHTML = '<li class="subtasks-empty"><i class="fas fa-check-circle"></i><br>No steps yet. Add steps to break down this task.</li>';
            return;
        }
        
        list.innerHTML = subtasks.map(subtask => `
            <li class="subtask-item ${subtask.is_completed ? 'completed' : ''}" data-subtask-id="${subtask.id}">
                <div class="subtask-checkbox ${subtask.is_completed ? 'checked' : ''}" onclick="app.modals.handleToggleSubtask(${subtask.id})"></div>
                <span class="subtask-text">${this.escapeHtml(subtask.title)}</span>
                <div class="subtask-actions">
                    <button type="button" class="subtask-action-btn delete" onclick="event.stopPropagation(); app.modals.handleDeleteSubtask(${subtask.id})" title="Delete step">
                        <i class="fas fa-trash"></i>
                    </button>
                </div>
            </li>
        `).join('');
    }

    updateSubtasksProgress(stats) {
        const progressBar = document.getElementById('subtasks-progress-bar');
        const progressText = document.getElementById('subtasks-progress-text');
        
        if (progressBar && stats) {
            const percentage = stats.percentage || 0;
            progressBar.style.width = `${percentage}%`;
        }
        
        if (progressText && stats) {
            progressText.textContent = `${stats.completed || 0} / ${stats.total || 0} completed`;
        }
    }

    async handleAddSubtask() {
        const input = document.getElementById('new-subtask-input');
        const taskId = document.getElementById('task-id').value;
        const title = input.value.trim();
        
        if (!title) {
            this.showToast('Please enter a step title', 'warning');
            return;
        }
        
        if (!taskId) {
            this.showToast('Please save the task first before adding steps', 'warning');
            return;
        }
        
        try {
            const response = await fetch(`${CONFIG.API_BASE_URL}subtasks`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                credentials: 'include',
                body: JSON.stringify({ task_id: taskId, title: title })
            });
            
            const result = await response.json();
            
            if (result.success) {
                input.value = '';
                await this.loadSubtasks(taskId);
                this.showToast('Step added successfully', 'success');
                
                // Check for onboarding task completion
                await app.checkOnboardingTaskCompletion('create_subtask');
                
                // Check achievements for subtask creation
                await app.achievements.checkAchievements('subtask_created', { subtask_id: result.data.id });
            } else {
                this.showToast(result.error || 'Failed to add step', 'error');
            }
        } catch (error) {
            logger.error('Failed to add subtask:', error);
            this.showToast('Failed to add step', 'error');
        }
    }

    async handleToggleSubtask(subtaskId) {
        const taskId = document.getElementById('task-id').value;
        
        try {
            const response = await fetch(`${CONFIG.API_BASE_URL}subtasks/${subtaskId}/toggle`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                credentials: 'include'
            });
            
            const result = await response.json();
            
            if (result.success) {
                await this.loadSubtasks(taskId);
            } else {
                this.showToast(result.error || 'Failed to toggle step', 'error');
            }
        } catch (error) {
            logger.error('Failed to toggle subtask:', error);
            this.showToast('Failed to toggle step', 'error');
        }
    }

    async handleDeleteSubtask(subtaskId) {
        if (!confirm('Are you sure you want to delete this step?')) {
            return;
        }
        
        const taskId = document.getElementById('task-id').value;
        
        try {
            const response = await fetch(`${CONFIG.API_BASE_URL}subtasks/${subtaskId}`, {
                method: 'DELETE',
                headers: { 'Content-Type': 'application/json' },
                credentials: 'include'
            });
            
            const result = await response.json();
            
            if (result.success) {
                await this.loadSubtasks(taskId);
                this.showToast('Step deleted successfully', 'success');
            } else {
                this.showToast(result.error || 'Failed to delete step', 'error');
            }
        } catch (error) {
            logger.error('Failed to delete subtask:', error);
            this.showToast('Failed to delete step', 'error');
        }
    }

    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    setButtonLoading(button, loading) {
        if (!button) return;
        
        const text = button.querySelector('.btn-text');
        const loadingEl = button.querySelector('.btn-loading');
        
        // If button doesn't have the expected structure, just disable it
        if (!text || !loadingEl) {
            if (loading) {
                button.disabled = true;
                button.style.opacity = '0.6';
                button.style.cursor = 'not-allowed';
            } else {
                button.disabled = false;
                button.style.opacity = '1';
                button.style.cursor = 'pointer';
            }
            return;
        }
        
        // Use the proper structure if available
        if (loading) {
            text.style.display = 'none';
            loadingEl.style.display = 'inline-flex';
            button.disabled = true;
        } else {
            text.style.display = 'inline';
            loadingEl.style.display = 'none';
            button.disabled = false;
        }
    }

    showToast(message, type = 'info') {
        const container = document.getElementById('notification-container');
        const toast = document.createElement('div');
        toast.className = `toast toast-${type}`;
        toast.innerHTML = `
            <div class="toast-content">
                <span class="toast-message">${message}</span>
                <button class="toast-close" onclick="this.parentElement.parentElement.remove()">
                    <i class="fas fa-times"></i>
                </button>
            </div>
        `;
        
        container.appendChild(toast);
        
        // Auto remove after 5 seconds
        setTimeout(() => {
            if (toast.parentElement) {
                toast.remove();
            }
        }, 5000);
    }

    // New modal methods for project-category binding, category deletion, and settings
    openProjectCategoriesModal(project) {
        const modal = document.getElementById('project-categories-modal');
        const title = document.getElementById('project-categories-modal-title');
        const projectName = document.getElementById('selected-project-name');
        const categoriesList = document.getElementById('project-categories-list');
        
        title.textContent = `Manage Categories - ${project.name}`;
        projectName.textContent = project.name;
        
        // Store current project
        this.currentProject = project;
        
        // Load and display categories
        this.loadProjectCategories(project.id);
        
        this.showModal('project-categories-modal');
    }

    async loadProjectCategories(projectId) {
        try {
            const categories = await this.api.getCategories();
            const projectCategories = await this.api.getProjectCategories(projectId);
            const categoriesList = document.getElementById('project-categories-list');
            
            categoriesList.innerHTML = '';
            
            categories.forEach(category => {
                const isSelected = projectCategories.some(pc => pc.category_id === category.id);
                const categoryItem = document.createElement('div');
                categoryItem.className = `category-item ${isSelected ? 'selected' : ''}`;
                categoryItem.innerHTML = `
                    <input type="checkbox" class="category-checkbox" ${isSelected ? 'checked' : ''} data-category-id="${category.id}">
                    <div class="category-info">
                        <div class="category-name">${category.name}</div>
                        <div class="category-task-count">${category.task_count || 0} tasks</div>
                    </div>
                `;
                
                categoryItem.addEventListener('click', (e) => {
                    if (e.target.type !== 'checkbox') {
                        const checkbox = categoryItem.querySelector('.category-checkbox');
                        checkbox.checked = !checkbox.checked;
                        categoryItem.classList.toggle('selected', checkbox.checked);
                    } else {
                        categoryItem.classList.toggle('selected', e.target.checked);
                    }
                });
                
                categoriesList.appendChild(categoryItem);
            });
        } catch (error) {
            logger.error('Error loading project categories:', error);
            this.showToast('Failed to load categories', 'error');
        }
    }

    async saveProjectCategories() {
        const checkboxes = document.querySelectorAll('#project-categories-list .category-checkbox');
        const selectedCategories = Array.from(checkboxes)
            .filter(cb => cb.checked)
            .map(cb => parseInt(cb.dataset.categoryId));
        
        try {
            await this.api.updateProjectCategories(this.currentProject.id, selectedCategories);
            this.showToast('Categories updated successfully', 'success');
            this.closeModal('project-categories-modal');
        } catch (error) {
            logger.error('Error saving project categories:', error);
            this.showToast('Failed to save categories', 'error');
        }
    }

    openDeleteCategoryModal(category) {
        const modal = document.getElementById('delete-category-modal');
        const categoryName = document.getElementById('category-name-to-delete');
        
        categoryName.textContent = category.name;
        this.categoryToDelete = category;
        
        this.showModal('delete-category-modal');
    }

    async confirmDeleteCategory() {
        if (!this.categoryToDelete) return;
        
        try {
            await this.api.deleteCategory(this.categoryToDelete.id);
            this.showToast('Category deleted successfully', 'success');
            this.closeModal('delete-category-modal');
            // Refresh the current view
            // TODO: Add view refresh logic
        } catch (error) {
            logger.error('Error deleting category:', error);
            this.showToast('Failed to delete category', 'error');
        }
    }

    openSettingsModal() {
        const modal = document.getElementById('settings-modal');
        this.loadCurrentSettings();
        this.showModal('settings-modal');
    }

    loadCurrentSettings() {
        // Load current settings from localStorage or user preferences
        const settings = JSON.parse(localStorage.getItem('taskListSettings') || '{}');
        
        // General settings
        document.getElementById('default-due-days').value = settings.defaultDueDays || '1';
        document.getElementById('tasks-per-page').value = settings.tasksPerPage || '25';
        document.getElementById('auto-save').checked = settings.autoSave !== false;
        
        // Appearance settings
        const currentTheme = settings.theme || 'auto';
        document.querySelectorAll('.theme-option').forEach(btn => {
            btn.classList.toggle('active', btn.dataset.theme === currentTheme);
        });
        document.getElementById('font-size').value = settings.fontSize || 'medium';
        document.getElementById('reduced-motion').checked = settings.reducedMotion || false;
        
        // Notification settings
        document.getElementById('browser-notifications').checked = settings.browserNotifications !== false;
        document.getElementById('task-reminders').checked = settings.taskReminders !== false;
        document.getElementById('reminder-time').value = settings.reminderTime || '60';
        document.getElementById('sound-notifications').checked = settings.soundNotifications || false;
    }

    async saveSettings() {
        const settings = {
            defaultDueDays: document.getElementById('default-due-days').value,
            tasksPerPage: document.getElementById('tasks-per-page').value,
            autoSave: document.getElementById('auto-save').checked,
            theme: document.querySelector('.theme-option.active')?.dataset.theme || 'auto',
            fontSize: document.getElementById('font-size').value,
            reducedMotion: document.getElementById('reduced-motion').checked,
            browserNotifications: document.getElementById('browser-notifications').checked,
            taskReminders: document.getElementById('task-reminders').checked,
            reminderTime: document.getElementById('reminder-time').value,
            soundNotifications: document.getElementById('sound-notifications').checked
        };
        
        try {
            localStorage.setItem('taskListSettings', JSON.stringify(settings));
            
            // Apply theme and font-size changes immediately
            this.applyThemeSettings(settings.theme);
            this.applyFontSizeSettings(settings.fontSize);
            
            this.showToast('Settings saved successfully', 'success');
            this.closeModal('settings-modal');
        } catch (error) {
            logger.error('Error saving settings:', error);
            this.showToast('Failed to save settings', 'error');
        }
    }

    setupSettingsEventListeners() {
        // Settings tabs
        document.querySelectorAll('.settings-tab').forEach(tab => {
            tab.addEventListener('click', () => {
                const tabName = tab.dataset.tab;
                
                // Update active tab
                document.querySelectorAll('.settings-tab').forEach(t => t.classList.remove('active'));
                tab.classList.add('active');
                
                // Update active panel
                document.querySelectorAll('.settings-panel').forEach(panel => {
                    panel.classList.toggle('active', panel.id === `${tabName}-settings`);
                });
            });
        });
        
        // Theme selection
        document.querySelectorAll('.theme-option').forEach(option => {
            option.addEventListener('click', () => {
                document.querySelectorAll('.theme-option').forEach(opt => opt.classList.remove('active'));
                option.classList.add('active');
            });
        });
        
        // Settings action buttons
        document.getElementById('save-settings')?.addEventListener('click', () => {
            this.saveSettings();
        });
        
        document.getElementById('save-project-categories')?.addEventListener('click', () => {
            this.saveProjectCategories();
        });
        
        document.getElementById('confirm-delete-category')?.addEventListener('click', () => {
            this.confirmDeleteCategory();
        });
        
        // Settings button in navigation
        document.getElementById('settings-btn')?.addEventListener('click', (e) => {
            e.preventDefault();
            this.openSettingsModal();
        });
    }

    // Category Management Modal Methods
    openCategoryModal(category = null) {
        const modal = document.getElementById('category-modal');
        const title = document.getElementById('category-modal-title');
        const form = document.getElementById('category-form');
        
        // Reset form
        form.reset();
        
        if (category) {
            title.textContent = 'Edit Category';
            document.getElementById('category-name').value = category.name;
            document.getElementById('category-description').value = category.description || '';
            document.getElementById('category-color').value = category.color || '#2563eb';
            document.getElementById('category-auto-bind').checked = category.auto_bind || false;
            this.currentCategory = category;
        } else {
            title.textContent = 'Add Category';
            this.currentCategory = null;
        }
        
        this.showModal('category-modal');
    }

    async autoBindCategoriesToProject(projectId) {
        try {
            const categories = this.state.get('categories');
            const autoBind = categories.filter(cat => cat.auto_bind);
            
            if (autoBind.length > 0) {
                // For each auto-bind category, update its project bindings to include the new project
                for (const category of autoBind) {
                    const currentProjects = this.state.get('projects')
                        .filter(p => p.category_ids && p.category_ids.includes(category.id))
                        .map(p => p.id);
                    
                    currentProjects.push(projectId);
                    
                    await this.api.updateCategoryProjects(category.id, currentProjects);
                }
            }
        } catch (error) {
            logger.error('Error auto-binding categories:', error);
            // Don't show error to user as this is a background operation
        }
    }

    openCategoryBindingModal(category) {
        const modal = document.getElementById('category-binding-modal');
        const categoryName = document.getElementById('binding-category-name');
        const categoryDescription = document.getElementById('binding-category-description');
        const categoryColor = document.getElementById('binding-category-color');
        const projectsList = document.getElementById('binding-projects-list');
        
        categoryName.textContent = category.name;
        categoryDescription.textContent = category.description || 'No description';
        categoryColor.style.backgroundColor = category.color || '#2563eb';
        
        this.currentCategory = category;
        
        // Load projects with checkboxes
        this.loadCategoryBindingProjects(category);
        
        this.showModal('category-binding-modal');
    }

    async loadCategoryBindingProjects(category) {
        try {
            const projects = this.state.get('projects') || [];
            const projectsList = document.getElementById('binding-projects-list');
            
            projectsList.innerHTML = '';
            
            projects.forEach(project => {
                const isSelected = project.category_ids && project.category_ids.includes(category.id);
                const projectItem = document.createElement('div');
                projectItem.className = 'checkbox-item';
                projectItem.innerHTML = `
                    <label class="checkbox-label">
                        <input type="checkbox" class="project-checkbox" ${isSelected ? 'checked' : ''} data-project-id="${project.id}">
                        <span class="checkbox-custom"></span>
                        <div class="project-info">
                            <span class="project-name">${this.escapeHtml(project.name)}</span>
                            ${project.description ? `<span class="project-description">${this.escapeHtml(project.description)}</span>` : ''}
                        </div>
                    </label>
                `;
                projectsList.appendChild(projectItem);
            });
        } catch (error) {
            logger.error('Error loading project bindings:', error);
            this.showToast('Failed to load projects', 'error');
        }
    }

    openDeleteCategoryModal(category) {
        const modal = document.getElementById('delete-category-modal');
        const categoryName = document.getElementById('category-name-to-delete');
        
        categoryName.textContent = category.name;
        this.currentCategory = category;
        
        this.showModal('delete-category-modal');
    }

    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    async saveCategoryBindings() {
        try {
            const checkboxes = document.querySelectorAll('#binding-projects-list .project-checkbox:checked');
            const selectedProjects = Array.from(checkboxes).map(cb => parseInt(cb.dataset.projectId));
            
            // Update category bindings via API
            await this.api.updateCategoryProjects(this.currentCategory.id, selectedProjects);
            
            this.showToast('Category bindings updated successfully!', 'success');
            this.closeModal('category-binding-modal');
            
            // Refresh projects data to get updated category_ids
            await app.loadProjects();
            
            // Refresh the categories view if we're on it
            if (this.state.get('currentView') === 'categories') {
                await app.ui.renderView('categories');
            }
            
        } catch (error) {
            logger.error('Error saving category bindings:', error);
            this.showToast('Failed to update category bindings', 'error');
        }
    }

    async confirmDeleteCategory() {
        try {
            await this.api.deleteCategory(this.currentCategory.id);
            
            this.showToast('Category deleted successfully!', 'success');
            this.closeModal('delete-category-modal');
            
            // Refresh data
            await app.loadCategories();
            
            if (this.state.get('currentView') === 'categories') {
                await app.ui.renderView('categories');
            }
            
        } catch (error) {
            logger.error('Error deleting category:', error);
            
            // Extract more specific error messages
            let errorMessage = 'Failed to delete category';
            
            if (error.message.includes('Cannot delete category that is used by tasks') || error.message.includes('409')) {
                errorMessage = 'Cannot delete category that is currently being used by tasks. Please remove the category from all tasks first, then try deleting again.';
            } else if (error.message.includes('404') || error.message.includes('not found')) {
                errorMessage = 'Category not found or you do not have permission to delete it.';
            } else if (error.message.includes('500')) {
                errorMessage = 'Server error while deleting category. Please try again.';
            } else if (error.message && error.message !== 'Failed to delete category') {
                // Use the API error message if it's descriptive
                errorMessage = error.message;
            }
            
            this.showToast(errorMessage, 'error');
        }
    }


}

// =============================================================================
// MAIN APPLICATION CLASS
// =============================================================================

class TaskListApp {
    constructor() {
        this.state = new AppState();
        this.api = new APIClient(CONFIG.API_BASE_URL);
        this.ui = new UIRenderer(this.state, this.api, this); // Pass app reference
        this.modals = new ModalManager(this.state, this.api);
        this.taskAssistant = new TaskAssistant(this);
        this.calendar = new CalendarManager(this);
        this.collaboration = new CollaborationManager(this);
        this.achievements = new AchievementSystem(this);
        this.keyboard = new KeyboardManager(this);
        this.devTools = new DevTools(this);
        this.sharing = new SharingManager(this.api, this.state);
        this.bulkActions = new BulkActions(this);
        this.quickCapture = new QuickCapture(this);
        this.focusMode = new FocusMode(this);
        
        this.initialized = false;
    }

    async init() {
        if (this.initialized) return;
        
        try {
            // Check if in share mode (guest user)
            const isShareMode = document.body.dataset.shareMode === '1';
            const shareProjectId = document.body.dataset.shareProjectId;
            const sharePermission = document.body.dataset.sharePermission;
            
            if (isShareMode) {
                logger.info('[App] Initializing in share mode', { shareProjectId, sharePermission });
                this.shareMode = {
                    enabled: true,
                    projectId: parseInt(shareProjectId),
                    permission: sharePermission,
                    isGuest: true
                };
                
                // Skip user settings and onboarding for guest users
            } else {
                // Initialize user settings first (for first-time users)
                const userSettings = await this.ui.initializeUserSettings();
                
                // Initialize Getting Started project for new users
                await this.initializeGettingStartedProject(userSettings);
                
                // Load and apply user preferences from database
                await this.ui.loadAndApplySettings();
            }
            
            await this.loadInitialData();
            
            // Initialize achievements AFTER data is loaded so we can check existing stats
            logger.info('[App] Initializing achievement system...');
            await this.achievements.init();
            
            // Restore saved filters from localStorage
            this.restoreSavedFilters();
            
            this.setupEventListeners();
            this.setupNavigation();
            
            // Set up URL handling
            this.setupURLHandling();
            
            // Load page from URL or default to dashboard
            this.loadPageFromURL();
            
            // Update user XP after view is loaded
            await this.updateUserXP();
            
            // Ensure navigation badges are updated after UI is ready
            setTimeout(() => this.updateNavigationBadges(), 100);
            
            // Initialize dev tools
            this.devTools.init();
            
            // Check if user is on mobile device
            this.checkMobileView();
            
            // Auto-complete welcome task on first load
            await this.checkOnboardingTaskCompletion('welcome');
            
            this.hideLoadingScreen();
            this.initialized = true;
            
            logger.debug('Task Manager initialized successfully');
        } catch (error) {
            logger.error('Failed to initialize app:', error);
            this.showError('Failed to load application. Please refresh the page.');
        }
    }

    checkMobileView() {
        const isMobile = window.innerWidth <= 768 || 
                        /Android|webOS|iPhone|iPad|iPod|BlackBerry|IEMobile|Opera Mini/i.test(navigator.userAgent);
        
        // Initialize wasMobile state for resize detection
        this.state.set('ui.wasMobile', isMobile);
        
        if (isMobile) {
            // Auto-complete mobile_view task
            this.checkOnboardingTaskCompletion('mobile_view');
        }
    }

    async loadInitialData() {
        try {
            logger.info('Loading initial data...');
            logger.debug('API Base URL:', this.api.baseUrl);
            
            // Loading status messages (some funny ones)
            const loadingMessages = [
                'Getting tasks...',
                'Loading settings...',
                'Fetching projects...',
                'Organizing categories...',
                'Calculating procrastination levels...',
                'Herding digital cats...',
                'Consulting the task oracle...',
                'Brewing productivity potion...'
            ];
            
            let messageIndex = 0;
            const messageInterval = setInterval(() => {
                if (messageIndex < loadingMessages.length) {
                    this.updateLoadingStatus(loadingMessages[messageIndex]);
                    messageIndex++;
                }
            }, 300);
            
            // Load data with individual error handling
            const results = await Promise.allSettled([
                this.loadTasks(), // Use loadTasks() to fetch tasks with subtasks
                this.api.getProjects(),
                this.api.getCategories()
            ]);
            
            clearInterval(messageInterval);

            // Handle tasks (already loaded by loadTasks())
            if (results[0].status === 'rejected') {
                logger.error('Failed to load tasks:', results[0].reason);
                this.state.set('tasks', []);
            }
            
            // Update navigation badges after loading tasks
            this.updateNavigationBadges();

            // Handle projects
            if (results[1].status === 'fulfilled') {
                let projects = results[1].value.data || [];
                
                // Filter to only shared project in share mode
                if (this.shareMode && this.shareMode.enabled) {
                    projects = projects.filter(p => p.id === this.shareMode.projectId);
                    logger.info('[Share Mode] Filtered to shared project:', projects);
                }
                
                this.state.set('projects', projects);
            } else {
                logger.error('Failed to load projects:', results[1].reason);
                this.state.set('projects', []);
            }

            // Handle categories
            if (results[2].status === 'fulfilled') {
                let categories = results[2].value.data || [];
                
                // In share mode, only show categories for the shared project
                if (this.shareMode && this.shareMode.enabled) {
                    const sharedProject = this.state.get('projects').find(p => p.id === this.shareMode.projectId);
                    if (sharedProject) {
                        // Categories belong to the project owner, so we keep all of their categories
                        // The UI will naturally filter by project when viewing tasks
                        logger.info('[Share Mode] Categories available:', categories.length);
                    }
                }
                
                this.state.set('categories', categories);
            } else {
                logger.error('Failed to load categories:', results[2].reason);
                this.state.set('categories', []);
            }

            // Load user data from PHP
            const userData = {
                id: window.BLOODWEB_USER?.id || CURRENT_USER?.id || 1,
                username: window.BLOODWEB_USER?.username || 'User',
                first_name: window.BLOODWEB_USER?.first_name || '',
                last_name: window.BLOODWEB_USER?.last_name || '',
                email: window.BLOODWEB_USER?.email || '',
                role: window.BLOODWEB_USER?.role || 'user',
                created_at: window.BLOODWEB_USER?.created_at || null,
                xp_total: 0,
                level_id: 1
            };
            this.state.set('user', userData);
            
            // Don't call updateUserXP here - it will be called after view is loaded
            logger.info('Initial data loaded successfully');
            
            // Apply user's default filter settings
            await this.applyUserFilterDefaults();
        } catch (error) {
            logger.error('Failed to load initial data:', error);
            // Don't throw - allow app to initialize with empty data
            this.state.set('tasks', []);
            this.state.set('projects', []);
            this.state.set('categories', []);
        }
    }

    async applyUserFilterDefaults() {
        try {
            // Get user settings to determine default filters
            const userSettings = await this.api.getUserSettings();
            
            // Update app state with user's preferred default filters
            const defaultFilters = {
                status: userSettings.TL_defaultStatusFilter || CONFIG.DEFAULTS.filters.status,
                priority: userSettings.TL_defaultPriorityFilter || CONFIG.DEFAULTS.filters.priority,
                project: CONFIG.DEFAULTS.filters.project, // Keep 'all' for these
                category: CONFIG.DEFAULTS.filters.category
            };
            
            this.state.set('filters', defaultFilters);
            
            // Store in localStorage for quick access - store as strings for consistency
            localStorage.setItem('TL_defaultStatusFilter', defaultFilters.status);
            localStorage.setItem('TL_defaultPriorityFilter', defaultFilters.priority);
            localStorage.setItem('TL_defaultTaskPriority', userSettings.TL_defaultTaskPriority || CONFIG.DEFAULTS.task.priority);
            localStorage.setItem('TL_defaultDueDays', userSettings.TL_defaultDueDays || CONFIG.DEFAULTS.task.dueDays.toString());
            
            // Store ALL other settings for app-wide access
            Object.entries(userSettings).forEach(([key, value]) => {
                if (!localStorage.getItem(key)) { // Don't overwrite what we just set
                    localStorage.setItem(key, typeof value === 'object' ? JSON.stringify(value) : String(value));
                }
            });
            
        } catch (error) {
            logger.error('Error loading user filter defaults:', error);
            // Fall back to CONFIG defaults if user settings can't be loaded
            this.state.set('filters', CONFIG.DEFAULTS.filters);
        }
    }

    restoreSavedFilters() {
        try {
            const savedFilters = localStorage.getItem('taskListFilters');
            if (savedFilters) {
                const filters = JSON.parse(savedFilters);
                logger.debug('Restoring saved filters:', filters);
                this.state.set('filters', filters);
            }
        } catch (error) {
            logger.error('Error restoring saved filters:', error);
        }
    }

    async updateUserXP() {
        try {
            const xpData = await this.api.getUserXP(CURRENT_USER.id);
            
            if (xpData.data) {
                // Update user state with XP data
                const currentUser = this.state.get('user') || {};
                const oldLevel = currentUser.level_id || 1;
                const newLevel = xpData.data.current_level || 1;
                
                currentUser.xp_total = xpData.data.total_xp || 0;
                currentUser.level_id = newLevel;
                this.state.set('user', currentUser);
                
                // Check if user leveled up
                if (newLevel > oldLevel) {
                    await this.checkOnboardingTaskCompletion('level_up');
                    // Check achievements for level milestones (checks all level-based achievements)
                    await this.achievements.checkStatAchievements('level_reached', newLevel);
                    
                    // Also check XP milestones
                    await this.achievements.checkStatAchievements('xp_earned', currentUser.xp_total);
                }
                
                const userLevelEl = document.getElementById('user-level');
                const userXpCurrentEl = document.getElementById('xp-amount');
                const userStatsEl = document.querySelector('.user-stats');
                
                if (userLevelEl) {
                    const level = xpData.data.current_level || xpData.data.user_level || xpData.data.level || 1;
                    userLevelEl.textContent = level;
                    
                    // Update XP progress bar on user-stats element
                    const totalXP = xpData.data.total_xp || 0;
                    const currentLevelXP = totalXP % 100;
                    const progressPercentage = currentLevelXP;
                    if (userStatsEl) {
                        userStatsEl.style.setProperty('--xp-progress', `${progressPercentage}%`);
                    }
                }
                
                if (userXpCurrentEl) {
                    const totalXP = xpData.data.total_xp || 0;
                    userXpCurrentEl.textContent = totalXP;
                }
                
                // Don't auto-refresh dashboard here - it causes routing issues
                // Dashboard will refresh itself when tasks are completed
            }
        } catch (error) {
            logger.error('Failed to load user XP:', error);
            // Set fallback values
            const userLevelEl = document.getElementById('user-level');
            const userXpCurrentEl = document.getElementById('xp-amount');
            const userStatsEl = document.querySelector('.user-stats');
            if (userLevelEl) {
                userLevelEl.textContent = '1';
            }
            if (userStatsEl) {
                userStatsEl.style.setProperty('--xp-progress', '0%');
            }
            if (userXpCurrentEl) {
                userXpCurrentEl.textContent = '0';
            }
        }
    }

    setupEventListeners() {
        // Store event handlers for cleanup
        if (!this.eventHandlers) {
            this.eventHandlers = {};
        }
        
        // Helper to add event listener with cleanup
        const addListener = (id, event, handler, options = {}) => {
            const element = document.getElementById(id);
            if (element) {
                // Remove old listener if exists
                if (this.eventHandlers[`${id}-${event}`]) {
                    element.removeEventListener(event, this.eventHandlers[`${id}-${event}`]);
                }
                this.eventHandlers[`${id}-${event}`] = handler;
                element.addEventListener(event, handler, options);
            }
        };
        
        // Sidebar toggle
        addListener('sidebar-toggle', 'click', () => {
            this.toggleSidebar();
        });

        // User menu toggle - handled by initUserMenu() called at startup
        // See line 7310+ for implementation

        // User profile click - navigate to account page
        addListener('user-profile', 'click', (e) => {
            e.preventDefault();
            this.navigateToView('account');
        });
        
        // Analytics tab switching
        document.querySelectorAll('.analytics-tab').forEach(tab => {
            tab.addEventListener('click', (e) => {
                const tabName = e.currentTarget.dataset.analyticsTab;
                this.switchAnalyticsTab(tabName);
            });
        });

        // Task modal mode toggle (Edit button)
        addListener('task-mode-toggle', 'click', () => {
            if (this.currentTaskModalTask) {
                this.modals.openTaskModal(this.currentTaskModalTask, 'edit');
            }
        });

        // Task view complete button
        addListener('task-view-complete', 'click', async () => {
            if (this.currentTaskModalTask) {
                await this.toggleTaskStatus(this.currentTaskModalTask.id);
                this.modals.closeModal('task-modal');
            }
        });

        // Task view delete button
        addListener('task-view-delete', 'click', async () => {
            if (this.currentTaskModalTask) {
                await this.deleteTask(this.currentTaskModalTask.id);
                this.modals.closeModal('task-modal');
            }
        });

        // Global click handler to close dropdowns and sidebar
        // Use capture phase and once:false to ensure proper cleanup
        if (!this.globalClickHandler) {
            this.globalClickHandler = (e) => {
                // Close sidebar if clicking outside (when sidebar is open)
                if (this.state.get('ui.sidebarOpen')) {
                    const sidebar = document.getElementById('sidebar');
                    const sidebarToggle = document.getElementById('sidebar-toggle');
                    
                    if (sidebar && !sidebar.contains(e.target) && !sidebarToggle.contains(e.target)) {
                        this.toggleSidebar();
                    }
                }
                
                // Close any other open dropdowns
            };
            document.addEventListener('click', this.globalClickHandler);
        }

        // Window resize handler
        if (!this.resizeHandler) {
            this.resizeHandler = () => {
                this.handleResize();
            };
            window.addEventListener('resize', this.resizeHandler);
        }
        
        // Setup search functionality
        this.setupSearchAutocomplete();
    }
    
    setupSearchAutocomplete() {
        const searchInput = document.getElementById('global-search');
        const searchResults = document.getElementById('search-results');
        if (!searchInput || !searchResults) return;
        
        let searchTimeout;
        
        searchInput.addEventListener('input', (e) => {
            clearTimeout(searchTimeout);
            const query = e.target.value.trim().toLowerCase();
            
            if (query.length === 0) {
                searchResults.style.display = 'none';
                return;
            }
            
            searchTimeout = setTimeout(() => {
                this.performSearch(query);
            }, 200); // Debounce search by 200ms
        });
        
        // Close search results when clicking outside
        document.addEventListener('click', (e) => {
            if (!searchInput.contains(e.target) && !searchResults.contains(e.target)) {
                searchResults.style.display = 'none';
            }
        });
        
        // Handle Enter key to navigate to first result
        searchInput.addEventListener('keydown', (e) => {
            if (e.key === 'Enter') {
                const firstResult = searchResults.querySelector('.search-result-item');
                if (firstResult) {
                    firstResult.click();
                }
            } else if (e.key === 'Escape') {
                searchResults.style.display = 'none';
                searchInput.blur();
            }
        });
    }
    
    performSearch(query) {
        const searchResults = document.getElementById('search-results');
        if (!searchResults) return;
        
        const tasks = this.state.get('tasks') || [];
        const projects = this.state.get('projects') || [];
        const categories = this.state.get('categories') || [];
        
        // Check user setting for showing completed items
        const showCompleted = document.getElementById('show-completed-in-search')?.checked ?? false;
        
        let results = [];
        
        // Search tasks (limit to 3), filter out completed if setting is false
        const taskMatches = tasks.filter(task => {
            const matchesQuery = task.title.toLowerCase().includes(query);
            const shouldShow = showCompleted || task.status !== 'completed';
            return matchesQuery && shouldShow;
        }).slice(0, 3);
        
        results.push(...taskMatches.map(task => ({
            type: 'task',
            id: task.id,
            title: task.title,
            query: query
        })));
        
        // Search projects (limit to 2)
        const projectMatches = projects.filter(project => 
            project.name.toLowerCase().includes(query)
        ).slice(0, 2);
        
        results.push(...projectMatches.map(project => ({
            type: 'project',
            id: project.id,
            title: project.name,
            query: query,
            color: project.color
        })));
        
        // Search categories (if room left, limit total to 5)
        if (results.length < 5) {
            const categoryMatches = categories.filter(category => 
                category.name.toLowerCase().includes(query)
            ).slice(0, 5 - results.length);
            
            results.push(...categoryMatches.map(category => ({
                type: 'category',
                id: category.id,
                title: category.name,
                query: query,
                color: category.color
            })));
        }
        
        // Limit to 5 total results
        results = results.slice(0, 5);
        
        if (results.length === 0) {
            searchResults.innerHTML = '<div class="search-no-results">No results found</div>';
        } else {
            searchResults.innerHTML = results.map(result => this.renderSearchResult(result)).join('');
        }
        
        searchResults.style.display = 'block';
    }
    
    renderSearchResult(result) {
        const highlightedTitle = this.highlightSearchTerm(result.title, result.query);
        const icon = result.type === 'task' ? 'check-circle' : 
                     result.type === 'project' ? 'folder' : 'tag';
        const typeLabel = result.type.charAt(0).toUpperCase() + result.type.slice(1);
        const colorStyle = result.color ? `border-left: 3px solid ${result.color};` : '';
        
        return `
            <div class="search-result-item" onclick="app.handleSearchResultClick('${result.type}', ${result.id})" style="${colorStyle}">
                <i class="fas fa-${icon} search-result-icon"></i>
                <div class="search-result-content">
                    <div class="search-result-title">${highlightedTitle}</div>
                    <div class="search-result-type">${typeLabel}</div>
                </div>
            </div>
        `;
    }
    
    highlightSearchTerm(text, query) {
        const regex = new RegExp(`(${query})`, 'gi');
        return text.replace(regex, '<mark>$1</mark>');
    }
    
    handleSearchResultClick(type, id) {
        const searchResults = document.getElementById('search-results');
        const searchInput = document.getElementById('global-search');
        
        if (searchResults) searchResults.style.display = 'none';
        if (searchInput) searchInput.value = '';
        
        if (type === 'task') {
            const task = this.state.get('tasks').find(t => t.id === id);
            if (task) {
                this.modals.openTaskModal(task, 'view');
            }
        } else if (type === 'project') {
            this.navigateToView('projects');
            setTimeout(() => {
                const projectElement = document.querySelector(`[data-project-id="${id}"]`);
                if (projectElement) {
                    projectElement.scrollIntoView({ behavior: 'smooth', block: 'center' });
                    projectElement.classList.add('highlight-flash');
                    setTimeout(() => projectElement.classList.remove('highlight-flash'), 2000);
                }
            }, 300);
        } else if (type === 'category') {
            // Navigate to tasks view with category filter
            this.state.set('filters', {
                projects: [],
                categories: [id],
                statuses: [],
                priorities: []
            });
            this.navigateToView('tasks');
        }
    }

    setupNavigation() {
        document.querySelectorAll('.nav-link').forEach(link => {
            link.addEventListener('click', (e) => {
                e.preventDefault();
                
                // Check if guest user trying to access restricted feature
                if (link.hasAttribute('data-guest-restricted')) {
                    this.ui.showNotification('Please create an account to access this feature', 'info');
                    return;
                }
                
                const view = link.dataset.view;
                this.navigateToView(view);
            });
        });
    }

    setupURLHandling() {
        // Handle browser back/forward buttons
        window.addEventListener('popstate', (event) => {
            this.handlePopState(event);
        });
    }

    async navigateToView(viewName, preserveFilters = false) {
        logger.debug('[navigateToView] Called with viewName:', viewName, 'preserveFilters:', preserveFilters);
        
        // Clear any active filters when switching views normally
        if (!preserveFilters) {
            this.state.set('taskFilter', null);
            this.state.set('projectFilter', null);
        }
        
        // Update active nav link
        document.querySelectorAll('.nav-link').forEach(link => {
            link.classList.toggle('active', link.dataset.view === viewName);
        });

        await this.ui.renderView(viewName);
        
        // Check for onboarding task completion
        if (viewName === 'settings') {
            await this.checkOnboardingTaskCompletion('visit_settings');
        } else if (viewName === 'account') {
            await this.checkOnboardingTaskCompletion('visit_account');
        } else if (viewName === 'analytics') {
            // Analytics tab events are already bound in setupEventListeners
        }
        
        // Update URL to reflect current view
        this.updateURL(viewName);
    }

    switchAnalyticsTab(tabName) {
        // Update tab buttons
        document.querySelectorAll('.analytics-tab').forEach(tab => {
            tab.classList.toggle('active', tab.dataset.analyticsTab === tabName);
        });
        
        // Update tab panels
        document.querySelectorAll('.analytics-panel').forEach(panel => {
            panel.classList.toggle('active', panel.dataset.analyticsPanel === tabName);
        });
    }

    // URL Management Methods
    getUrlParams() {
        const params = new URLSearchParams(window.location.search);
        return {
            page: params.get('page') || 'dashboard'
        };
    }

    updateURL(viewName) {
        const url = new URL(window.location);
        url.searchParams.set('page', viewName);
        
        // Update URL without triggering page reload
        window.history.pushState({ view: viewName }, '', url);
    }

    loadPageFromURL() {
        const params = this.getUrlParams();
        const viewName = params.page;
        
        logger.debug('[loadPageFromURL] URL params:', params, 'viewName:', viewName);
        
        // Validate that the view exists
        const validViews = ['dashboard', 'focus', 'tasks', 'projects', 'categories', 'achievements', 'analytics', 'calendar', 'changelog', 'accountsettings', 'account', 'settings'];
        if (validViews.includes(viewName)) {
            logger.debug('[loadPageFromURL] Navigating to valid view:', viewName);
            this.navigateToView(viewName);
        } else {
            // Default to dashboard if invalid view
            logger.debug('[loadPageFromURL] Invalid view, defaulting to dashboard');
            this.navigateToView('dashboard');
        }
    }

    handlePopState(event) {
        // Handle browser back/forward buttons
        if (event.state && event.state.view) {
            // Don't update URL again to avoid infinite loop
            this.navigateToViewWithoutURL(event.state.view);
        } else {
            // Fallback to reading from URL
            const params = this.getUrlParams();
            this.navigateToViewWithoutURL(params.page);
        }
    }

    navigateToViewWithoutURL(viewName, preserveFilters = false) {
        // Same as navigateToView but doesn't update URL (used for back/forward navigation)
        if (!preserveFilters) {
            this.state.set('taskFilter', null);
            this.state.set('projectFilter', null);
        }
        
        // Update active nav link
        document.querySelectorAll('.nav-link').forEach(link => {
            link.classList.toggle('active', link.dataset.view === viewName);
        });

        this.ui.renderView(viewName);
    }

    toggleSidebar() {
        const sidebar = document.getElementById('sidebar');
        const isOpen = this.state.get('ui.sidebarOpen');
        
        sidebar.classList.toggle('open', !isOpen);
        this.state.set('ui.sidebarOpen', !isOpen);
    }

    handleResize() {
        // Handle responsive behavior
        const isMobile = window.innerWidth < 768;
        const wasMobile = this.state.get('ui.wasMobile');
        
        // Close sidebar on mobile if open
        if (isMobile && this.state.get('ui.sidebarOpen')) {
            this.toggleSidebar();
        }
        
        // Re-render dashboard if we crossed the mobile breakpoint and on dashboard view
        if (wasMobile !== isMobile) {
            this.state.set('ui.wasMobile', isMobile);
            const currentView = this.state.get('currentView');
            if (currentView === 'dashboard') {
                this.ui.renderView('dashboard');
            }
        }
    }

    hideLoadingScreen() {
        const loadingScreen = document.getElementById('loading-screen');
        
        if (loadingScreen) {
            loadingScreen.style.display = 'none';
        }
        
        document.body.classList.remove('loading');
    }

    updateLoadingStatus(message) {
        const statusElement = document.getElementById('loading-status');
        if (statusElement) {
            statusElement.textContent = message;
        }
    }

    showError(message) {
        const errorBoundary = document.getElementById('error-boundary');
        if (errorBoundary) {
            const errorText = errorBoundary.querySelector('p');
            if (errorText) {
                errorText.textContent = message;
            }
            errorBoundary.style.display = 'flex';
        } else {
            logger.error('Error boundary element not found. Error message:', message);
            // Fallback to alert if no error boundary
            alert('Application Error: ' + message);
        }
    }

    // Public API methods for global access
    async loadTasks() {
        logger.debug('=== loadTasks() called ===');
        const tasks = await this.api.getTasks();
        const tasksData = tasks.data || [];
        const tasksMeta = tasks.meta || {};
        
        logger.debug(`Loading subtasks for ${tasksData.length} tasks...`);
        logger.debug('Tasks metadata:', tasksMeta);
        
        // Fetch subtasks for each task
        for (const task of tasksData) {
            try {
                const response = await fetch(`${CONFIG.API_BASE_URL}subtasks?task_id=${task.id}`, {
                    method: 'GET',
                    headers: { 'Content-Type': 'application/json' },
                    credentials: 'include'
                });
                const result = await response.json();
                if (result.success && result.data && result.data.subtasks) {
                    task.subtasks = result.data.subtasks;
                    logger.debug(`Task ${task.id} (${task.title}) has ${task.subtasks.length} subtasks`);
                } else {
                    task.subtasks = [];
                }
            } catch (error) {
                logger.error(`Failed to load subtasks for task ${task.id}:`, error);
                task.subtasks = [];
            }
        }
        
        this.state.set('tasks', tasksData);
        this.state.set('tasksMeta', tasksMeta);
        this.updateNavigationBadges();
    }

    updateNavigationBadges() {
        const tasks = this.state.get('tasks');
        logger.debug('Updating badges - Total tasks:', tasks ? tasks.length : 'null');
        
        // Count active (non-completed) tasks
        const activeTasks = tasks ? tasks.filter(task => task.status !== 'completed').length : 0;
        logger.debug('Active tasks count:', activeTasks);
        
        // Update task count badge
        const tasksBadge = document.getElementById('active-tasks-count');
        if (tasksBadge) {
            tasksBadge.textContent = activeTasks;
            tasksBadge.style.display = activeTasks > 0 ? 'inline' : 'none';
            logger.debug('Updated badge to:', activeTasks);
        } else {
            logger.debug('Badge element not found');
        }
    }

    async loadMoreTasks() {
        try {
            const currentTasks = this.state.get('tasks') || [];
            const tasksMeta = this.state.get('tasksMeta') || {};
            
            if (!tasksMeta.has_next) {
                this.modals.showToast('No more tasks to load', 'info');
                return;
            }
            
            const nextPage = (tasksMeta.page || 1) + 1;
            logger.info(`Loading more tasks - page ${nextPage}`);
            
            const response = await this.api.getTasks({ page: nextPage, per_page: 100 });
            const newTasks = response.data || [];
            const newMeta = response.meta || {};
            
            // Load subtasks for new tasks
            for (const task of newTasks) {
                try {
                    const subtasksResponse = await fetch(`${CONFIG.API_BASE_URL}subtasks?task_id=${task.id}`, {
                        method: 'GET',
                        headers: { 'Content-Type': 'application/json' },
                        credentials: 'include'
                    });
                    const result = await subtasksResponse.json();
                    if (result.success && result.data && result.data.subtasks) {
                        task.subtasks = result.data.subtasks;
                    } else {
                        task.subtasks = [];
                    }
                } catch (error) {
                    logger.error(`Failed to load subtasks for task ${task.id}:`, error);
                    task.subtasks = [];
                }
            }
            
            // Append new tasks to existing ones
            const allTasks = [...currentTasks, ...newTasks];
            this.state.set('tasks', allTasks);
            this.state.set('tasksMeta', newMeta);
            
            // Re-render current view
            await this.ui.renderView(this.state.get('currentView'));
            
            this.modals.showToast(`Loaded ${newTasks.length} more tasks`, 'success');
        } catch (error) {
            logger.error('Failed to load more tasks:', error);
            this.modals.showToast('Failed to load more tasks', 'error');
        }
    }

    toggleDevBar() {
        const devBar = document.getElementById('dev-bar');
        const trigger = document.getElementById('dev-bar-trigger');
        
        if (!devBar) return;
        
        const isVisible = devBar.style.display !== 'none';
        
        if (isVisible) {
            devBar.style.display = 'none';
            if (trigger) trigger.style.display = 'flex';
            // Save closed state
            localStorage.setItem('devToolsOpen', 'false');
        } else {
            devBar.style.display = 'block';
            if (trigger) trigger.style.display = 'none';
            // Update stats when opening
            this.devTools.updateStats();
            // Populate category selector
            this.devTools.populateCategorySelector();
            // Save open state
            localStorage.setItem('devToolsOpen', 'true');
        }
    }

    async loadProjects() {
        const projects = await this.api.getProjects();
        this.state.set('projects', projects.data || []);
    }

    async loadCategories() {
        const categories = await this.api.getCategories();
        this.state.set('categories', categories.data || []);
    }

    async toggleTaskStatus(taskId) {
        try {
            // Check if task is already completed
            const tasks = this.state.get('tasks') || [];
            const task = tasks.find(t => t.id == taskId);
            
            if (task && task.status === 'completed') {
                this.modals.showToast('Task is already completed', 'info');
                return;
            }
            
            // Prevent multiple simultaneous completions
            if (this.completingTasks && this.completingTasks.has(taskId)) {
                return; // Already in progress
            }
            
            // Track completion in progress
            if (!this.completingTasks) this.completingTasks = new Set();
            this.completingTasks.add(taskId);
            
            const result = await this.api.completeTask(taskId);
            await this.loadTasks();
            
            // Check for onboarding task completion
            await this.checkOnboardingTaskCompletion('complete_task');
            await this.updateUserXP();
            
            // Check achievements
            if (this.achievements) {
                await this.achievements.checkAchievements('task_completed', { taskId, result });
            }
            
            // Refresh the current view to update any task counts or data
            const currentView = this.state.get('currentView');
            if (currentView === 'tasks' || currentView === 'projects') {
                await this.ui.renderView(currentView);
            }
            
            // Show XP toast with breakdown
            this.showXPToast(result);
        } catch (error) {
            logger.error('Failed to complete task:', error);
            if (error.message && error.message.includes('already completed')) {
                this.modals.showToast('Task is already completed', 'info');
            } else {
                this.modals.showToast('Failed to complete task', 'error');
            }
        } finally {
            // Remove from completion tracking
            if (this.completingTasks) {
                this.completingTasks.delete(taskId);
            }
        }
    }

    async deleteTask(taskId) {
        if (!confirm('Are you sure you want to delete this task?')) return;
        
        try {
            await this.api.deleteTask(taskId);
            await this.loadTasks();
            
            // Update navigation badges
            this.updateNavigationBadges();
            
            // Refresh the current view to update any task counts or data
            const currentView = this.state.get('currentView');
            await this.ui.renderView(currentView);
            
            this.modals.showToast('Task deleted successfully', 'success');
        } catch (error) {
            logger.error('Failed to delete task:', error);
            this.modals.showToast('Failed to delete task', 'error');
        }
    }

    editTask(taskId, mode = 'view') {
        const task = this.state.get('tasks').find(t => t.id === taskId);
        if (task) {
            this.modals.openTaskModal(task, mode);
        }
    }

    showXPToast(result) {
        // Get task info for toast
        const task = result.task || {};
        const taskTitle = task.title || '';
        const taskId = task.id || result.task_id || '';
        
        // Get first 3 words of title
        const titleWords = taskTitle.split(' ').slice(0, 3).join(' ');
        const titlePreview = titleWords + (taskTitle.split(' ').length > 3 ? '...' : '');
        
        if (!result || !result.task_xp_earned) {
            this.modals.showToast(`✅ Task completed${titlePreview ? ': ' + titlePreview : ''}`, 'success');
            return;
        }

        // Build toast message with task info and XP
        let message = `✅ Task #${taskId}: ${titlePreview}\n+${result.task_xp_earned} XP earned`;
        let details = [];
        
        // Add task XP breakdown
        if (result.task_xp_breakdown) {
            for (const [source, points] of Object.entries(result.task_xp_breakdown)) {
                if (points > 0) {
                    const sourceName = source.replace('_', ' ').replace(/\b\w/g, l => l.toUpperCase());
                    details.push(`${sourceName}: +${points}`);
                }
            }
        }

        // Check for project completion bonus
        if (result.project_completed && result.project_xp_earned) {
            message = `Task & Project completed! +${result.total_xp_earned} XP`;
            details.push(`--- Project Bonus ---`);
            
            if (result.project_xp_breakdown) {
                for (const [source, points] of Object.entries(result.project_xp_breakdown)) {
                    if (points > 0) {
                        const sourceName = source.replace('_', ' ').replace(/\b\w/g, l => l.toUpperCase());
                        details.push(`${sourceName}: +${points}`);
                    }
                }
            }
        }

        // Show detailed breakdown in console for debugging
        if (details.length > 0) {
            logger.debug('XP Breakdown:', details);
        }

        // Show enhanced toast with tooltip or extended message
        const detailText = details.length > 3 ? 
            `(${details.slice(0, 2).join(', ')}, +${details.length - 2} more)` : 
            details.length > 0 ? `(${details.join(', ')})` : '';
            
        this.modals.showToast(`${message} ${detailText}`, 'success');
    }

    async toggleProjectCompletion(projectId) {
        try {
            const project = this.state.get('projects').find(p => p.id === projectId);
            if (!project) {
                throw new Error('Project not found');
            }

            // Call API to toggle completion
            await fetch(`${this.api.baseUrl}projects/${projectId}/toggle-completion`, {
                method: 'PATCH',
                credentials: 'include'
            });

            // Reload projects and refresh view
            await this.loadProjects();
            
            if (this.state.get('currentView') === 'projects') {
                await this.ui.renderView('projects');
            }

            const message = project.is_completed 
                ? 'Project marked as active' 
                : 'Project marked as complete';
            this.modals.showToast(message, 'success');
        } catch (error) {
            logger.error('Failed to toggle project completion:', error);
            this.modals.showToast('Failed to update project', 'error');
        }
    }

    async deleteProject(projectId) {
        if (!confirm('Are you sure you want to permanently delete this project? All associated tasks will also be deleted.')) return;
        
        try {
            await this.api.deleteProject(projectId);
            await this.loadProjects();
            await this.loadTasks();
            
            if (this.state.get('currentView') === 'projects') {
                await this.ui.renderView('projects');
            }
            
            this.modals.showToast('Project deleted successfully', 'success');
        } catch (error) {
            logger.error('Failed to delete project:', error);
            this.modals.showToast('Failed to delete project', 'error');
        }
    }

    editProject(projectId) {
        const project = this.state.get('projects').find(p => p.id == projectId);
        if (project) {
            this.modals.openProjectModal(project);
        }
    }

    async toggleProjectCompletion(projectId) {
        try {
            const project = this.state.get('projects').find(p => p.id === projectId);
            if (!project) {
                this.modals.showToast('Project not found', 'error');
                return;
            }

            const wasCompleted = project.is_completed;
            
            // Call API to toggle completion
            const response = await fetch(`${this.api.baseUrl}projects/${projectId}/toggle-completion`, {
                method: 'PATCH',
                credentials: 'include',
                headers: {
                    'Content-Type': 'application/json'
                }
            });

            if (!response.ok) {
                throw new Error('Failed to toggle project completion');
            }

            const result = await response.json();
            const updatedProject = result.data;

            // If project was just marked as completed (not uncompleted), award XP
            if (!wasCompleted && updatedProject.is_completed) {
                // Check achievements for project completion
                await this.achievements.checkAchievements('project_completed', { project_id: projectId });
                
                // Calculate project XP
                const tasks = this.state.get('tasks').filter(t => t.project_id === projectId);
                const completedTasks = tasks.filter(t => t.status === 'completed');
                
                // Award project completion XP via API
                const xpResponse = await fetch(`${this.api.baseUrl}xp/award-project-completion`, {
                    method: 'POST',
                    credentials: 'include',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({
                        project_id: projectId,
                        completed_tasks_count: completedTasks.length
                    })
                });

                if (xpResponse.ok) {
                    const xpResult = await xpResponse.json();
                    if (xpResult.data && xpResult.data.xp_earned) {
                        const xpAmount = xpResult.data.xp_earned;
                        const breakdown = xpResult.data.xp_breakdown;
                        
                        // Build XP breakdown message
                        let xpDetails = '';
                        if (breakdown) {
                            const bonuses = [];
                            if (breakdown.base) bonuses.push(`Base: ${breakdown.base}`);
                            if (breakdown.description_bonus) bonuses.push(`Description: +${breakdown.description_bonus}`);
                            if (breakdown.task_completion_bonus) bonuses.push(`Tasks: +${breakdown.task_completion_bonus}`);
                            
                            if (bonuses.length > 0) {
                                xpDetails = ` (${bonuses.join(', ')})`;
                            }
                        }
                        
                        this.modals.showToast(`🎉 Project Completed! +${xpAmount} XP${xpDetails}`, 'success', 5000);
                        
                        // Update user XP display
                        await this.updateUserXP();
                    }
                }
            }

            // Reload projects and refresh view
            await this.loadProjects();
            
            if (this.state.get('currentView') === 'projects') {
                await this.ui.renderView('projects');
            }

            const statusText = updatedProject.is_completed ? 'completed' : 'active';
            this.modals.showToast(`Project marked as ${statusText}`, 'success');

        } catch (error) {
            logger.error('Failed to toggle project completion:', error);
            this.modals.showToast('Failed to update project status', 'error');
        }
    }

    openTaskModal() {
        this.modals.openTaskModal();
    }

    openProjectModal() {
        this.modals.openProjectModal();
    }

    // View navigation method for global access
    showView(viewName) {
        this.state.set('currentView', viewName);
        this.navigateToView(viewName);
    }

    // Filter management methods
    async clearTaskFilter() {
        this.state.set('taskFilter', null);
        await this.ui.renderView('tasks');
        this.modals.showToast('Showing all tasks', 'info');
    }

    toggleFilterPanel() {
        const panel = document.getElementById('filter-panel');
        const toggle = document.getElementById('filter-toggle');
        
        if (!panel || !toggle) return;
        
        const isVisible = panel.style.display !== 'none';
        
        if (isVisible) {
            panel.style.display = 'none';
            toggle.classList.remove('active');
            localStorage.setItem('filterPanelVisible', 'false');
        } else {
            panel.style.display = 'block';
            toggle.classList.add('active');
            localStorage.setItem('filterPanelVisible', 'true');
        }
    }

    async toggleFilterPill(pill) {
        const type = pill.dataset.filterType;
        const value = pill.dataset.filterValue;
        const filters = this.state.get('filters');
        
        const numValue = !isNaN(value) ? parseInt(value) : value;
        
        if (type === 'project') {
            if (filters.projects.includes(numValue)) {
                filters.projects = filters.projects.filter(p => p !== numValue);
                pill.classList.remove('active');
            } else {
                filters.projects.push(numValue);
                pill.classList.add('active');
            }
        } else if (type === 'category') {
            if (filters.categories.includes(numValue)) {
                filters.categories = filters.categories.filter(c => c !== numValue);
                pill.classList.remove('active');
            } else {
                filters.categories.push(numValue);
                pill.classList.add('active');
            }
        } else if (type === 'status') {
            if (filters.statuses.includes(value)) {
                filters.statuses = filters.statuses.filter(s => s !== value);
                pill.classList.remove('active');
            } else {
                filters.statuses.push(value);
                pill.classList.add('active');
            }
        } else if (type === 'priority') {
            if (filters.priorities.includes(value)) {
                filters.priorities = filters.priorities.filter(p => p !== value);
                pill.classList.remove('active');
            } else {
                filters.priorities.push(value);
                pill.classList.add('active');
            }
        }
        
        this.state.set('filters', filters);
        localStorage.setItem('taskListFilters', JSON.stringify(filters));
        
        await this.ui.renderView('tasks');
        this.updateFilterBanner();
    }

    async sortTasks(sortBy) {
        if (!sortBy) return;
        
        const tasks = this.state.get('tasks');
        let sorted = [...tasks];
        
        switch(sortBy) {
            case 'date_desc':
                sorted.sort((a, b) => new Date(b.created_at) - new Date(a.created_at));
                break;
            case 'date_asc':
                sorted.sort((a, b) => new Date(a.created_at) - new Date(b.created_at));
                break;
            case 'priority':
                const priorityOrder = { urgent: 0, high: 1, medium: 2, low: 3 };
                sorted.sort((a, b) => (priorityOrder[a.priority] || 4) - (priorityOrder[b.priority] || 4));
                break;
            case 'title':
                sorted.sort((a, b) => a.title.localeCompare(b.title));
                break;
        }
        
        this.state.set('tasks', sorted);
        this.state.set('taskSort', sortBy);
        localStorage.setItem('taskSort', sortBy);
        await this.ui.renderView('tasks');
        
        // Restore dropdown value
        const sortSelect = document.getElementById('task-sort');
        if (sortSelect) {
            sortSelect.value = sortBy;
        }
        
        this.modals.showToast('Tasks sorted', 'success');
    }

    async clearAllFilters() {
        const filters = {
            projects: [],
            categories: [],
            statuses: [],
            priorities: []
        };
        
        this.state.set('filters', filters);
        localStorage.setItem('taskListFilters', JSON.stringify(filters));
        
        // Remove active class from all pills
        document.querySelectorAll('.filter-pill.active').forEach(pill => {
            pill.classList.remove('active');
        });
        
        await this.ui.renderView('tasks');
        this.updateFilterBanner();
        this.modals.showToast('All filters cleared', 'info');
    }

    updateFilterBanner() {
        const toggle = document.getElementById('filter-toggle');
        const banner = document.getElementById('filter-banner-container');
        if (!toggle) return;
        
        const filters = this.state.get('filters');
        const filterCount = (filters.projects?.length || 0) + 
                           (filters.categories?.length || 0) + 
                           (filters.statuses?.length || 0) + 
                           (filters.priorities?.length || 0);
        
        // Update toggle button to show filter count
        if (filterCount > 0) {
            toggle.innerHTML = `
                <i class="fas fa-filter"></i>
                <span>Advanced Filters</span>
                <span class="filter-count-badge">${filterCount}</span>
                <i class="fas fa-chevron-down filter-arrow"></i>
            `;
            
            // Show clear filters button in banner
            if (banner) {
                banner.innerHTML = `
                    <button class="clear-all-filters-btn" onclick="app.clearAllFilters()">
                        <i class="fas fa-times-circle"></i> Clear Filters (${filterCount})
                    </button>
                `;
            }
        } else {
            toggle.innerHTML = `
                <i class="fas fa-filter"></i>
                <span>Advanced Filters</span>
                <i class="fas fa-chevron-down filter-arrow"></i>
            `;
            
            if (banner) {
                banner.innerHTML = '';
            }
        }
    }

    async removeFilterTag(type, id) {
        const filters = this.state.get('filters');
        const numId = !isNaN(id) ? parseInt(id) : id;
        
        if (type === 'project') {
            filters.projects = filters.projects.filter(p => p !== numId);
        } else if (type === 'category') {
            filters.categories = filters.categories.filter(c => c !== numId);
        } else if (type === 'status') {
            filters.statuses = filters.statuses.filter(s => s !== id);
        } else if (type === 'priority') {
            filters.priorities = filters.priorities.filter(p => p !== id);
        }
        
        this.state.set('filters', filters);
        localStorage.setItem('taskListFilters', JSON.stringify(filters));
        
        // Update checkbox state in panel
        const checkbox = document.querySelector(
            `.filter-checkbox-input[data-filter-type="${type}"][data-filter-value="${id}"]`
        );
        if (checkbox) {
            checkbox.checked = false;
        }
        
        await this.ui.renderView('tasks');
        this.updateFilterBanner();
    }

    async setStatusFilter(status) {
        const filters = this.state.get('filters');
        filters.statuses = [status];
        this.state.set('filters', filters);
        localStorage.setItem('taskListFilters', JSON.stringify(filters));
        await this.ui.renderView('tasks');
        this.updateFilterBanner();
    }

    async sortProjects(sortBy) {
        if (!sortBy) return;
        
        const projects = this.state.get('projects');
        const tasks = this.state.get('tasks');
        let sorted = [...projects];
        
        switch(sortBy) {
            case 'name':
                sorted.sort((a, b) => a.name.localeCompare(b.name));
                this.modals.showToast('Projects sorted by name', 'info');
                break;
            case 'date':
                sorted.sort((a, b) => new Date(b.created_at) - new Date(a.created_at));
                this.modals.showToast('Projects sorted by date', 'info');
                break;
            case 'tasks':
                sorted.sort((a, b) => {
                    const aCount = tasks.filter(t => t.project_id === a.id).length;
                    const bCount = tasks.filter(t => t.project_id === b.id).length;
                    return bCount - aCount;
                });
                this.modals.showToast('Projects sorted by task count', 'info');
                break;
        }
        
        localStorage.setItem('projectSort', sortBy);
        this.state.set('projects', sorted);
        await this.ui.renderView('projects');
    }

    async sortCategories(sortBy) {
        if (!sortBy) return;
        
        const categories = this.state.get('categories');
        const tasks = this.state.get('tasks');
        const projects = this.state.get('projects');
        let sorted = [...categories];
        
        switch(sortBy) {
            case 'alpha':
                sorted.sort((a, b) => a.name.localeCompare(b.name));
                this.modals.showToast('Categories sorted alphabetically', 'info');
                break;
            case 'newest':
                sorted.sort((a, b) => new Date(b.created_at) - new Date(a.created_at));
                this.modals.showToast('Categories sorted by date', 'info');
                break;
            case 'projects':
                sorted.sort((a, b) => {
                    const aCount = projects.filter(p => p.category_id === a.id).length;
                    const bCount = projects.filter(p => p.category_id === b.id).length;
                    return bCount - aCount;
                });
                this.modals.showToast('Categories sorted by project count', 'info');
                break;
            case 'tasks':
                sorted.sort((a, b) => {
                    const aCount = tasks.filter(t => t.category_id === a.id).length;
                    const bCount = tasks.filter(t => t.category_id === b.id).length;
                    return bCount - aCount;
                });
                this.modals.showToast('Categories sorted by task count', 'info');
                break;
        }
        
        localStorage.setItem('categorySort', sortBy);
        this.state.set('categories', sorted);
        await this.ui.renderView('categories');
    }

    async clearProjectFilter() {
        this.state.set('projectFilter', null);
        await this.ui.renderView('projects');
        this.modals.showToast('Showing all projects', 'info');
    }

    // Category Management Methods
    openCategoryModal() {
        this.modals.openCategoryModal();
    }

    async editCategory(categoryId) {
        const category = this.state.get('categories').find(c => c.id === categoryId);
        if (category) {
            this.modals.openCategoryModal(category);
        }
    }

    async manageCategoryProjects(categoryId) {
        const category = this.state.get('categories').find(c => c.id === categoryId);
        if (category) {
            this.modals.openCategoryBindingModal(category);
        }
    }

    async deleteCategory(categoryId) {
        const category = this.state.get('categories').find(c => c.id === categoryId);
        if (category) {
            this.modals.openDeleteCategoryModal(category);
        }
    }

    // Navigation methods for category stats
    async navigateToTasks(categoryId) {
        try {
            // Switch to tasks view
            this.state.set('currentView', 'tasks');
            
            // Set filter to show only tasks with this category
            this.state.set('taskFilter', { categoryId });
            
            // Update UI
            await this.ui.renderView('tasks');
            
            // Update navigation active state
            document.querySelectorAll('.nav-link').forEach(link => link.classList.remove('active'));
            document.querySelector('[data-view="tasks"]').classList.add('active');
            
            // Show toast to inform user
            const categoryName = this.state.get('categories').find(c => c.id === categoryId)?.name || 'Category';
            this.modals.showToast(`Showing tasks in "${categoryName}"`, 'info');
            
        } catch (error) {
            logger.error('Error navigating to tasks:', error);
            this.modals.showToast('Failed to filter tasks by category', 'error');
        }
    }

    async navigateToProjects(categoryId) {
        try {
            // Switch to projects view
            this.state.set('currentView', 'projects');
            
            // Set filter to show only projects with this category
            this.state.set('projectFilter', { categoryId });
            
            // Update UI
            await this.ui.renderView('projects');
            
            // Update navigation active state
            document.querySelectorAll('.nav-link').forEach(link => link.classList.remove('active'));
            document.querySelector('[data-view="projects"]').classList.add('active');
            
            // Show toast to inform user
            const categoryName = this.state.get('categories').find(c => c.id === categoryId)?.name || 'Category';
            this.modals.showToast(`Showing projects in "${categoryName}"`, 'info');
            
        } catch (error) {
            logger.error('Error navigating to projects:', error);
            this.modals.showToast('Failed to filter projects by category', 'error');
        }
    }

    async initializeGettingStartedProject(userSettings) {
        try {
            // Check if this is a first-time user
            if (!userSettings || !userSettings._isFirstTime) {
                logger.debug('Not a first-time user, skipping Getting Started project');
                return;
            }

            // Check if Getting Started project already exists
            const projects = await this.api.getProjects();
            const existingGettingStarted = projects.data?.find(p => p.name === 'Getting Started');
            
            if (existingGettingStarted) {
                logger.debug('Getting Started project already exists');
                return;
            }

            logger.debug('Creating Getting Started project for new user...');

            // Create Getting Started project
            const gettingStartedProject = {
                name: 'Getting Started',
                description: 'Learn how to use Task Manager with these guided tasks. Complete them to get familiar with the system!',
                color: '#2563eb'
            };

            const projectResponse = await this.api.createProject(gettingStartedProject);
            const projectId = projectResponse.data.id;

            // Create General category if needed
            const categories = await this.api.getCategories();
            let generalCategoryId;
            
            const existingGeneralCategory = categories.data?.find(c => c.name === 'General');
            if (existingGeneralCategory) {
                generalCategoryId = existingGeneralCategory.id;
            } else {
                const generalCategory = {
                    name: 'General',
                    description: '📋 General tasks',
                    color: '#6366f1',
                    auto_bind: true
                };
                const categoryResponse = await this.api.createCategory(generalCategory);
                generalCategoryId = categoryResponse.data.id;
            }
            // Create Onboarding category
            let onboardingCategoryId;
            const existingOnboardingCategory = categories.data?.find(c => c.name === 'Onboarding');
            if (existingOnboardingCategory) {
                onboardingCategoryId = existingOnboardingCategory.id;
            } else {
                const onboardingCategory = {
                    name: 'Onboarding',
                    description: '🎓 Onboarding tasks - will be deleted when complete',
                    color: '#f59e0b',
                    auto_bind: false
                };
                const categoryResponse = await this.api.createCategory(onboardingCategory);
                onboardingCategoryId = categoryResponse.data.id;
            }


            // Bind both categories to Getting Started project
            await fetch(`${this.api.baseUrl}categories/${generalCategoryId}/bind`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                credentials: 'include',
                body: JSON.stringify({ project_ids: [parseInt(projectId)] })
            });
            await fetch(`${this.api.baseUrl}categories/${onboardingCategoryId}/bind`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                credentials: 'include',
                body: JSON.stringify({ project_ids: [parseInt(projectId)] })
            });

            // Create getting started tasks
            const onboardingTasks = [
                {
                    title: 'Create your first project',
                    description: 'Learn how to organize your work by creating a new project. Click the "New Project" button to get started.',
                    project_id: projectId,
                    category_id: onboardingCategoryId,
                    priority: 'high',
                    status: 'pending',
                    onboarding_action: 'create_project'
                },
                {
                    title: 'Create your first task',
                    description: 'Tasks are the building blocks of your projects. Try creating a new task to see how it works.',
                    project_id: projectId,
                    category_id: onboardingCategoryId,
                    priority: 'high',
                    status: 'pending',
                    onboarding_action: 'create_task'
                },
                {
                    title: 'Create a new category',
                    description: 'Categories help you organize tasks by type or theme. Create a category to better organize your work.',
                    project_id: projectId,
                    category_id: onboardingCategoryId,
                    priority: 'medium',
                    status: 'pending',
                    onboarding_action: 'create_category'
                },
                {
                    title: 'Explore the Settings tab',
                    description: 'Customize TaskList to work the way you want. Check out the Settings tab to see what options are available.',
                    project_id: projectId,
                    category_id: onboardingCategoryId,
                    priority: 'medium',
                    status: 'pending',
                    onboarding_action: 'visit_settings'
                },
                {
                    title: 'Explore the Account tab',
                    description: 'View your progress, XP, and achievements in the Account tab. See how the gamification system works!',
                    project_id: projectId,
                    category_id: onboardingCategoryId,
                    priority: 'medium',
                    status: 'pending',
                    onboarding_action: 'visit_account'
                },
                {
                    title: 'Complete your first task',
                    description: 'Mark a task as complete to see how the system tracks your progress and awards XP.',
                    project_id: projectId,
                    category_id: onboardingCategoryId,
                    priority: 'low',
                    status: 'pending',
                    onboarding_action: 'complete_task'
                },
                {
                    title: 'Level up for the first time',
                    description: 'Complete tasks to earn XP and level up! This happens automatically as you use the system.',
                    project_id: projectId,
                    category_id: onboardingCategoryId,
                    priority: 'low',
                    status: 'pending',
                    onboarding_action: 'level_up'
                }
            ];

            // Create all onboarding tasks
            for (const task of onboardingTasks) {
                await this.api.createTask(task);
            }

            logger.debug('Getting Started project and tasks created successfully');
            
        } catch (error) {
            logger.error('Error creating Getting Started project:', error);
            // Don't throw error to prevent app initialization failure
        }
    }

    async checkOnboardingTaskCompletion(action) {
        try {
            // Get all tasks to find onboarding tasks
            const tasks = this.state.get('tasks') || [];
            logger.debug(`Checking onboarding completion for action: ${action}`);
            logger.debug(`Total tasks loaded: ${tasks.length}`);
            
            // Debug: log all tasks with onboarding_action
            const onboardingTasks = tasks.filter(task => task.onboarding_action);
            logger.debug(`Found ${onboardingTasks.length} onboarding tasks:`, onboardingTasks.map(t => ({
                title: t.title,
                action: t.onboarding_action,
                status: t.status
            })));
            
            const onboardingTask = tasks.find(task => 
                task.onboarding_action === action && 
                task.status !== 'completed'
            );

            if (onboardingTask) {
                logger.debug(`Found matching onboarding task: ${onboardingTask.title}`);
                // Complete the onboarding task (this awards XP automatically via API)
                const response = await this.api.updateTask(onboardingTask.id, { status: 'completed' });
                
                logger.debug('=== XP DEBUG START ===');
                logger.debug('Full Response:', JSON.stringify(response, null, 2));
                logger.debug('Response type:', typeof response);
                logger.debug('Response keys:', Object.keys(response));
                logger.debug('=== XP DEBUG END ===');
                
                // Reload tasks to reflect the change
                await this.loadTasks();
                
                // Update user XP to reflect the points earned
                await this.updateUserXP();
                
                // Refresh the current view to show the updated task
                const currentView = this.state.get('currentView');
                if (currentView) {
                    await this.ui.renderView(currentView);
                }
                
                // Check both possible response structures
                const xpData = response.data?.xp_earned ? response.data : response;
                
                // Show XP notification if XP was earned
                if (xpData && xpData.xp_earned) {
                    const xpAmount = xpData.xp_earned;
                    const breakdown = xpData.xp_breakdown;
                    
                    // Build XP breakdown message
                    let xpDetails = '';
                    if (breakdown) {
                        const bonuses = [];
                        if (breakdown.title_bonus) bonuses.push(`Title: +${breakdown.title_bonus}`);
                        if (breakdown.description_bonus) bonuses.push(`Description: +${breakdown.description_bonus}`);
                        if (breakdown.category_bonus) bonuses.push(`Category: +${breakdown.category_bonus}`);
                        if (breakdown.priority_bonus) bonuses.push(`Priority: +${breakdown.priority_bonus}`);
                        if (breakdown.complexity_bonus) bonuses.push(`Complexity: +${breakdown.complexity_bonus}`);
                        if (breakdown.early_completion) bonuses.push(`Early: +${breakdown.early_completion}`);
                        
                        if (bonuses.length > 0) {
                            xpDetails = ` (${bonuses.join(', ')})`;
                        }
                    }
                    
                    this.modals.showToast(`🎉 +${xpAmount} XP${xpDetails}`, 'success');
                } else {
                    logger.warn('No XP data in response:', response);
                }
                
                // Show a congratulatory message
                this.modals.showToast(`✅ Onboarding: ${onboardingTask.title}`, 'success');
                
                logger.debug(`Onboarding task completed: ${action}`);
                
                // Check if all other onboarding tasks are now complete
                await this.checkAllOnboardingComplete();
            } else {
                logger.debug(`No matching onboarding task found for action: ${action}`);
            }
        } catch (error) {
            logger.error('Error checking onboarding task completion:', error);
        }
    }

    async checkAllOnboardingComplete() {
        try {
            const tasks = this.state.get('tasks') || [];
            const onboardingTasks = tasks.filter(task => task.onboarding_action);
            
            // Get all onboarding tasks except the final "complete_onboarding" task
            const regularOnboardingTasks = onboardingTasks.filter(t => 
                t.onboarding_action !== 'complete_onboarding'
            );
            
            // Check if all regular onboarding tasks are completed
            const allComplete = regularOnboardingTasks.every(t => t.status === 'completed');
            
            if (allComplete && regularOnboardingTasks.length > 0) {
                // Auto-complete the final onboarding task
                const finalTask = onboardingTasks.find(t => 
                    t.onboarding_action === 'complete_onboarding' && 
                    t.status !== 'completed'
                );
                
                if (finalTask) {
                    logger.debug('All onboarding tasks complete! Completing final task.');
                    const response = await this.api.updateTask(finalTask.id, { status: 'completed' });
                    await this.loadTasks();
                    
                    // Update user XP to reflect the final task completion
                    await this.updateUserXP();
                    
                    // Refresh the view to show completion
                    const currentView = this.state.get('currentView');
                    if (currentView) {
                        await this.ui.renderView(currentView);
                    }
                    
                    // Show XP notification for final task
                    if (response.data && response.data.xp_earned) {
                        this.modals.showToast(`🎉 +${response.data.xp_earned} XP`, 'success');
                    }
                    
                    this.modals.showToast('🎊 Congratulations! You\'ve completed the onboarding!', 'success');
                    
                    // Mark the Getting Started project as completed after a short delay
                    setTimeout(async () => {
                        try {
                            logger.debug('Completing Getting Started project...');
                            const projects = await this.api.getProjects();
                            const gettingStarted = projects.data?.find(p => p.name === 'Getting Started');
                            
                            if (gettingStarted) {
                                // Mark project as completed (this will award XP via toggleProjectCompletion)
                                await this.toggleProjectCompletion(gettingStarted.id);
                                logger.debug('✓ Getting Started project marked as completed');
                                
                                // Delete the Onboarding category
                                const categories = await this.api.getCategories();
                                const onboardingCategory = categories.data?.find(c => c.name === 'Onboarding');
                                
                                if (onboardingCategory) {
                                    // Check if any other tasks use this category
                                    const allTasks = await this.api.getTasks();
                                    const tasksUsingOnboarding = allTasks.data?.filter(t => 
                                        t.category_id === onboardingCategory.id &&
                                        t.project_id !== gettingStarted.id
                                    );
                                    
                                    if (!tasksUsingOnboarding || tasksUsingOnboarding.length === 0) {
                                        await fetch(`${this.api.baseUrl}categories/${onboardingCategory.id}`, {
                                            method: 'DELETE',
                                            credentials: 'include'
                                        });
                                        logger.debug('✓ Onboarding category deleted');
                                        this.modals.showToast('✅ Getting Started project completed and Onboarding category cleaned up!', 'success');
                                    } else {
                                        logger.debug('Onboarding category still in use by other tasks');
                                        this.modals.showToast('✅ Getting Started project completed!', 'success');
                                    }
                                } else {
                                    this.modals.showToast('✅ Getting Started project completed!', 'success');
                                }
                                
                                // Reload data to reflect changes
                                await this.loadProjects();
                                await this.loadCategories();
                                await this.ui.renderView(currentView);
                            }
                        } catch (error) {
                            logger.error('Error completing Getting Started project:', error);
                        }
                    }, 3000); // 3 second delay to let user see the completion message
                }
            }
        } catch (error) {
            logger.error('Error checking all onboarding complete:', error);
        }
    }

    // Temporary method to manually create Getting Started project for testing
    async createGettingStartedProjectManually() {
        try {
            logger.debug('Manually creating Getting Started project...');

            // Check if Getting Started project already exists
            const projects = await this.api.getProjects();
            const existingGettingStarted = projects.data?.find(p => p.name === 'Getting Started');
            
            if (existingGettingStarted) {
                // Also check if onboarding tasks exist
                const tasks = await this.api.getTasks();
                const onboardingTasks = tasks.data?.filter(t => 
                    t.project_id === existingGettingStarted.id && 
                    t.onboarding_action
                );
                
                if (onboardingTasks && onboardingTasks.length >= 7) {
                    this.modals.showToast('Getting Started project already exists with all tasks!', 'info');
                    return;
                } else {
                    this.modals.showToast('Getting Started project exists but missing tasks. Please delete it first.', 'warning');
                    return;
                }
            }

            // Create Getting Started project
            const gettingStartedProject = {
                name: 'Getting Started',
                description: 'Learn how to use Task Manager with these guided tasks. Complete them to get familiar with the system!',
                color: '#2563eb'
            };

            const projectResponse = await this.api.createProject(gettingStartedProject);
            const projectId = projectResponse.data.id;

            // Create General category if needed
            const categories = await this.api.getCategories();
            let generalCategoryId;
            
            const existingGeneralCategory = categories.data?.find(c => c.name === 'General');
            if (existingGeneralCategory) {
                generalCategoryId = existingGeneralCategory.id;
            } else {
                const generalCategory = {
                    name: 'General',
                    description: '📋 General tasks',
                    color: '#6366f1',
                    auto_bind: true
                };
                const categoryResponse = await this.api.createCategory(generalCategory);
                generalCategoryId = categoryResponse.data.id;
            }
            // Create Onboarding category
            let onboardingCategoryId;
            const existingOnboardingCategory = categories.data?.find(c => c.name === 'Onboarding');
            if (existingOnboardingCategory) {
                onboardingCategoryId = existingOnboardingCategory.id;
            } else {
                const onboardingCategory = {
                    name: 'Onboarding',
                    description: '🎓 Onboarding tasks - will be deleted when complete',
                    color: '#f59e0b',
                    auto_bind: false
                };
                const categoryResponse = await this.api.createCategory(onboardingCategory);
                onboardingCategoryId = categoryResponse.data.id;
            }


            // Bind both categories to Getting Started project
            await fetch(`${this.api.baseUrl}categories/${generalCategoryId}/bind`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                credentials: 'include',
                body: JSON.stringify({ project_ids: [parseInt(projectId)] })
            });
            await fetch(`${this.api.baseUrl}categories/${onboardingCategoryId}/bind`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                credentials: 'include',
                body: JSON.stringify({ project_ids: [parseInt(projectId)] })
            });

            // Create getting started tasks
            const onboardingTasks = [
                {
                    title: 'Create your first project',
                    description: 'Learn how to organize your work by creating a new project. Click the "New Project" button to get started.',
                    project_id: projectId,
                    category_id: onboardingCategoryId,
                    priority: 'high',
                    status: 'pending',
                    onboarding_action: 'create_project'
                },
                {
                    title: 'Create your first task',
                    description: 'Tasks are the building blocks of your projects. Try creating a new task to see how it works.',
                    project_id: projectId,
                    category_id: onboardingCategoryId,
                    priority: 'high',
                    status: 'pending',
                    onboarding_action: 'create_task'
                },
                {
                    title: 'Create a new category',
                    description: 'Categories help you organize tasks by type or theme. Create a category to better organize your work.',
                    project_id: projectId,
                    category_id: onboardingCategoryId,
                    priority: 'medium',
                    status: 'pending',
                    onboarding_action: 'create_category'
                },
                {
                    title: 'Explore the Settings tab',
                    description: 'Customize TaskList to work the way you want. Check out the Settings tab to see what options are available.',
                    project_id: projectId,
                    category_id: onboardingCategoryId,
                    priority: 'medium',
                    status: 'pending',
                    onboarding_action: 'visit_settings'
                },
                {
                    title: 'Explore the Account tab',
                    description: 'View your progress, XP, and achievements in the Account tab. See how the gamification system works!',
                    project_id: projectId,
                    category_id: onboardingCategoryId,
                    priority: 'medium',
                    status: 'pending',
                    onboarding_action: 'visit_account'
                },
                {
                    title: 'Complete your first task',
                    description: 'Mark a task as complete to see how the system tracks your progress and awards XP.',
                    project_id: projectId,
                    category_id: onboardingCategoryId,
                    priority: 'low',
                    status: 'pending',
                    onboarding_action: 'complete_task'
                },
                {
                    title: 'Level up for the first time',
                    description: 'Complete tasks to earn XP and level up! This happens automatically as you use the system.',
                    project_id: projectId,
                    category_id: onboardingCategoryId,
                    priority: 'low',
                    status: 'pending',
                    onboarding_action: 'level_up'
                }
            ];

            // Create all onboarding tasks
            for (const task of onboardingTasks) {
                await this.api.createTask(task);
            }

            this.modals.showToast('🎉 Getting Started project created successfully!', 'success');
            logger.debug('Getting Started project and tasks created successfully');
            
            // Reload data to show the new project
            await this.loadTasks();
            await this.loadProjects();
            await this.loadCategories();
            
            // Navigate to projects view to see the new project
            await this.navigateToView('projects');
            
        } catch (error) {
            logger.error('Error creating Getting Started project manually:', error);
            this.modals.showToast('Failed to create Getting Started project: ' + error.message, 'error');
        }
    }
}

// =============================================================================
// DEVELOPER TOOLS
// =============================================================================

class DevTools {
    constructor(app) {
        this.app = app;
        this.isVisible = false;
    }

    init() {
        // Auto-enable for admins
        const isAdmin = CURRENT_USER.role && CURRENT_USER.role.includes('admin');
        const devBarEnabled = localStorage.getItem('devBarEnabled') === 'true' || isAdmin;
        
        if (devBarEnabled || isAdmin) {
            this.showTrigger();
            // Save preference
            if (isAdmin) {
                localStorage.setItem('devBarEnabled', 'true');
            }
        }
        
        // Restore DevTools state if it was open
        const wasOpen = localStorage.getItem('devToolsOpen') === 'true';
        if (wasOpen) {
            const devBar = document.getElementById('dev-bar');
            const trigger = document.getElementById('dev-bar-trigger');
            if (devBar && trigger) {
                devBar.style.display = 'block';
                trigger.style.display = 'none';
                
                // Restore active tab
                const savedTab = localStorage.getItem('devToolsActiveTab') || 'general';
                this.switchTab(savedTab);
            }
        }
        
        // Update stats
        this.updateStats();
        
        // Listen for state changes to update stats
        if (this.app.state) {
            setInterval(() => this.updateStats(), 2000);
        }
    }

    showTrigger() {
        const trigger = document.getElementById('dev-bar-trigger');
        if (trigger) {
            trigger.style.display = 'flex';
        }
    }

    hideTrigger() {
        const trigger = document.getElementById('dev-bar-trigger');
        if (trigger) {
            trigger.style.display = 'none';
        }
    }

    populateCategorySelector() {
        const categorySelect = document.getElementById('gen-task-category');
        if (!categorySelect) return;
        
        const categories = this.app.state.get('categories') || [];
        
        // Clear existing options except "Random Category"
        categorySelect.innerHTML = '<option value="random">Random Category</option>';
        
        // Add category options
        categories.forEach(category => {
            const option = document.createElement('option');
            option.value = category.id;
            option.textContent = category.name;
            categorySelect.appendChild(option);
        });
    }

    updateStats() {
        const tasks = this.app.state.get('tasks') || [];
        const projects = this.app.state.get('projects') || [];
        const categories = this.app.state.get('categories') || [];
        const user = this.app.state.get('user') || {};
        
        // Count test items (names starting with [TEST])
        const testTasks = tasks.filter(t => t.title && t.title.startsWith('[TEST]'));
        const testProjects = projects.filter(p => p.name && p.name.startsWith('[TEST]'));
        
        document.getElementById('dev-tasks-count').textContent = tasks.length;
        document.getElementById('dev-test-tasks-count').textContent = `/ ${testTasks.length} test`;
        document.getElementById('dev-projects-count').textContent = projects.length;
        document.getElementById('dev-test-projects-count').textContent = `/ ${testProjects.length} test`;
        document.getElementById('dev-categories-count').textContent = categories.length;
        document.getElementById('dev-level').textContent = user.level || 1;
        document.getElementById('dev-xp-total').textContent = user.xp_total || 0;
        document.getElementById('dev-username').textContent = CURRENT_USER.username || '-';
        
        // Update cache metrics if available
        if (this.app.api && this.app.api.apiCache) {
            const metrics = this.app.api.getCacheMetrics();
            const cacheStatsEl = document.getElementById('dev-cache-stats');
            if (cacheStatsEl) {
                cacheStatsEl.innerHTML = `
                    <strong>Cache:</strong> ${metrics.size} entries | 
                    Hit Rate: ${metrics.hitRate} | 
                    Size: ${this.app.api.apiCache.getFormattedSize()}
                `;
            }
        }
    }

    viewState() {
        const state = {
            tasks: this.app.state.get('tasks'),
            projects: this.app.state.get('projects'),
            categories: this.app.state.get('categories'),
            filters: this.app.state.get('filters'),
            user: this.app.state.get('user'),
            currentView: this.app.state.get('currentView')
        };
        
        logger.debug('=== APPLICATION STATE ===');
        logger.debug(state);
        console.table({
            'Total Tasks': state.tasks.length,
            'Total Projects': state.projects.length,
            'Total Categories': state.categories.length,
            'Current View': state.currentView,
            'User XP': state.user?.xp_total || 0
        });
        
        alert('State logged to console (F12)');
    }

    clearCache() {
        if (confirm('Clear all localStorage data? This will reset filters and preferences.')) {
            const keysToKeep = ['devBarEnabled']; // Keep dev bar enabled
            const allKeys = Object.keys(localStorage);
            
            allKeys.forEach(key => {
                if (!keysToKeep.includes(key)) {
                    localStorage.removeItem(key);
                }
            });
            
            logger.debug('Cache cleared!');
            alert('Cache cleared! Page will reload.');
            window.location.reload();
        }
    }

    async testAPI() {
        logger.debug('=== API TEST ===');
        logger.debug('API Base URL:', this.app.api.baseUrl);
        
        try {
            const response = await fetch(`${this.app.api.baseUrl}tasks`, {
                method: 'GET',
                credentials: 'include'
            });
            const data = await response.json();
            
            logger.debug('API Response:', data);
            logger.debug('Status:', response.status);
            logger.debug('Success:', data.success);
            
            alert(`API Test: ${response.status} - ${data.success ? 'SUCCESS' : 'FAILED'}\nCheck console for details`);
        } catch (error) {
            logger.error('API Test Error:', error);
            alert('API Test FAILED - Check console');
        }
    }

    exportData() {
        const data = {
            tasks: this.app.state.get('tasks'),
            projects: this.app.state.get('projects'),
            categories: this.app.state.get('categories'),
            user: this.app.state.get('user'),
            exportedAt: new Date().toISOString()
        };
        
        const blob = new Blob([JSON.stringify(data, null, 2)], { type: 'application/json' });
        const url = URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = `tasklist-export-${Date.now()}.json`;
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
        URL.revokeObjectURL(url);
        
        logger.debug('Data exported:', data);
    }

    openConsole() {
        logger.debug('%c=== DEVELOPER CONSOLE ===', 'color: #2563eb; font-size: 16px; font-weight: bold;');
        logger.debug('%cAvailable commands:', 'color: #10b981; font-weight: bold;');
        logger.debug('  app - Access the main application instance');
        logger.debug('  app.state.get(key) - Get state value');
        logger.debug('  app.state.set(key, value) - Set state value');
        logger.debug('  app.loadTasks() - Reload tasks');
        logger.debug('  app.devTools.viewState() - View full state');
        logger.debug('%cPress F12 to open developer tools', 'color: #f59e0b;');
        
        alert('Console commands logged!\nOpen DevTools (F12) to use them.');
    }

    async reloadData() {
        logger.debug('Reloading all data...');
        try {
            await this.app.loadInitialData();
            await this.app.ui.renderView(this.app.state.get('currentView'));
            this.updateStats();
            logger.debug('Data reloaded successfully!');
            alert('Data reloaded successfully!');
        } catch (error) {
            logger.error('Failed to reload data:', error);
            alert('Failed to reload data - check console');
        }
    }

    async resetOnboarding() {
        if (!confirm('This will DELETE the existing "Getting Started" project and recreate it with all onboarding tasks.\n\nContinue?')) {
            return;
        }

        logger.debug('=== RESETTING ONBOARDING ===');
        
        try {
            // Step 1: Find and delete ALL existing Getting Started projects
            const projects = await this.app.api.getProjects();
            const gettingStartedProjects = projects.data?.filter(p => p.name === 'Getting Started') || [];
            
            if (gettingStartedProjects.length > 0) {
                logger.debug(`Deleting ${gettingStartedProjects.length} existing Getting Started project(s)...`);
                for (const project of gettingStartedProjects) {
                    await fetch(`${this.app.api.baseUrl}projects/${project.id}`, {
                        method: 'DELETE',
                        credentials: 'include'
                    });
                    logger.debug(`✓ Deleted old project ID: ${project.id}`);
                }
            }

            // Step 2: Create new Getting Started project
            logger.debug('Creating new Getting Started project...');
            const projectResponse = await this.app.api.createProject({
                name: 'Getting Started',
                description: '🎯 Complete these tasks to learn Task Manager! Each task teaches you a feature. Delete this project when you\'re done with onboarding.',
                color: '#2563eb'
            });
            const projectId = projectResponse.data.id;
            logger.debug('✓ Created project:', projectId);

            // Step 3: Ensure General category exists
            const categories = await this.app.api.getCategories();
            let generalCategoryId = categories.data?.find(c => c.name === 'General')?.id;
            
            if (!generalCategoryId) {
                logger.debug('Creating General category...');
                const generalCatResponse = await this.app.api.createCategory({
                    name: 'General',
                    description: '📋 General tasks',
                    color: '#6366f1',
                    auto_bind: true
                });
                generalCategoryId = generalCatResponse.data.id;
                logger.debug('✓ Created General category:', generalCategoryId);
            }

            // Step 4: Create Onboarding category
            logger.debug('Creating Onboarding category...');
            let onboardingCategoryId = categories.data?.find(c => c.name === 'Onboarding')?.id;
            
            if (!onboardingCategoryId) {
                const onboardingCatResponse = await this.app.api.createCategory({
                    name: 'Onboarding',
                    description: '🎓 Onboarding tasks - will be deleted when complete',
                    color: '#f59e0b',
                    auto_bind: false
                });
                onboardingCategoryId = onboardingCatResponse.data.id;
                logger.debug('✓ Created Onboarding category:', onboardingCategoryId);
            }

            // Step 5: Bind both categories to Getting Started project
            logger.debug('Binding categories to Getting Started project...');
            await fetch(`${this.app.api.baseUrl}categories/${generalCategoryId}/bind`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                credentials: 'include',
                body: JSON.stringify({ project_ids: [parseInt(projectId)] })
            });
            await fetch(`${this.app.api.baseUrl}categories/${onboardingCategoryId}/bind`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                credentials: 'include',
                body: JSON.stringify({ project_ids: [parseInt(projectId)] })
            });
            logger.debug('✓ Bound categories to project');

            // Step 5: Create comprehensive onboarding tasks
            logger.debug('Creating onboarding tasks...');
            const tasks = [
                // Phase 1: First Steps
                { title: '👋 Welcome to Task Manager!', description: 'Click this task to open it and explore. You can edit details, add subtasks (steps), and mark it complete. Task Manager helps you organize work with projects, categories, priorities, and XP rewards!', priority: 'high', complexity: 'simple', onboarding_action: 'welcome' },
                { title: '📊 Check out the Dashboard', description: 'Click "Dashboard" in the sidebar. The dashboard shows project stats, recent tasks, and your XP progress at a glance. It\'s your command center!', priority: 'high', complexity: 'simple', onboarding_action: 'visit_dashboard' },
                { title: '✅ Complete your first task', description: 'Click the "Complete" button on any task to mark it done. You\'ll earn 10 XP! Tasks can be marked as pending, in-progress, or completed.', priority: 'high', complexity: 'simple', onboarding_action: 'complete_task' },
                
                // Phase 2: Building Blocks
                { title: '📁 Create a project', description: 'Click "+ New Project" in the sidebar. Projects organize related tasks - like "Work", "Personal", or "Learning". Give it a name and color!', priority: 'high', complexity: 'simple', onboarding_action: 'create_project' },
                { title: '🏷️ Create a category', description: 'Categories tag tasks by type. Click "+ New Category" and try "Ideas", "Bugs", or "Urgent". Categories have colors too!', priority: 'high', complexity: 'simple', onboarding_action: 'create_category' },
                { title: '📝 Create a custom task', description: 'Click the blue "+" button (FAB) in the corner, or use "+ New Task". Add a title, description, set priority (low/medium/high/urgent), assign to a project, add a due date, and set complexity!', priority: 'high', complexity: 'moderate', onboarding_action: 'create_task' },
                
                // Phase 3: Power Features
                { title: '🔢 Break tasks into steps', description: 'Open any task and click "Add Step" to create subtasks. Break complex work into smaller pieces. Each completed step earns 2-5 XP (bonus for descriptive steps)!', priority: 'medium', complexity: 'moderate', onboarding_action: 'create_subtask' },
                { title: '🔍 Filter and organize', description: 'Use the filter dropdowns in Tasks view. Filter by status, priority, project, or category. Filters persist across page refreshes!', priority: 'medium', complexity: 'moderate', onboarding_action: 'use_filter' },
                { title: '🎯 Master task properties', description: 'Edit any task and try these features: Set a due date to track deadlines, choose priority (Low/Medium/High/Urgent) to focus on what matters, and set complexity (Simple/Moderate/Complex/Very Complex) to estimate effort. These properties help you organize and prioritize your work!', priority: 'medium', complexity: 'moderate', onboarding_action: 'set_task_properties' },
                
                // Phase 4: Gamification
                { title: '⭐ View your XP & level', description: 'Click "Account" in the sidebar. See your total XP, current level, and XP history. You earn 10 XP per task completion, 2-5 XP per subtask!', priority: 'medium', complexity: 'simple', onboarding_action: 'visit_account' },
                { title: '🎮 Level up!', description: 'Complete tasks to gain XP and level up. Each level requires more XP than the last. Watch your progress bar grow!', priority: 'medium', complexity: 'simple', onboarding_action: 'level_up' },
                
                // Phase 5: Customization
                { title: '⚙️ Customize settings', description: 'Visit Settings to set defaults: priority levels, due date offsets, filter preferences. Control how new tasks are created!', priority: 'low', complexity: 'simple', onboarding_action: 'visit_settings' },
                { title: '🎨 Color-code your workspace', description: 'Edit projects and categories to change colors. Make your workspace visually organized and easy to scan!', priority: 'low', complexity: 'simple', onboarding_action: 'edit_colors' },
                { title: '📱 Try mobile view', description: 'Task Manager is fully responsive! Open it on your phone or tablet. The sidebar collapses, filters adapt, and everything works seamlessly.', priority: 'low', complexity: 'simple', onboarding_action: 'mobile_view' },
                
                // Final
                { title: '🎉 Onboarding complete!', description: 'You\'ve learned Task Manager! Now add your own tasks, projects, and goals. You can delete this Getting Started project anytime. Happy tasking! 🚀', priority: 'low', complexity: 'simple', onboarding_action: 'complete_onboarding' }
            ];

            let created = 0;
            for (const task of tasks) {
                await this.app.api.createTask({
                    title: task.title,
                    description: task.description,
                    project_id: parseInt(projectId),
                    category_id: parseInt(onboardingCategoryId),
                    priority: task.priority,
                    complexity: task.complexity,
                    status: 'pending',
                    onboarding_action: task.onboarding_action || null
                });
                created++;
            }
            
            logger.debug(`✓ Created ${created} onboarding tasks`);
            logger.debug('=== ONBOARDING RESET COMPLETE ===');
            
            // Reload data to show new content
            await this.reloadData();
            alert(`✅ Onboarding reset complete!\n\n• Created "Getting Started" project\n• Added ${created} onboarding tasks\n• Ready to learn Task Manager!`);
            
        } catch (error) {
            logger.error('Failed to reset onboarding:', error);
            alert('❌ Failed to reset onboarding. Check console for details.');
        }
    }

    async removeOnboarding() {
        if (!confirm('This will PERMANENTLY DELETE:\n\n• "Getting Started" project\n• All onboarding tasks\n• "Onboarding" category\n\nThis cannot be undone. Continue?')) {
            return;
        }

        logger.debug('=== REMOVING ONBOARDING ===');
        
        try {
            // Step 1: Find and delete Getting Started project(s)
            const projects = await this.app.api.getProjects();
            const gettingStartedProjects = projects.data?.filter(p => p.name === 'Getting Started') || [];
            
            if (gettingStartedProjects.length > 0) {
                logger.debug(`Deleting ${gettingStartedProjects.length} Getting Started project(s)...`);
                for (const project of gettingStartedProjects) {
                    // Delete the project (tasks will cascade delete via foreign key constraints)
                    await fetch(`${this.app.api.baseUrl}projects/${project.id}`, {
                        method: 'DELETE',
                        credentials: 'include'
                    });
                    logger.debug(`✓ Deleted project ID: ${project.id} (tasks cascade deleted)`);
                }
            } else {
                logger.debug('⚠️ No "Getting Started" projects found');
            }

            // Step 2: Find and delete Onboarding category
            const categories = await this.app.api.getCategories();
            const onboardingCategory = categories.data?.find(c => c.name === 'Onboarding');
            
            if (onboardingCategory) {
                logger.debug('Deleting Onboarding category...');
                await fetch(`${this.app.api.baseUrl}categories/${onboardingCategory.id}`, {
                    method: 'DELETE',
                    credentials: 'include'
                });
                logger.debug(`✓ Deleted Onboarding category ID: ${onboardingCategory.id}`);
            } else {
                logger.debug('⚠️ No "Onboarding" category found');
            }

            // Step 3: Delete any orphaned onboarding tasks (just in case)
            const tasks = await this.app.api.getTasks();
            const onboardingTasks = tasks.data?.filter(t => 
                t.title?.includes('Getting Started') || 
                t.onboarding_action || 
                t.project_name === 'Getting Started'
            ) || [];
            
            if (onboardingTasks.length > 0) {
                logger.debug(`Found ${onboardingTasks.length} orphaned onboarding tasks, deleting...`);
                for (const task of onboardingTasks) {
                    await fetch(`${this.app.api.baseUrl}tasks/${task.id}`, {
                        method: 'DELETE',
                        credentials: 'include'
                    });
                }
                logger.debug(`✓ Deleted ${onboardingTasks.length} orphaned tasks`);
            }

            logger.debug('=== ONBOARDING REMOVAL COMPLETE ===');
            
            // Reload data to reflect deletions
            await this.reloadData();
            
            const message = gettingStartedProjects.length > 0 || onboardingCategory || onboardingTasks.length > 0
                ? `✅ Onboarding removed successfully!\n\n• Deleted ${gettingStartedProjects.length} project(s)\n• Deleted Onboarding category\n• Deleted ${onboardingTasks.length} orphaned task(s)`
                : '⚠️ No onboarding content found to remove.';
            
            alert(message);
            
        } catch (error) {
            logger.error('Failed to remove onboarding:', error);
            alert('❌ Failed to remove onboarding. Check console for details.');
        }
    }

    async createTestTasks() {
        const countInput = document.getElementById('test-task-count');
        const projectSelect = document.getElementById('test-project-select');
        const categorySelect = document.getElementById('test-category-select');
        
        const count = parseInt(countInput?.value || 10);
        const projectMode = projectSelect?.value || 'random';
        const categoryMode = categorySelect?.value || 'random';
        
        if (count < 1 || count > 500) {
            alert('Please enter a number between 1 and 500');
            return;
        }

        if (!confirm(`Create ${count} test tasks?\nProject: ${projectMode}\nCategory: ${categoryMode}`)) {
            return;
        }

        logger.info(`Creating ${count} test tasks...`);
        const startTime = Date.now();
        let created = 0;
        let failed = 0;
        let testProjectId = null;

        try {
            // Get available projects and categories
            const projects = await this.app.api.getProjects();
            const categories = await this.app.api.getCategories();
            
            if (!categories.data || categories.data.length === 0) {
                alert('❌ No categories found. Please create a category first.');
                return;
            }
            
            // Handle project selection
            if (projectMode === 'test') {
                // Create [TEST] project
                const testProject = await this.app.api.createProject({
                    name: '[TEST] Project',
                    description: 'Test project for dev tools. Safe to delete.',
                    color: '#9333ea'
                });
                testProjectId = testProject.data.id;
                logger.info(`Created [TEST] Project with ID: ${testProjectId}`);
            } else if (!projects.data || projects.data.length === 0) {
                alert('❌ No projects found. Please create a project first or select "[TEST] Project" option.');
                return;
            }
            
            const priorities = ['low', 'medium', 'high', 'urgent'];
            const statuses = ['pending', 'in_progress', 'pending'];
            const projectIds = projectMode === 'test' ? [testProjectId] : projects.data.map(p => p.id);
            const categoryIds = categories.data.map(c => c.id);
            
            for (let i = 1; i <= count; i++) {
                try {
                    const priority = priorities[Math.floor(Math.random() * priorities.length)];
                    const status = statuses[Math.floor(Math.random() * statuses.length)];
                    const projectId = projectIds[Math.floor(Math.random() * projectIds.length)];
                    const categoryId = categoryIds[Math.floor(Math.random() * categoryIds.length)];
                    
                    const taskData = {
                        title: `[TEST] Sample Task ${i}`,
                        description: `This is a test task #${i} created by the dev tools for testing purposes. You can safely delete this.`,
                        project_id: projectId,
                        category_id: categoryId,
                        priority: priority,
                        status: status,
                        due_date: new Date(Date.now() + (Math.random() * 30) * 24 * 60 * 60 * 1000).toISOString().split('T')[0]
                    };

                    await this.app.api.createTask(taskData);
                    created++;
                    
                    // Update progress every 10 tasks
                    if (created % 10 === 0) {
                        logger.debug(`Created ${created}/${count} test tasks...`);
                    }
                } catch (error) {
                    logger.error(`Failed to create test task ${i}:`, error);
                    failed++;
                }
            }

            const duration = ((Date.now() - startTime) / 1000).toFixed(2);
            logger.info(`Test task creation complete: ${created} created, ${failed} failed in ${duration}s`);
            
            // Reload tasks to show new test data
            await this.reloadData();
            
            alert(`✅ Created ${created} test tasks successfully!\n${failed > 0 ? `\n⚠️ ${failed} failed` : ''}\n⏱️ Time: ${duration}s`);
            
        } catch (error) {
            logger.error('Failed to create test tasks:', error);
            alert(`❌ Failed to create test tasks.\n\n✅ Created: ${created}\n❌ Failed: ${failed}`);
        }
    }

    async cleanupTestTasks() {
        try {
            // Find all test tasks
            const tasks = await this.app.api.getTasks();
            const testTasks = tasks.data?.filter(t => t.title?.startsWith('[TEST]')) || [];
            
            // Find test project
            const projects = await this.app.api.getProjects();
            const testProject = projects.data?.find(p => p.name === '[TEST] Project');
            
            if (testTasks.length === 0 && !testProject) {
                alert('⚠️ No test data found to cleanup.');
                return;
            }

            const message = testProject 
                ? `Delete ${testTasks.length} test tasks and [TEST] Project?\n\nThis cannot be undone.`
                : `Delete ${testTasks.length} test tasks?\n\nThis cannot be undone.`;

            if (!confirm(message)) {
                return;
            }

            logger.info(`Cleaning up ${testTasks.length} test tasks${testProject ? ' and [TEST] Project' : ''}...`);
            const startTime = Date.now();
            let deleted = 0;
            let failed = 0;

            // Delete test tasks
            for (const task of testTasks) {
                try {
                    await fetch(`${this.app.api.baseUrl}tasks/${task.id}`, {
                        method: 'DELETE',
                        credentials: 'include'
                    });
                    deleted++;
                    
                    // Update progress every 10 deletions
                    if (deleted % 10 === 0) {
                        logger.debug(`Deleted ${deleted}/${testTasks.length} test tasks...`);
                    }
                } catch (error) {
                    logger.error(`Failed to delete test task ${task.id}:`, error);
                    failed++;
                }
            }

            // Delete test project if it exists
            if (testProject) {
                try {
                    await fetch(`${this.app.api.baseUrl}projects/${testProject.id}`, {
                        method: 'DELETE',
                        credentials: 'include'
                    });
                    logger.info(`Deleted [TEST] Project (ID: ${testProject.id})`);
                } catch (error) {
                    logger.error('Failed to delete [TEST] Project:', error);
                }
            }

            const duration = ((Date.now() - startTime) / 1000).toFixed(2);
            logger.info(`Test data cleanup complete: ${deleted} tasks deleted, ${failed} failed in ${duration}s`);
            
            // Reload data to reflect deletions (ignore errors from subtasks loading)
            try {
                await this.reloadData();
            } catch (reloadError) {
                logger.warn('Error reloading data after cleanup, but test data was deleted:', reloadError);
            }
            
            const resultMessage = testProject
                ? `✅ Deleted ${deleted} test tasks and [TEST] Project successfully!\n${failed > 0 ? `\n⚠️ ${failed} failed` : ''}\n⏱️ Time: ${duration}s`
                : `✅ Deleted ${deleted} test tasks successfully!\n${failed > 0 ? `\n⚠️ ${failed} failed` : ''}\n⏱️ Time: ${duration}s`;
            
            alert(resultMessage);
            
        } catch (error) {
            logger.error('Failed to cleanup test tasks:', error);
            alert('❌ Failed to cleanup test tasks. Check console for details.');
        }
    }

    // =========================================================================
    // TAB SYSTEM
    // =========================================================================

    switchTab(tabName) {
        // Update tab buttons
        document.querySelectorAll('.dev-tab').forEach(tab => {
            if (tab.dataset.tab === tabName) {
                tab.classList.add('active');
            } else {
                tab.classList.remove('active');
            }
        });

        // Update tab contents
        document.querySelectorAll('.dev-tab-content').forEach(content => {
            content.style.display = 'none';
        });

        const activeContent = document.getElementById(`dev-tab-${tabName}`);
        if (activeContent) {
            activeContent.style.display = 'block';
        }
        
        // Save active tab preference
        localStorage.setItem('devToolsActiveTab', tabName);
    }

    // =========================================================================
    // ACHIEVEMENT METHODS
    // =========================================================================

    async adjustAchievementStat(statKey, amount) {
        try {
            // Special handling for streak_days - need to update user_streaks table
            if (statKey === 'streak_days') {
                const currentStreak = this.app.achievements?.streak?.current || 0;
                const newStreak = Math.max(0, currentStreak + amount);
                
                // Update user_streaks table via API
                await this.app.api.post('achievements/streak/set', {
                    current_streak: newStreak
                });
                
                logger.info(`Adjusted streak_days to ${newStreak}`);
                alert(`✅ Adjusted streak_days to ${newStreak} days`);
                
                // Reload achievements to reflect new streak
                if (this.app.achievements) {
                    await this.app.achievements.init();
                }
                
                // Reload current view if on achievements page
                if (this.app.state.get('currentView') === 'achievements') {
                    await this.app.ui.renderView('achievements');
                }
                
                this.updateStats();
                return;
            }
            
            // Find all achievements that track this stat type
            const relatedAchievements = this.app.achievements?.achievements.filter(a => 
                a.criteria?.type === statKey
            ) || [];
            
            if (relatedAchievements.length === 0) {
                logger.warn(`No achievements found for stat: ${statKey}`);
                alert(`⚠️ No achievements track ${statKey}`);
                return;
            }
            
            // Update progress for all related achievements by the amount
            for (const achievement of relatedAchievements) {
                const currentProgress = achievement.current_progress || 0;
                const newProgress = Math.max(0, currentProgress + amount);
                const target = achievement.criteria?.target || 1;
                const achievementKey = achievement.criteria?.type;
                
                if (!achievementKey) continue;
                
                await this.app.api.post('achievements/update', {
                    achievement_key: achievementKey,
                    progress: newProgress,
                    target: target,
                    increment: false
                });
            }
            
            logger.info(`Adjusted ${statKey} by ${amount} for ${relatedAchievements.length} achievements`);
            alert(`✅ Adjusted ${statKey} by ${amount > 0 ? '+' : ''}${amount}`);
            
            // Recheck achievements and reload
            if (this.app.achievements) {
                await this.app.achievements.init();
            }
            
            this.updateStats();
        } catch (error) {
            logger.error(`Failed to adjust stat ${statKey}:`, error);
            alert(`❌ Failed to adjust ${statKey}`);
        }
    }

    async setAchievementStat(statKey) {
        const inputId = `stat-${statKey.replace(/_/g, '-')}`;
        const input = document.getElementById(inputId);
        
        if (!input || !input.value) {
            alert('Please enter a value');
            return;
        }
        
        const value = parseInt(input.value);
        if (isNaN(value) || value < 0) {
            alert('Please enter a valid positive number');
            return;
        }
        
        try {
            // Map stat keys to achievement codes
            // Find all achievements that track this stat type
            const relatedAchievements = this.app.achievements?.achievements.filter(a => 
                a.criteria?.type === statKey
            ) || [];
            
            // Set progress for all related achievements to the value
            for (const achievement of relatedAchievements) {
                const target = achievement.criteria?.target || 1;
                const achievementKey = achievement.criteria?.type;
                
                if (!achievementKey) continue;
                
                await this.app.api.post('achievements/update', {
                    achievement_key: achievementKey,
                    progress: value,
                    target: target,
                    increment: false
                });
            }
            
            logger.info(`Set ${statKey} to ${value} for ${relatedAchievements.length} achievements`);
            alert(`✅ Set ${statKey} to ${value}`);
            
            // Recheck achievements and reload
            if (this.app.achievements) {
                await this.app.achievements.init();
            }
            
            input.value = '';
            this.updateStats();
        } catch (error) {
            logger.error(`Failed to set stat ${statKey}:`, error);
            alert(`❌ Failed to set ${statKey}`);
        }
    }

    async recheckAllAchievements() {
        if (!this.app.achievements) {
            alert('Achievement system not initialized');
            return;
        }
        
        try {
            logger.info('Rechecking all achievements...');
            await this.app.achievements.checkExistingAchievements();
            alert('✅ All achievements rechecked!');
        } catch (error) {
            logger.error('Failed to recheck achievements:', error);
            alert('❌ Failed to recheck achievements');
        }
    }

    viewAchievements() {
        if (!this.app.achievements) {
            alert('Achievement system not initialized');
            return;
        }
        
        logger.debug('=== ALL ACHIEVEMENTS ===');
        logger.debug(this.app.achievements.achievements);
        console.table(this.app.achievements.achievements.map(a => ({
            ID: a.id,
            Name: a.name,
            Category: a.category,
            XP: a.xp_reward,
            Unlocked: a.is_unlocked ? '✓' : '✗'
        })));
        
        alert('Achievement list logged to console (F12)');
    }

    async resetAchievements() {
        if (!confirm('⚠️ Reset ALL achievement progress?\n\nThis will:\n- Remove all unlocked achievements\n- Reset all progress counters\n- Cannot be undone\n\nContinue?')) {
            return;
        }
        
        try {
            await this.app.api.post('achievements/reset-all');
            logger.info('All achievement progress reset');
            
            if (this.app.achievements) {
                await this.app.achievements.init();
            }
            
            alert('✅ All achievement progress has been reset!');
            this.updateStats();
        } catch (error) {
            logger.error('Failed to reset achievements:', error);
            alert('❌ Failed to reset achievements');
        }
    }

    async unlockAchievement() {
        const input = document.getElementById('achievement-id-input');
        const achievementId = parseInt(input?.value);
        
        if (!achievementId || achievementId < 1 || achievementId > 99) {
            alert('Please enter a valid Achievement ID (1-99)');
            return;
        }
        
        try {
            const achievement = this.app.achievements?.achievements.find(a => a.id === achievementId);
            if (!achievement) {
                alert(`Achievement ID ${achievementId} not found`);
                return;
            }
            
            // Get target and achievement key from criteria object
            const target = achievement.criteria?.target || 1;
            const achievementKey = achievement.criteria?.type;
            
            if (!achievementKey) {
                alert('❌ Achievement has no criteria type');
                return;
            }
            
            const payload = {
                achievement_key: achievementKey,
                progress: target,
                target: target,
                increment: false
            };
            
            console.log('🔍 DevTools unlock payload:', payload);
            console.log('🔍 Achievement object:', achievement);
            
            await this.app.api.post('achievements/update', payload);
            
            logger.info(`Unlocked achievement: ${achievement.name} (ID: ${achievementId})`);
            this.app.modals.showToast(`✅ Unlocked: ${achievement.name}`, 'success');
            
            if (this.app.achievements) {
                await this.app.achievements.init();
            }
            
            // Reload achievements view if currently viewing
            if (this.app.state.get('currentView') === 'achievements') {
                await this.app.ui.renderView('achievements');
            }
        } catch (error) {
            logger.error('Failed to unlock achievement:', error);
            alert('❌ Failed to unlock achievement');
        }
    }

    async lockAchievement() {
        const input = document.getElementById('achievement-id-input');
        const achievementId = parseInt(input?.value);
        
        if (!achievementId || achievementId < 1 || achievementId > 99) {
            alert('Please enter a valid Achievement ID (1-99)');
            return;
        }
        
        try {
            const achievement = this.app.achievements?.achievements.find(a => a.id === achievementId);
            if (!achievement) {
                alert(`Achievement ID ${achievementId} not found`);
                return;
            }
            
            // Get target and achievement key from criteria object
            const target = achievement.criteria?.target || 1;
            const achievementKey = achievement.criteria?.type;
            
            if (!achievementKey) {
                alert('❌ Achievement has no criteria type');
                return;
            }
            
            // Reset progress to 0
            await this.app.api.post('achievements/update', {
                achievement_key: achievementKey,
                progress: 0,
                target: target,
                increment: false
            });
            
            logger.info(`Locked achievement: ${achievement.name} (ID: ${achievementId})`);
            alert(`🔒 Locked: ${achievement.name}`);
            
            if (this.app.achievements) {
                await this.app.achievements.init();
            }
        } catch (error) {
            logger.error('Failed to lock achievement:', error);
            alert('❌ Failed to lock achievement');
        }
    }

    async unlockAdminSecret() {
        try {
            const adminAchievement = this.app.achievements?.achievements.find(a => a.id === 99);
            if (!adminAchievement) {
                alert('Admin secret achievement not found');
                return;
            }
            
            await this.app.api.post('achievements/update', {
                achievement_key: adminAchievement.code,
                progress: 1,
                target: 1,
                increment: false
            });
            
            logger.info('Unlocked admin secret achievement');
            alert('👑 Admin Secret Unlocked!');
            
            if (this.app.achievements) {
                await this.app.achievements.init();
            }
        } catch (error) {
            logger.error('Failed to unlock admin secret:', error);
            alert('❌ Failed to unlock admin secret');
        }
    }

    async lockAdminSecret() {
        try {
            const adminAchievement = this.app.achievements?.achievements.find(a => a.id === 99);
            if (!adminAchievement) {
                alert('Admin secret achievement not found');
                return;
            }
            
            const achievementKey = adminAchievement.criteria?.type || 'admin_secret';
            
            await this.app.api.post('achievements/update', {
                achievement_key: achievementKey,
                progress: 0,
                target: 1,
                increment: false
            });
            
            logger.info('Locked admin secret achievement');
            alert('🔒 Admin Secret Locked');
            
            if (this.app.achievements) {
                await this.app.achievements.init();
            }
        } catch (error) {
            logger.error('Failed to lock admin secret:', error);
            alert('❌ Failed to lock admin secret');
        }
    }

    // =========================================================================
    // TESTING METHODS
    // =========================================================================

    async generateTestTasks() {
        const count = parseInt(document.getElementById('gen-task-count')?.value) || 10;
        const descWords = parseInt(document.getElementById('gen-task-desc-words')?.value) || 20;
        const projectSel = document.getElementById('gen-task-project')?.value || 'random';
        const categorySel = document.getElementById('gen-task-category')?.value || 'random';
        const priority = document.getElementById('gen-task-priority')?.value || 'random';
        const status = document.getElementById('gen-task-status')?.value || 'pending';
        
        if (count > 100) {
            alert('Maximum 100 tasks at once');
            return;
        }
        
        try {
            logger.info(`Generating ${count} test tasks...`);
            
            // Get projects and categories for random selection
            const projects = this.app.state.get('projects') || [];
            const categories = this.app.state.get('categories') || [];
            
            // Find or create [TEST] Project if needed
            let testProject = projects.find(p => p.name === '[TEST] Project');
            if (projectSel === 'test' && !testProject) {
                const result = await this.app.api.createProject({
                    name: '[TEST] Project',
                    description: 'Auto-generated test project',
                    color: '#2563eb'
                });
                testProject = result.data;
                logger.info('Created [TEST] Project');
            }
            
            let created = 0;
            let failed = 0;
            
            for (let i = 0; i < count; i++) {
                const taskPriority = priority === 'random' 
                    ? ['high', 'medium', 'low'][Math.floor(Math.random() * 3)]
                    : priority;
                
                const taskStatus = status === 'random'
                    ? ['pending', 'in-progress', 'completed'][Math.floor(Math.random() * 3)]
                    : status;
                
                const description = descWords > 0 ? this.generateLoremIpsum(descWords) : null;
                
                // Determine project_id
                let project_id = null;
                if (projectSel === 'test' && testProject) {
                    project_id = testProject.id;
                } else if (projectSel === 'random' && projects.length > 0) {
                    project_id = projects[Math.floor(Math.random() * projects.length)].id;
                }
                
                // Determine category_id
                let category_id = null;
                if (categorySel === 'random' && categories.length > 0) {
                    category_id = categories[Math.floor(Math.random() * categories.length)].id;
                } else if (categorySel !== 'random') {
                    category_id = parseInt(categorySel);
                }
                
                try {
                    await this.app.api.createTask({
                        title: `[TEST] Generated Task ${i + 1}`,
                        description: description,
                        priority: taskPriority,
                        status: taskStatus,
                        project_id: project_id,
                        category_id: category_id
                    });
                    created++;
                } catch (error) {
                    logger.error(`Failed to create task ${i + 1}:`, error);
                    failed++;
                }
            }
            
            await this.reloadData();
            alert(`✅ Generated ${created} test tasks!\n${failed > 0 ? `⚠️ ${failed} failed` : ''}`);
            
        } catch (error) {
            logger.error('Failed to generate test tasks:', error);
            alert('❌ Failed to generate test tasks');
        }
    }

    async generateTestProjects() {
        const count = parseInt(document.getElementById('gen-project-count')?.value) || 5;
        
        if (count > 20) {
            alert('Maximum 20 projects at once');
            return;
        }
        
        try {
            logger.info(`Generating ${count} test projects...`);
            let created = 0;
            let failed = 0;
            
            const colors = ['#2563eb', '#7c3aed', '#db2777', '#dc2626', '#ea580c', '#16a34a', '#0891b2'];
            
            for (let i = 0; i < count; i++) {
                const color = colors[Math.floor(Math.random() * colors.length)];
                
                try {
                    await this.app.api.createProject({
                        name: `[TEST] Project ${i + 1}`,
                        description: this.generateLoremIpsum(15),
                        color: color
                    });
                    created++;
                } catch (error) {
                    logger.error(`Failed to create project ${i + 1}:`, error);
                    failed++;
                }
            }
            
            await this.reloadData();
            alert(`✅ Generated ${created} test projects!\n${failed > 0 ? `⚠️ ${failed} failed` : ''}`);
            
        } catch (error) {
            logger.error('Failed to generate test projects:', error);
            alert('❌ Failed to generate test projects');
        }
    }

    async cleanupTestProjects() {
        try {
            const projects = await this.app.api.getProjects();
            const testProjects = projects.data?.filter(p => p.name?.startsWith('[TEST]')) || [];
            
            if (testProjects.length === 0) {
                alert('⚠️ No test projects found to cleanup.');
                return;
            }

            if (!confirm(`Delete ${testProjects.length} test projects?\n\nThis cannot be undone.`)) {
                return;
            }

            logger.info(`Cleaning up ${testProjects.length} test projects...`);
            let deleted = 0;
            let failed = 0;

            for (const project of testProjects) {
                try {
                    await fetch(`${this.app.api.baseUrl}projects/${project.id}`, {
                        method: 'DELETE',
                        credentials: 'include'
                    });
                    deleted++;
                } catch (error) {
                    logger.error(`Failed to delete test project ${project.id}:`, error);
                    failed++;
                }
            }

            await this.reloadData();
            alert(`✅ Deleted ${deleted} test projects!\n${failed > 0 ? `⚠️ ${failed} failed` : ''}`);
            
        } catch (error) {
            logger.error('Failed to cleanup test projects:', error);
            alert('❌ Failed to cleanup test projects');
        }
    }

    async cleanupAllTestData() {
        try {
            const tasks = await this.app.api.getTasks();
            const projects = await this.app.api.getProjects();
            
            const testTasks = tasks.data?.filter(t => t.title?.startsWith('[TEST]')) || [];
            const testProjects = projects.data?.filter(p => p.name?.startsWith('[TEST]')) || [];
            
            if (testTasks.length === 0 && testProjects.length === 0) {
                alert('⚠️ No test data found to cleanup.');
                return;
            }

            if (!confirm(`Delete ALL test data?\n\n• ${testTasks.length} test tasks\n• ${testProjects.length} test projects\n\nThis cannot be undone!`)) {
                return;
            }

            logger.info(`Cleaning up all test data: ${testTasks.length} tasks, ${testProjects.length} projects...`);
            const startTime = Date.now();
            let deletedTasks = 0;
            let deletedProjects = 0;
            let failed = 0;

            // Delete test tasks
            for (const task of testTasks) {
                try {
                    await fetch(`${this.app.api.baseUrl}tasks/${task.id}`, {
                        method: 'DELETE',
                        credentials: 'include'
                    });
                    deletedTasks++;
                } catch (error) {
                    logger.error(`Failed to delete task ${task.id}:`, error);
                    failed++;
                }
            }

            // Delete test projects
            for (const project of testProjects) {
                try {
                    await fetch(`${this.app.api.baseUrl}projects/${project.id}`, {
                        method: 'DELETE',
                        credentials: 'include'
                    });
                    deletedProjects++;
                } catch (error) {
                    logger.error(`Failed to delete project ${project.id}:`, error);
                    failed++;
                }
            }

            const duration = ((Date.now() - startTime) / 1000).toFixed(2);
            
            await this.reloadData();
            alert(`✅ Cleanup complete!\n\n• ${deletedTasks} tasks deleted\n• ${deletedProjects} projects deleted${failed > 0 ? `\n⚠️ ${failed} failed` : ''}\n⏱️ Time: ${duration}s`);
            
        } catch (error) {
            logger.error('Failed to cleanup all test data:', error);
            alert('❌ Failed to cleanup test data');
        }
    }

    generateLoremIpsum(wordCount) {
        const words = [
            'lorem', 'ipsum', 'dolor', 'sit', 'amet', 'consectetur', 'adipiscing', 'elit',
            'sed', 'do', 'eiusmod', 'tempor', 'incididunt', 'ut', 'labore', 'et', 'dolore',
            'magna', 'aliqua', 'enim', 'ad', 'minim', 'veniam', 'quis', 'nostrud', 'exercitation',
            'ullamco', 'laboris', 'nisi', 'aliquip', 'ex', 'ea', 'commodo', 'consequat'
        ];
        
        let result = [];
        for (let i = 0; i < wordCount; i++) {
            result.push(words[Math.floor(Math.random() * words.length)]);
        }
        
        return result.join(' ').charAt(0).toUpperCase() + result.join(' ').slice(1) + '.';
    }

    updateDebugInfo() {
        const debugDiv = document.getElementById('dev-debug-info');
        if (!debugDiv) return;
        
        const stats = {
            'Total Tasks': this.app.state.get('tasks')?.length || 0,
            'Total Projects': this.app.state.get('projects')?.length || 0,
            'Total Achievements': this.app.achievements?.achievements?.length || 0,
            'Unlocked Achievements': this.app.achievements?.achievements?.filter(a => a.is_unlocked)?.length || 0,
            'Total XP': this.app.state.get('user')?.xp_total || 0,
            'Current Level': this.app.achievements?.currentLevel || 1,
            'Achievement Multiplier': this.app.achievements?.achievementMultiplier?.toFixed(2) || '1.00',
            'Current Streak': this.app.achievements?.streak?.current || 0,
            'Longest Streak': this.app.achievements?.streak?.longest || 0
        };
        
        debugDiv.innerHTML = Object.entries(stats)
            .map(([key, value]) => `<div>${key}: <strong>${value}</strong></div>`)
            .join('');
    }
}

// =============================================================================
// SHARING MANAGER
// =============================================================================

class SharingManager {
    constructor(api, state) {
        this.api = api;
        this.state = state;
        this.currentProjectId = null;
        this.searchTimeout = null;
        this.initializeEventListeners();
    }

    initializeEventListeners() {
        // Share tabs
        document.querySelectorAll('.share-tab').forEach(tab => {
            tab.addEventListener('click', (e) => {
                const tabName = e.currentTarget.dataset.tab;
                this.switchShareTab(tabName);
            });
        });

        // User search
        const searchInput = document.getElementById('user-search-input');
        if (searchInput) {
            searchInput.addEventListener('input', (e) => {
                clearTimeout(this.searchTimeout);
                this.searchTimeout = setTimeout(() => {
                    this.searchUsers(e.target.value);
                }, 300);
            });
        }

        // Generate public link
        const generateBtn = document.getElementById('generate-link-btn');
        if (generateBtn) {
            generateBtn.addEventListener('click', () => this.generatePublicLink());
        }
    }

    switchShareTab(tabName) {
        // Update tab buttons
        document.querySelectorAll('.share-tab').forEach(tab => {
            tab.classList.toggle('active', tab.dataset.tab === tabName);
        });

        // Update tab content
        document.querySelectorAll('.share-tab-content').forEach(content => {
            const isActive = content.id === `${tabName}-tab`;
            content.classList.toggle('active', isActive);
            content.style.display = isActive ? 'block' : 'none';
        });
    }

    async openShareModal(projectId) {
        this.currentProjectId = projectId;
        const project = this.state.get('projects').find(p => p.id === projectId);
        
        if (!project) {
            app.modals.showToast('Project not found', 'error');
            return;
        }

        // Set project name
        document.getElementById('share-project-name').textContent = project.name;

        // Check if user is a guest (in share mode)
        const isGuestUser = document.body.dataset.shareMode === '1';
        
        if (!isGuestUser) {
            // Only load shares for authenticated users
            await this.loadProjectShares();
        }

        // Open modal
        app.modals.showModal('share-modal');
    }

    async loadProjectShares() {
        try {
            const response = await fetch(`/app/tasklist/api/sharing.php?action=project_shares&project_id=${this.currentProjectId}`);
            const data = await response.json();

            if (!response.ok) {
                throw new Error(data.error || 'Failed to load shares');
            }

            this.renderCollaborators(data.user_shares);
            this.renderPublicLinks(data.public_links);
        } catch (error) {
            logger.error('Error loading project shares:', error);
            app.modals.showToast('Failed to load shares', 'error');
        }
    }

    renderCollaborators(collaborators) {
        const container = document.getElementById('collaborators-list');
        const emptyState = document.getElementById('collaborators-empty');

        if (!collaborators || collaborators.length === 0) {
            emptyState.style.display = 'block';
            container.querySelectorAll('.collaborator-item').forEach(item => item.remove());
            return;
        }

        emptyState.style.display = 'none';

        const html = collaborators.map(collab => `
            <div class="collaborator-item ${!collab.is_accepted ? 'collaborator-pending' : ''}" data-share-id="${collab.id}">
                <div class="collaborator-info">
                    <div class="collaborator-avatar">
                        ${this.getInitials(collab.full_name || collab.username)}
                    </div>
                    <div class="collaborator-details">
                        <div class="collaborator-name">${collab.full_name || collab.username}</div>
                        <div class="collaborator-email">${collab.email}</div>
                    </div>
                </div>
                <div class="collaborator-controls">
                    ${!collab.is_accepted ? `
                        <span class="pending-badge">Pending</span>
                    ` : `
                        <select class="permission-select" data-share-id="${collab.id}">
                            <option value="view" ${collab.permission_level === 'view' ? 'selected' : ''}>View Only</option>
                            <option value="edit" ${collab.permission_level === 'edit' ? 'selected' : ''}>Can Edit</option>
                            <option value="manage" ${collab.permission_level === 'manage' ? 'selected' : ''}>Can Manage</option>
                        </select>
                    `}
                    <button class="btn btn-icon btn-danger" onclick="app.sharing.revokeShare(${collab.id})" title="Remove access">
                        <i class="fas fa-times"></i>
                    </button>
                </div>
            </div>
        `).join('');

        // Remove old items and add new
        container.querySelectorAll('.collaborator-item').forEach(item => item.remove());
        container.insertAdjacentHTML('beforeend', html);

        // Add event listeners to permission selects
        container.querySelectorAll('.permission-select').forEach(select => {
            select.addEventListener('change', (e) => {
                const shareId = parseInt(e.target.dataset.shareId);
                const newPermission = e.target.value;
                this.updatePermission(shareId, newPermission);
            });
        });
    }

    renderPublicLinks(links) {
        const container = document.getElementById('public-links-list');
        const emptyState = document.getElementById('public-links-empty');

        // Update link count badge
        const linkCountBadge = document.getElementById('link-count-badge');
        if (linkCountBadge) {
            linkCountBadge.textContent = links ? links.length : 0;
        }

        if (!links || links.length === 0) {
            emptyState.style.display = 'block';
            container.querySelectorAll('.public-link-item').forEach(item => item.remove());
            return;
        }

        emptyState.style.display = 'none';

        const html = links.map(link => {
            const isExpired = link.expires_at && new Date(link.expires_at) < new Date();
            const fullUrl = `${window.location.origin}/app/tasklist/shared.php?token=${link.token}`;
            const oppositePermission = link.permission_level === 'view' ? 'edit' : 'view';
            const linkName = link.name || 'Unnamed Link';
            
            return `
                <div class="public-link-item" data-token-id="${link.id}">
                    <div class="link-header">
                        <div class="link-meta">
                            <h5 style="margin: 0; font-size: 1rem; font-weight: 600; color: var(--text-primary);">
                                ${this.escapeHtml(linkName)}
                            </h5>
                        </div>
                        <div class="link-meta">
                            <span class="permission-badge ${link.permission_level}">
                                <i class="fas fa-${link.permission_level === 'view' ? 'eye' : 'edit'}"></i>
                                ${link.permission_level === 'view' ? 'View Only' : 'Can Edit'}
                            </span>
                            <span class="link-stats">
                                <i class="fas fa-eye"></i> ${link.access_count || 0}
                            </span>
                            ${link.last_accessed_at ? `
                                <span class="link-stats">
                                    <i class="fas fa-history"></i> ${this.formatDate(link.last_accessed_at)}
                                </span>
                            ` : ''}
                        </div>
                        <div class="link-actions">
                            <button class="btn btn-icon btn-edit-link" 
                                    onclick="app.sharing.editLink(${link.id})" 
                                    title="Edit link">
                                <i class="fas fa-edit"></i>
                            </button>
                            <button class="btn btn-icon btn-toggle-permission" 
                                    onclick="app.sharing.toggleLinkPermission(${link.id}, '${oppositePermission}')" 
                                    title="Change to ${oppositePermission === 'view' ? 'View Only' : 'Can Edit'}">
                                <i class="fas fa-${oppositePermission === 'view' ? 'eye' : 'edit'}"></i>
                            </button>
                            <button class="btn btn-icon btn-revoke-link" 
                                    onclick="app.sharing.revokePublicLink(${link.id})" 
                                    title="Revoke link">
                                <i class="fas fa-trash"></i>
                            </button>
                        </div>
                    </div>
                    <div class="link-url-container">
                        <input type="text" class="link-url" value="${fullUrl}" readonly>
                        <button class="btn btn-sm btn-primary" onclick="app.sharing.copyLink(this)" title="Copy link">
                            <i class="fas fa-copy"></i> Copy
                        </button>
                    </div>
                    <div class="link-footer">
                        ${link.expires_at ? `
                            <div class="link-expiry ${isExpired ? 'expired' : ''}">
                                <i class="fas fa-clock"></i>
                                ${isExpired ? 'Expired' : 'Expires'} ${this.formatDate(link.expires_at)}
                            </div>
                        ` : `
                            <div class="link-expiry">
                                <i class="fas fa-infinity"></i> Never expires
                            </div>
                        `}
                        <div class="link-created">
                            <i class="fas fa-calendar-plus"></i>
                            Created ${this.formatDate(link.created_at)}
                        </div>
                    </div>
                </div>
            `;
        }).join('');

        // Remove old items and add new (insert before empty state)
        container.querySelectorAll('.public-link-item').forEach(item => item.remove());
        emptyState.insertAdjacentHTML('beforebegin', html);
    }

    async searchUsers(query) {
        if (!query || query.length < 2) {
            document.getElementById('user-search-results').style.display = 'none';
            return;
        }

        try {
            const response = await fetch(`/app/tasklist/api/sharing.php?action=search_users&q=${encodeURIComponent(query)}&project_id=${this.currentProjectId}`);
            const users = await response.json();

            const resultsContainer = document.getElementById('user-search-results');
            
            if (!users || users.length === 0) {
                resultsContainer.innerHTML = '<div class="user-search-result" style="justify-content: center; color: var(--text-tertiary);">No users found</div>';
                resultsContainer.style.display = 'block';
                return;
            }

            const html = users.map(user => `
                <div class="user-search-result" data-user-id="${user.id}">
                    <div class="user-info">
                        <div class="user-info-name">${user.full_name || user.username}</div>
                        <div class="user-info-email">${user.email}</div>
                    </div>
                    <button class="btn btn-primary btn-sm" onclick="app.sharing.inviteUser(${user.id}, '${(user.full_name || user.username).replace(/'/g, "\\'")}')">
                        <i class="fas fa-user-plus"></i> Invite
                    </button>
                </div>
            `).join('');

            resultsContainer.innerHTML = html;
            resultsContainer.style.display = 'block';
        } catch (error) {
            logger.error('Error searching users:', error);
        }
    }

    async inviteUser(userId, userName) {
        try {
            const response = await fetch('/app/tasklist/api/sharing.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    action: 'share_with_user',
                    project_id: this.currentProjectId,
                    user_id: userId,
                    permission_level: 'view'
                })
            });

            const result = await response.json();

            if (!response.ok) {
                throw new Error(result.error || 'Failed to share project');
            }

            app.modals.showToast(`Invited ${userName} to project`, 'success');
            
            // Clear search and reload
            document.getElementById('user-search-input').value = '';
            document.getElementById('user-search-results').style.display = 'none';
            await this.loadProjectShares();
        } catch (error) {
            logger.error('Error inviting user:', error);
            app.modals.showToast(error.message || 'Failed to invite user', 'error');
        }
    }

    async updatePermission(shareId, newPermission) {
        try {
            const response = await fetch('/app/tasklist/api/sharing.php', {
                method: 'PUT',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    share_id: shareId,
                    permission_level: newPermission
                })
            });

            const result = await response.json();

            if (!response.ok) {
                throw new Error(result.error || 'Failed to update permission');
            }

            app.modals.showToast('Permission updated', 'success');
        } catch (error) {
            logger.error('Error updating permission:', error);
            app.modals.showToast(error.message || 'Failed to update permission', 'error');
            // Reload to reset select
            await this.loadProjectShares();
        }
    }

    async revokeShare(shareId) {
        if (!confirm('Are you sure you want to remove this collaborator?')) {
            return;
        }

        try {
            const response = await fetch(`/app/tasklist/api/sharing.php?share_id=${shareId}`, {
                method: 'DELETE'
            });

            const result = await response.json();

            if (!response.ok) {
                throw new Error(result.error || 'Failed to revoke access');
            }

            app.modals.showToast('Access revoked', 'success');
            await this.loadProjectShares();
        } catch (error) {
            logger.error('Error revoking share:', error);
            app.modals.showToast(error.message || 'Failed to revoke access', 'error');
        }
    }

    async generatePublicLink() {
        const name = document.getElementById('link-name').value.trim();
        const permission = document.getElementById('link-permission').value;
        const expiration = document.getElementById('link-expiration').value;

        try {
            const response = await fetch('/app/tasklist/api/sharing.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    action: 'create_public_link',
                    project_id: this.currentProjectId,
                    name: name || null,
                    permission_level: permission,
                    expires_days: expiration || null
                })
            });

            const result = await response.json();

            if (!response.ok) {
                throw new Error(result.error || 'Failed to generate link');
            }

            app.modals.showToast('Public link created', 'success');
            
            // Clear the name input
            document.getElementById('link-name').value = '';
            
            await this.loadProjectShares();
            
            // Switch to public links tab
            this.switchShareTab('public-links');
        } catch (error) {
            logger.error('Error generating link:', error);
            app.modals.showToast(error.message || 'Failed to generate link', 'error');
        }
    }

    async toggleLinkPermission(tokenId, newPermission) {
        try {
            const response = await fetch('/app/tasklist/api/sharing.php', {
                method: 'PUT',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    action: 'update_token_permission',
                    token_id: tokenId,
                    permission_level: newPermission
                })
            });

            const result = await response.json();

            if (!response.ok) {
                throw new Error(result.error || 'Failed to update link permission');
            }

            app.modals.showToast(`Link updated to ${newPermission === 'view' ? 'View Only' : 'Can Edit'}`, 'success');
            await this.loadProjectShares();
        } catch (error) {
            logger.error('Error updating link permission:', error);
            app.modals.showToast(error.message || 'Failed to update permission', 'error');
        }
    }

    async editLink(tokenId) {
        // Find the link data
        const linkElement = document.querySelector(`[data-token-id="${tokenId}"]`);
        if (!linkElement) return;

        const currentNameRaw = linkElement.querySelector('h5').textContent.trim();
        const currentName = currentNameRaw === 'Unnamed Link' ? '' : currentNameRaw;
        
        // For now, just allow editing the name with a prompt
        const newName = prompt('Edit link name:', currentName);
        
        if (newName === null) return; // Cancelled
        
        try {
            const response = await fetch('/app/tasklist/api/sharing.php', {
                method: 'PUT',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    action: 'update_token_name',
                    token_id: tokenId,
                    name: newName.trim() || null
                })
            });

            const result = await response.json();

            if (!response.ok) {
                throw new Error(result.error || 'Failed to update link name');
            }

            app.modals.showToast('Link name updated', 'success');
            await this.loadProjectShares();
        } catch (error) {
            logger.error('Error updating link name:', error);
            app.modals.showToast(error.message || 'Failed to update link', 'error');
        }
    }

    async revokePublicLink(tokenId) {
        if (!confirm('Are you sure you want to revoke this public link? Anyone with the link will lose access.')) {
            return;
        }

        try {
            const response = await fetch(`/app/tasklist/api/sharing.php?token_id=${tokenId}`, {
                method: 'DELETE'
            });

            const result = await response.json();

            if (!response.ok) {
                throw new Error(result.error || 'Failed to revoke link');
            }

            app.modals.showToast('Public link revoked', 'success');
            await this.loadProjectShares();
        } catch (error) {
            logger.error('Error revoking link:', error);
            app.modals.showToast(error.message || 'Failed to revoke link', 'error');
        }
    }

    copyLink(button) {
        const input = button.parentElement.querySelector('.link-url');
        input.select();
        input.setSelectionRange(0, 99999); // For mobile
        
        try {
            navigator.clipboard.writeText(input.value).then(() => {
                const originalText = button.innerHTML;
                button.innerHTML = '<i class="fas fa-check"></i> Copied!';
                button.classList.add('btn-success');
                button.classList.remove('btn-primary');
                
                setTimeout(() => {
                    button.innerHTML = originalText;
                    button.classList.remove('btn-success');
                    button.classList.add('btn-primary');
                }, 2000);
            });
        } catch (err) {
            // Fallback for older browsers
            document.execCommand('copy');
            app.modals.showToast('Link copied to clipboard', 'success');
        }
    }

    getInitials(name) {
        if (!name) return '?';
        const parts = name.split(' ');
        if (parts.length >= 2) {
            return `${parts[0][0]}${parts[1][0]}`.toUpperCase();
        }
        return name.substring(0, 2).toUpperCase();
    }

    formatDate(dateString) {
        if (!dateString) return '';
        const date = new Date(dateString);
        const now = new Date();
        const diffMs = date - now;
        const diffDays = Math.ceil(diffMs / (1000 * 60 * 60 * 24));

        if (diffDays < 0) {
            return date.toLocaleDateString();
        } else if (diffDays === 0) {
            return 'today';
        } else if (diffDays === 1) {
            return 'tomorrow';
        } else if (diffDays < 7) {
            return `in ${diffDays} days`;
        } else {
            return date.toLocaleDateString();
        }
    }

    escapeHtml(text) {
        if (!text) return '';
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
}

// =============================================================================
// APPLICATION INITIALIZATION
// =============================================================================

// Global app instance
let app;

// Initialize when DOM is ready
document.addEventListener('DOMContentLoaded', async () => {
    // Load version data first
    await loadVersionData();
    
    // Update CONFIG with loaded version
    if (APP_VERSION) {
        CONFIG.VERSION = APP_VERSION.version;
    }
    
    // Initialize user menu dropdown immediately
    initUserMenu();
    
    // Initialize version badge click handler
    initVersionBadge();
    
    app = new TaskListApp();
    window.app = app; // Make available globally for debugging
    
    try {
        await app.init();
    } catch (error) {
        logger.error('Critical initialization error:', error);
        
        // Show fallback error UI
        document.body.innerHTML = `
            <div class="critical-error">
                <h1>Application Error</h1>
                <p>Failed to initialize Task Manager. Please check your connection and try again.</p>
                <button onclick="window.location.reload()">Reload Page</button>
            </div>
        `;
    }
});

// Version badge click handler
function initVersionBadge() {
    const versionBadge = document.getElementById('version-badge');
    
    if (!versionBadge) {
        console.error('Version badge element not found');
        return;
    }
    
    // Handle click to navigate to changelog
    const handleVersionClick = (e) => {
        e.preventDefault();
        e.stopPropagation();
        
        // Wait for app to be initialized
        if (window.app) {
            window.app.navigateToView('changelog');
        } else {
            console.warn('App not yet initialized');
        }
    };
    
    versionBadge.addEventListener('click', handleVersionClick);
    
    // Handle keyboard accessibility (Enter or Space)
    versionBadge.addEventListener('keydown', (e) => {
        if (e.key === 'Enter' || e.key === ' ') {
            handleVersionClick(e);
        }
    });
    
    // Add hover effect cursor
    versionBadge.style.cursor = 'pointer';
}

// User menu dropdown functionality
function initUserMenu() {
    const userProfile = document.getElementById('user-profile');
    const menuToggle = document.getElementById('user-menu-toggle');
    const dropdownMenu = document.getElementById('user-dropdown-menu');
    const settingsMenuItem = document.getElementById('settings-menu-item');
    const logoutMenuItem = document.getElementById('logout-menu-item');

    console.log('initUserMenu called', {
        userProfile: !!userProfile,
        menuToggle: !!menuToggle,
        dropdownMenu: !!dropdownMenu
    });

    if (!userProfile || !menuToggle || !dropdownMenu) {
        console.error('User menu elements not found');
        return;
    }

    // Toggle menu on click
    menuToggle.addEventListener('click', (e) => {
        e.preventDefault();
        e.stopPropagation();
        const isOpen = userProfile.classList.toggle('menu-open');
        console.log('Menu toggled:', isOpen);
    });

    // Also allow clicking the entire user profile area to toggle
    userProfile.addEventListener('click', (e) => {
        if (e.target.closest('.user-dropdown-menu')) {
            return; // Don't toggle if clicking inside the dropdown
        }
        e.stopPropagation();
    });

    // Close menu when clicking outside
    document.addEventListener('click', (e) => {
        if (!userProfile.contains(e.target)) {
            userProfile.classList.remove('menu-open');
        }
    });

    // Close menu when clicking a menu item
    dropdownMenu.querySelectorAll('.user-menu-item').forEach(item => {
        item.addEventListener('click', () => {
            userProfile.classList.remove('menu-open');
        });
    });
    
    // Keyboard shortcut: '/' focuses search
    document.addEventListener('keydown', (e) => {
        const globalSearch = document.getElementById('global-search');
        const isTyping = ['INPUT', 'TEXTAREA', 'SELECT'].includes(document.activeElement?.tagName);
        
        if (e.key === '/' && !isTyping && globalSearch) {
            e.preventDefault();
            globalSearch.focus();
        }
    });    // Handle settings click - open settings modal if available
    if (settingsMenuItem) {
        settingsMenuItem.addEventListener('click', (e) => {
            e.preventDefault();
            if (app && app.modals && app.modals.open) {
                app.modals.open('settings-modal');
            }
        });
    }

    // Handle logout click
    if (logoutMenuItem) {
        logoutMenuItem.addEventListener('click', async (e) => {
            e.preventDefault();
            
            try {
                const res = await fetch('/auth/manipulate_users.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    cache: 'no-store',
                    body: JSON.stringify({ method: 'LOGOUT' })
                });
                await res.json().catch(() => ({}));
                
                // Show toast notification
                if (app && app.modals && app.modals.showToast) {
                    app.modals.showToast('Logged out successfully! Redirecting...', 'success');
                }
                
                // Redirect after a short delay
                setTimeout(() => {
                    window.location.href = '/auth/login.php';
                }, 1500);
            } catch (error) {
                console.error('Logout error:', error);
                if (app && app.modals && app.modals.showToast) {
                    app.modals.showToast('Logout failed. Please try again.', 'error');
                }
            }
        });
    }
}