-- ============================================================================
-- PROJECT SHARING SCHEMA
-- ============================================================================
-- Author: Jack Ewers / BloodWeb
-- Date: November 18, 2025
-- Description: Database schema for project sharing and collaboration
-- ============================================================================

-- Table: project_shares
-- Purpose: Store sharing relationships between users and projects
CREATE TABLE IF NOT EXISTS `project_shares` (
    `id` INT(11) NOT NULL AUTO_INCREMENT,
    `project_id` INT(11) NOT NULL,
    `shared_by_user_id` INT(11) NOT NULL COMMENT 'User who created the share',
    `shared_with_user_id` INT(11) NOT NULL COMMENT 'User receiving access',
    `permission_level` ENUM('view', 'edit', 'manage') NOT NULL DEFAULT 'view' COMMENT 'Access level: view-only, edit tasks, manage collaborators',
    `is_accepted` TINYINT(1) NOT NULL DEFAULT 0 COMMENT 'Has recipient accepted the invitation',
    `created_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `accepted_at` TIMESTAMP NULL DEFAULT NULL,
    `updated_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    UNIQUE KEY `unique_share` (`project_id`, `shared_with_user_id`),
    KEY `idx_project` (`project_id`),
    KEY `idx_shared_with` (`shared_with_user_id`),
    KEY `idx_shared_by` (`shared_by_user_id`),
    CONSTRAINT `fk_share_project` FOREIGN KEY (`project_id`) REFERENCES `projects` (`id`) ON DELETE CASCADE,
    CONSTRAINT `fk_share_by_user` FOREIGN KEY (`shared_by_user_id`) REFERENCES `users` (`id`) ON DELETE CASCADE,
    CONSTRAINT `fk_share_with_user` FOREIGN KEY (`shared_with_user_id`) REFERENCES `users` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Table: share_tokens
-- Purpose: Manage public share links with token-based access
CREATE TABLE IF NOT EXISTS `share_tokens` (
    `id` INT(11) NOT NULL AUTO_INCREMENT,
    `project_id` INT(11) NOT NULL,
    `created_by_user_id` INT(11) NOT NULL,
    `token` VARCHAR(64) NOT NULL COMMENT 'Unique secure token for public access',
    `permission_level` ENUM('view', 'edit') NOT NULL DEFAULT 'view' COMMENT 'Public link permissions (no manage)',
    `is_active` TINYINT(1) NOT NULL DEFAULT 1,
    `expires_at` TIMESTAMP NULL DEFAULT NULL COMMENT 'Optional expiration date',
    `access_count` INT(11) NOT NULL DEFAULT 0 COMMENT 'Track number of accesses',
    `last_accessed_at` TIMESTAMP NULL DEFAULT NULL,
    `created_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `updated_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    UNIQUE KEY `unique_token` (`token`),
    KEY `idx_project` (`project_id`),
    KEY `idx_created_by` (`created_by_user_id`),
    KEY `idx_token_active` (`token`, `is_active`),
    CONSTRAINT `fk_token_project` FOREIGN KEY (`project_id`) REFERENCES `projects` (`id`) ON DELETE CASCADE,
    CONSTRAINT `fk_token_user` FOREIGN KEY (`created_by_user_id`) REFERENCES `users` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Table: share_activity_log
-- Purpose: Track sharing activities for audit and notifications
CREATE TABLE IF NOT EXISTS `share_activity_log` (
    `id` INT(11) NOT NULL AUTO_INCREMENT,
    `project_id` INT(11) NOT NULL,
    `user_id` INT(11) NOT NULL COMMENT 'User who performed the action',
    `action_type` ENUM('invited', 'accepted', 'revoked', 'permission_changed', 'link_created', 'link_revoked', 'accessed_via_link') NOT NULL,
    `target_user_id` INT(11) NULL COMMENT 'User affected by action (for invites)',
    `details` JSON NULL COMMENT 'Additional action details',
    `created_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    KEY `idx_project` (`project_id`),
    KEY `idx_user` (`user_id`),
    KEY `idx_target_user` (`target_user_id`),
    KEY `idx_created` (`created_at`),
    CONSTRAINT `fk_activity_project` FOREIGN KEY (`project_id`) REFERENCES `projects` (`id`) ON DELETE CASCADE,
    CONSTRAINT `fk_activity_user` FOREIGN KEY (`user_id`) REFERENCES `users` (`id`) ON DELETE CASCADE,
    CONSTRAINT `fk_activity_target_user` FOREIGN KEY (`target_user_id`) REFERENCES `users` (`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Add columns to projects table for sharing metadata (if they don't exist)
ALTER TABLE `projects` 
    ADD COLUMN IF NOT EXISTS `is_shared` TINYINT(1) NOT NULL DEFAULT 0 COMMENT 'Quick flag if project has any shares',
    ADD COLUMN IF NOT EXISTS `share_settings` JSON NULL COMMENT 'Sharing preferences (allow_public_links, require_approval, etc.)';

-- Add index if it doesn't exist
CREATE INDEX IF NOT EXISTS `idx_is_shared` ON `projects` (`is_shared`, `user_id`);

-- Add columns to tasks table to track collaboration (if they don't exist)
ALTER TABLE `tasks` 
    ADD COLUMN IF NOT EXISTS `created_by_user_id` INT(11) NULL COMMENT 'Original task creator',
    ADD COLUMN IF NOT EXISTS `last_modified_by_user_id` INT(11) NULL COMMENT 'Last user to edit task';

-- Add indexes if they don't exist
CREATE INDEX IF NOT EXISTS `idx_created_by` ON `tasks` (`created_by_user_id`);
CREATE INDEX IF NOT EXISTS `idx_modified_by` ON `tasks` (`last_modified_by_user_id`);

-- ============================================================================
-- HELPER VIEWS
-- ============================================================================

-- View: user_shared_projects
-- Purpose: Easy access to all projects shared with a user
CREATE OR REPLACE VIEW `user_shared_projects` AS
SELECT 
    ps.id AS share_id,
    p.id AS project_id,
    p.name AS project_name,
    p.color AS project_color,
    p.icon AS project_icon,
    p.is_completed AS is_completed,
    ps.permission_level,
    ps.is_accepted,
    owner.id AS owner_id,
    owner.username AS owner_username,
    owner.email AS owner_email,
    ps.shared_with_user_id AS user_id,
    ps.created_at AS shared_at,
    ps.accepted_at,
    (SELECT COUNT(*) FROM tasks WHERE project_id = p.id) AS task_count,
    (SELECT COUNT(*) FROM tasks WHERE project_id = p.id AND is_completed = 1) AS completed_task_count
FROM project_shares ps
INNER JOIN projects p ON ps.project_id = p.id
INNER JOIN users owner ON p.user_id = owner.id
WHERE ps.is_accepted = 1;

-- View: project_collaborators
-- Purpose: List all collaborators for a project
CREATE OR REPLACE VIEW `project_collaborators` AS
SELECT 
    ps.project_id,
    u.id AS user_id,
    u.username,
    u.email,
    CONCAT_WS(' ', u.first_name, u.last_name) AS full_name,
    ps.permission_level,
    ps.is_accepted,
    ps.created_at AS invited_at,
    ps.accepted_at,
    'direct' AS access_type
FROM project_shares ps
INNER JOIN users u ON ps.shared_with_user_id = u.id
UNION ALL
SELECT 
    p.id AS project_id,
    u.id AS user_id,
    u.username,
    u.email,
    CONCAT_WS(' ', u.first_name, u.last_name) AS full_name,
    'manage' AS permission_level,
    1 AS is_accepted,
    p.created_at AS invited_at,
    p.created_at AS accepted_at,
    'owner' AS access_type
FROM projects p
INNER JOIN users u ON p.user_id = u.id;

-- ============================================================================
-- INDEXES FOR PERFORMANCE
-- ============================================================================

-- Composite indexes for common queries
CREATE INDEX `idx_share_project_accepted` ON `project_shares` (`project_id`, `is_accepted`);
CREATE INDEX `idx_share_user_accepted` ON `project_shares` (`shared_with_user_id`, `is_accepted`);
CREATE INDEX `idx_token_project_active` ON `share_tokens` (`project_id`, `is_active`);

-- ============================================================================
-- SAMPLE DATA (for testing)
-- ============================================================================

-- Note: Insert sample shares after projects and users exist
-- Example:
-- INSERT INTO project_shares (project_id, shared_by_user_id, shared_with_user_id, permission_level, is_accepted)
-- VALUES (1, 1, 2, 'edit', 1);

-- Example: Generate a public share token
-- INSERT INTO share_tokens (project_id, created_by_user_id, token, permission_level)
-- VALUES (1, 1, SHA2(CONCAT(UUID(), UNIX_TIMESTAMP()), 256), 'view');
