<?php
/**
 * Project Sharing API
 * 
 * Handles sharing projects with users and generating public links
 * 
 * @author Jack Ewers / BloodWeb
 * @version 2.0.0
 * @date 2025-11-18
 */

// Enable error reporting for development
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Set JSON response headers
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-API-Key');

// Handle preflight OPTIONS requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Include dependencies
require_once 'Database.php';
require_once 'Models.php';
// Include session bootstrap for authentication
require_once $_SERVER['DOCUMENT_ROOT'] . '/auth/session_bootstrap.php';

$db = DatabaseConnection::getInstance()->getPDO();
$method = $_SERVER['REQUEST_METHOD'];
$user_id = $_SESSION['user_id'] ?? null;
$action = $_GET['action'] ?? null;

// Public actions that don't require authentication
$public_actions = ['public_project'];

// Check authentication (except for public actions)
if (!in_array($action, $public_actions) && !$user_id) {
    http_response_code(401);
    echo json_encode(['error' => 'Unauthorized']);
    exit;
}

/**
 * Generate a secure random token for public shares
 */
function generateShareToken() {
    return bin2hex(random_bytes(32)); // 64 character hex string
}

/**
 * Check if user has permission to manage shares for a project
 */
function canManageShares($db, $project_id, $user_id) {
    // User must be owner OR have 'manage' permission
    $stmt = $db->prepare("
        SELECT 1 FROM projects WHERE id = ? AND user_id = ?
        UNION
        SELECT 1 FROM project_shares 
        WHERE project_id = ? AND shared_with_user_id = ? 
        AND permission_level = 'manage' AND is_accepted = 1
        LIMIT 1
    ");
    $stmt->execute([$project_id, $user_id, $project_id, $user_id]);
    return $stmt->fetch() !== false;
}

/**
 * Get user's permission level for a project
 */
function getUserPermission($db, $project_id, $user_id) {
    // Check if owner
    $stmt = $db->prepare("SELECT 1 FROM projects WHERE id = ? AND user_id = ?");
    $stmt->execute([$project_id, $user_id]);
    if ($stmt->fetch()) {
        return 'manage'; // Owners have full management rights
    }
    
    // Check shared permissions
    $stmt = $db->prepare("
        SELECT permission_level 
        FROM project_shares 
        WHERE project_id = ? AND shared_with_user_id = ? AND is_accepted = 1
    ");
    $stmt->execute([$project_id, $user_id]);
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    
    return $result ? $result['permission_level'] : null;
}

try {
    switch ($method) {
        case 'GET':
            if (isset($_GET['action'])) {
                switch ($_GET['action']) {
                    
                    // Get all shares for a project
                    case 'project_shares':
                        $project_id = $_GET['project_id'] ?? null;
                        
                        if (!$project_id || !canManageShares($db, $project_id, $user_id)) {
                            http_response_code(403);
                            echo json_encode(['error' => 'Not authorized to view shares']);
                            exit;
                        }
                        
                        // Get user shares
                        $stmt = $db->prepare("
                            SELECT 
                                ps.id,
                                ps.shared_with_user_id AS user_id,
                                u.username,
                                u.email,
                                CONCAT_WS(' ', u.first_name, u.last_name) AS full_name,
                                ps.permission_level,
                                ps.is_accepted,
                                ps.created_at,
                                ps.accepted_at
                            FROM project_shares ps
                            INNER JOIN users u ON ps.shared_with_user_id = u.id
                            WHERE ps.project_id = ?
                            ORDER BY ps.created_at DESC
                        ");
                        $stmt->execute([$project_id]);
                        $user_shares = $stmt->fetchAll(PDO::FETCH_ASSOC);
                        
                        // Get public link tokens (only active ones)
                        $stmt = $db->prepare("
                            SELECT 
                                id,
                                token,
                                name,
                                permission_level,
                                is_active,
                                expires_at,
                                access_count,
                                last_accessed_at,
                                created_at
                            FROM share_tokens
                            WHERE project_id = ? AND is_active = 1
                            ORDER BY created_at DESC
                        ");
                        $stmt->execute([$project_id]);
                        $public_links = $stmt->fetchAll(PDO::FETCH_ASSOC);
                        
                        echo json_encode([
                            'user_shares' => $user_shares,
                            'public_links' => $public_links
                        ]);
                        break;
                    
                    // Get projects shared with current user
                    case 'shared_with_me':
                        $stmt = $db->prepare("
                            SELECT * FROM user_shared_projects
                            WHERE user_id = ?
                            ORDER BY shared_at DESC
                        ");
                        $stmt->execute([$user_id]);
                        $projects = $stmt->fetchAll(PDO::FETCH_ASSOC);
                        
                        echo json_encode($projects);
                        break;
                    
                    // Search users to share with
                    case 'search_users':
                        $query = $_GET['q'] ?? '';
                        $project_id = $_GET['project_id'] ?? null;
                        
                        if (strlen($query) < 2) {
                            echo json_encode([]);
                            exit;
                        }
                        
                        // Search users, excluding current user and already shared users
                        $stmt = $db->prepare("
                            SELECT 
                                u.id,
                                u.username,
                                u.email,
                                CONCAT_WS(' ', u.first_name, u.last_name) AS full_name
                            FROM users u
                            WHERE u.id != ?
                            AND u.id NOT IN (
                                SELECT shared_with_user_id 
                                FROM project_shares 
                                WHERE project_id = ?
                            )
                            AND (
                                u.username LIKE ? 
                                OR u.email LIKE ? 
                                OR CONCAT_WS(' ', u.first_name, u.last_name) LIKE ?
                            )
                            LIMIT 10
                        ");
                        $search = "%{$query}%";
                        $stmt->execute([$user_id, $project_id, $search, $search, $search]);
                        $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
                        
                        echo json_encode($users);
                        break;
                    
                    // Get project via public token
                    case 'public_project':
                        $token = $_GET['token'] ?? null;
                        
                        if (!$token) {
                            http_response_code(400);
                            echo json_encode(['error' => 'Token required']);
                            exit;
                        }
                        
                        // Validate token and get project
                        $stmt = $db->prepare("
                            SELECT 
                                st.permission_level,
                                p.*,
                                u.username AS owner_username,
                                CONCAT_WS(' ', u.first_name, u.last_name) AS owner_full_name
                            FROM share_tokens st
                            INNER JOIN projects p ON st.project_id = p.id
                            INNER JOIN users u ON p.user_id = u.id
                            WHERE st.token = ? 
                            AND st.is_active = 1
                            AND (st.expires_at IS NULL OR st.expires_at > NOW())
                        ");
                        $stmt->execute([$token]);
                        $result = $stmt->fetch(PDO::FETCH_ASSOC);
                        
                        if (!$result) {
                            http_response_code(404);
                            echo json_encode(['error' => 'Invalid or expired link']);
                            exit;
                        }
                        
                        // Update access count
                        $stmt = $db->prepare("
                            UPDATE share_tokens 
                            SET access_count = access_count + 1,
                                last_accessed_at = NOW()
                            WHERE token = ?
                        ");
                        $stmt->execute([$token]);
                        
                        // Log access
                        $stmt = $db->prepare("
                            INSERT INTO share_activity_log 
                            (project_id, user_id, action_type, details)
                            VALUES (?, ?, 'accessed_via_link', ?)
                        ");
                        $stmt->execute([
                            $result['id'],
                            $result['user_id'],
                            json_encode(['token' => $token])
                        ]);
                        
                        echo json_encode($result);
                        break;
                    
                    default:
                        http_response_code(400);
                        echo json_encode(['error' => 'Invalid action']);
                }
            }
            break;
        
        case 'POST':
            $data = json_decode(file_get_contents('php://input'), true);
            $action = $data['action'] ?? null;
            
            switch ($action) {
                
                // Share project with a user
                case 'share_with_user':
                    $project_id = $data['project_id'] ?? null;
                    $target_user_id = $data['user_id'] ?? null;
                    $permission = $data['permission_level'] ?? 'view';
                    
                    if (!$project_id || !$target_user_id) {
                        http_response_code(400);
                        echo json_encode(['error' => 'Missing required fields']);
                        exit;
                    }
                    
                    if (!canManageShares($db, $project_id, $user_id)) {
                        http_response_code(403);
                        echo json_encode(['error' => 'Not authorized']);
                        exit;
                    }
                    
                    // Validate permission level
                    if (!in_array($permission, ['view', 'edit', 'manage'])) {
                        http_response_code(400);
                        echo json_encode(['error' => 'Invalid permission level']);
                        exit;
                    }
                    
                    // Create share
                    $stmt = $db->prepare("
                        INSERT INTO project_shares 
                        (project_id, shared_by_user_id, shared_with_user_id, permission_level)
                        VALUES (?, ?, ?, ?)
                    ");
                    $stmt->execute([$project_id, $user_id, $target_user_id, $permission]);
                    
                    // Update project shared flag
                    $stmt = $db->prepare("UPDATE projects SET is_shared = 1 WHERE id = ?");
                    $stmt->execute([$project_id]);
                    
                    // Log activity
                    $stmt = $db->prepare("
                        INSERT INTO share_activity_log 
                        (project_id, user_id, action_type, target_user_id, details)
                        VALUES (?, ?, 'invited', ?, ?)
                    ");
                    $stmt->execute([
                        $project_id,
                        $user_id,
                        $target_user_id,
                        json_encode(['permission' => $permission])
                    ]);
                    
                    echo json_encode([
                        'success' => true,
                        'share_id' => $db->lastInsertId()
                    ]);
                    break;
                
                // Create public share link
                case 'create_public_link':
                    $project_id = $data['project_id'] ?? null;
                    $name = $data['name'] ?? null;
                    $permission = $data['permission_level'] ?? 'view';
                    $expires_days = $data['expires_days'] ?? null;
                    
                    if (!$project_id) {
                        http_response_code(400);
                        echo json_encode(['error' => 'Project ID required']);
                        exit;
                    }
                    
                    if (!canManageShares($db, $project_id, $user_id)) {
                        http_response_code(403);
                        echo json_encode(['error' => 'Not authorized']);
                        exit;
                    }
                    
                    // Validate permission (public links can't have 'manage')
                    if (!in_array($permission, ['view', 'edit'])) {
                        http_response_code(400);
                        echo json_encode(['error' => 'Invalid permission level']);
                        exit;
                    }
                    
                    // Generate token
                    $token = generateShareToken();
                    
                    // Calculate expiration if specified
                    $expires_at = null;
                    if ($expires_days) {
                        $expires_at = date('Y-m-d H:i:s', strtotime("+{$expires_days} days"));
                    }
                    
                    // Create token
                    $stmt = $db->prepare("
                        INSERT INTO share_tokens 
                        (project_id, created_by_user_id, token, name, permission_level, expires_at)
                        VALUES (?, ?, ?, ?, ?, ?)
                    ");
                    $stmt->execute([$project_id, $user_id, $token, $name, $permission, $expires_at]);
                    
                    // Update project shared flag
                    $stmt = $db->prepare("UPDATE projects SET is_shared = 1 WHERE id = ?");
                    $stmt->execute([$project_id]);
                    
                    // Log activity
                    $stmt = $db->prepare("
                        INSERT INTO share_activity_log 
                        (project_id, user_id, action_type, details)
                        VALUES (?, ?, 'link_created', ?)
                    ");
                    $stmt->execute([
                        $project_id,
                        $user_id,
                        json_encode(['permission' => $permission, 'expires_at' => $expires_at])
                    ]);
                    
                    echo json_encode([
                        'success' => true,
                        'token' => $token,
                        'url' => "/app/tasklist/shared.php?token={$token}"
                    ]);
                    break;
                
                // Accept share invitation
                case 'accept_share':
                    $share_id = $data['share_id'] ?? null;
                    
                    if (!$share_id) {
                        http_response_code(400);
                        echo json_encode(['error' => 'Share ID required']);
                        exit;
                    }
                    
                    // Verify share is for current user
                    $stmt = $db->prepare("
                        SELECT project_id 
                        FROM project_shares 
                        WHERE id = ? AND shared_with_user_id = ? AND is_accepted = 0
                    ");
                    $stmt->execute([$share_id, $user_id]);
                    $share = $stmt->fetch(PDO::FETCH_ASSOC);
                    
                    if (!$share) {
                        http_response_code(404);
                        echo json_encode(['error' => 'Share not found']);
                        exit;
                    }
                    
                    // Accept share
                    $stmt = $db->prepare("
                        UPDATE project_shares 
                        SET is_accepted = 1, accepted_at = NOW()
                        WHERE id = ?
                    ");
                    $stmt->execute([$share_id]);
                    
                    // Log activity
                    $stmt = $db->prepare("
                        INSERT INTO share_activity_log 
                        (project_id, user_id, action_type)
                        VALUES (?, ?, 'accepted')
                    ");
                    $stmt->execute([$share['project_id'], $user_id]);
                    
                    echo json_encode(['success' => true]);
                    break;
                
                default:
                    http_response_code(400);
                    echo json_encode(['error' => 'Invalid action']);
            }
            break;
        
        case 'PUT':
            $data = json_decode(file_get_contents('php://input'), true);
            $action = $data['action'] ?? null;
            
            // Update public link permission
            if ($action === 'update_token_permission') {
                $token_id = $data['token_id'] ?? null;
                $new_permission = $data['permission_level'] ?? null;
                
                if (!$token_id || !$new_permission) {
                    http_response_code(400);
                    echo json_encode(['error' => 'Missing required fields']);
                    exit;
                }
                
                if (!in_array($new_permission, ['view', 'edit'])) {
                    http_response_code(400);
                    echo json_encode(['error' => 'Invalid permission level']);
                    exit;
                }
                
                // Verify user can manage this token
                $stmt = $db->prepare("
                    SELECT st.project_id
                    FROM share_tokens st
                    WHERE st.id = ?
                ");
                $stmt->execute([$token_id]);
                $token = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if (!$token || !canManageShares($db, $token['project_id'], $user_id)) {
                    http_response_code(403);
                    echo json_encode(['error' => 'Not authorized']);
                    exit;
                }
                
                // Update token permission
                $stmt = $db->prepare("
                    UPDATE share_tokens 
                    SET permission_level = ?, updated_at = NOW()
                    WHERE id = ?
                ");
                $stmt->execute([$new_permission, $token_id]);
                
                // Log activity
                $stmt = $db->prepare("
                    INSERT INTO share_activity_log 
                    (project_id, user_id, action_type, details)
                    VALUES (?, ?, 'permission_changed', ?)
                ");
                $stmt->execute([
                    $token['project_id'],
                    $user_id,
                    json_encode(['type' => 'public_link', 'new_permission' => $new_permission])
                ]);
                
                echo json_encode(['success' => true]);
                exit;
            }
            
            // Update public link name
            if ($action === 'update_token_name') {
                $token_id = $data['token_id'] ?? null;
                $new_name = $data['name'] ?? null;
                
                if (!$token_id) {
                    http_response_code(400);
                    echo json_encode(['error' => 'Missing token_id']);
                    exit;
                }
                
                // Verify user can manage this token
                $stmt = $db->prepare("
                    SELECT st.project_id
                    FROM share_tokens st
                    WHERE st.id = ?
                ");
                $stmt->execute([$token_id]);
                $token = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if (!$token || !canManageShares($db, $token['project_id'], $user_id)) {
                    http_response_code(403);
                    echo json_encode(['error' => 'Not authorized']);
                    exit;
                }
                
                // Update token name
                $stmt = $db->prepare("
                    UPDATE share_tokens 
                    SET name = ?, updated_at = NOW()
                    WHERE id = ?
                ");
                $stmt->execute([$new_name, $token_id]);
                
                echo json_encode(['success' => true]);
                exit;
            }
            
            // Update user share permission
            $share_id = $data['share_id'] ?? null;
            $new_permission = $data['permission_level'] ?? null;
            
            if (!$share_id || !$new_permission) {
                http_response_code(400);
                echo json_encode(['error' => 'Missing required fields']);
                exit;
            }
            
            if (!in_array($new_permission, ['view', 'edit', 'manage'])) {
                http_response_code(400);
                echo json_encode(['error' => 'Invalid permission level']);
                exit;
            }
            
            // Verify user can manage this share
            $stmt = $db->prepare("
                SELECT ps.project_id, ps.shared_with_user_id
                FROM project_shares ps
                WHERE ps.id = ?
            ");
            $stmt->execute([$share_id]);
            $share = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$share || !canManageShares($db, $share['project_id'], $user_id)) {
                http_response_code(403);
                echo json_encode(['error' => 'Not authorized']);
                exit;
            }
            
            // Update permission
            $stmt = $db->prepare("
                UPDATE project_shares 
                SET permission_level = ?
                WHERE id = ?
            ");
            $stmt->execute([$new_permission, $share_id]);
            
            // Log activity
            $stmt = $db->prepare("
                INSERT INTO share_activity_log 
                (project_id, user_id, action_type, target_user_id, details)
                VALUES (?, ?, 'permission_changed', ?, ?)
            ");
            $stmt->execute([
                $share['project_id'],
                $user_id,
                $share['shared_with_user_id'],
                json_encode(['new_permission' => $new_permission])
            ]);
            
            echo json_encode(['success' => true]);
            break;
        
        case 'DELETE':
            // Revoke share or deactivate public link
            $share_id = $_GET['share_id'] ?? null;
            $token_id = $_GET['token_id'] ?? null;
            
            if ($share_id) {
                // Revoke user share
                $stmt = $db->prepare("
                    SELECT ps.project_id, ps.shared_with_user_id
                    FROM project_shares ps
                    WHERE ps.id = ?
                ");
                $stmt->execute([$share_id]);
                $share = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if (!$share || !canManageShares($db, $share['project_id'], $user_id)) {
                    http_response_code(403);
                    echo json_encode(['error' => 'Not authorized']);
                    exit;
                }
                
                // Delete share
                $stmt = $db->prepare("DELETE FROM project_shares WHERE id = ?");
                $stmt->execute([$share_id]);
                
                // Log activity
                $stmt = $db->prepare("
                    INSERT INTO share_activity_log 
                    (project_id, user_id, action_type, target_user_id)
                    VALUES (?, ?, 'revoked', ?)
                ");
                $stmt->execute([
                    $share['project_id'],
                    $user_id,
                    $share['shared_with_user_id']
                ]);
                
                echo json_encode(['success' => true]);
                
            } elseif ($token_id) {
                // Deactivate public link
                $stmt = $db->prepare("
                    SELECT project_id 
                    FROM share_tokens 
                    WHERE id = ?
                ");
                $stmt->execute([$token_id]);
                $token = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if (!$token || !canManageShares($db, $token['project_id'], $user_id)) {
                    http_response_code(403);
                    echo json_encode(['error' => 'Not authorized']);
                    exit;
                }
                
                // Deactivate token
                $stmt = $db->prepare("
                    UPDATE share_tokens 
                    SET is_active = 0 
                    WHERE id = ?
                ");
                $stmt->execute([$token_id]);
                
                // Log activity
                $stmt = $db->prepare("
                    INSERT INTO share_activity_log 
                    (project_id, user_id, action_type)
                    VALUES (?, ?, 'link_revoked')
                ");
                $stmt->execute([$token['project_id'], $user_id]);
                
                echo json_encode(['success' => true]);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'Share ID or Token ID required']);
            }
            break;
        
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
    }
    
} catch (PDOException $e) {
    error_log("Sharing API Error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => 'Database error occurred']);
} catch (Exception $e) {
    error_log("Sharing API Error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => 'An error occurred']);
}
