<?php
/**
 * TaskList v2 API Router and Controller
 * 
 * This file handles all API requests with proper routing, authentication,
 * validation, and error handling.
 */

// Enable error reporting for development (log to file, not display)
error_reporting(E_ALL);
ini_set('display_errors', 0); // Don't display errors (they break JSON responses)
ini_set('log_errors', 1);     // Log errors to PHP error log

// Set JSON response headers
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-API-Key');

// Handle preflight OPTIONS requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Include dependencies
require_once 'Database.php';
require_once 'Models.php';
// Include session bootstrap for authentication
require_once $_SERVER['DOCUMENT_ROOT'] . '/auth/session_bootstrap.php';

/**
 * API Response Helper Class
 */
class APIResponse {
    public static function success($data = null, $meta = null, $code = 200) {
        http_response_code($code);
        $response = ['success' => true];
        
        if ($data !== null) {
            $response['data'] = $data;
        }
        
        if ($meta !== null) {
            $response['meta'] = $meta;
        }
        
        echo json_encode($response, JSON_PRETTY_PRINT);
        exit();
    }
    
    public static function error($message, $code = 400, $details = null) {
        http_response_code($code);
        
        $response = [
            'success' => false,
            'error' => [
                'message' => $message,
                'code' => self::getErrorCode($code)
            ]
        ];
        
        if ($details !== null) {
            $response['error']['details'] = $details;
        }
        
        echo json_encode($response, JSON_PRETTY_PRINT);
        exit();
    }
    
    private static function getErrorCode($httpCode) {
        $codes = [
            400 => 'VALIDATION_ERROR',
            401 => 'AUTHENTICATION_REQUIRED',
            403 => 'PERMISSION_DENIED',
            404 => 'RESOURCE_NOT_FOUND',
            409 => 'DUPLICATE_RESOURCE',
            422 => 'UNPROCESSABLE_ENTITY',
            429 => 'RATE_LIMIT_EXCEEDED',
            500 => 'INTERNAL_ERROR'
        ];
        
        return $codes[$httpCode] ?? 'UNKNOWN_ERROR';
    }
}

/**
 * API Authentication Helper
 */
class APIAuth {
    public static function authenticate() {
        // Check for test mode (development only)
        if (isset($_GET['test']) && $_GET['test'] === '1' &&
            (isset($_SERVER['QUERY_STRING']) && $_SERVER['QUERY_STRING'] === 'test=1')) {
            return 3; // Return the bloodweb development user ID
        }
        
        // Check for existing session (only if session is started)
        if (session_status() === PHP_SESSION_ACTIVE && 
            isset($_SESSION['user_id']) && !empty($_SESSION['user_id'])) {
            return (int) $_SESSION['user_id'];
        }
        
        // Check for share token access (anonymous users with link)
        if (session_status() === PHP_SESSION_ACTIVE && 
            isset($_SESSION['share_token']) && 
            isset($_SESSION['share_project_id'])) {
            // Return a special value to indicate share token access
            // We'll use negative numbers to differentiate from real user IDs
            return -1 * (int) $_SESSION['share_project_id'];
        }
        
        // Use the BloodWeb session system
        if (!defined('AUTH_USER_ID')) {
            require_once $_SERVER['DOCUMENT_ROOT'] . '/auth/session_bootstrap.php';
        }
        
        if (defined('AUTH_USER_ID') && AUTH_USER_ID > 0) {
            return (int) AUTH_USER_ID;
        }
        
        // Check for API key in headers or query params
        $apiKey = self::getApiKey();
        if ($apiKey) {
            $userId = self::validateApiKey($apiKey);
            if ($userId) {
                return $userId;
            }
        }
        
        APIResponse::error('Authentication required', 401);
    }
    
    /**
     * Check if the current user has share token access
     */
    public static function isShareTokenAccess() {
        return session_status() === PHP_SESSION_ACTIVE && 
               isset($_SESSION['share_token']) && 
               isset($_SESSION['share_project_id']);
    }
    
    /**
     * Get share token permission level (view or edit)
     */
    public static function getSharePermission() {
        if (self::isShareTokenAccess()) {
            return $_SESSION['share_permission'] ?? 'view';
        }
        return null;
    }
    
    /**
     * Get shared project ID
     */
    public static function getSharedProjectId() {
        if (self::isShareTokenAccess()) {
            return (int) $_SESSION['share_project_id'];
        }
        return null;
    }
    
    private static function getApiKey() {
        // Check X-API-Key header
        if (isset($_SERVER['HTTP_X_API_KEY'])) {
            return $_SERVER['HTTP_X_API_KEY'];
        }
        
        // Check Authorization header (Bearer token)
        if (isset($_SERVER['HTTP_AUTHORIZATION'])) {
            $auth = $_SERVER['HTTP_AUTHORIZATION'];
            if (strpos($auth, 'Bearer ') === 0) {
                return substr($auth, 7);
            }
        }
        
        // Check query parameter
        if (isset($_GET['api_key'])) {
            return $_GET['api_key'];
        }
        
        return null;
    }
    
    private static function validateApiKey($apiKey) {
        // TODO: Implement API key validation against database
        // For now, return default user for development (user id 3 = bloodweb)
        return 3;
    }
}

/**
 * Main API Router Class
 */
class APIRouter {
    private $method;
    private $path;
    private $userId;
    
    public function __construct() {
        $this->method = $_SERVER['REQUEST_METHOD'];
        $this->path = $this->parsePath();
        $this->userId = APIAuth::authenticate();
    }
    
    /**
     * Check if current user is a guest (share token access)
     * Guest users have negative user_id
     */
    private function isGuestUser() {
        return $this->userId < 0;
    }
    
    /**
     * Get the shared project ID for guest users
     */
    private function getSharedProjectId() {
        return isset($_SESSION['share_project_id']) ? (int)$_SESSION['share_project_id'] : null;
    }
    
    private function parsePath() {
        $requestUri = $_SERVER['REQUEST_URI'];
        $basePath = '/TaskList/api/';
        
        // Remove base path and query string
        $path = str_replace($basePath, '', parse_url($requestUri, PHP_URL_PATH));
        
        // Remove leading/trailing slashes and split
        return array_filter(explode('/', trim($path, '/')));
    }
    
    public function route() {
        try {
            if (empty($this->path)) {
                APIResponse::success(['message' => 'TaskList v2 API', 'version' => '2.0.0']);
            }
            
            $resource = $this->path[0];
            $id = $this->path[1] ?? null;
            $action = $this->path[2] ?? null;
            
            switch ($resource) {
                case 'tasks':
                    $this->handleTasks($id, $action);
                    break;
                    
                case 'bulk-tasks':
                    $this->handleBulkTasks();
                    break;
                    
                case 'projects':
                    $this->handleProjects($id, $action);
                    break;
                    
                case 'categories':
                    $this->handleCategories($id, $action);
                    break;
                    
                case 'subtasks':
                    $this->handleSubtasks($id, $action);
                    break;
                    
                case 'analytics':
                    $this->handleAnalytics($id, $action);
                    break;
                    
                case 'xp':
                    $this->handleXP($id, $action);
                    break;
                    
                case 'achievements':
                    $this->handleAchievements($id, $action);
                    break;
                    
                case 'calendar':
                    $this->handleCalendar($id, $action);
                    break;
                    
                case 'search':
                    $this->handleSearch();
                    break;
                    
                case 'user':
                    $this->handleUser($id, $action);
                    break;
                    
                default:
                    APIResponse::error('Resource not found', 404);
            }
            
        } catch (Exception $e) {
            error_log("API Error: " . $e->getMessage());
            APIResponse::error('Internal server error', 500);
        }
    }
    
    private function handleTasks($id, $action) {
        $taskModel = new Task();
        
        // Check if this is share token access
        $isShareAccess = APIAuth::isShareTokenAccess();
        $sharedProjectId = APIAuth::getSharedProjectId();
        $sharePermission = APIAuth::getSharePermission();
        
        switch ($this->method) {
            case 'GET':
                if ($id) {
                    if ($action) {
                        APIResponse::error('Invalid endpoint', 404);
                    }
                    
                    $task = $taskModel->find($id);
                    
                    // Check access: either owns task OR has share access to its project
                    $hasAccess = false;
                    if ($this->userId > 0 && $task['user_id'] === $this->userId) {
                        $hasAccess = true;
                    } elseif ($isShareAccess && $task['project_id'] == $sharedProjectId) {
                        $hasAccess = true;
                    }
                    
                    if (!$task || !$hasAccess) {
                        APIResponse::error('Task not found', 404);
                    }
                    
                    APIResponse::success($task);
                } else {
                    // Get tasks with filters
                    $filters = $_GET;
                    $page = (int) ($_GET['page'] ?? 1);
                    $perPage = min((int) ($_GET['per_page'] ?? 20), 100);
                    
                    // If share access, override filters to only get shared project tasks
                    if ($isShareAccess) {
                        $filters['project_id'] = $sharedProjectId;
                        // Get the owner's user_id from the project
                        $db = DatabaseConnection::getInstance()->getPDO();
                        $stmt = $db->prepare("SELECT user_id FROM projects WHERE id = ?");
                        $stmt->execute([$sharedProjectId]);
                        $project = $stmt->fetch(PDO::FETCH_ASSOC);
                        if ($project) {
                            $result = $taskModel->getByUser($project['user_id'], $filters, $page, $perPage);
                        } else {
                            APIResponse::success(['data' => [], 'meta' => ['total' => 0]]);
                        }
                    } else {
                        $result = $taskModel->getByUser($this->userId, $filters, $page, $perPage);
                    }
                    APIResponse::success($result['data'], $result['meta']);
                }
                break;
                
            case 'POST':
                // Check if share access has edit permission
                if ($isShareAccess && $sharePermission !== 'edit') {
                    APIResponse::error('You do not have permission to edit this project', 403);
                }
                
                if ($id && $action) {
                    // Handle task actions
                    $task = $taskModel->find($id);
                    
                    // Check access
                    $hasAccess = false;
                    if ($this->userId > 0 && $task['user_id'] === $this->userId) {
                        $hasAccess = true;
                    } elseif ($isShareAccess && $task['project_id'] == $sharedProjectId) {
                        $hasAccess = true;
                    }
                    
                    if (!$task || !$hasAccess) {
                        APIResponse::error('Task not found', 404);
                    }
                    
                    switch ($action) {
                        case 'complete':
                            // Check if task is already completed
                            if ($task['status'] === 'completed') {
                                APIResponse::error('Task is already completed', 400);
                            }
                            
                            // For guest users, use the actual task owner's user_id
                            $userIdForCompletion = $this->isGuestUser() ? $task['user_id'] : $this->userId;
                            
                            $success = $taskModel->markComplete($id, $userIdForCompletion);
                            if ($success) {
                                $response = ['message' => 'Task marked as completed'];
                                
                                // Guest users don't get XP
                                if (!$this->isGuestUser()) {
                                    // Get project and category data for XP calculation
                                    $projectModel = new Project();
                                    $categoryModel = new Category();
                                    
                                    $project = $projectModel->find($task['project_id']);
                                    $category = $categoryModel->find($task['category_id']);
                                    
                                    // Calculate advanced XP for task completion
                                    $xpModel = new XPLog();
                                    $taskXpResult = $xpModel->calculateTaskXP($task, $project, $category);
                                    
                                    // Create clean description
                                    $description = "Completed: {$task['title']}";
                                    
                                    // Award task completion XP
                                    $xpModel->awardXP($this->userId, 'task_completed', $taskXpResult['total_xp'], $id, $description, null, $taskXpResult['breakdown']);
                                    
                                    // Check for project completion
                                    $projectCompletion = $projectModel->checkAndAwardProjectCompletion($task['project_id'], $this->userId);
                                    
                                    $response['task_xp_earned'] = $taskXpResult['total_xp'];
                                    $response['task_xp_breakdown'] = $taskXpResult['breakdown'];
                                    
                                    if ($projectCompletion) {
                                        $response['project_completed'] = true;
                                        $response['project_xp_earned'] = $projectCompletion['xp_earned'];
                                        $response['project_xp_breakdown'] = $projectCompletion['xp_breakdown'];
                                        $response['total_xp_earned'] = $taskXpResult['total_xp'] + $projectCompletion['xp_earned'];
                                    } else {
                                        $response['total_xp_earned'] = $taskXpResult['total_xp'];
                                    }
                                } else {
                                    // Guest users get 0 XP
                                    $response['task_xp_earned'] = 0;
                                    $response['total_xp_earned'] = 0;
                                }
                                
                                APIResponse::success($response);
                            } else {
                                APIResponse::error('Failed to update task', 500);
                            }
                            break;
                            
                        case 'start':
                            $success = $taskModel->markInProgress($id, $this->userId);
                            if ($success) {
                                APIResponse::success(['message' => 'Task marked as in progress']);
                            } else {
                                APIResponse::error('Failed to update task', 500);
                            }
                            break;
                            
                        default:
                            APIResponse::error('Invalid action', 404);
                    }
                } else {
                    // Create new task
                    $data = json_decode(file_get_contents('php://input'), true);
                    
                    // Validate required fields
                    $validator = new Validator($data, [
                        'title' => 'required|max:255',
                        'project_id' => 'required|numeric',
                        'category_id' => 'required|numeric',
                        'priority' => 'in:low,medium,high,urgent',
                        'complexity' => 'in:none,simple,moderate,complex,very_complex',
                        'due_date' => 'date'
                    ]);
                    
                    if (!$validator->validate()) {
                        APIResponse::error('Validation failed', 422, $validator->getErrors());
                    }
                    
                    // Add user_id to data
                    $data['user_id'] = $this->userId;
                    
                    // Verify project and category belong to user
                    $projectModel = new Project();
                    $categoryModel = new Category();
                    
                    $project = $projectModel->find($data['project_id']);
                    $category = $categoryModel->find($data['category_id']);
                    
                    // Check project access
                    $hasProjectAccess = false;
                    if ($this->userId > 0 && $project && $project['user_id'] === $this->userId) {
                        $hasProjectAccess = true;
                    } elseif ($isShareAccess && $project && $project['id'] == $sharedProjectId) {
                        $hasProjectAccess = true;
                    }
                    
                    if (!$project || !$hasProjectAccess) {
                        APIResponse::error('Invalid project', 422);
                    }
                    
                    // For share access, verify category belongs to the project owner
                    if ($isShareAccess) {
                        if (!$category || $category['user_id'] !== $project['user_id']) {
                            APIResponse::error('Invalid category', 422);
                        }
                        // Set user_id to project owner for shared tasks
                        $data['user_id'] = $project['user_id'];
                    } else {
                        if (!$category || $category['user_id'] !== $this->userId) {
                            APIResponse::error('Invalid category', 422);
                        }
                    }
                    
                    $task = $taskModel->create($data);
                    APIResponse::success($task, null, 201);
                }
                break;
                
            case 'PUT':
                // Check if share access has edit permission
                if ($isShareAccess && $sharePermission !== 'edit') {
                    APIResponse::error('You do not have permission to edit this project', 403);
                }
                
                if (!$id) {
                    APIResponse::error('Task ID required', 400);
                }
                
                $task = $taskModel->find($id);
                
                // Check access
                $hasAccess = false;
                if ($this->userId > 0 && $task['user_id'] === $this->userId) {
                    $hasAccess = true;
                } elseif ($isShareAccess && $task['project_id'] == $sharedProjectId) {
                    $hasAccess = true;
                }
                
                if (!$task || !$hasAccess) {
                    APIResponse::error('Task not found', 404);
                }
                
                $data = json_decode(file_get_contents('php://input'), true);
                
                // Validate fields if provided
                $rules = [];
                if (isset($data['title'])) $rules['title'] = 'required|max:255';
                if (isset($data['priority'])) $rules['priority'] = 'in:low,medium,high,urgent';
                if (isset($data['complexity'])) $rules['complexity'] = 'in:none,simple,moderate,complex,very_complex';
                if (isset($data['status'])) $rules['status'] = 'in:pending,in_progress,completed,cancelled,on_hold';
                if (isset($data['due_date'])) $rules['due_date'] = 'date';
                
                if (!empty($rules)) {
                    $validator = new Validator($data, $rules);
                    if (!$validator->validate()) {
                        APIResponse::error('Validation failed', 422, $validator->getErrors());
                    }
                }
                
                // Check if status is changing to completed
                $wasCompleted = $task['status'] === 'completed';
                $isBeingCompleted = isset($data['status']) && $data['status'] === 'completed' && !$wasCompleted;
                
                $updatedTask = $taskModel->update($id, $data);
                
                // Award XP if task was just completed (skip for guest users)
                if ($isBeingCompleted && !$this->isGuestUser()) {
                    try {
                        error_log("=== AWARDING XP FOR TASK {$id} ===");
                        
                        // Get project and category data for XP calculation
                        $projectModel = new Project();
                        $categoryModel = new Category();
                        
                        $project = $projectModel->find($task['project_id']);
                        $category = $categoryModel->find($task['category_id']);
                        
                        error_log("Project: " . json_encode($project));
                        error_log("Category: " . json_encode($category));
                        
                        // Calculate advanced XP for task completion
                        $xpModel = new XPLog();
                        $taskXpResult = $xpModel->calculateTaskXP($updatedTask, $project, $category);
                        
                        error_log("XP Result: " . json_encode($taskXpResult));
                        
                        // Create clean description
                        $description = "Completed: {$updatedTask['title']}";
                        
                        // Award task completion XP (project_id should be NULL for task completions, not the task's project_id)
                        $awardResult = $xpModel->awardXP($this->userId, 'task_completed', $taskXpResult['total_xp'], $id, $description, null, $taskXpResult['breakdown']);
                        
                        error_log("Award XP result: " . json_encode($awardResult));
                        error_log("=== XP AWARDED SUCCESSFULLY ===");
                    } catch (Exception $e) {
                        error_log("ERROR AWARDING XP: " . $e->getMessage());
                        error_log("Stack trace: " . $e->getTraceAsString());
                    }
                    
                    // Check if user has auto-complete projects enabled (skip for guest users)
                    if (!$this->isGuestUser() && $task['project_id']) {
                        $userModel = new User();
                        $user = $userModel->find($this->userId);
                        $settings = $user['settings'] ?? [];
                        $autoCompleteProjects = $settings['TL_autoCompleteProjects'] ?? false;
                    } else {
                        $autoCompleteProjects = false;
                    }
                    
                    // If auto-complete is enabled, check if all project tasks are complete
                    if ($autoCompleteProjects) {
                        $project = $projectModel->find($task['project_id']);
                        
                        // Only auto-complete if project is not already completed and not archived
                        if ($project && !$project['is_completed'] && !$project['is_archived']) {
                            // Check if all tasks in this project are completed
                            $allTasks = $taskModel->getByUser($this->userId, ['project' => $task['project_id']])['data'];
                            $allComplete = true;
                            
                            foreach ($allTasks as $projectTask) {
                                if ($projectTask['status'] !== 'completed') {
                                    $allComplete = false;
                                    break;
                                }
                            }
                            
                            // If all tasks complete, mark project as completed and award XP
                            if ($allComplete && count($allTasks) > 0) {
                                error_log("Auto-completing project {$task['project_id']} - all tasks done");
                                $projectModel->update($task['project_id'], ['is_completed' => true]);
                                
                                // Award project completion XP
                                $xpModel = new XPLog();
                                $completedTasksCount = count($allTasks);
                                $projectXpResult = $xpModel->calculateProjectXP($project, $completedTasksCount);
                                
                                // Check if XP already awarded
                                $existingXP = $xpModel->db->selectOne(
                                    "SELECT * FROM xp_log WHERE user_id = ? AND action_type = 'project_completed' AND project_id = ?",
                                    [$this->userId, $task['project_id']]
                                );
                                
                                if (!$existingXP) {
                                    $description = "Auto-completed project: {$project['name']}";
                                    $xpModel->awardXP(
                                        $this->userId,
                                        'project_completed',
                                        $projectXpResult['total_xp'],
                                        null,
                                        $description,
                                        $task['project_id'],
                                        $projectXpResult['breakdown']
                                    );
                                    
                                    $updatedTask['project_auto_completed'] = true;
                                    $updatedTask['project_xp_earned'] = $projectXpResult['total_xp'];
                                }
                            }
                        }
                    }
                    
                    // Check for project completion
                    $projectCompletion = $projectModel->checkAndAwardProjectCompletion($task['project_id'], $this->userId);
                    
                    $updatedTask['xp_earned'] = $taskXpResult['total_xp'];
                    $updatedTask['xp_breakdown'] = $taskXpResult['breakdown'];
                    
                    if ($projectCompletion) {
                        $updatedTask['project_completed'] = true;
                        $updatedTask['project_xp_earned'] = $projectCompletion['xp_earned'];
                        $updatedTask['total_xp_earned'] = $taskXpResult['total_xp'] + $projectCompletion['xp_earned'];
                    }
                }
                
                APIResponse::success($updatedTask);
                break;
                
            case 'DELETE':
                // Check if share access has edit permission
                if ($isShareAccess && $sharePermission !== 'edit') {
                    APIResponse::error('You do not have permission to edit this project', 403);
                }
                
                if (!$id) {
                    APIResponse::error('Task ID required', 400);
                }
                
                $task = $taskModel->find($id);
                
                // Check access
                $hasAccess = false;
                if ($this->userId > 0 && $task['user_id'] === $this->userId) {
                    $hasAccess = true;
                } elseif ($isShareAccess && $task['project_id'] == $sharedProjectId) {
                    $hasAccess = true;
                }
                
                if (!$task || !$hasAccess) {
                    APIResponse::error('Task not found', 404);
                }
                
                $success = $taskModel->delete($id);
                if ($success) {
                    APIResponse::success(['message' => 'Task deleted successfully'], null, 204);
                } else {
                    APIResponse::error('Failed to delete task', 500);
                }
                break;
                
            default:
                APIResponse::error('Method not allowed', 405);
        }
    }
    
    private function handleBulkTasks() {
        $taskModel = new Task();
        
        // Only allow POST for bulk operations
        if ($this->method !== 'POST') {
            APIResponse::error('Method not allowed', 405);
        }
        
        // Get request body
        $input = json_decode(file_get_contents('php://input'), true);
        
        if (!$input || !isset($input['task_ids']) || !is_array($input['task_ids'])) {
            APIResponse::error('task_ids array required', 400);
        }
        
        $taskIds = array_map('intval', $input['task_ids']);
        
        if (empty($taskIds)) {
            APIResponse::error('No task IDs provided', 400);
        }
        
        // Get action from query param
        $action = $_GET['action'] ?? null;
        
        switch ($action) {
            case 'bulk-update':
                if (!isset($input['updates']) || !is_array($input['updates'])) {
                    APIResponse::error('updates object required', 400);
                }
                
                $updates = $input['updates'];
                $successCount = 0;
                $failedIds = [];
                
                foreach ($taskIds as $taskId) {
                    $task = $taskModel->find($taskId);
                    
                    // Verify ownership
                    if (!$task || $task['user_id'] !== $this->userId) {
                        $failedIds[] = $taskId;
                        continue;
                    }
                    
                    // Merge current task with updates
                    $updatedData = array_merge($task, $updates);
                    
                    // Ensure required fields
                    $updatedData['id'] = $taskId;
                    $updatedData['user_id'] = $this->userId;
                    
                    $success = $taskModel->update($taskId, $updatedData);
                    
                    if ($success) {
                        $successCount++;
                    } else {
                        $failedIds[] = $taskId;
                    }
                }
                
                APIResponse::success([
                    'message' => "Updated {$successCount} of " . count($taskIds) . " tasks",
                    'success_count' => $successCount,
                    'failed_count' => count($failedIds),
                    'failed_ids' => $failedIds
                ]);
                break;
                
            case 'bulk-delete':
                $successCount = 0;
                $failedIds = [];
                
                foreach ($taskIds as $taskId) {
                    $task = $taskModel->find($taskId);
                    
                    // Verify ownership
                    if (!$task || $task['user_id'] !== $this->userId) {
                        $failedIds[] = $taskId;
                        continue;
                    }
                    
                    $success = $taskModel->delete($taskId);
                    
                    if ($success) {
                        $successCount++;
                    } else {
                        $failedIds[] = $taskId;
                    }
                }
                
                APIResponse::success([
                    'message' => "Deleted {$successCount} of " . count($taskIds) . " tasks",
                    'success_count' => $successCount,
                    'failed_count' => count($failedIds),
                    'failed_ids' => $failedIds
                ]);
                break;
                
            default:
                APIResponse::error('Invalid bulk action. Use bulk-update or bulk-delete', 400);
        }
    }
    
    private function handleProjects($id, $action) {
        $projectModel = new Project();
        
        switch ($this->method) {
            case 'GET':
                if ($id) {
                    if ($action === 'categories') {
                        $categories = $projectModel->getCategories($id);
                        APIResponse::success($categories);
                    } else {
                        $project = $projectModel->getWithStats($this->userId, $id);
                        if (!$project) {
                            APIResponse::error('Project not found', 404);
                        }
                        APIResponse::success($project);
                    }
                } else {
                    // Guest users only see the shared project
                    if ($this->isGuestUser()) {
                        $sharedProjectId = $this->getSharedProjectId();
                        if (!$sharedProjectId) {
                            APIResponse::success([]);
                            return;
                        }
                        
                        // Get the project - just use basic find, no stats for guests
                        $project = $projectModel->find($sharedProjectId);
                        
                        if ($project) {
                            // Add category IDs
                            $categoryIds = $projectModel->getCategories($sharedProjectId);
                            $project['category_ids'] = array_map(function($cat) {
                                return (int)$cat['id'];
                            }, $categoryIds);
                            
                            // Add basic stats manually
                            $db = DatabaseConnection::getInstance()->getPDO();
                            $stmt = $db->prepare("
                                SELECT 
                                    COUNT(*) as total_tasks,
                                    SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed_tasks,
                                    SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending_tasks,
                                    SUM(CASE WHEN status = 'in_progress' THEN 1 ELSE 0 END) as in_progress_tasks
                                FROM tasks 
                                WHERE project_id = ?
                            ");
                            $stmt->execute([$sharedProjectId]);
                            $stats = $stmt->fetch(PDO::FETCH_ASSOC);
                            
                            $project['total_tasks'] = (int)($stats['total_tasks'] ?? 0);
                            $project['completed_tasks'] = (int)($stats['completed_tasks'] ?? 0);
                            $project['pending_tasks'] = (int)($stats['pending_tasks'] ?? 0);
                            $project['in_progress_tasks'] = (int)($stats['in_progress_tasks'] ?? 0);
                            $project['completion_percentage'] = $project['total_tasks'] > 0 
                                ? round(($project['completed_tasks'] / $project['total_tasks']) * 100, 1) 
                                : 0;
                        }
                        
                        APIResponse::success($project ? [$project] : []);
                    } else {
                        // Regular users see all their projects
                        $includeArchived = isset($_GET['include_archived']) && $_GET['include_archived'] === 'true';
                        $includeStats = isset($_GET['include_stats']) && $_GET['include_stats'] === 'true';
                        
                        if ($includeStats) {
                            $projects = $projectModel->getWithStats($this->userId);
                        } else {
                            $projects = $projectModel->getByUser($this->userId, $includeArchived);
                        }
                        
                        APIResponse::success($projects);
                    }
                }
                break;
                
            case 'POST':
                if ($id && $action === 'categories') {
                    // Add category to project
                    $data = json_decode(file_get_contents('php://input'), true);
                    
                    if (!isset($data['category_id'])) {
                        APIResponse::error('Category ID required', 400);
                    }
                    
                    $project = $projectModel->find($id);
                    if (!$project || $project['user_id'] !== $this->userId) {
                        APIResponse::error('Project not found', 404);
                    }
                    
                    $categoryModel = new Category();
                    $category = $categoryModel->find($data['category_id']);
                    if (!$category || $category['user_id'] !== $this->userId) {
                        APIResponse::error('Category not found', 404);
                    }
                    
                    $projectModel->addCategory($id, $data['category_id']);
                    APIResponse::success(['message' => 'Category added to project'], null, 201);
                } else {
                    // Create new project
                    $data = json_decode(file_get_contents('php://input'), true);
                    
                    $validator = new Validator($data, [
                        'name' => 'required|max:255'
                    ]);
                    
                    if (!$validator->validate()) {
                        APIResponse::error('Validation failed', 422, $validator->getErrors());
                    }
                    
                    $data['user_id'] = $this->userId;
                    $project = $projectModel->create($data);
                    APIResponse::success($project, null, 201);
                }
                break;
                
            case 'PUT':
                if (!$id) {
                    APIResponse::error('Project ID required', 400);
                }
                
                $project = $projectModel->find($id);
                if (!$project || $project['user_id'] !== $this->userId) {
                    APIResponse::error('Project not found', 404);
                }
                
                $data = json_decode(file_get_contents('php://input'), true);
                $updatedProject = $projectModel->update($id, $data);
                APIResponse::success($updatedProject);
                break;
                
            case 'PATCH':
                if (!$id) {
                    APIResponse::error('Project ID required', 400);
                }
                
                $project = $projectModel->find($id);
                if (!$project || $project['user_id'] !== $this->userId) {
                    APIResponse::error('Project not found', 404);
                }
                
                // Handle specific actions
                if ($action === 'toggle-completion') {
                    // Toggle is_completed status
                    $newStatus = !$project['is_completed'];
                    $updatedProject = $projectModel->update($id, ['is_completed' => $newStatus]);
                    APIResponse::success($updatedProject);
                } else {
                    // Generic PATCH for partial updates
                    $data = json_decode(file_get_contents('php://input'), true);
                    $updatedProject = $projectModel->update($id, $data);
                    APIResponse::success($updatedProject);
                }
                break;
                
            case 'DELETE':
                if ($action) {
                    // Remove category from project
                    $categoryId = $action;
                    $success = $projectModel->removeCategory($id, $categoryId);
                    if ($success) {
                        APIResponse::success(['message' => 'Category removed from project'], null, 204);
                    } else {
                        APIResponse::error('Failed to remove category', 500);
                    }
                } else {
                    // Archive project
                    if (!$id) {
                        APIResponse::error('Project ID required', 400);
                    }
                    
                    $project = $projectModel->find($id);
                    if (!$project || $project['user_id'] !== $this->userId) {
                        APIResponse::error('Project not found', 404);
                    }
                    
                    $success = $projectModel->update($id, ['is_archived' => true]);
                    if ($success) {
                        APIResponse::success(['message' => 'Project archived successfully'], null, 204);
                    } else {
                        APIResponse::error('Failed to archive project', 500);
                    }
                }
                break;
                
            default:
                APIResponse::error('Method not allowed', 405);
        }
    }
    
    private function handleCategories($id, $action) {
        $categoryModel = new Category();
        
        switch ($this->method) {
            case 'GET':
                if ($id) {
                    $category = $categoryModel->find($id);
                    if (!$category || $category['user_id'] !== $this->userId) {
                        APIResponse::error('Category not found', 404);
                    }
                    APIResponse::success($category);
                } else {
                    // Guest users only see categories for the shared project
                    if ($this->isGuestUser()) {
                        $sharedProjectId = $this->getSharedProjectId();
                        if (!$sharedProjectId) {
                            APIResponse::success([]);
                            return;
                        }
                        
                        $projectModel = new Project();
                        $categories = $projectModel->getCategories($sharedProjectId);
                        APIResponse::success($categories);
                    } else {
                        // Regular users see all their categories
                        $categories = $categoryModel->getByUser($this->userId);
                        APIResponse::success($categories);
                    }
                }
                break;
                
            case 'POST':
                $data = json_decode(file_get_contents('php://input'), true);
                
                $validator = new Validator($data, [
                    'name' => 'required|max:255'
                ]);
                
                if (!$validator->validate()) {
                    APIResponse::error('Validation failed', 422, $validator->getErrors());
                }
                
                $data['user_id'] = $this->userId;
                
                // Convert checkbox values to proper booleans
                if (isset($data['auto_bind'])) {
                    $data['auto_bind'] = ($data['auto_bind'] === 'on' || $data['auto_bind'] === true || $data['auto_bind'] === '1') ? 1 : 0;
                }
                
                $category = $categoryModel->create($data);
                APIResponse::success($category, null, 201);
                break;
                
            case 'PUT':
                if (!$id) {
                    APIResponse::error('Category ID required', 400);
                }
                
                $category = $categoryModel->find($id);
                if (!$category || $category['user_id'] !== $this->userId) {
                    APIResponse::error('Category not found', 404);
                }
                
                if ($action === 'projects') {
                    // Handle category-project binding
                    $data = json_decode(file_get_contents('php://input'), true);
                    
                    if (!isset($data['project_ids']) || !is_array($data['project_ids'])) {
                        APIResponse::error('project_ids array is required', 400);
                    }
                    
                    $success = $categoryModel->updateProjectBindings($id, $data['project_ids'], $this->userId);
                    
                    if ($success) {
                        APIResponse::success(['message' => 'Category project bindings updated successfully']);
                    } else {
                        APIResponse::error('Failed to update category project bindings', 500);
                    }
                } else {
                    // Handle regular category update
                    $data = json_decode(file_get_contents('php://input'), true);
                    
                    // Convert checkbox values to proper booleans
                    if (isset($data['auto_bind'])) {
                        $data['auto_bind'] = ($data['auto_bind'] === 'on' || $data['auto_bind'] === true || $data['auto_bind'] === '1') ? 1 : 0;
                    }
                    
                    $updatedCategory = $categoryModel->update($id, $data);
                    APIResponse::success($updatedCategory);
                }
                break;
                
            case 'DELETE':
                if (!$id) {
                    APIResponse::error('Category ID required', 400);
                }
                
                $category = $categoryModel->find($id);
                if (!$category || $category['user_id'] !== $this->userId) {
                    APIResponse::error('Category not found', 404);
                }
                
                // Check if category is used by any tasks
                $db = DatabaseConnection::getInstance();
                $tasksUsing = $db->count('tasks', ['category_id = ?'], [$id]);
                
                if ($tasksUsing > 0) {
                    APIResponse::error('Cannot delete category that is used by tasks', 409);
                }
                
                $success = $categoryModel->delete($id);
                if ($success) {
                    APIResponse::success(['message' => 'Category deleted successfully']);
                } else {
                    APIResponse::error('Failed to delete category', 500);
                }
                break;
                
            default:
                APIResponse::error('Method not allowed', 405);
        }
    }
    
    private function handleSubtasks($id, $action) {
        $subtaskModel = new Subtask();
        $taskModel = new Task();
        
        // Check if this is share token access
        $isShareAccess = APIAuth::isShareTokenAccess();
        $sharedProjectId = APIAuth::getSharedProjectId();
        
        switch ($this->method) {
            case 'GET':
                if ($id) {
                    // GET /subtasks/{id} - Get a specific subtask
                    $subtask = $subtaskModel->find($id);
                    if (!$subtask) {
                        APIResponse::error('Subtask not found', 404);
                    }
                    
                    // Verify user owns the parent task OR has share access to its project
                    $task = $taskModel->find($subtask['task_id']);
                    $hasAccess = false;
                    if ($this->userId > 0 && $task['user_id'] === $this->userId) {
                        $hasAccess = true;
                    } elseif ($isShareAccess && $task['project_id'] == $sharedProjectId) {
                        $hasAccess = true;
                    }
                    
                    if (!$task || !$hasAccess) {
                        APIResponse::error('Permission denied', 403);
                    }
                    
                    APIResponse::success($subtask);
                } else {
                    // GET /subtasks?task_id=X - Get subtasks for a task
                    $taskId = $_GET['task_id'] ?? null;
                    if (!$taskId) {
                        APIResponse::error('task_id parameter required', 400);
                    }
                    
                    // Verify user owns the task OR has share access to its project
                    $task = $taskModel->find($taskId);
                    $hasAccess = false;
                    if ($this->userId > 0 && $task['user_id'] === $this->userId) {
                        $hasAccess = true;
                    } elseif ($isShareAccess && $task['project_id'] == $sharedProjectId) {
                        $hasAccess = true;
                    }
                    
                    if (!$task || !$hasAccess) {
                        APIResponse::error('Task not found or permission denied', 403);
                    }
                    
                    $subtasks = $subtaskModel->getByTask($taskId);
                    $stats = $subtaskModel->getTaskStats($taskId);
                    
                    APIResponse::success([
                        'subtasks' => $subtasks,
                        'stats' => $stats
                    ]);
                }
                break;
                
            case 'POST':
                // Check if share access has edit permission
                $sharePermission = APIAuth::getSharePermission();
                if ($isShareAccess && $sharePermission !== 'edit') {
                    APIResponse::error('You do not have permission to edit this project', 403);
                }
                
                if ($id && $action === 'toggle') {
                    // POST /subtasks/{id}/toggle - Toggle completion status
                    $subtask = $subtaskModel->find($id);
                    if (!$subtask) {
                        APIResponse::error('Subtask not found', 404);
                    }
                    
                    // Verify user owns the parent task OR has share access to its project
                    $task = $taskModel->find($subtask['task_id']);
                    $hasAccess = false;
                    if ($this->userId > 0 && $task['user_id'] === $this->userId) {
                        $hasAccess = true;
                    } elseif ($isShareAccess && $task['project_id'] == $sharedProjectId) {
                        $hasAccess = true;
                    }
                    
                    if (!$task || !$hasAccess) {
                        APIResponse::error('Permission denied', 403);
                    }
                    
                    $success = $subtaskModel->toggle($id);
                    if ($success) {
                        $updated = $subtaskModel->find($id);
                        $stats = $subtaskModel->getTaskStats($subtask['task_id']);
                        
                        APIResponse::success([
                            'subtask' => $updated,
                            'stats' => $stats,
                            'message' => 'Subtask ' . ($updated['is_completed'] ? 'completed' : 'uncompleted')
                        ]);
                    } else {
                        APIResponse::error('Failed to toggle subtask', 500);
                    }
                } else if ($id && $action === 'reorder') {
                    // POST /subtasks/{task_id}/reorder - Reorder subtasks
                    $data = json_decode(file_get_contents('php://input'), true);
                    $subtaskIds = $data['subtask_ids'] ?? null;
                    
                    if (!$subtaskIds || !is_array($subtaskIds)) {
                        APIResponse::error('subtask_ids array required', 400);
                    }
                    
                    // Verify user owns the task OR has share access to its project
                    $task = $taskModel->find($id);
                    $hasAccess = false;
                    if ($this->userId > 0 && $task['user_id'] === $this->userId) {
                        $hasAccess = true;
                    } elseif ($isShareAccess && $task['project_id'] == $sharedProjectId) {
                        $hasAccess = true;
                    }
                    
                    if (!$task || !$hasAccess) {
                        APIResponse::error('Task not found or permission denied', 403);
                    }
                    
                    $success = $subtaskModel->reorder($id, $subtaskIds);
                    if ($success) {
                        APIResponse::success(['message' => 'Subtasks reordered successfully']);
                    } else {
                        APIResponse::error('Failed to reorder subtasks', 500);
                    }
                } else {
                    // POST /subtasks - Create new subtask
                    $data = json_decode(file_get_contents('php://input'), true);
                    
                    $taskId = $data['task_id'] ?? null;
                    $title = trim($data['title'] ?? '');
                    
                    if (!$taskId || !$title) {
                        APIResponse::error('task_id and title are required', 400);
                    }
                    
                    // Verify user owns the task OR has share access to its project
                    $task = $taskModel->find($taskId);
                    $hasAccess = false;
                    if ($this->userId > 0 && $task['user_id'] === $this->userId) {
                        $hasAccess = true;
                    } elseif ($isShareAccess && $task['project_id'] == $sharedProjectId) {
                        $hasAccess = true;
                    }
                    
                    if (!$task || !$hasAccess) {
                        APIResponse::error('Task not found or permission denied', 403);
                    }
                    
                    // Get max sort order for this task
                    $db = DatabaseConnection::getInstance();
                    $maxOrder = $db->selectOne(
                        "SELECT COALESCE(MAX(sort_order), -1) as max_order FROM subtasks WHERE task_id = ?",
                        [$taskId]
                    );
                    
                    $subtaskData = [
                        'task_id' => $taskId,
                        'title' => $title,
                        'is_completed' => false,
                        'sort_order' => $maxOrder['max_order'] + 1
                    ];
                    
                    $subtask = $subtaskModel->create($subtaskData);
                    if ($subtask) {
                        $stats = $subtaskModel->getTaskStats($taskId);
                        
                        APIResponse::success([
                            'subtask' => $subtask,
                            'stats' => $stats,
                            'message' => 'Subtask created successfully'
                        ], null, 201);
                    } else {
                        APIResponse::error('Failed to create subtask', 500);
                    }
                }
                break;
                
            case 'PUT':
                if (!$id) {
                    APIResponse::error('Subtask ID required', 400);
                }
                
                $subtask = $subtaskModel->find($id);
                if (!$subtask) {
                    APIResponse::error('Subtask not found', 404);
                }
                
                // Verify user owns the parent task
                $task = $taskModel->find($subtask['task_id']);
                if (!$task || $task['user_id'] !== $this->userId) {
                    APIResponse::error('Permission denied', 403);
                }
                
                $data = json_decode(file_get_contents('php://input'), true);
                $updateData = [];
                
                if (isset($data['title']) && trim($data['title'])) {
                    $updateData['title'] = trim($data['title']);
                }
                
                if (isset($data['is_completed'])) {
                    $updateData['is_completed'] = (bool)$data['is_completed'];
                }
                
                if (isset($data['sort_order'])) {
                    $updateData['sort_order'] = (int)$data['sort_order'];
                }
                
                if (empty($updateData)) {
                    APIResponse::error('No valid fields to update', 400);
                }
                
                $success = $subtaskModel->update($id, $updateData);
                if ($success) {
                    $updated = $subtaskModel->find($id);
                    $stats = $subtaskModel->getTaskStats($subtask['task_id']);
                    
                    APIResponse::success([
                        'subtask' => $updated,
                        'stats' => $stats,
                        'message' => 'Subtask updated successfully'
                    ]);
                } else {
                    APIResponse::error('Failed to update subtask', 500);
                }
                break;
                
            case 'DELETE':
                if (!$id) {
                    APIResponse::error('Subtask ID required', 400);
                }
                
                $subtask = $subtaskModel->find($id);
                if (!$subtask) {
                    APIResponse::error('Subtask not found', 404);
                }
                
                // Verify user owns the parent task
                $task = $taskModel->find($subtask['task_id']);
                if (!$task || $task['user_id'] !== $this->userId) {
                    APIResponse::error('Permission denied', 403);
                }
                
                $taskId = $subtask['task_id'];
                $success = $subtaskModel->delete($id);
                if ($success) {
                    $stats = $subtaskModel->getTaskStats($taskId);
                    
                    APIResponse::success([
                        'stats' => $stats,
                        'message' => 'Subtask deleted successfully'
                    ]);
                } else {
                    APIResponse::error('Failed to delete subtask', 500);
                }
                break;
                
            default:
                APIResponse::error('Method not allowed', 405);
        }
    }
    
    private function handleXP($id, $action) {
        $xpModel = new XPLog();
        $userModel = new User();
        
        // Handle case where URL is /xp/log or /xp/award-project-completion (no ID)
        if ($id === 'log' && $action === null) {
            $action = 'log';
            $id = null;
        } else if ($id === 'award-project-completion' && $action === null) {
            $action = 'award-project-completion';
            $id = null;
        }
        
        switch ($this->method) {
            case 'GET':
                // Guest users don't have XP tracking - return default values
                if ($this->isGuestUser()) {
                    APIResponse::success([
                        'level' => 1,
                        'xp' => 0,
                        'xp_for_next_level' => 100,
                        'xp_progress' => 0,
                        'recent_xp' => []
                    ]);
                    return;
                }
                
                $user = $userModel->find($this->userId);
                if (!$user) {
                    APIResponse::error('User not found', 404);
                }
                
                $recentXP = $xpModel->getByUser($this->userId, 10);
                
                // Get total XP from all entries
                $totalXP = $xpModel->getTotalXP($this->userId);
                
                // Scaled level calculation with tier-based progression
                // Makes it harder to predict total XP needed
                // Formula: 100 + n*(10 + tier_multiplier)
                // Tier 0 (L1-9):   multiplier = 0   -> 100+n*10  (110, 120, 130...)
                // Tier 1 (L10-19): multiplier = 10  -> 100+n*20  (300, 320, 340...)
                // Tier 2 (L20-29): multiplier = 30  -> 100+n*40  (900, 940, 980...)
                // Tier 3 (L30-39): multiplier = 70  -> 100+n*80  (2500, 2580...)
                // Each tier's multiplier increases non-linearly
                $currentLevel = 1;
                $xpAccumulator = 0;
                
                while (true) {
                    $tier = floor(($currentLevel - 1) / 10);
                    
                    // Tier multipliers: 0, 10, 30, 70, 150, 310... (roughly doubles+)
                    $tierMultiplier = 0;
                    for ($i = 0; $i < $tier; $i++) {
                        $tierMultiplier += (10 * pow(2, $i));
                    }
                    
                    $xpForNextLevel = 100 + ($currentLevel * (10 + $tierMultiplier));
                    
                    if ($xpAccumulator + $xpForNextLevel > $totalXP) {
                        break;
                    }
                    $xpAccumulator += $xpForNextLevel;
                    $currentLevel++;
                }
                
                $tier = floor(($currentLevel - 1) / 10);
                $tierMultiplier = 0;
                for ($i = 0; $i < $tier; $i++) {
                    $tierMultiplier += (10 * pow(2, $i));
                }
                $xpForCurrentLevel = 100 + ($currentLevel * (10 + $tierMultiplier));
                $xpInCurrentLevel = $totalXP - $xpAccumulator;
                $xpToNextLevel = $xpForCurrentLevel - $xpInCurrentLevel;
                $levelProgress = ($xpInCurrentLevel / $xpForCurrentLevel) * 100;
                
                APIResponse::success([
                    'total_xp' => $totalXP,
                    'current_level' => $currentLevel,
                    'xp_to_next_level' => max(0, $xpToNextLevel),
                    'xp_for_current_level' => $xpForCurrentLevel,
                    'xp_in_current_level' => $xpInCurrentLevel,
                    'level_progress' => round($levelProgress, 1),
                    'recent_xp' => $recentXP
                ]);
                break;
                
            case 'POST':
                if ($action === 'log') {
                    // Log XP manually (for achievements, etc)
                    $data = json_decode(file_get_contents('php://input'), true);
                    
                    if (!isset($data['source_type']) || !isset($data['xp_amount'])) {
                        APIResponse::error('Missing required fields: source_type, xp_amount', 400);
                    }
                    
                    $sourceType = $data['source_type'];
                    $sourceId = $data['source_id'] ?? null;
                    $xpAmount = (int) $data['xp_amount'];
                    $description = $data['description'] ?? '';
                    
                    if ($xpAmount <= 0) {
                        APIResponse::error('XP amount must be positive', 400);
                    }
                    
                    // Log the XP
                    $success = $xpModel->awardXP(
                        $this->userId,
                        $sourceType,
                        $xpAmount,
                        null, // task_id
                        $description,
                        null, // project_id
                        null  // breakdown
                    );
                    
                    if ($success) {
                        // Get updated totals
                        $totalXP = $xpModel->getTotalXP($this->userId);
                        
                        APIResponse::success([
                            'xp_logged' => $xpAmount,
                            'total_xp' => $totalXP,
                            'message' => 'XP logged successfully'
                        ], null, 201);
                    } else {
                        APIResponse::error('Failed to log XP', 500);
                    }
                } else if ($action === 'award-project-completion') {
                    $data = json_decode(file_get_contents('php://input'), true);
                    
                    if (!isset($data['project_id'])) {
                        APIResponse::error('Project ID required', 400);
                    }
                    
                    $projectModel = new Project();
                    $project = $projectModel->find($data['project_id']);
                    
                    if (!$project || $project['user_id'] !== $this->userId) {
                        APIResponse::error('Project not found', 404);
                    }
                    
                    // Check if XP was already awarded for this project completion
                    $existingXP = $xpModel->db->selectOne(
                        "SELECT * FROM xp_log WHERE user_id = ? AND action_type = 'project_completed' AND project_id = ?",
                        [$this->userId, $data['project_id']]
                    );
                    
                    if ($existingXP) {
                        // Already awarded XP for this project
                        APIResponse::success([
                            'xp_earned' => 0,
                            'message' => 'XP already awarded for this project',
                            'xp_breakdown' => []
                        ]);
                    }
                    
                    $completedTasksCount = $data['completed_tasks_count'] ?? 0;
                    
                    // Calculate project XP
                    $xpResult = $xpModel->calculateProjectXP($project, $completedTasksCount);
                    $totalXP = $xpResult['total_xp'];
                    $breakdown = $xpResult['breakdown'];
                    
                    // Award XP
                    $description = "Completed project: {$project['name']}";
                    $success = $xpModel->awardXP(
                        $this->userId,
                        'project_completed',
                        $totalXP,
                        null, // task_id
                        $description,
                        $data['project_id'], // project_id
                        $breakdown
                    );
                    
                    if ($success) {
                        APIResponse::success([
                            'xp_earned' => $totalXP,
                            'xp_breakdown' => $breakdown,
                            'message' => 'Project completion XP awarded'
                        ], null, 201);
                    } else {
                        APIResponse::error('Failed to award XP', 500);
                    }
                } else {
                    APIResponse::error('Invalid action', 404);
                }
                break;
                
            default:
                APIResponse::error('Method not allowed', 405);
        }
    }
    
    private function handleAchievements($id, $action) {
        // Guest users don't track achievements
        if ($this->isGuestUser()) {
            // Return appropriate empty responses based on the request
            if ($this->method === 'GET') {
                APIResponse::success([]);
            } else {
                APIResponse::success(['message' => 'Achievements not tracked for guest users']);
            }
            return;
        }
        
        $progressModel = new AchievementProgress();
        
        // Handle case where URL is /achievements/update (no ID)
        // In this case, $id = 'update' and $action = null
        if ($id === 'update' && $action === null) {
            $action = 'update';
            $id = null;
        } else if ($id === 'progress' && $action === null) {
            $action = 'progress';
            $id = null;
        } else if ($id === 'reset-all' && $action === null) {
            $action = 'reset-all';
            $id = null;
        } else if ($id === 'streak' && $action === null) {
            $action = 'streak';
            $id = null;
        } else if ($id === 'streak' && $action === 'set') {
            // Keep $id and $action as is for /achievements/streak/set
            // This will be handled in POST case
        }
        
        switch ($this->method) {
            case 'GET':
                if ($action === 'streak') {
                    // Get/Update streak data
                    $streak = $this->updateUserStreak();
                    APIResponse::success(['streak' => $streak]);
                } else if ($action === 'progress') {
                    // Get user's achievement progress
                    $progress = $progressModel->getByUser($this->userId);
                    APIResponse::success(['achievements' => $progress]);
                } else {
                    // Legacy: Get unlocked achievements from achievements_v2
                    $achievementModel = new Achievement();
                    $achievements = $achievementModel->getByUser($this->userId);
                    APIResponse::success($achievements);
                }
                break;
                
            case 'POST':
                if ($action === 'update') {
                    // Update achievement progress
                    $data = json_decode(file_get_contents('php://input'), true);
                    
                    if (!isset($data['achievement_key']) || !isset($data['target'])) {
                        APIResponse::error('Missing required fields: achievement_key, target', 400);
                    }
                    
                    $progress = $data['progress'] ?? 1;
                    $target = $data['target'];
                    $increment = $data['increment'] ?? false;
                    
                    if ($increment) {
                        $result = $progressModel->incrementProgress($this->userId, $data['achievement_key'], $target, $progress);
                    } else {
                        $result = $progressModel->updateProgress($this->userId, $data['achievement_key'], $progress, $target);
                    }
                    
                    if ($result) {
                        // Check if unlocked
                        $updated = $progressModel->getProgress($this->userId, $data['achievement_key']);
                        APIResponse::success([
                            'progress' => $updated,
                            'unlocked' => $updated['is_unlocked']
                        ]);
                    } else {
                        APIResponse::error('Failed to update progress', 500);
                    }
                } else if ($action === 'set' && $id === 'streak') {
                    // Manually set streak (dev tool)
                    $data = json_decode(file_get_contents('php://input'), true);
                    
                    if (!$data || !isset($data['current_streak'])) {
                        APIResponse::error('Missing required field: current_streak', 400);
                        return;
                    }
                    
                    $currentStreak = max(0, (int)$data['current_streak']);
                    $db = DatabaseConnection::getInstance();
                    
                    if (!$db) {
                        APIResponse::error('Database connection failed', 500);
                        return;
                    }
                    
                    // Check if user_streaks record exists
                    $existing = $db->selectOne(
                        "SELECT * FROM user_streaks WHERE user_id = ?",
                        [$this->userId]
                    );
                    
                    if ($existing) {
                        $longestStreak = max($currentStreak, (int)$existing['longest_streak']);
                        $db->update(
                            "UPDATE user_streaks 
                             SET current_streak = ?, longest_streak = ?, last_activity_date = CURDATE()
                             WHERE user_id = ?",
                            [$currentStreak, $longestStreak, $this->userId]
                        );
                    } else {
                        $db->insert(
                            "INSERT INTO user_streaks (user_id, current_streak, longest_streak, last_activity_date, streak_start_date) 
                             VALUES (?, ?, ?, CURDATE(), CURDATE())",
                            [$this->userId, $currentStreak, $currentStreak]
                        );
                    }
                    
                    // Check streak achievements
                    $this->checkStreakAchievements($currentStreak);
                    
                    APIResponse::success([
                        'message' => 'Streak updated',
                        'current_streak' => $currentStreak,
                        'longest_streak' => $longestStreak ?? $currentStreak
                    ]);
                } else if ($action === 'reset-all') {
                    // Reset all achievement progress for user (dev tool)
                    $db = DatabaseConnection::getInstance();
                    $result = $db->delete(
                        "DELETE FROM achievement_progress WHERE user_id = ?",
                        [$this->userId]
                    );
                    
                    APIResponse::success([
                        'message' => 'All achievement progress reset',
                        'deleted' => $result
                    ]);
                } else {
                    APIResponse::error('Invalid action', 400);
                }
                break;
                
            default:
                APIResponse::error('Method not allowed', 405);
        }
    }
    
    private function handleCalendar($id, $action) {
        try {
            $eventModel = new CalendarEvent();
            
            switch ($this->method) {
                case 'GET':
                    if ($id === 'events') {
                        // Get events with optional date range filtering
                        $startDate = $_GET['start_date'] ?? null;
                        $endDate = $_GET['end_date'] ?? null;
                        $date = $_GET['date'] ?? null;
                        
                        if ($date) {
                            // Get events for specific date
                            $events = $eventModel->getByDate($this->userId, $date);
                        } else if ($startDate && $endDate) {
                            // Get events in date range
                            $events = $eventModel->getByDateRange($this->userId, $startDate, $endDate);
                        } else {
                            // Get all events for current month by default
                            $startDate = date('Y-m-01');
                            $endDate = date('Y-m-t');
                            $events = $eventModel->getByDateRange($this->userId, $startDate, $endDate);
                        }
                    
                    APIResponse::success($events);
                } else if ($id && is_numeric($id)) {
                    // Get single event
                    $event = $eventModel->find($id);
                    if (!$event || $event['user_id'] !== $this->userId) {
                        APIResponse::error('Event not found', 404);
                    }
                    APIResponse::success($event);
                } else if ($id === 'upcoming-reminders') {
                    // Get upcoming events with reminders
                    $minutesAhead = $_GET['minutes'] ?? 60;
                    $events = $eventModel->getUpcomingWithReminders($this->userId, $minutesAhead);
                    APIResponse::success($events);
                } else {
                    APIResponse::error('Invalid endpoint', 404);
                }
                break;
                
            case 'POST':
                if ($id === 'events') {
                    // Create new event
                    $data = json_decode(file_get_contents('php://input'), true);
                    
                    if (!isset($data['title']) || !isset($data['event_date'])) {
                        APIResponse::error('Missing required fields: title, event_date', 400);
                    }
                    
                    $data['user_id'] = $this->userId;
                    
                    // Set default color if not provided
                    if (!isset($data['color'])) {
                        $data['color'] = '#3b82f6';
                    }
                    
                    $eventId = $eventModel->create($data);
                    
                    if ($eventId) {
                        $event = $eventModel->find($eventId);
                        APIResponse::success($event, null, 201);
                    } else {
                        APIResponse::error('Failed to create event', 500);
                    }
                } else {
                    APIResponse::error('Invalid endpoint', 404);
                }
                break;
                
            case 'PUT':
                if ($id && is_numeric($id)) {
                    // Update event
                    $event = $eventModel->find($id);
                    if (!$event || $event['user_id'] !== $this->userId) {
                        APIResponse::error('Event not found', 404);
                    }
                    
                    $data = json_decode(file_get_contents('php://input'), true);
                    
                    // Prevent updating user_id
                    unset($data['user_id']);
                    
                    if ($action === 'link-task') {
                        // Link event to task
                        if (!isset($data['task_id'])) {
                            APIResponse::error('Missing task_id', 400);
                        }
                        $success = $eventModel->linkToTask($id, $data['task_id']);
                    } else if ($action === 'unlink-task') {
                        // Unlink event from task
                        $success = $eventModel->unlinkFromTask($id);
                    } else {
                        // Regular update
                        $success = $eventModel->update($id, $data);
                    }
                    
                    if ($success) {
                        $updatedEvent = $eventModel->find($id);
                        APIResponse::success($updatedEvent);
                    } else {
                        APIResponse::error('Failed to update event', 500);
                    }
                } else {
                    APIResponse::error('Invalid event ID', 400);
                }
                break;
                
            case 'DELETE':
                if ($id && is_numeric($id)) {
                    // Delete event
                    $event = $eventModel->find($id);
                    if (!$event || $event['user_id'] !== $this->userId) {
                        APIResponse::error('Event not found', 404);
                    }
                    
                    $success = $eventModel->delete($id);
                    
                    if ($success) {
                        APIResponse::success(['message' => 'Event deleted successfully']);
                    } else {
                        APIResponse::error('Failed to delete event', 500);
                    }
                } else {
                    APIResponse::error('Invalid event ID', 400);
                }
                break;
                
            default:
                APIResponse::error('Method not allowed', 405);
            }
        } catch (Exception $e) {
            error_log("Calendar API Error: " . $e->getMessage());
            error_log("Calendar API Stack: " . $e->getTraceAsString());
            APIResponse::error('Failed to process calendar request: ' . $e->getMessage(), 500);
        }
    }
    
    /**
     * Update and return user streak data
     * Tracks consecutive days of activity
     */
    private function updateUserStreak() {
        // Guest users don't track streaks
        if ($this->isGuestUser()) {
            return [
                'current' => 0,
                'longest' => 0,
                'last_activity' => null,
                'streak_start' => null
            ];
        }
        
        try {
            $db = DatabaseConnection::getInstance();
            $today = date('Y-m-d');
            
            // Check if user_streaks table exists, if not return default
            $tableCheck = $db->selectOne("SHOW TABLES LIKE 'user_streaks'");
            if (!$tableCheck) {
                error_log('user_streaks table does not exist');
                return [
                    'current' => 0,
                    'longest' => 0,
                    'last_activity' => null,
                    'streak_start' => null
                ];
            }
            
            // Get current streak data
            $streak = $db->selectOne(
                "SELECT * FROM user_streaks WHERE user_id = ?",
                [$this->userId]
            );
            
            if (!$streak) {
                // First time - create streak record
                $db->insert(
                    "INSERT INTO user_streaks (user_id, current_streak, longest_streak, last_activity_date, streak_start_date) 
                     VALUES (?, 1, 1, ?, ?)",
                    [$this->userId, $today, $today]
                );
                
                return [
                    'current' => 1,
                    'longest' => 1,
                    'last_activity' => $today,
                    'streak_start' => $today
                ];
            }
            
            $lastActivity = $streak['last_activity_date'];
            $currentStreak = (int)$streak['current_streak'];
            $longestStreak = (int)$streak['longest_streak'];
            
            // If already logged in today, return current data
            if ($lastActivity === $today) {
                return [
                    'current' => $currentStreak,
                    'longest' => $longestStreak,
                    'last_activity' => $lastActivity,
                    'streak_start' => $streak['streak_start_date']
                ];
            }
            
            // Calculate days since last activity
            $lastDate = new DateTime($lastActivity);
            $currentDate = new DateTime($today);
            $daysDiff = $currentDate->diff($lastDate)->days;
            
            if ($daysDiff === 1) {
                // Consecutive day - increment streak
                $currentStreak++;
                $longestStreak = max($longestStreak, $currentStreak);
                
                $db->update(
                    "UPDATE user_streaks 
                     SET current_streak = ?, longest_streak = ?, last_activity_date = ?
                     WHERE user_id = ?",
                    [$currentStreak, $longestStreak, $today, $this->userId]
                );
            } else {
                // Streak broken - reset to 1
                $currentStreak = 1;
                
                $db->update(
                    "UPDATE user_streaks 
                     SET current_streak = 1, last_activity_date = ?, streak_start_date = ?
                     WHERE user_id = ?",
                    [$today, $today, $this->userId]
                );
            }
            
            // Check streak achievements
            $this->checkStreakAchievements($currentStreak);
            
            return [
                'current' => $currentStreak,
                'longest' => $longestStreak,
                'last_activity' => $today,
                'streak_start' => $streak['streak_start_date']
            ];
            
        } catch (Exception $e) {
            error_log('Error updating user streak: ' . $e->getMessage());
            // Return default values on error
            return [
                'current' => 0,
                'longest' => 0,
                'last_activity' => null,
                'streak_start' => null
            ];
        }
    }
    
    /**
     * Check and update streak-based achievements
     */
    private function checkStreakAchievements($currentStreak) {
        // Guest users don't track achievements
        if ($this->isGuestUser()) {
            return;
        }
        
        $progressModel = new AchievementProgress();
        
        // Streak achievement IDs and their targets
        $streakAchievements = [
            15 => 3,   // 3-day streak
            16 => 7,   // 7-day streak
            17 => 30,  // 30-day streak
            18 => 100, // 100-day streak
            19 => 365  // 365-day streak
        ];
        
        foreach ($streakAchievements as $achievementId => $target) {
            if ($currentStreak >= $target) {
                // Get the achievement key from the ID
                $db = DatabaseConnection::getInstance();
                $achievement = $db->selectOne(
                    "SELECT code FROM achievement_definitions WHERE id = ?",
                    [$achievementId]
                );
                
                if ($achievement) {
                    $progressModel->updateProgress(
                        $this->userId,
                        $achievement['code'],
                        $currentStreak,
                        $target
                    );
                }
            }
        }
    }
    
    private function handleAnalytics($id, $action) {
        if ($this->method !== 'GET') {
            APIResponse::error('Method not allowed', 405);
        }
        
        $taskModel = new Task();
        $db = DatabaseConnection::getInstance();
        
        if ($id === 'overview') {
            // Get user analytics overview
            $sql = "
                SELECT 
                    COUNT(*) as total_tasks,
                    COUNT(CASE WHEN status = 'completed' THEN 1 END) as completed_tasks,
                    COUNT(CASE WHEN status = 'pending' THEN 1 END) as pending_tasks,
                    COUNT(CASE WHEN status = 'in_progress' THEN 1 END) as in_progress_tasks,
                    COUNT(CASE WHEN status IN ('pending', 'in_progress') THEN 1 END) as active_tasks,
                    COALESCE(SUM(actual_hours), 0) as total_hours_logged,
                    COALESCE(AVG(CASE WHEN status = 'completed' THEN actual_hours END), 0) as avg_completion_time
                FROM tasks 
                WHERE user_id = ?
            ";
            
            $stats = $db->selectOne($sql, [$this->userId]);
            
            // Get user's XP from xp_log (same calculation as XP API for consistency)
            $xpModel = new XPLog();
            $stats['xp_total'] = $xpModel->getTotalXP($this->userId);
            
            if ($stats['total_tasks'] > 0) {
                $stats['completion_rate'] = round(($stats['completed_tasks'] / $stats['total_tasks']) * 100, 1);
            } else {
                $stats['completion_rate'] = 0;
            }
            
            // Get productivity score (example calculation)
            $stats['productivity_score'] = min(10, round($stats['completion_rate'] / 10, 1));
            
            // Get current streak (example)
            $stats['current_streak'] = 0; // TODO: Implement streak calculation
            $stats['best_streak'] = 0; // TODO: Implement best streak tracking
            
            APIResponse::success($stats);
        } else {
            APIResponse::error('Analytics endpoint not found', 404);
        }
    }
    
    private function handleSearch() {
        if ($this->method !== 'GET') {
            APIResponse::error('Method not allowed', 405);
        }
        
        $query = $_GET['q'] ?? '';
        $type = $_GET['type'] ?? 'all';
        $limit = min((int) ($_GET['limit'] ?? 20), 50);
        
        if (empty($query)) {
            APIResponse::error('Search query required', 400);
        }
        
        $db = DatabaseConnection::getInstance();
        $results = ['tasks' => [], 'projects' => [], 'categories' => []];
        
        $searchTerm = '%' . $query . '%';
        
        if ($type === 'all' || $type === 'tasks') {
            $sql = "
                SELECT 'task' as type, id, title as name, description, 'tasks' as table_name
                FROM tasks 
                WHERE user_id = ? AND (title LIKE ? OR description LIKE ?)
                ORDER BY title
                LIMIT ?
            ";
            $results['tasks'] = $db->select($sql, [$this->userId, $searchTerm, $searchTerm, $limit]);
        }
        
        if ($type === 'all' || $type === 'projects') {
            $sql = "
                SELECT 'project' as type, id, name, description, 'projects' as table_name
                FROM projects 
                WHERE user_id = ? AND (name LIKE ? OR description LIKE ?) AND is_archived = false
                ORDER BY name
                LIMIT ?
            ";
            $results['projects'] = $db->select($sql, [$this->userId, $searchTerm, $searchTerm, $limit]);
        }
        
        if ($type === 'all' || $type === 'categories') {
            $sql = "
                SELECT 'category' as type, id, name, description, 'categories' as table_name
                FROM categories 
                WHERE user_id = ? AND (name LIKE ? OR description LIKE ?) AND is_active = true
                ORDER BY name
                LIMIT ?
            ";
            $results['categories'] = $db->select($sql, [$this->userId, $searchTerm, $searchTerm, $limit]);
        }
        
        APIResponse::success($results);
    }
    
    private function handleUser($id, $action) {
        // Handle /user/settings as /user/{action} when no action is provided
        if ($id === 'settings' && $action === null) {
            $this->handleUserSettings();
            return;
        }
        
        // Handle /user/profile as /user/{action} when no action is provided
        if ($id === 'profile' && $action === null) {
            $this->handleUserProfile();
            return;
        }
        
        // Handle /user/stats as /user/{action} when no action is provided
        if ($id === 'stats' && $action === null) {
            $this->handleUserStats();
            return;
        }
        
        // Handle /user/xp-history as /user/{action} when no action is provided
        if ($id === 'xp-history' && $action === null) {
            $this->handleUserXPHistory();
            return;
        }
        
        switch ($action) {
            case 'settings':
                $this->handleUserSettings();
                break;
                
            case 'profile':
                $this->handleUserProfile();
                break;
                
            case 'stats':
                $this->handleUserStats();
                break;
                
            case 'xp-history':
                $this->handleUserXPHistory();
                break;
                
            default:
                APIResponse::error('User endpoint not found', 404);
        }
    }
    
    private function handleUserSettings() {
        $db = DatabaseConnection::getInstance();
        
        switch ($this->method) {
            case 'GET':
                // Guest users get default settings
                if ($this->isGuestUser()) {
                    APIResponse::success([]);
                    return;
                }
                
                // Get user settings from user.settings JSON field
                $sql = "SELECT settings FROM users WHERE id = ?";
                $result = $db->select($sql, [$this->userId]);
                
                if (empty($result)) {
                    APIResponse::error('User not found', 404);
                }
                
                $settingsJson = $result[0]['settings'] ?? '{}';
                $settings = json_decode($settingsJson, true);
                
                // If JSON decode fails, return empty object
                if (json_last_error() !== JSON_ERROR_NONE) {
                    $settings = [];
                }
                
                APIResponse::success($settings);
                break;
                
            case 'POST':
                // Guest users can't update settings
                if ($this->isGuestUser()) {
                    APIResponse::success(['message' => 'Settings not saved for guest users']);
                    return;
                }
                
                // Update user settings
                $input = json_decode(file_get_contents('php://input'), true);
                
                if (!is_array($input)) {
                    APIResponse::error('Invalid JSON data', 400);
                }
                
                try {
                    // First, get current settings
                    $sql = "SELECT settings FROM users WHERE id = ?";
                    $result = $db->select($sql, [$this->userId]);
                    
                    if (empty($result)) {
                        APIResponse::error('User not found', 404);
                    }
                    
                    // Parse existing settings
                    $currentSettingsJson = $result[0]['settings'] ?? '{}';
                    $currentSettings = json_decode($currentSettingsJson, true);
                    
                    if (json_last_error() !== JSON_ERROR_NONE) {
                        $currentSettings = [];
                    }
                    
                    // Merge new settings with existing ones
                    $updatedSettings = array_merge($currentSettings, $input);
                    
                    // Validate all setting keys
                    foreach (array_keys($input) as $key) {
                        if (!preg_match('/^[a-zA-Z0-9_]+$/', $key)) {
                            throw new Exception("Invalid setting key format: $key");
                        }
                    }
                    
                    // Update user settings
                    $settingsJson = json_encode($updatedSettings);
                    $sql = "UPDATE users SET settings = ? WHERE id = ?";
                    $db->update($sql, [$settingsJson, $this->userId]);
                    
                    APIResponse::success([
                        'message' => 'Settings updated successfully',
                        'settings' => $updatedSettings
                    ]);
                    
                } catch (Exception $e) {
                    error_log("Settings update error: " . $e->getMessage());
                    APIResponse::error('Failed to update settings: ' . $e->getMessage(), 500);
                }
                break;
                
            default:
                APIResponse::error('Method not allowed', 405);
        }
    }
    
    private function handleUserProfile() {
        if ($this->method !== 'GET') {
            APIResponse::error('Method not allowed', 405);
        }
        
        $db = DatabaseConnection::getInstance();
        
        try {
            // Get user profile information
            $sql = "SELECT id, username, email, created_at FROM users WHERE id = ?";
            $result = $db->select($sql, [$this->userId]);
            
            if (empty($result)) {
                APIResponse::error('User not found', 404);
            }
            
            $user = $result[0];
            
            // Format the response
            $profile = [
                'id' => (int) $user['id'],
                'username' => $user['username'],
                'email' => $user['email'],
                'created_at' => $user['created_at']
            ];
            
            APIResponse::success($profile);
            
        } catch (Exception $e) {
            error_log("User profile error: " . $e->getMessage());
            APIResponse::error('Failed to fetch user profile', 500);
        }
    }
    
    private function handleUserStats() {
        if ($this->method !== 'GET') {
            APIResponse::error('Method not allowed', 405);
        }
        
        $db = DatabaseConnection::getInstance();
        
        try {
            // Get total XP
            $sql = "SELECT COALESCE(SUM(xp_amount), 0) as totalXP FROM xp_log WHERE user_id = ?";
            $xpResult = $db->select($sql, [$this->userId]);
            $totalXP = (int) ($xpResult[0]['totalXP'] ?? 0);
            
            // Get total tasks completed
            $sql = "SELECT COUNT(*) as totalTasks FROM tasks WHERE user_id = ? AND is_completed = true";
            $tasksResult = $db->select($sql, [$this->userId]);
            $totalTasks = (int) ($tasksResult[0]['totalTasks'] ?? 0);
            
            // Get total projects completed
            $sql = "SELECT COUNT(*) as totalProjects FROM projects WHERE user_id = ? AND is_active = false";
            $projectsResult = $db->select($sql, [$this->userId]);
            $totalProjects = (int) ($projectsResult[0]['totalProjects'] ?? 0);
            
            // Calculate current streak (simplified - consecutive days with XP earned)
            $sql = "SELECT DATE(earned_at) as earned_date 
                   FROM xp_log 
                   WHERE user_id = ? 
                   ORDER BY earned_at DESC 
                   LIMIT 30";
            $streakResult = $db->select($sql, [$this->userId]);
            
            $currentStreak = 0;
            $lastDate = null;
            
            foreach ($streakResult as $row) {
                $earnedDate = new DateTime($row['earned_date']);
                $today = new DateTime();
                $today->setTime(0, 0, 0);
                
                if ($lastDate === null) {
                    // First iteration
                    if ($earnedDate->format('Y-m-d') === $today->format('Y-m-d') || 
                        $earnedDate->format('Y-m-d') === $today->sub(new DateInterval('P1D'))->format('Y-m-d')) {
                        $currentStreak = 1;
                        $lastDate = $earnedDate;
                    } else {
                        break;
                    }
                } else {
                    // Check if this date is consecutive
                    $expectedDate = clone $lastDate;
                    $expectedDate->sub(new DateInterval('P1D'));
                    
                    if ($earnedDate->format('Y-m-d') === $expectedDate->format('Y-m-d')) {
                        $currentStreak++;
                        $lastDate = $earnedDate;
                    } else {
                        break;
                    }
                }
            }
            
            $stats = [
                'totalXP' => $totalXP,
                'totalTasksCompleted' => $totalTasks,
                'totalProjectsCompleted' => $totalProjects,
                'currentStreak' => $currentStreak
            ];
            
            APIResponse::success($stats);
            
        } catch (Exception $e) {
            error_log("User stats error: " . $e->getMessage());
            APIResponse::error('Failed to fetch user statistics', 500);
        }
    }
    
    private function handleUserXPHistory() {
        if ($this->method !== 'GET') {
            APIResponse::error('Method not allowed', 405);
        }
        
        $db = DatabaseConnection::getInstance();
        
        try {
            // Get recent XP history (last 20 entries)
            $sql = "SELECT xp_gained, action_type, created_at, description, breakdown, task_id, project_id 
                   FROM xp_log 
                   WHERE user_id = ? 
                   ORDER BY created_at DESC 
                   LIMIT 20";
            $result = $db->select($sql, [$this->userId]);
            
            // Format the results
            $history = [];
            foreach ($result as $entry) {
                $historyItem = [
                    'xp_amount' => (int) $entry['xp_gained'],
                    'action_type' => $entry['action_type'],
                    'earned_at' => $entry['created_at'],
                    'description' => $entry['description'],
                    'task_id' => $entry['task_id'],
                    'project_id' => $entry['project_id']
                ];
                
                // Add breakdown if available
                if ($entry['breakdown']) {
                    $historyItem['breakdown'] = json_decode($entry['breakdown'], true);
                }
                
                $history[] = $historyItem;
            }
            
            APIResponse::success($history);
            
        } catch (Exception $e) {
            error_log("User XP history error: " . $e->getMessage());
            APIResponse::error('Failed to fetch XP history', 500);
        }
    }
}

// Initialize and route the request
try {
    $router = new APIRouter();
    $router->route();
} catch (Exception $e) {
    error_log("API Router Error: " . $e->getMessage());
    APIResponse::error('Internal server error', 500);
}

