<?php
/**
 * Focus Mode API Endpoint
 * 
 * Handles focus mode operations including:
 * - Getting priority tasks for today
 * - Logging focus sessions
 * - Tracking daily streaks
 * 
 * @author Jack Ewers / BloodWeb
 * @version 1.0.0
 */

// Enable error reporting for development
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Set JSON response headers
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Handle preflight OPTIONS requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Include dependencies
require_once 'Database.php';
require_once 'Models.php';
require_once $_SERVER['DOCUMENT_ROOT'] . '/auth/session_bootstrap.php';

/**
 * FocusMode API Handler
 */
class FocusModeAPI {
    private $db;
    private $userId;
    
    public function __construct() {
        $this->db = DatabaseConnection::getInstance()->getPDO();
        $this->userId = $this->authenticate();
        
        if (!$this->userId) {
            $this->error('Authentication required', 401);
        }
    }
    
    private function authenticate() {
        // Check for test mode
        if (isset($_GET['test']) && $_GET['test'] === '1') {
            return 3; // Development user ID
        }
        
        // Check session
        if (session_status() === PHP_SESSION_ACTIVE && isset($_SESSION['user_id'])) {
            return $_SESSION['user_id'];
        }
        
        return null;
    }
    
    public function handleRequest() {
        $action = $_GET['action'] ?? null;
        $method = $_SERVER['REQUEST_METHOD'];
        
        try {
            switch ($action) {
                case 'focus-tasks':
                    if ($method === 'GET') {
                        $this->getFocusTasks();
                    } else {
                        $this->error('Method not allowed', 405);
                    }
                    break;
                    
                case 'log-session':
                    if ($method === 'POST') {
                        $this->logSession();
                    } else {
                        $this->error('Method not allowed', 405);
                    }
                    break;
                    
                case 'streak':
                    if ($method === 'GET') {
                        $this->getStreak();
                    } else {
                        $this->error('Method not allowed', 405);
                    }
                    break;
                    
                default:
                    $this->error('Invalid action', 400);
            }
        } catch (Exception $e) {
            error_log("Focus API Error: " . $e->getMessage());
            $this->error('Internal server error', 500);
        }
    }
    
    /**
     * Get top 5 priority tasks for today
     */
    private function getFocusTasks() {
        $today = date('Y-m-d');
        
        $sql = "SELECT 
                    id,
                    title,
                    description,
                    status,
                    priority,
                    due_date,
                    project_id,
                    category_id,
                    created_at,
                    updated_at
                FROM tasks
                WHERE user_id = :user_id
                    AND status != 'completed'
                    AND status != 'cancelled'
                    AND (due_date IS NULL OR due_date <= :today)
                ORDER BY 
                    CASE priority
                        WHEN 'urgent' THEN 1
                        WHEN 'high' THEN 2
                        WHEN 'medium' THEN 3
                        WHEN 'low' THEN 4
                        ELSE 5
                    END,
                    due_date ASC,
                    created_at ASC
                LIMIT 5";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute([
            ':user_id' => $this->userId,
            ':today' => $today
        ]);
        
        $tasks = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $this->success(['tasks' => $tasks]);
    }
    
    /**
     * Log a focus session
     */
    private function logSession() {
        $input = json_decode(file_get_contents('php://input'), true);
        
        if (!$input) {
            $this->error('Invalid JSON', 400);
        }
        
        $duration = $input['duration'] ?? null;
        $completedTasks = $input['completed_tasks'] ?? [];
        $xpEarned = $input['xp_earned'] ?? 0;
        
        if (!$duration || !is_numeric($duration)) {
            $this->error('duration (in seconds) required', 400);
        }
        
        // Create focus_sessions table if it doesn't exist
        $this->createFocusSessionsTable();
        
        // Insert session log
        $sql = "INSERT INTO focus_sessions 
                (user_id, duration, completed_tasks, xp_earned, session_date, created_at)
                VALUES (:user_id, :duration, :completed_tasks, :xp_earned, :session_date, NOW())";
        
        $stmt = $this->db->prepare($sql);
        $success = $stmt->execute([
            ':user_id' => $this->userId,
            ':duration' => (int)$duration,
            ':completed_tasks' => json_encode($completedTasks),
            ':xp_earned' => (int)$xpEarned,
            ':session_date' => date('Y-m-d')
        ]);
        
        if (!$success) {
            $this->error('Failed to log session', 500);
        }
        
        $sessionId = $this->db->lastInsertId();
        
        // Update user XP
        $xpModel = new XP();
        $xpModel->addXP($this->userId, $xpEarned, 'focus_session', "Focus session completed");
        
        // Get today's session count and total XP
        $statsQuery = "SELECT 
                        COUNT(*) as session_count,
                        SUM(xp_earned) as total_xp,
                        SUM(duration) as total_duration
                       FROM focus_sessions
                       WHERE user_id = :user_id
                       AND session_date = :today";
        
        $statsStmt = $this->db->prepare($statsQuery);
        $statsStmt->execute([
            ':user_id' => $this->userId,
            ':today' => date('Y-m-d')
        ]);
        
        $stats = $statsStmt->fetch(PDO::FETCH_ASSOC);
        
        $this->success([
            'session_id' => $sessionId,
            'xp_earned' => $xpEarned,
            'today_sessions' => (int)$stats['session_count'],
            'today_xp' => (int)$stats['total_xp'],
            'today_duration' => (int)$stats['total_duration']
        ]);
    }
    
    /**
     * Get user's daily streak
     */
    private function getStreak() {
        // Create focus_sessions table if it doesn't exist
        $this->createFocusSessionsTable();
        
        // Get distinct dates where user had sessions
        $sql = "SELECT DISTINCT session_date
                FROM focus_sessions
                WHERE user_id = :user_id
                ORDER BY session_date DESC
                LIMIT 365";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute([':user_id' => $this->userId]);
        
        $dates = $stmt->fetchAll(PDO::FETCH_COLUMN);
        
        // Calculate current streak
        $currentStreak = 0;
        $today = date('Y-m-d');
        $yesterday = date('Y-m-d', strtotime('-1 day'));
        
        if (!empty($dates)) {
            $mostRecent = $dates[0];
            
            // Only count if session was today or yesterday
            if ($mostRecent === $today || $mostRecent === $yesterday) {
                $currentStreak = 1;
                $expectedDate = date('Y-m-d', strtotime($mostRecent . ' -1 day'));
                
                for ($i = 1; $i < count($dates); $i++) {
                    if ($dates[$i] === $expectedDate) {
                        $currentStreak++;
                        $expectedDate = date('Y-m-d', strtotime($expectedDate . ' -1 day'));
                    } else {
                        break;
                    }
                }
            }
        }
        
        // Calculate longest streak
        $longestStreak = 0;
        $tempStreak = 0;
        
        if (!empty($dates)) {
            $tempStreak = 1;
            
            for ($i = 0; $i < count($dates) - 1; $i++) {
                $current = strtotime($dates[$i]);
                $next = strtotime($dates[$i + 1]);
                $diff = ($current - $next) / 86400; // Days difference
                
                if ($diff === 1) {
                    $tempStreak++;
                } else {
                    $longestStreak = max($longestStreak, $tempStreak);
                    $tempStreak = 1;
                }
            }
            
            $longestStreak = max($longestStreak, $tempStreak);
        }
        
        // Get total sessions
        $totalQuery = "SELECT COUNT(*) as total_sessions,
                              SUM(duration) as total_duration,
                              SUM(xp_earned) as total_xp
                       FROM focus_sessions
                       WHERE user_id = :user_id";
        
        $totalStmt = $this->db->prepare($totalQuery);
        $totalStmt->execute([':user_id' => $this->userId]);
        $totals = $totalStmt->fetch(PDO::FETCH_ASSOC);
        
        $this->success([
            'current_streak' => $currentStreak,
            'longest_streak' => $longestStreak,
            'total_sessions' => (int)$totals['total_sessions'],
            'total_duration' => (int)$totals['total_duration'],
            'total_xp' => (int)$totals['total_xp'],
            'has_session_today' => !empty($dates) && $dates[0] === $today
        ]);
    }
    
    /**
     * Create focus_sessions table if it doesn't exist
     */
    private function createFocusSessionsTable() {
        $sql = "CREATE TABLE IF NOT EXISTS focus_sessions (
            id INT AUTO_INCREMENT PRIMARY KEY,
            user_id INT NOT NULL,
            duration INT NOT NULL COMMENT 'Duration in seconds',
            completed_tasks JSON DEFAULT NULL,
            xp_earned INT DEFAULT 0,
            session_date DATE NOT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            INDEX idx_user_date (user_id, session_date),
            INDEX idx_session_date (session_date)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        $this->db->exec($sql);
    }
    
    private function success($data = null, $code = 200) {
        http_response_code($code);
        echo json_encode([
            'success' => true,
            'data' => $data
        ], JSON_PRETTY_PRINT);
        exit();
    }
    
    private function error($message, $code = 400) {
        http_response_code($code);
        echo json_encode([
            'success' => false,
            'error' => [
                'message' => $message,
                'code' => $code
            ]
        ], JSON_PRETTY_PRINT);
        exit();
    }
}

// Initialize and handle request
$api = new FocusModeAPI();
$api->handleRequest();
