<?php
/**
 * Model Classes for TaskList v2
 * 
 * These classes represent the main entities in the TaskList application
 * and provide methods for database operations.
 */

require_once 'Database.php';

/**
 * User Model
 */
class User extends BaseModel {
    protected $table = 'users';
    protected $fillable = [
        'username', 'email', 'password_hash', 'first_name', 'last_name',
        'imgID', 'phonenumber', 'birthday', 'settings'
    ];
    protected $hidden = ['password_hash'];
    protected $casts = [
        'settings' => 'json',
        'is_active' => 'bool'
    ];
    
    public function findByUsername($username) {
        $sql = "SELECT * FROM `{$this->table}` WHERE `username` = ?";
        $result = $this->db->selectOne($sql, [$username]);
        return $result ? $this->cast($result) : null;
    }
    
    public function findByEmail($email) {
        $sql = "SELECT * FROM `{$this->table}` WHERE `email` = ?";
        $result = $this->db->selectOne($sql, [$email]);
        return $result ? $this->cast($result) : null;
    }
    
    public function verifyPassword($userId, $password) {
        $sql = "SELECT password_hash FROM `{$this->table}` WHERE `id` = ?";
        $result = $this->db->selectOne($sql, [$userId]);
        
        if ($result) {
            return password_verify($password, $result['password_hash']);
        }
        
        return false;
    }
    
    public function updateXP($userId, $xpAmount) {
        $sql = "UPDATE `{$this->table}` SET 
                `xp_total` = `xp_total` + ?
                WHERE `id` = ?";
        
        return $this->db->update($sql, [$xpAmount, $userId]);
    }
}

/**
 * Project Model
 */
class Project extends BaseModel {
    protected $table = 'projects';
    protected $fillable = [
        'user_id', 'name', 'description', 'color', 'icon', 'sort_order', 'is_archived', 'is_completed'
    ];
    protected $casts = [
        'user_id' => 'int',
        'sort_order' => 'int',
        'is_archived' => 'bool',
        'is_completed' => 'bool'
    ];
    
    public function getByUser($userId, $includeArchived = false, $includeCompleted = true) {
        $where = ['user_id = ?'];
        $params = [$userId];
        
        if (!$includeArchived) {
            $where[] = 'is_archived = ?';
            $params[] = false;
        }
        
        // Order by completion status (active first, then completed), then by sort_order and name
        $orderBy = 'is_completed ASC, sort_order ASC, name ASC';
        
        $projects = $this->all($where, $params, $orderBy);
        
        // Add category IDs and sharing status to each project
        foreach ($projects as &$project) {
            $categoryIds = $this->db->select(
                "SELECT category_id FROM project_categories WHERE project_id = ?",
                [$project['id']]
            );
            $project['category_ids'] = array_map('intval', array_column($categoryIds, 'category_id'));
            
            // Check if project has any active shares (user shares or public links)
            $shareCount = $this->db->selectOne(
                "SELECT 
                    (SELECT COUNT(*) FROM project_shares WHERE project_id = ? AND is_accepted = 1) +
                    (SELECT COUNT(*) FROM share_tokens WHERE project_id = ? AND is_active = 1) as total",
                [$project['id'], $project['id']]
            );
            $project['is_shared'] = ($shareCount['total'] ?? 0) > 0;
        }
        
        return $projects;
    }
    
    public function getWithStats($userId, $projectId = null) {
        $sql = "
            SELECT 
                p.*,
                COALESCE(ps.total_tasks, 0) as total_tasks,
                COALESCE(ps.completed_tasks, 0) as completed_tasks,
                COALESCE(ps.pending_tasks, 0) as pending_tasks,
                COALESCE(ps.in_progress_tasks, 0) as in_progress_tasks,
                COALESCE(ps.total_estimated_hours, 0) as total_estimated_hours,
                COALESCE(ps.total_actual_hours, 0) as total_actual_hours,
                CASE 
                    WHEN ps.total_tasks > 0 THEN ROUND((ps.completed_tasks / ps.total_tasks) * 100, 1)
                    ELSE 0 
                END as completion_percentage
            FROM projects p
            LEFT JOIN project_stats ps ON p.id = ps.id
            WHERE p.user_id = ?
        ";
        
        $params = [$userId];
        
        if ($projectId) {
            $sql .= " AND p.id = ?";
            $params[] = $projectId;
            $project = $this->db->selectOne($sql, $params);
            
            if ($project) {
                // Add category IDs to the project
                $categoryIds = $this->db->select(
                    "SELECT category_id FROM project_categories WHERE project_id = ?",
                    [$project['id']]
                );
                $project['category_ids'] = array_map('intval', array_column($categoryIds, 'category_id'));
                
                // Check if project has any active shares
                $shareCount = $this->db->selectOne(
                    "SELECT 
                        (SELECT COUNT(*) FROM project_shares WHERE project_id = ? AND is_accepted = 1) +
                        (SELECT COUNT(*) FROM share_tokens WHERE project_id = ? AND is_active = 1) as total",
                    [$project['id'], $project['id']]
                );
                $project['is_shared'] = ($shareCount['total'] ?? 0) > 0;
            }
            
            return $project;
        } else {
            $sql .= " ORDER BY p.sort_order ASC, p.name ASC";
            $projects = $this->db->select($sql, $params);
            
            // Add category IDs and sharing status to each project
            foreach ($projects as &$project) {
                $categoryIds = $this->db->select(
                    "SELECT category_id FROM project_categories WHERE project_id = ?",
                    [$project['id']]
                );
                $project['category_ids'] = array_map('intval', array_column($categoryIds, 'category_id'));
                
                // Check if project has any active shares
                $shareCount = $this->db->selectOne(
                    "SELECT 
                        (SELECT COUNT(*) FROM project_shares WHERE project_id = ? AND is_accepted = 1) +
                        (SELECT COUNT(*) FROM share_tokens WHERE project_id = ? AND is_active = 1) as total",
                    [$project['id'], $project['id']]
                );
                $project['is_shared'] = ($shareCount['total'] ?? 0) > 0;
            }
            
            return $projects;
        }
    }
    
    public function getCategories($projectId) {
        $sql = "
            SELECT c.*
            FROM categories c
            INNER JOIN project_categories pc ON c.id = pc.category_id
            WHERE pc.project_id = ?
            ORDER BY c.sort_order ASC, c.name ASC
        ";
        
        return $this->db->select($sql, [$projectId]);
    }
    
    public function addCategory($projectId, $categoryId) {
        $sql = "INSERT IGNORE INTO project_categories (project_id, category_id) VALUES (?, ?)";
        return $this->db->insert($sql, [$projectId, $categoryId]);
    }
    
    public function removeCategory($projectId, $categoryId) {
        $sql = "DELETE FROM project_categories WHERE project_id = ? AND category_id = ?";
        return $this->db->delete($sql, [$projectId, $categoryId]);
    }
    
    /**
     * Check if project is completed and award XP if newly completed
     */
    public function checkAndAwardProjectCompletion($projectId, $userId) {
        // Get project details
        $project = $this->find($projectId);
        if (!$project || $project['user_id'] !== $userId) {
            return false;
        }
        
        // Check if all tasks in project are completed
        $sql = "
            SELECT 
                COUNT(*) as total_tasks,
                COUNT(CASE WHEN status = 'completed' THEN 1 END) as completed_tasks
            FROM tasks 
            WHERE project_id = ? AND user_id = ?
        ";
        
        $taskStats = $this->db->selectOne($sql, [$projectId, $userId]);
        
        if ($taskStats['total_tasks'] == 0) {
            return false; // No tasks in project
        }
        
        // Check if project is now complete
        $isProjectComplete = $taskStats['completed_tasks'] == $taskStats['total_tasks'];
        
        if (!$isProjectComplete) {
            return false;
        }
        
        // Check if we've already awarded XP for this project completion
        $xpLogModel = new XPLog();
        $existingAward = $this->db->selectOne(
            "SELECT id FROM xp_log WHERE project_id = ? AND action_type = 'project_completed' AND user_id = ?",
            [$projectId, $userId]
        );
        
        if ($existingAward) {
            return false; // Already awarded
        }
        
        // Calculate and award project completion XP
        $xpResult = $xpLogModel->calculateProjectXP($project, $taskStats['completed_tasks']);
        
        // Create clean description
        $description = "Completed Project: {$project['name']}";
        
        // Award XP
        $xpLogModel->awardXP($userId, 'project_completed', $xpResult['total_xp'], null, $description, $projectId, $xpResult['breakdown']);
        
        return [
            'project_completed' => true,
            'xp_earned' => $xpResult['total_xp'],
            'xp_breakdown' => $xpResult['breakdown']
        ];
    }
}

/**
 * Category Model
 */
class Category extends BaseModel {
    protected $table = 'categories';
    protected $fillable = [
        'user_id', 'name', 'description', 'emoji', 'color', 'sort_order', 'is_active', 'auto_bind'
    ];
    protected $casts = [
        'user_id' => 'int',
        'sort_order' => 'int',
        'is_active' => 'bool',
        'auto_bind' => 'bool'
    ];
    
    public function getByUser($userId, $activeOnly = true) {
        $where = ['user_id = ?'];
        $params = [$userId];
        
        if ($activeOnly) {
            $where[] = 'is_active = ?';
            $params[] = true;
        }
        
        return $this->all($where, $params, 'sort_order ASC, name ASC');
    }
    
    public function getByProject($projectId) {
        $sql = "
            SELECT c.*
            FROM categories c
            INNER JOIN project_categories pc ON c.id = pc.category_id
            WHERE pc.project_id = ? AND c.is_active = true
            ORDER BY c.sort_order ASC, c.name ASC
        ";
        
        return $this->db->select($sql, [$projectId]);
    }
    
    public function getProjects($categoryId) {
        $sql = "
            SELECT p.*
            FROM projects p
            INNER JOIN project_categories pc ON p.id = pc.project_id
            WHERE pc.category_id = ? AND p.is_archived = false
            ORDER BY p.sort_order ASC, p.name ASC
        ";
        
        return $this->db->select($sql, [$categoryId]);
    }
    
    public function updateProjectBindings($categoryId, $projectIds, $userId) {
        try {
            // Begin transaction
            $this->db->beginTransaction();
            
            // First, verify the category belongs to the user
            $category = $this->find($categoryId);
            if (!$category || $category['user_id'] !== $userId) {
                throw new Exception('Category not found or access denied');
            }
            
            // Remove all existing bindings for this category
            $sql = "DELETE FROM project_categories WHERE category_id = ?";
            $this->db->delete($sql, [$categoryId]);
            
            // Add new bindings, but only for projects owned by the user
            foreach ($projectIds as $projectId) {
                // Verify project belongs to user
                $project = $this->db->selectOne(
                    "SELECT id FROM projects WHERE id = ? AND user_id = ?",
                    [$projectId, $userId]
                );
                
                if ($project) {
                    $sql = "INSERT INTO project_categories (project_id, category_id) VALUES (?, ?)";
                    $this->db->insert($sql, [$projectId, $categoryId]);
                }
            }
            
            $this->db->commit();
            return true;
            
        } catch (Exception $e) {
            $this->db->rollback();
            error_log("Failed to update project bindings: " . $e->getMessage());
            return false;
        }
    }
}

/**
 * Task Model
 */
class Task extends BaseModel {
    protected $table = 'tasks';
    protected $fillable = [
        'user_id', 'project_id', 'category_id', 'title', 'description',
        'priority', 'complexity', 'status', 'due_date', 'estimated_hours', 'actual_hours',
        'tags', 'sort_order', 'onboarding_action'
    ];
    protected $casts = [
        'user_id' => 'int',
        'project_id' => 'int',
        'category_id' => 'int',
        'estimated_hours' => 'float',
        'actual_hours' => 'float',
        'tags' => 'json',
        'sort_order' => 'int'
    ];
    
    public function getByUser($userId, $filters = [], $page = 1, $perPage = 20) {
        $where = ['t.user_id = ?'];
        $params = [$userId];
        
        // Apply filters
        if (!empty($filters['project_id'])) {
            $where[] = 't.project_id = ?';
            $params[] = $filters['project_id'];
        }
        
        if (!empty($filters['category_id'])) {
            $where[] = 't.category_id = ?';
            $params[] = $filters['category_id'];
        }
        
        if (!empty($filters['status'])) {
            $where[] = 't.status = ?';
            $params[] = $filters['status'];
        }
        
        if (!empty($filters['priority'])) {
            $where[] = 't.priority = ?';
            $params[] = $filters['priority'];
        }
        
        if (!empty($filters['due_before'])) {
            $where[] = 't.due_date <= ?';
            $params[] = $filters['due_before'];
        }
        
        if (!empty($filters['due_after'])) {
            $where[] = 't.due_date >= ?';
            $params[] = $filters['due_after'];
        }
        
        if (!empty($filters['search'])) {
            $where[] = '(t.title LIKE ? OR t.description LIKE ?)';
            $searchTerm = '%' . $filters['search'] . '%';
            $params[] = $searchTerm;
            $params[] = $searchTerm;
        }
        
        // Build ORDER BY clause
        $orderBy = 't.sort_order ASC, ';
        if (!empty($filters['sort'])) {
            $sortField = $filters['sort'];
            $sortOrder = !empty($filters['order']) && strtolower($filters['order']) === 'desc' ? 'DESC' : 'ASC';
            $orderBy = "t.$sortField $sortOrder, ";
        }
        $orderBy .= 't.created_at DESC';
        
        // Use task_details view for complete information
        $sql = "
            SELECT 
                t.*,
                p.name as project_name,
                p.color as project_color,
                c.name as category_name,
                c.emoji as category_emoji,
                c.color as category_color
            FROM tasks t
            INNER JOIN projects p ON t.project_id = p.id
            INNER JOIN categories c ON t.category_id = c.id
        ";
        
        if (!empty($where)) {
            $sql .= " WHERE " . implode(" AND ", $where);
        }
        
        $sql .= " ORDER BY $orderBy";
        
        // Manual pagination for complex query
        $offset = ($page - 1) * $perPage;
        $sql .= " LIMIT $perPage OFFSET $offset";
        
        $results = $this->db->select($sql, $params);
        
        // Get total count
        $countSql = "
            SELECT COUNT(*)
            FROM tasks t
            INNER JOIN projects p ON t.project_id = p.id
            INNER JOIN categories c ON t.category_id = c.id
        ";
        if (!empty($where)) {
            $countSql .= " WHERE " . implode(" AND ", $where);
        }
        
        $stmt = $this->db->getPDO()->prepare($countSql);
        $stmt->execute($params);
        $total = (int) $stmt->fetchColumn();
        
        return [
            'data' => $results,
            'meta' => [
                'total' => $total,
                'page' => $page,
                'per_page' => $perPage,
                'total_pages' => ceil($total / $perPage),
                'has_next' => $page < ceil($total / $perPage),
                'has_prev' => $page > 1
            ]
        ];
    }
    
    public function markComplete($taskId, $userId) {
        $sql = "UPDATE `{$this->table}` 
                SET `status` = 'completed', `completed_at` = NOW(), `updated_at` = NOW()
                WHERE `id` = ? AND `user_id` = ?";
        
        return $this->db->update($sql, [$taskId, $userId]) > 0;
    }
    
    public function markInProgress($taskId, $userId) {
        $sql = "UPDATE `{$this->table}` 
                SET `status` = 'in_progress', `updated_at` = NOW()
                WHERE `id` = ? AND `user_id` = ?";
        
        return $this->db->update($sql, [$taskId, $userId]) > 0;
    }
    
    public function updateActualHours($taskId, $hours, $userId) {
        $sql = "UPDATE `{$this->table}` 
                SET `actual_hours` = ?, `updated_at` = NOW()
                WHERE `id` = ? AND `user_id` = ?";
        
        return $this->db->update($sql, [$hours, $taskId, $userId]) > 0;
    }
    
    public function getDueSoon($userId, $days = 7) {
        $sql = "
            SELECT t.*, p.name as project_name, c.name as category_name
            FROM tasks t
            INNER JOIN projects p ON t.project_id = p.id
            INNER JOIN categories c ON t.category_id = c.id
            WHERE t.user_id = ? 
            AND t.status IN ('pending', 'in_progress')
            AND t.due_date BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL ? DAY)
            ORDER BY t.due_date ASC, t.priority DESC
        ";
        
        return $this->db->select($sql, [$userId, $days]);
    }
    
    public function getOverdue($userId) {
        $sql = "
            SELECT t.*, p.name as project_name, c.name as category_name
            FROM tasks t
            INNER JOIN projects p ON t.project_id = p.id
            INNER JOIN categories c ON t.category_id = c.id
            WHERE t.user_id = ? 
            AND t.status IN ('pending', 'in_progress')
            AND t.due_date < CURDATE()
            ORDER BY t.due_date ASC, t.priority DESC
        ";
        
        return $this->db->select($sql, [$userId]);
    }
}

/**
 * XP Log Model
 */
class XPLog extends BaseModel {
    protected $table = 'xp_log';
    protected $fillable = [
        'user_id', 'action_type', 'xp_gained', 'task_id', 'project_id', 'description', 'breakdown'
    ];
    protected $casts = [
        'user_id' => 'int',
        'xp_gained' => 'int',
        'task_id' => 'int',
        'project_id' => 'int',
        'breakdown' => 'json'
    ];
    
    // XP Configuration Constants
    const BASE_XP = [
        'task' => 10,
        'project' => 100
    ];
    
    const IDEAL_RANGES = [
        'task' => [
            'title' => ['min' => 3, 'max' => 8],
            'description' => ['min' => 10, 'max' => 50]
        ],
        'project' => [
            'description' => ['min' => 20, 'max' => 100]
        ]
    ];
    
    public function getByUser($userId, $limit = 20) {
        $where = ['user_id = ?'];
        $params = [$userId];
        
        return $this->all($where, $params, 'created_at DESC', $limit);
    }
    
    public function getTotalXP($userId) {
        $sql = "SELECT COALESCE(SUM(xp_gained), 0) FROM `{$this->table}` WHERE user_id = ?";
        $stmt = $this->db->getPDO()->prepare($sql);
        $stmt->execute([$userId]);
        return (int) $stmt->fetchColumn();
    }
    
    public function awardXP($userId, $actionType, $xpAmount, $taskId = null, $description = null, $projectId = null, $breakdown = null) {
        // Create the XP log entry with breakdown
        $logData = [
            'user_id' => $userId,
            'action_type' => $actionType,
            'xp_gained' => $xpAmount,
            'task_id' => $taskId,
            'project_id' => $projectId,
            'description' => $description,
            'breakdown' => $breakdown
        ];
        
        try {
            $logId = $this->create($logData);
            
            if ($logId) {
                // Update user's total XP
                $userModel = new User();
                $userModel->updateXP($userId, $xpAmount);
                return true;
            }
            
            return false;
        } catch (Exception $e) {
            error_log("XP Award Error: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Calculate XP for task completion based on various factors
     */
    public function calculateTaskXP($task, $project = null, $category = null) {
        $breakdown = [];
        
        // Base XP for task completion
        $baseXP = self::BASE_XP['task'];
        $breakdown['base'] = $baseXP;
        
        // Title score (0-3 points based on how close to ideal range)
        $titleScore = $this->calculateRangeScore($task['title'], 'title');
        if ($titleScore > 0) {
            $breakdown['title_bonus'] = $titleScore;
        }
        
        // Description score (0-3 points based on how close to ideal range)
        $descScore = $this->calculateRangeScore($task['description'] ?? '', 'description');
        if ($descScore > 0) {
            $breakdown['description_bonus'] = $descScore;
        }
        
        // Category bonus (+1 if task has a category - simplified)
        if ($task['category_id'] && $task['category_id'] > 0) {
            $breakdown['category_bonus'] = 1;
        }
        
        // Early completion bonus (+2 if completed before due date)
        if ($task['due_date'] && strtotime($task['due_date']) > time()) {
            $breakdown['early_completion'] = 2;
        }
        
        // Complexity bonus (0-4 points)
        $complexityBonus = $this->getComplexityBonus($task['complexity'] ?? 'none');
        if ($complexityBonus > 0) {
            $breakdown['complexity_bonus'] = $complexityBonus;
        }
        
        // Priority bonus (1-4 points)
        $priorityBonus = $this->getPriorityBonus($task['priority'] ?? 'medium');
        if ($priorityBonus > 1) { // Only show bonus if above base priority
            $breakdown['priority_bonus'] = $priorityBonus - 1; // Show bonus above base
        }
        
        // Calculate base total XP
        $baseTotal = array_sum($breakdown);
        
        // Apply achievement multiplier: 1 + (achievements_unlocked * 0.01)
        $achievementMultiplier = $this->getAchievementMultiplier($task['user_id']);
        $totalXP = round($baseTotal * $achievementMultiplier);
        
        if ($achievementMultiplier > 1.0) {
            $breakdown['achievement_multiplier'] = sprintf('%.2fx', $achievementMultiplier);
            $breakdown['multiplied_total'] = $totalXP;
        }
        
        return [
            'total_xp' => $totalXP,
            'breakdown' => $breakdown
        ];
    }
    
    /**
     * Get achievement multiplier based on unlocked achievements
     * Formula: 1 + (unlocked_achievements * 0.01)
     * Example: 10 achievements = 1.10x multiplier
     */
    private function getAchievementMultiplier($userId) {
        try {
            $db = DatabaseConnection::getInstance();
            $sql = "SELECT COUNT(*) as count FROM achievement_progress WHERE user_id = ? AND is_unlocked = 1";
            $result = $db->selectOne($sql, [$userId]);
            $unlockedCount = $result['count'] ?? 0;
            return 1.0 + ($unlockedCount * 0.01);
        } catch (Exception $e) {
            error_log("Error getting achievement multiplier: " . $e->getMessage());
            return 1.0; // Default to no multiplier on error
        }
    }
    
    /**
     * Calculate XP for project completion
     */
    public function calculateProjectXP($project, $completedTasksCount) {
        $breakdown = [];
        
        // Base XP for project completion
        $baseXP = self::BASE_XP['project'];
        $breakdown['base'] = $baseXP;
        
        // Project description score (0-10 points based on how close to ideal range)
        $descScore = $this->calculateProjectRangeScore($project['description'] ?? '');
        if ($descScore > 0) {
            $breakdown['description_bonus'] = $descScore;
        }
        
        // Task completion bonus (completed_tasks / 10, rounded up)
        $taskBonus = ceil($completedTasksCount / 10);
        if ($taskBonus > 0) {
            $breakdown['task_completion_bonus'] = $taskBonus;
        }
        
        // Calculate base total XP
        $baseTotal = array_sum($breakdown);
        
        // Apply achievement multiplier
        $achievementMultiplier = $this->getAchievementMultiplier($project['user_id']);
        $totalXP = round($baseTotal * $achievementMultiplier);
        
        if ($achievementMultiplier > 1.0) {
            $breakdown['achievement_multiplier'] = sprintf('%.2fx', $achievementMultiplier);
            $breakdown['multiplied_total'] = $totalXP;
        }
        
        return [
            'total_xp' => $totalXP,
            'breakdown' => $breakdown
        ];
    }
    
    /**
     * Calculate score based on how close word count is to ideal range
     * Returns 0-3 for tasks
     */
    private function calculateRangeScore($text, $type) {
        if (!$text) return 0;
        
        $wordCount = str_word_count(trim($text));
        
        // Get ideal ranges from constants
        $idealMin = self::IDEAL_RANGES['task'][$type]['min'];
        $idealMax = self::IDEAL_RANGES['task'][$type]['max'];
        
        // Perfect score if within range
        if ($wordCount >= $idealMin && $wordCount <= $idealMax) {
            return 3;
        }
        
        // Calculate how far off we are
        $distance = 0;
        if ($wordCount < $idealMin) {
            $distance = $idealMin - $wordCount;
        } else {
            $distance = $wordCount - $idealMax;
        }
        
        // Score based on distance (closer = better)
        if ($distance <= 2) return 2;
        if ($distance <= 5) return 1;
        return 0;
    }
    
    /**
     * Calculate project description score (0-10 points)
     */
    private function calculateProjectRangeScore($description) {
        if (!$description) return 0;
        
        $wordCount = str_word_count(trim($description));
        
        $idealMin = self::IDEAL_RANGES['project']['description']['min'];
        $idealMax = self::IDEAL_RANGES['project']['description']['max'];
        
        // Perfect score if within range
        if ($wordCount >= $idealMin && $wordCount <= $idealMax) {
            return 10;
        }
        
        // Calculate how far off we are
        $distance = 0;
        if ($wordCount < $idealMin) {
            $distance = $idealMin - $wordCount;
        } else {
            $distance = $wordCount - $idealMax;
        }
        
        // Score based on distance (0-9 depending on how close)
        $rangeSize = $idealMax - $idealMin;
        $maxAllowedDistance = $rangeSize;
        
        if ($distance >= $maxAllowedDistance) return 0;
        
        $score = 9 - floor(($distance / $maxAllowedDistance) * 9);
        return max(0, $score);
    }
    
    /**
     * Get complexity bonus (0-4 points)
     */
    private function getComplexityBonus($complexity) {
        $bonuses = [
            'none' => 0,
            'simple' => 1,
            'moderate' => 2,
            'complex' => 3,
            'very_complex' => 4
        ];
        
        return $bonuses[$complexity] ?? 0;
    }
    
    /**
     * Get priority bonus (1-4 points)
     */
    private function getPriorityBonus($priority) {
        $bonuses = [
            'low' => 1,
            'medium' => 2,
            'high' => 3,
            'urgent' => 4
        ];
        
        return $bonuses[$priority] ?? 2;
    }
}

/**
 * Achievement Model
 */
class Achievement extends BaseModel {
    protected $table = 'achievements_v2';
    protected $fillable = [
        'user_id', 'achievement_type', 'title', 'description', 'icon', 'xp_reward'
    ];
    protected $casts = [
        'user_id' => 'int',
        'xp_reward' => 'int'
    ];
    
    public function getByUser($userId) {
        $where = ['user_id = ?'];
        $params = [$userId];
        
        return $this->all($where, $params, 'unlocked_at DESC');
    }
}

/**
 * Achievement Progress Model
 * Tracks user progress towards unlocking achievements
 */
class AchievementProgress extends BaseModel {
    protected $table = 'achievement_progress';
    protected $fillable = [
        'user_id', 'achievement_key', 'progress', 'target', 'is_unlocked', 'unlocked_at'
    ];
    protected $casts = [
        'user_id' => 'int',
        'progress' => 'int',
        'target' => 'int',
        'is_unlocked' => 'bool'
    ];
    
    /**
     * Get all progress for a user
     */
    public function getByUser($userId) {
        $where = ['user_id = ?'];
        $params = [$userId];
        return $this->all($where, $params, 'created_at DESC');
    }
    
    /**
     * Get specific achievement progress
     */
    public function getProgress($userId, $achievementKey) {
        $where = ['user_id = ? AND achievement_key = ?'];
        $params = [$userId, $achievementKey];
        return $this->one($where, $params);
    }
    
    /**
     * Update or create progress
     */
    public function updateProgress($userId, $achievementKey, $progress, $target) {
        $existing = $this->getProgress($userId, $achievementKey);
        
        if ($existing) {
            // Check if should be unlocked
            $isUnlocked = $progress >= $target;
            $unlockedAt = ($isUnlocked && !$existing['is_unlocked']) ? date('Y-m-d H:i:s') : $existing['unlocked_at'];
            
            return $this->update($existing['id'], [
                'progress' => $progress,
                'is_unlocked' => $isUnlocked,
                'unlocked_at' => $unlockedAt
            ]);
        } else {
            $isUnlocked = $progress >= $target;
            return $this->create([
                'user_id' => $userId,
                'achievement_key' => $achievementKey,
                'progress' => $progress,
                'target' => $target,
                'is_unlocked' => $isUnlocked,
                'unlocked_at' => $isUnlocked ? date('Y-m-d H:i:s') : null
            ]);
        }
    }
    
    /**
     * Increment progress
     */
    public function incrementProgress($userId, $achievementKey, $target, $increment = 1) {
        $existing = $this->getProgress($userId, $achievementKey);
        $newProgress = ($existing['progress'] ?? 0) + $increment;
        return $this->updateProgress($userId, $achievementKey, $newProgress, $target);
    }
}

/**
 * Audit Log Model
 */
class AuditLog extends BaseModel {
    protected $table = 'audit_logs';
    protected $fillable = [
        'user_id', 'table_name', 'record_id', 'action', 'old_values', 'new_values'
    ];
    protected $casts = [
        'user_id' => 'int',
        'record_id' => 'int',
        'old_values' => 'json',
        'new_values' => 'json'
    ];
    
    public function getByUser($userId, $limit = 50) {
        $where = ['user_id = ?'];
        $params = [$userId];
        
        return $this->all($where, $params, 'created_at DESC', $limit);
    }
    
    public function logAction($userId, $tableName, $recordId, $action, $oldValues = null, $newValues = null) {
        return $this->create([
            'user_id' => $userId,
            'table_name' => $tableName,
            'record_id' => $recordId,
            'action' => $action,
            'old_values' => $oldValues,
            'new_values' => $newValues
        ]);
    }
}

/**
 * Subtask Model
 * Represents individual steps/subtasks within a parent task
 */
class Subtask extends BaseModel {
    protected $table = 'subtasks';
    protected $fillable = [
        'task_id', 'title', 'is_completed', 'sort_order'
    ];
    protected $casts = [
        'task_id' => 'int',
        'is_completed' => 'bool',
        'sort_order' => 'int'
    ];
    
    /**
     * Get all subtasks for a specific task
     */
    public function getByTask($taskId) {
        $where = ['task_id = ?'];
        $params = [$taskId];
        
        return $this->all($where, $params, 'sort_order ASC, id ASC');
    }
    
    /**
     * Toggle subtask completion status
     */
    public function toggle($subtaskId) {
        $subtask = $this->find($subtaskId);
        if (!$subtask) {
            return false;
        }
        
        $newStatus = !$subtask['is_completed'];
        return $this->update($subtaskId, ['is_completed' => $newStatus]);
    }
    
    /**
     * Get completion stats for a task
     */
    public function getTaskStats($taskId) {
        $sql = "SELECT 
                    COUNT(*) as total,
                    SUM(CASE WHEN is_completed = 1 THEN 1 ELSE 0 END) as completed
                FROM {$this->table}
                WHERE task_id = ?";
        
        $result = $this->db->selectOne($sql, [$taskId]);
        
        return [
            'total' => (int)$result['total'],
            'completed' => (int)$result['completed'],
            'remaining' => (int)$result['total'] - (int)$result['completed'],
            'percentage' => $result['total'] > 0 
                ? round(($result['completed'] / $result['total']) * 100) 
                : 0
        ];
    }
    
    /**
     * Reorder subtasks
     */
    public function reorder($taskId, $subtaskIds) {
        $order = 0;
        foreach ($subtaskIds as $subtaskId) {
            $this->update($subtaskId, [
                'sort_order' => $order++
            ]);
        }
        return true;
    }
}

/**
 * Calendar Event Model
 */
class CalendarEvent extends BaseModel {
    protected $table = 'calendar_events';
    protected $fillable = [
        'user_id', 'title', 'description', 'event_date', 'event_time', 
        'end_time', 'is_all_day', 'location', 'color', 'reminder_minutes',
        'recurrence_rule', 'task_id'
    ];
    protected $casts = [
        'user_id' => 'int',
        'is_all_day' => 'bool',
        'reminder_minutes' => 'int',
        'task_id' => 'int'
    ];
    
    /**
     * Get events for a user within a date range
     */
    public function getByDateRange($userId, $startDate, $endDate) {
        $sql = "SELECT * FROM `{$this->table}` 
                WHERE user_id = ? 
                AND event_date BETWEEN ? AND ?
                ORDER BY event_date ASC, event_time ASC";
        
        return $this->db->select($sql, [$userId, $startDate, $endDate]);
    }
    
    /**
     * Get events for a specific date
     */
    public function getByDate($userId, $date) {
        $sql = "SELECT * FROM `{$this->table}` 
                WHERE user_id = ? 
                AND event_date = ?
                ORDER BY event_time ASC";
        
        return $this->db->select($sql, [$userId, $date]);
    }
    
    /**
     * Get upcoming events with reminders
     */
    public function getUpcomingWithReminders($userId, $minutesAhead = 60) {
        $sql = "SELECT * FROM `{$this->table}` 
                WHERE user_id = ? 
                AND reminder_minutes IS NOT NULL
                AND CONCAT(event_date, ' ', IFNULL(event_time, '00:00:00')) 
                    BETWEEN NOW() 
                    AND DATE_ADD(NOW(), INTERVAL ? MINUTE)
                ORDER BY event_date ASC, event_time ASC";
        
        return $this->db->select($sql, [$userId, $minutesAhead]);
    }
    
    /**
     * Get events linked to a task
     */
    public function getByTask($taskId) {
        $sql = "SELECT * FROM `{$this->table}` 
                WHERE task_id = ?
                ORDER BY event_date ASC, event_time ASC";
        
        return $this->db->select($sql, [$taskId]);
    }
    
    /**
     * Link event to a task
     */
    public function linkToTask($eventId, $taskId) {
        return $this->update($eventId, ['task_id' => $taskId]);
    }
    
    /**
     * Unlink event from task
     */
    public function unlinkFromTask($eventId) {
        return $this->update($eventId, ['task_id' => null]);
    }
}
