<?php
/**
 * Database Connection and Query Layer for TaskList v2
 * 
 * This class provides a secure, efficient database abstraction layer
 * with prepared statements, connection pooling, and error handling.
 */

class DatabaseConnection {
    private static $instance = null;
    private $pdo;
    private $config;
    
    private function __construct() {
        // Load global database credentials
        require_once __DIR__ . '/../../../PHP/GLOB_Connect_BLOODWEB.php';
        
        $this->config = [
            'host' => $GLOBALS['server_name'] ?? 'localhost',
            'dbname' => $GLOBALS['DBNAME'] ?? 'bloodweb',
            'username' => $GLOBALS['user_name'] ?? 'bloodweb',
            'password' => $GLOBALS['password'] ?? 'BLOODWEBMYSQL',
            'charset' => 'utf8mb4',
            'options' => [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false,
                PDO::ATTR_PERSISTENT => true,
                PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8mb4 COLLATE utf8mb4_unicode_ci"
            ]
        ];
        
        $this->connect();
    }
    
    private function connect() {
        try {
            $dsn = "mysql:host={$this->config['host']};dbname={$this->config['dbname']};charset={$this->config['charset']}";
            $this->pdo = new PDO($dsn, $this->config['username'], $this->config['password'], $this->config['options']);
            
            // Set timezone to UTC for consistent datetime handling
            $this->pdo->exec("SET time_zone = '+00:00'");
            
        } catch (PDOException $e) {
            error_log("Database connection failed: " . $e->getMessage());
            throw new Exception("Database connection failed", 500);
        }
    }
    
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    public function getPDO() {
        return $this->pdo;
    }
    
    /**
     * Execute a SELECT query with parameters
     */
    public function select($sql, $params = []) {
        try {
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            return $stmt->fetchAll();
        } catch (PDOException $e) {
            error_log("Database select error: " . $e->getMessage());
            throw new Exception("Database query failed", 500);
        }
    }
    
    /**
     * Execute a SELECT query and return single row
     */
    public function selectOne($sql, $params = []) {
        try {
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            return $stmt->fetch();
        } catch (PDOException $e) {
            error_log("Database selectOne error: " . $e->getMessage());
            throw new Exception("Database query failed", 500);
        }
    }
    
    /**
     * Execute an INSERT query and return last insert ID
     */
    public function insert($sql, $params = []) {
        try {
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            return $this->pdo->lastInsertId();
        } catch (PDOException $e) {
            error_log("Database insert error: " . $e->getMessage());
            throw new Exception("Database insert failed", 500);
        }
    }
    
    /**
     * Execute an UPDATE query and return affected rows
     */
    public function update($sql, $params = []) {
        try {
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            return $stmt->rowCount();
        } catch (PDOException $e) {
            error_log("Database update error: " . $e->getMessage());
            throw new Exception("Database update failed", 500);
        }
    }
    
    /**
     * Execute a DELETE query and return affected rows
     */
    public function delete($sql, $params = []) {
        try {
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            return $stmt->rowCount();
        } catch (PDOException $e) {
            error_log("Database delete error: " . $e->getMessage());
            throw new Exception("Database delete failed", 500);
        }
    }
    
    /**
     * Begin a database transaction
     */
    public function beginTransaction() {
        return $this->pdo->beginTransaction();
    }
    
    /**
     * Commit a database transaction
     */
    public function commit() {
        return $this->pdo->commit();
    }
    
    /**
     * Rollback a database transaction
     */
    public function rollback() {
        return $this->pdo->rollback();
    }
    
    /**
     * Execute multiple queries in a transaction
     */
    public function transaction(callable $callback) {
        try {
            $this->beginTransaction();
            $result = $callback($this);
            $this->commit();
            return $result;
        } catch (Exception $e) {
            $this->rollback();
            throw $e;
        }
    }
    
    /**
     * Get table record count with optional WHERE conditions
     */
    public function count($table, $where = [], $params = []) {
        $sql = "SELECT COUNT(*) FROM `$table`";
        
        if (!empty($where)) {
            $sql .= " WHERE " . implode(" AND ", $where);
        }
        
        try {
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            return (int) $stmt->fetchColumn();
        } catch (PDOException $e) {
            error_log("Database count error: " . $e->getMessage());
            throw new Exception("Database count failed", 500);
        }
    }
    
    /**
     * Check if a record exists
     */
    public function exists($table, $where = [], $params = []) {
        return $this->count($table, $where, $params) > 0;
    }
}

/**
 * Base Model class for TaskList entities
 */
abstract class BaseModel {
    protected $db;
    protected $table;
    protected $primaryKey = 'id';
    protected $fillable = [];
    protected $hidden = [];
    protected $casts = [];
    
    public function __construct() {
        $this->db = DatabaseConnection::getInstance();
    }
    
    /**
     * Find record by ID
     */
    public function find($id) {
        $sql = "SELECT * FROM `{$this->table}` WHERE `{$this->primaryKey}` = ?";
        $result = $this->db->selectOne($sql, [$id]);
        
        if ($result) {
            return $this->cast($result);
        }
        
        return null;
    }
    
    /**
     * Get all records with optional conditions
     */
    public function all($where = [], $params = [], $orderBy = null, $limit = null) {
        $sql = "SELECT * FROM `{$this->table}`";
        
        if (!empty($where)) {
            $sql .= " WHERE " . implode(" AND ", $where);
        }
        
        if ($orderBy) {
            $sql .= " ORDER BY $orderBy";
        }
        
        if ($limit) {
            $sql .= " LIMIT $limit";
        }
        
        $results = $this->db->select($sql, $params);
        
        return array_map([$this, 'cast'], $results);
    }
    
    /**
     * Get single record with optional conditions
     */
    public function one($where = [], $params = []) {
        $sql = "SELECT * FROM `{$this->table}`";
        
        if (!empty($where)) {
            $sql .= " WHERE " . implode(" AND ", $where);
        }
        
        $sql .= " LIMIT 1";
        
        $result = $this->db->selectOne($sql, $params);
        
        if ($result) {
            return $this->cast($result);
        }
        
        return null;
    }
    
    /**
     * Create new record
     */
    public function create($data) {
        $data = $this->filterFillable($data);
        $data = $this->prepareForDatabase($data);
        
        // Debug logging for XP log inserts
        if ($this->table === 'xp_log') {
            error_log("XP Log Insert Data: " . json_encode($data));
            error_log("Project ID value: " . var_export($data['project_id'] ?? 'NOT SET', true));
            error_log("Project ID type: " . gettype($data['project_id'] ?? null));
        }
        
        $columns = array_keys($data);
        $placeholders = array_fill(0, count($columns), '?');
        
        $sql = "INSERT INTO `{$this->table}` (`" . implode('`, `', $columns) . "`) VALUES (" . implode(', ', $placeholders) . ")";
        
        $id = $this->db->insert($sql, array_values($data));
        
        return $this->find($id);
    }
    
    /**
     * Update record by ID
     */
    public function update($id, $data) {
        $data = $this->filterFillable($data);
        $data = $this->prepareForDatabase($data);
        
        if (empty($data)) {
            return $this->find($id);
        }
        
        $sets = array_map(function($column) {
            return "`$column` = ?";
        }, array_keys($data));
        
        $sql = "UPDATE `{$this->table}` SET " . implode(', ', $sets) . " WHERE `{$this->primaryKey}` = ?";
        
        $params = array_values($data);
        $params[] = $id;
        
        $this->db->update($sql, $params);
        
        return $this->find($id);
    }
    
    /**
     * Delete record by ID
     */
    public function delete($id) {
        $sql = "DELETE FROM `{$this->table}` WHERE `{$this->primaryKey}` = ?";
        return $this->db->delete($sql, [$id]) > 0;
    }
    
    /**
     * Soft delete record (if table has deleted_at column)
     */
    public function softDelete($id) {
        $sql = "UPDATE `{$this->table}` SET `deleted_at` = NOW() WHERE `{$this->primaryKey}` = ?";
        return $this->db->update($sql, [$id]) > 0;
    }
    
    /**
     * Filter data to only include fillable fields
     */
    protected function filterFillable($data) {
        if (empty($data) || empty($this->fillable)) {
            return $data ?: [];
        }
        
        return array_intersect_key($data, array_flip($this->fillable));
    }
    
    /**
     * Prepare data for database insertion
     */
    protected function prepareForDatabase($data) {
        foreach ($data as $key => $value) {
            // Skip casting for NULL values - preserve them for foreign keys
            if ($value === null) {
                continue;
            }
            
            if (isset($this->casts[$key])) {
                switch ($this->casts[$key]) {
                    case 'json':
                        $data[$key] = json_encode($value);
                        break;
                    case 'bool':
                        // Convert boolean to integer for database
                        $data[$key] = $value ? 1 : 0;
                        break;
                    case 'int':
                        $data[$key] = (int) $value;
                        break;
                    case 'float':
                        $data[$key] = (float) $value;
                        break;
                    case 'datetime':
                        if ($value instanceof DateTime) {
                            $data[$key] = $value->format('Y-m-d H:i:s');
                        }
                        break;
                    case 'date':
                        if ($value instanceof DateTime) {
                            $data[$key] = $value->format('Y-m-d');
                        }
                        break;
                }
            }
        }
        
        return $data;
    }
    
    /**
     * Cast database values to appropriate types
     */
    protected function cast($data) {
        if (!$data) return $data;
        
        foreach ($this->casts as $key => $type) {
            if (isset($data[$key])) {
                switch ($type) {
                    case 'json':
                        $data[$key] = json_decode($data[$key], true);
                        break;
                    case 'int':
                        $data[$key] = (int) $data[$key];
                        break;
                    case 'float':
                        $data[$key] = (float) $data[$key];
                        break;
                    case 'bool':
                        $data[$key] = (bool) $data[$key];
                        break;
                }
            }
        }
        
        // Remove hidden fields
        foreach ($this->hidden as $field) {
            unset($data[$field]);
        }
        
        return $data;
    }
    
    /**
     * Get paginated results
     */
    public function paginate($page = 1, $perPage = 20, $where = [], $params = [], $orderBy = null) {
        $offset = ($page - 1) * $perPage;
        
        // Get total count
        $countSql = "SELECT COUNT(*) FROM `{$this->table}`";
        if (!empty($where)) {
            $countSql .= " WHERE " . implode(" AND ", $where);
        }
        
        $stmt = $this->db->getPDO()->prepare($countSql);
        $stmt->execute($params);
        $total = (int) $stmt->fetchColumn();
        
        // Get paginated data
        $sql = "SELECT * FROM `{$this->table}`";
        if (!empty($where)) {
            $sql .= " WHERE " . implode(" AND ", $where);
        }
        if ($orderBy) {
            $sql .= " ORDER BY $orderBy";
        }
        $sql .= " LIMIT $perPage OFFSET $offset";
        
        $results = $this->db->select($sql, $params);
        $data = array_map([$this, 'cast'], $results);
        
        return [
            'data' => $data,
            'meta' => [
                'total' => $total,
                'page' => $page,
                'per_page' => $perPage,
                'total_pages' => ceil($total / $perPage),
                'has_next' => $page < ceil($total / $perPage),
                'has_prev' => $page > 1
            ]
        ];
    }
}

/**
 * Validation Helper Class
 */
class Validator {
    private $data;
    private $rules;
    private $errors = [];
    
    public function __construct($data, $rules) {
        $this->data = $data;
        $this->rules = $rules;
    }
    
    public function validate() {
        foreach ($this->rules as $field => $ruleString) {
            $rules = explode('|', $ruleString);
            $value = $this->data[$field] ?? null;
            
            foreach ($rules as $rule) {
                $this->applyRule($field, $value, $rule);
            }
        }
        
        return empty($this->errors);
    }
    
    private function applyRule($field, $value, $rule) {
        if (strpos($rule, ':') !== false) {
            [$ruleName, $parameter] = explode(':', $rule, 2);
        } else {
            $ruleName = $rule;
            $parameter = null;
        }
        
        switch ($ruleName) {
            case 'required':
                if (empty($value) && $value !== '0') {
                    $this->errors[$field] = "$field is required";
                }
                break;
                
            case 'email':
                if ($value && !filter_var($value, FILTER_VALIDATE_EMAIL)) {
                    $this->errors[$field] = "$field must be a valid email";
                }
                break;
                
            case 'min':
                if ($value && strlen($value) < $parameter) {
                    $this->errors[$field] = "$field must be at least $parameter characters";
                }
                break;
                
            case 'max':
                if ($value && strlen($value) > $parameter) {
                    $this->errors[$field] = "$field must not exceed $parameter characters";
                }
                break;
                
            case 'numeric':
                if ($value && !is_numeric($value)) {
                    $this->errors[$field] = "$field must be numeric";
                }
                break;
                
            case 'date':
                if ($value && !strtotime($value)) {
                    $this->errors[$field] = "$field must be a valid date";
                }
                break;
                
            case 'in':
                $allowedValues = explode(',', $parameter);
                if ($value && !in_array($value, $allowedValues)) {
                    $this->errors[$field] = "$field must be one of: " . implode(', ', $allowedValues);
                }
                break;
        }
    }
    
    public function getErrors() {
        return $this->errors;
    }
}
