/**
 * VirtualScroller - Efficient rendering for large lists
 * Uses IntersectionObserver to render only visible items
 * @class VirtualScroller
 */
class VirtualScroller {
    constructor(options = {}) {
        this.container = options.container;
        this.items = options.items || [];
        this.renderItem = options.renderItem;
        this.itemHeight = options.itemHeight || 100; // Estimated height
        this.bufferSize = options.bufferSize || 5; // Extra items to render above/below
        this.batchSize = options.batchSize || 20; // Items per batch
        this.loadMore = options.loadMore; // Callback for loading more
        
        this.visibleItems = new Set();
        this.renderedItems = new Map();
        this.scrollPosition = 0;
        this.isLoading = false;
        this.hasMore = true;
        
        this.sentinelTop = null;
        this.sentinelBottom = null;
        this.observer = null;
        
        this.init();
    }

    /**
     * Initialize virtual scroller
     */
    init() {
        if (!this.container) {
            logger.error('[VirtualScroller] Container not found');
            return;
        }

        // Create scroll container wrapper
        this.scrollContainer = document.createElement('div');
        this.scrollContainer.className = 'virtual-scroll-container';
        
        // Create content wrapper
        this.contentWrapper = document.createElement('div');
        this.contentWrapper.className = 'virtual-scroll-content';
        
        // Create sentinel elements for intersection observer
        this.sentinelTop = document.createElement('div');
        this.sentinelTop.className = 'virtual-scroll-sentinel';
        this.sentinelTop.dataset.sentinel = 'top';
        
        this.sentinelBottom = document.createElement('div');
        this.sentinelBottom.className = 'virtual-scroll-sentinel';
        this.sentinelBottom.dataset.sentinel = 'bottom';
        
        // Append elements
        this.contentWrapper.appendChild(this.sentinelTop);
        this.contentWrapper.appendChild(this.sentinelBottom);
        this.scrollContainer.appendChild(this.contentWrapper);
        this.container.appendChild(this.scrollContainer);
        
        // Setup intersection observer
        this.setupObserver();
        
        // Restore scroll position if saved
        this.restoreScrollPosition();
        
        // Initial render
        this.render();
    }

    /**
     * Setup IntersectionObserver for sentinels
     */
    setupObserver() {
        const options = {
            root: this.scrollContainer,
            rootMargin: `${this.itemHeight * this.bufferSize}px`,
            threshold: 0
        };

        this.observer = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    const sentinel = entry.target.dataset.sentinel;
                    
                    if (sentinel === 'bottom' && !this.isLoading && this.hasMore) {
                        this.loadMoreItems();
                    }
                }
            });
        }, options);

        this.observer.observe(this.sentinelTop);
        this.observer.observe(this.sentinelBottom);
    }

    /**
     * Render visible items
     */
    render() {
        if (!this.items || this.items.length === 0) {
            this.renderEmptyState();
            return;
        }

        // Clear existing items
        const existingItems = this.contentWrapper.querySelectorAll('.virtual-scroll-item');
        existingItems.forEach(item => item.remove());

        // Calculate visible range
        const startIndex = 0;
        const endIndex = Math.min(this.batchSize, this.items.length);

        // Render initial batch
        for (let i = startIndex; i < endIndex; i++) {
            this.renderItemAtIndex(i);
        }

        // Update hasMore flag
        this.hasMore = endIndex < this.items.length;

        logger.debug(`[VirtualScroller] Rendered ${endIndex - startIndex} items, hasMore: ${this.hasMore}`);
    }

    /**
     * Render a single item at index
     * @param {number} index - Item index
     */
    renderItemAtIndex(index) {
        if (index < 0 || index >= this.items.length) return;

        const item = this.items[index];
        const itemElement = document.createElement('div');
        itemElement.className = 'virtual-scroll-item';
        itemElement.dataset.index = index;
        itemElement.innerHTML = this.renderItem(item, index);

        // Insert before bottom sentinel
        this.contentWrapper.insertBefore(itemElement, this.sentinelBottom);
        
        this.renderedItems.set(index, itemElement);
    }

    /**
     * Load more items
     */
    async loadMoreItems() {
        if (this.isLoading || !this.hasMore) return;

        this.isLoading = true;
        this.showLoadingIndicator();

        try {
            const currentCount = this.contentWrapper.querySelectorAll('.virtual-scroll-item').length;
            const nextBatch = this.items.slice(currentCount, currentCount + this.batchSize);

            if (nextBatch.length === 0) {
                this.hasMore = false;
                this.hideLoadingIndicator();
                return;
            }

            // Render next batch
            for (let i = 0; i < nextBatch.length; i++) {
                const globalIndex = currentCount + i;
                this.renderItemAtIndex(globalIndex);
            }

            // Check if more items available
            this.hasMore = currentCount + nextBatch.length < this.items.length;

            logger.debug(`[VirtualScroller] Loaded ${nextBatch.length} more items, total: ${currentCount + nextBatch.length}`);

        } catch (error) {
            logger.error('[VirtualScroller] Error loading more items:', error);
        } finally {
            this.isLoading = false;
            this.hideLoadingIndicator();
        }
    }

    /**
     * Update items and re-render
     * @param {Array} newItems - New items array
     */
    updateItems(newItems) {
        this.items = newItems;
        this.hasMore = true;
        this.renderedItems.clear();
        this.render();
    }

    /**
     * Show loading indicator
     */
    showLoadingIndicator() {
        let loader = this.contentWrapper.querySelector('.virtual-scroll-loader');
        if (!loader) {
            loader = document.createElement('div');
            loader.className = 'virtual-scroll-loader';
            loader.innerHTML = `
                <div class="loader-spinner"></div>
                <span>Loading more tasks...</span>
            `;
            this.contentWrapper.insertBefore(loader, this.sentinelBottom);
        }
        loader.style.display = 'flex';
    }

    /**
     * Hide loading indicator
     */
    hideLoadingIndicator() {
        const loader = this.contentWrapper.querySelector('.virtual-scroll-loader');
        if (loader) {
            loader.style.display = 'none';
        }
    }

    /**
     * Render empty state
     */
    renderEmptyState() {
        this.contentWrapper.innerHTML = `
            <div class="virtual-scroll-empty">
                <i class="fas fa-inbox"></i>
                <p>No items to display</p>
            </div>
        `;
    }

    /**
     * Save scroll position to localStorage
     */
    saveScrollPosition() {
        if (this.scrollContainer) {
            const position = this.scrollContainer.scrollTop;
            const key = this.container.id || 'virtual-scroller';
            localStorage.setItem(`TL_scroll_${key}`, position.toString());
        }
    }

    /**
     * Restore scroll position from localStorage
     */
    restoreScrollPosition() {
        if (this.scrollContainer) {
            const key = this.container.id || 'virtual-scroller';
            const savedPosition = localStorage.getItem(`TL_scroll_${key}`);
            
            if (savedPosition) {
                this.scrollContainer.scrollTop = parseInt(savedPosition);
                logger.debug(`[VirtualScroller] Restored scroll position: ${savedPosition}`);
            }
        }
    }

    /**
     * Scroll to top
     */
    scrollToTop() {
        if (this.scrollContainer) {
            this.scrollContainer.scrollTop = 0;
        }
    }

    /**
     * Scroll to item by index
     * @param {number} index - Item index
     */
    scrollToItem(index) {
        const itemElement = this.renderedItems.get(index);
        if (itemElement) {
            itemElement.scrollIntoView({ behavior: 'smooth', block: 'center' });
        }
    }

    /**
     * Get currently visible items
     * @returns {Array} Array of visible item indices
     */
    getVisibleIndices() {
        const indices = [];
        const items = this.contentWrapper.querySelectorAll('.virtual-scroll-item');
        
        items.forEach(item => {
            const rect = item.getBoundingClientRect();
            const containerRect = this.scrollContainer.getBoundingClientRect();
            
            if (rect.top < containerRect.bottom && rect.bottom > containerRect.top) {
                indices.push(parseInt(item.dataset.index));
            }
        });
        
        return indices;
    }

    /**
     * Destroy virtual scroller and cleanup
     */
    destroy() {
        if (this.observer) {
            this.observer.disconnect();
        }
        
        this.saveScrollPosition();
        
        if (this.scrollContainer && this.container) {
            this.container.removeChild(this.scrollContainer);
        }
        
        this.renderedItems.clear();
        this.visibleItems.clear();
    }

    /**
     * Refresh current view
     */
    refresh() {
        this.render();
    }

    /**
     * Get scroller stats
     * @returns {Object} Stats object
     */
    getStats() {
        return {
            totalItems: this.items.length,
            renderedItems: this.renderedItems.size,
            visibleItems: this.getVisibleIndices().length,
            hasMore: this.hasMore,
            isLoading: this.isLoading
        };
    }
}
