# Version Management System

## Overview
Centralized version tracking system for Task Manager with built-in changelog display.

## Files

### version.js
Central source of truth for version information and changelog.

**Structure:**
```javascript
const APP_VERSION = {
    version: '2.1.0',           // Semantic version number
    releaseDate: '2025-12-27',  // ISO date format
    codename: 'FocusFlow',       // Release codename
    buildNumber: 'b2025.12.27',  // Optional build identifier
    name: 'Task Manager',        // Product name
    fullName: 'Task Manager — FocusFlow' // Full display name
};

const CHANGELOG = [
    {
        version: '2.1.0',
        releaseDate: '2025-12-27',
        codename: 'FocusFlow',
        changes: {
            added: [...],      // New features
            changed: [...],    // Changes to existing features
            fixed: [...],      // Bug fixes
            removed: [...],    // Removed features
            security: [...]    // Security fixes
        }
    },
    // Previous versions...
];
```

### version.php
PHP helper that reads version from version.js for use in templates.

**Usage:**
```php
require_once __DIR__ . '/version.php';
echo $APP_VERSION; // Outputs: 2.1.0
echo $APP_CODENAME; // Outputs: FocusFlow
echo $APP_FULL_NAME; // Outputs: Task Manager — FocusFlow
```

## How to Update Version

1. **Update version.js**
   - Update `APP_VERSION` object with new version, date, codename
   - Add new entry to top of `CHANGELOG` array
   - Categorize changes appropriately (added, changed, fixed, removed, security)

2. **That's it!**
   - All files automatically use the updated version from version.js
   - No need to update individual file headers or constants

## Version Display

### Header Badge
The version badge in the header (top-left) is clickable:
- Click to view full changelog
- Keyboard accessible (Tab + Enter/Space)
- Hover effect indicates clickability

### Changelog View
Accessible via:
- Clicking version badge in header
- Direct URL: `?page=changelog`
- JavaScript: `app.ui.navigateToView('changelog')`

**Features:**
- Timeline view of all versions
- Color-coded change categories
- Latest version badge
- Release dates and codenames
- Responsive design

## Files Using Version

All version references are centralized through version.js:

- **index.php**: Dynamically displays version in header via version.php
- **main.js**: Uses APP_VERSION.version for CONFIG.VERSION
- **service-worker.js**: Uses CACHE_VERSION for cache naming
- **CalendarManager.js**: Updated to current version in file header
- **KeyboardManager.js**: Updated to current version in file header

## Best Practices

1. **Semantic Versioning**: Follow MAJOR.MINOR.PATCH format
   - MAJOR: Breaking changes
   - MINOR: New features (backwards compatible)
   - PATCH: Bug fixes

2. **Changelog Categories**:
   - `added`: New features, new capabilities
   - `changed`: Changes to existing features
   - `fixed`: Bug fixes
   - `removed`: Features removed/deprecated
   - `security`: Security vulnerability fixes

3. **Release Dates**: Use ISO format (YYYY-MM-DD) for consistency

4. **Codenames**: Optional but fun - use themes (Greek gods, constellations, etc.)

## Examples

### Adding a New Version
```javascript
// At the top of CHANGELOG array in version.js
{
    version: '2.1.0',
    releaseDate: '2025-12-15',
    codename: 'Odyssey',
    changes: {
        added: [
            'Dark mode theme support',
            'Task templates for quick creation',
            'Export tasks to CSV/PDF'
        ],
        changed: [
            'Improved search algorithm for faster results',
            'Updated task modal with better UX'
        ],
        fixed: [
            'Calendar events not showing correct timezone',
            'XP calculation error for subtasks'
        ]
    }
}
```

### Migration Notes
When migrating from hardcoded versions:
1. Update version.js with current version
2. Replace hardcoded strings with APP_VERSION references
3. Test all version displays (header, about, API, etc.)
4. Update build/deployment scripts if needed

## API Version Endpoint
Consider adding a version endpoint for API clients:
```php
// PHP/api/version.php
require_once __DIR__ . '/../version.php';
echo json_encode([
    'version' => $APP_VERSION,
    'api_version' => 'v2'
]);
```

## Future Enhancements
- Automatic version bump scripts
- Git tag integration
- Release notes generator
- Version deprecation warnings
- Update notifications
