/**
 * TaskAssistant - Intelligent task suggestion system
 * Provides smart categorization, due date suggestions, and task templates
 * @class TaskAssistant
 */
class TaskAssistant {
    constructor(app) {
        this.app = app;
        this.categoryKeywords = this.buildCategoryKeywords();
        this.templates = this.buildTemplates();
        this.dueDatePatterns = this.buildDueDatePatterns();
    }

    /**
     * Build keyword mapping for categories
     * Maps common keywords to category suggestions
     */
    buildCategoryKeywords() {
        return {
            work: ['meeting', 'email', 'call', 'presentation', 'report', 'deadline', 'client', 'project', 'business', 'office', 'conference'],
            personal: ['buy', 'shopping', 'grocery', 'clean', 'organize', 'home', 'family', 'doctor', 'appointment', 'personal'],
            health: ['workout', 'exercise', 'gym', 'run', 'walk', 'doctor', 'medicine', 'health', 'fitness', 'diet', 'nutrition'],
            learning: ['learn', 'study', 'read', 'course', 'tutorial', 'practice', 'research', 'book', 'article', 'video', 'education'],
            finance: ['pay', 'bill', 'invoice', 'budget', 'expense', 'tax', 'bank', 'money', 'financial', 'savings', 'invest'],
            creative: ['write', 'draw', 'design', 'paint', 'create', 'photo', 'video', 'music', 'art', 'creative', 'blog'],
            social: ['call', 'message', 'text', 'email', 'meet', 'party', 'birthday', 'event', 'hangout', 'friend', 'family'],
            maintenance: ['fix', 'repair', 'update', 'upgrade', 'install', 'maintain', 'backup', 'check', 'service', 'replace']
        };
    }

    /**
     * Build common task templates
     */
    buildTemplates() {
        return {
            'Meeting Preparation': {
                title: 'Prepare for [Meeting Name]',
                description: '- Review agenda\n- Prepare materials\n- Send calendar invite\n- Prepare questions',
                priority: 'high',
                category: 'work'
            },
            'Weekly Review': {
                title: 'Weekly Review & Planning',
                description: '- Review completed tasks\n- Plan next week priorities\n- Update project status\n- Clear inbox',
                priority: 'medium',
                category: 'work'
            },
            'Exercise Routine': {
                title: 'Workout Session',
                description: '- Warm up (10 min)\n- Main workout (30 min)\n- Cool down & stretch (10 min)\n- Log progress',
                priority: 'medium',
                category: 'health'
            },
            'Grocery Shopping': {
                title: 'Grocery Shopping',
                description: '- Make shopping list\n- Check pantry inventory\n- Plan meals for week\n- Shop at store',
                priority: 'low',
                category: 'personal'
            },
            'Bill Payment': {
                title: 'Pay [Bill Name]',
                description: '- Check due date\n- Verify amount\n- Process payment\n- Save receipt',
                priority: 'high',
                category: 'finance'
            },
            'Learning Goal': {
                title: 'Learn [Topic/Skill]',
                description: '- Find resources (courses/books)\n- Set learning schedule\n- Practice daily\n- Track progress',
                priority: 'medium',
                category: 'learning'
            },
            'Content Creation': {
                title: 'Create [Content Type]',
                description: '- Research topic\n- Create outline\n- Write/Design content\n- Review and edit\n- Publish',
                priority: 'medium',
                category: 'creative'
            },
            'Home Maintenance': {
                title: 'Fix/Maintain [Item]',
                description: '- Assess problem\n- Gather tools/materials\n- Perform repair/maintenance\n- Test functionality',
                priority: 'medium',
                category: 'maintenance'
            }
        };
    }

    /**
     * Build date parsing patterns
     */
    buildDueDatePatterns() {
        return [
            { pattern: /\b(today|now)\b/i, offset: 0 },
            { pattern: /\btomorrow\b/i, offset: 1 },
            { pattern: /\b(next week|in a week)\b/i, offset: 7 },
            { pattern: /\bin (\d+) days?\b/i, offsetMultiplier: 1 },
            { pattern: /\bin (\d+) weeks?\b/i, offsetMultiplier: 7 },
            { pattern: /\bin (\d+) months?\b/i, offsetMultiplier: 30 },
            { pattern: /\b(monday|mon)\b/i, dayOfWeek: 1 },
            { pattern: /\b(tuesday|tue)\b/i, dayOfWeek: 2 },
            { pattern: /\b(wednesday|wed)\b/i, dayOfWeek: 3 },
            { pattern: /\b(thursday|thu)\b/i, dayOfWeek: 4 },
            { pattern: /\b(friday|fri)\b/i, dayOfWeek: 5 },
            { pattern: /\b(saturday|sat)\b/i, dayOfWeek: 6 },
            { pattern: /\b(sunday|sun)\b/i, dayOfWeek: 0 }
        ];
    }

    /**
     * Suggest categories based on task title and description
     * @param {string} title - Task title
     * @param {string} description - Task description
     * @returns {Array} Array of suggested categories with confidence scores
     */
    suggestCategories(title = '', description = '') {
        const text = `${title} ${description}`.toLowerCase();
        const suggestions = [];
        const categories = this.app.state.get('categories');

        // Score each category based on keyword matches
        for (const [categoryKey, keywords] of Object.entries(this.categoryKeywords)) {
            let score = 0;
            let matchedKeywords = [];

            for (const keyword of keywords) {
                if (text.includes(keyword)) {
                    score += 1;
                    matchedKeywords.push(keyword);
                }
            }

            if (score > 0) {
                // Find matching category from user's categories
                const matchingCategory = categories.find(c => 
                    c.name.toLowerCase().includes(categoryKey) ||
                    categoryKey.includes(c.name.toLowerCase())
                );

                suggestions.push({
                    categoryKey,
                    categoryId: matchingCategory?.id,
                    categoryName: matchingCategory?.name || categoryKey,
                    score,
                    matchedKeywords,
                    confidence: Math.min(score * 20, 100) // Convert to percentage
                });
            }
        }

        // Sort by score descending
        suggestions.sort((a, b) => b.score - a.score);

        return suggestions.slice(0, 3); // Return top 3 suggestions
    }

    /**
     * Suggest due date based on task title and description
     * @param {string} title - Task title
     * @param {string} description - Task description
     * @returns {Object|null} Suggested due date with confidence
     */
    suggestDueDate(title = '', description = '') {
        const text = `${title} ${description}`.toLowerCase();
        
        for (const { pattern, offset, offsetMultiplier, dayOfWeek } of this.dueDatePatterns) {
            const match = text.match(pattern);
            
            if (match) {
                const today = new Date();
                today.setHours(0, 0, 0, 0);
                let suggestedDate = new Date(today);

                if (offset !== undefined) {
                    // Fixed offset (today, tomorrow, next week)
                    suggestedDate.setDate(today.getDate() + offset);
                } else if (offsetMultiplier !== undefined && match[1]) {
                    // Variable offset (in X days/weeks/months)
                    const multiplier = parseInt(match[1]);
                    suggestedDate.setDate(today.getDate() + (multiplier * offsetMultiplier));
                } else if (dayOfWeek !== undefined) {
                    // Specific day of week
                    const currentDay = today.getDay();
                    const daysUntilTarget = (dayOfWeek - currentDay + 7) % 7;
                    const daysToAdd = daysUntilTarget === 0 ? 7 : daysUntilTarget; // Next occurrence
                    suggestedDate.setDate(today.getDate() + daysToAdd);
                }

                return {
                    date: suggestedDate.toISOString().split('T')[0],
                    matchedText: match[0],
                    confidence: 'high'
                };
            }
        }

        // Check for priority keywords that imply urgency
        if (/\b(urgent|asap|immediately|critical|important)\b/i.test(text)) {
            const today = new Date();
            return {
                date: today.toISOString().split('T')[0],
                matchedText: 'urgent keyword',
                confidence: 'medium'
            };
        }

        return null;
    }

    /**
     * Suggest priority based on task content
     * @param {string} title - Task title
     * @param {string} description - Task description
     * @returns {string} Suggested priority (low, medium, high)
     */
    suggestPriority(title = '', description = '') {
        const text = `${title} ${description}`.toLowerCase();

        // High priority indicators
        const highPriorityKeywords = ['urgent', 'asap', 'immediately', 'critical', 'important', 'deadline', 'emergency'];
        if (highPriorityKeywords.some(keyword => text.includes(keyword))) {
            return 'high';
        }

        // Low priority indicators
        const lowPriorityKeywords = ['someday', 'maybe', 'optional', 'nice to have', 'eventually', 'when possible'];
        if (lowPriorityKeywords.some(keyword => text.includes(keyword))) {
            return 'low';
        }

        // Default to medium
        return 'medium';
    }

    /**
     * Find related tasks based on similarity
     * @param {string} title - Task title
     * @param {string} description - Task description
     * @param {number} limit - Maximum number of related tasks
     * @returns {Array} Array of related tasks
     */
    findRelatedTasks(title = '', description = '', limit = 3) {
        const tasks = this.app.state.get('tasks');
        const searchText = `${title} ${description}`.toLowerCase();
        const searchWords = searchText.split(/\s+/).filter(word => word.length > 3);

        if (searchWords.length === 0) {
            return [];
        }

        const relatedTasks = tasks.map(task => {
            const taskText = `${task.title} ${task.description || ''}`.toLowerCase();
            let score = 0;

            // Count matching words
            for (const word of searchWords) {
                if (taskText.includes(word)) {
                    score += 1;
                }
            }

            // Bonus for same category or project
            // (Will be applied when we have the actual task data)

            return { task, score };
        })
        .filter(item => item.score > 0)
        .sort((a, b) => b.score - a.score)
        .slice(0, limit)
        .map(item => item.task);

        return relatedTasks;
    }

    /**
     * Get task template by name
     * @param {string} templateName - Name of the template
     * @returns {Object|null} Template data
     */
    getTemplate(templateName) {
        return this.templates[templateName] || null;
    }

    /**
     * Get all available templates
     * @returns {Array} Array of template names
     */
    getTemplateNames() {
        return Object.keys(this.templates);
    }

    /**
     * Generate comprehensive suggestions for a task
     * @param {string} title - Task title
     * @param {string} description - Task description
     * @returns {Object} All suggestions (category, due date, priority, related tasks)
     */
    generateSuggestions(title = '', description = '') {
        return {
            categories: this.suggestCategories(title, description),
            dueDate: this.suggestDueDate(title, description),
            priority: this.suggestPriority(title, description),
            relatedTasks: this.findRelatedTasks(title, description),
            timestamp: Date.now()
        };
    }

    /**
     * Render suggestions UI in a container
     * @param {Object} suggestions - Suggestions object
     * @param {HTMLElement} container - Container to render suggestions
     */
    renderSuggestions(suggestions, container) {
        if (!container) return;

        let html = '<div class="task-suggestions">';

        // Category suggestions
        if (suggestions.categories && suggestions.categories.length > 0) {
            html += '<div class="suggestion-section">';
            html += '<h4><i class="fas fa-tag"></i> Suggested Categories</h4>';
            html += '<div class="suggestion-items">';
            suggestions.categories.forEach(cat => {
                html += `
                    <button class="suggestion-item" onclick="app.taskAssistant.applyCategorySuggestion(${cat.categoryId})">
                        <span class="suggestion-label">${cat.categoryName}</span>
                        <span class="suggestion-confidence">${cat.confidence}%</span>
                    </button>
                `;
            });
            html += '</div></div>';
        }

        // Due date suggestion
        if (suggestions.dueDate) {
            html += '<div class="suggestion-section">';
            html += '<h4><i class="fas fa-calendar"></i> Suggested Due Date</h4>';
            html += '<div class="suggestion-items">';
            html += `
                <button class="suggestion-item" onclick="app.taskAssistant.applyDueDateSuggestion('${suggestions.dueDate.date}')">
                    <span class="suggestion-label">${new Date(suggestions.dueDate.date).toLocaleDateString()}</span>
                    <span class="suggestion-note">${suggestions.dueDate.matchedText}</span>
                </button>
            `;
            html += '</div></div>';
        }

        // Priority suggestion
        if (suggestions.priority) {
            html += '<div class="suggestion-section">';
            html += '<h4><i class="fas fa-flag"></i> Suggested Priority</h4>';
            html += '<div class="suggestion-items">';
            html += `
                <button class="suggestion-item priority-${suggestions.priority}" onclick="app.taskAssistant.applyPrioritySuggestion('${suggestions.priority}')">
                    <span class="suggestion-label">${suggestions.priority.toUpperCase()}</span>
                </button>
            `;
            html += '</div></div>';
        }

        // Related tasks
        if (suggestions.relatedTasks && suggestions.relatedTasks.length > 0) {
            html += '<div class="suggestion-section">';
            html += '<h4><i class="fas fa-link"></i> Related Tasks</h4>';
            html += '<div class="suggestion-items related-tasks">';
            suggestions.relatedTasks.forEach(task => {
                html += `
                    <div class="suggestion-item related-task">
                        <span class="suggestion-label">${this.escapeHtml(task.title)}</span>
                        <button class="btn-icon" onclick="app.ui.renderView('tasks'); app.highlightTask(${task.id})">
                            <i class="fas fa-external-link-alt"></i>
                        </button>
                    </div>
                `;
            });
            html += '</div></div>';
        }

        html += '</div>';
        container.innerHTML = html;
    }

    /**
     * Apply category suggestion to form
     * @param {number} categoryId - Category ID to apply
     */
    applyCategorySuggestion(categoryId) {
        const categorySelect = document.getElementById('task-category');
        if (categorySelect && categoryId) {
            categorySelect.value = categoryId;
            categorySelect.dispatchEvent(new Event('change'));
        }
    }

    /**
     * Apply due date suggestion to form
     * @param {string} date - Date string to apply
     */
    applyDueDateSuggestion(date) {
        const dueDateInput = document.getElementById('task-due-date');
        if (dueDateInput && date) {
            dueDateInput.value = date;
            dueDateInput.dispatchEvent(new Event('change'));
        }
    }

    /**
     * Apply priority suggestion to form
     * @param {string} priority - Priority to apply
     */
    applyPrioritySuggestion(priority) {
        const prioritySelect = document.getElementById('task-priority');
        if (prioritySelect && priority) {
            prioritySelect.value = priority;
            prioritySelect.dispatchEvent(new Event('change'));
        }
    }

    /**
     * Escape HTML for safe rendering
     * @param {string} text - Text to escape
     * @returns {string} Escaped text
     */
    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
}
