/**
 * SkeletonLoader - Generates skeleton loader components
 * Provides reusable skeleton templates for various UI components
 * @class SkeletonLoader
 */
class SkeletonLoader {
    /**
     * Generate a skeleton task card
     * @param {number} index - Optional index for unique key
     * @returns {string} HTML string for skeleton task
     */
    static task(index = 0) {
        return `
            <div class="skeleton-task" data-skeleton="task-${index}">
                <div class="skeleton-task-header">
                    <div class="skeleton skeleton-checkbox"></div>
                    <div class="skeleton skeleton-title"></div>
                </div>
                <div class="skeleton skeleton-text w-75"></div>
                <div class="skeleton skeleton-text w-50"></div>
                <div class="skeleton-task-meta">
                    <div class="skeleton skeleton-badge"></div>
                    <div class="skeleton skeleton-badge"></div>
                    <div class="skeleton skeleton-badge"></div>
                </div>
            </div>
        `;
    }

    /**
     * Generate multiple skeleton tasks
     * @param {number} count - Number of skeleton tasks to generate
     * @returns {string} HTML string for multiple skeleton tasks
     */
    static tasks(count = 5) {
        return `
            <div class="skeleton-container">
                ${Array.from({ length: count }, (_, i) => this.task(i)).join('')}
            </div>
        `;
    }

    /**
     * Generate a skeleton stat card
     * @param {number} index - Optional index for unique key
     * @returns {string} HTML string for skeleton stat
     */
    static stat(index = 0) {
        return `
            <div class="skeleton-stat" data-skeleton="stat-${index}">
                <div class="skeleton skeleton-icon"></div>
                <div class="skeleton-stat-content">
                    <div class="skeleton skeleton-text w-50"></div>
                    <div class="skeleton skeleton-title"></div>
                </div>
            </div>
        `;
    }

    /**
     * Generate skeleton stats grid (for dashboard)
     * @param {number} count - Number of stat cards (default 5)
     * @returns {string} HTML string for skeleton stats
     */
    static statsGrid(count = 5) {
        return `
            <div class="stats-grid">
                ${Array.from({ length: count }, (_, i) => this.stat(i)).join('')}
            </div>
        `;
    }

    /**
     * Generate skeleton featured stat (for mobile)
     * @returns {string} HTML string for skeleton featured stat
     */
    static featuredStat() {
        return `
            <div class="stats-featured">
                <div class="skeleton-stat featured-stat">
                    <div class="skeleton skeleton-icon" style="width: 64px; height: 64px;"></div>
                    <div class="skeleton-stat-content">
                        <div class="skeleton skeleton-text w-50"></div>
                        <div class="skeleton skeleton-title" style="height: 48px;"></div>
                    </div>
                </div>
                <div class="stats-grid" style="grid-template-columns: repeat(2, 1fr);">
                    ${this.stat(1)}
                    ${this.stat(2)}
                    ${this.stat(3)}
                    ${this.stat(4)}
                </div>
            </div>
        `;
    }

    /**
     * Generate a skeleton project card
     * @param {number} index - Optional index for unique key
     * @returns {string} HTML string for skeleton project
     */
    static project(index = 0) {
        return `
            <div class="skeleton-project" data-skeleton="project-${index}">
                <div class="skeleton-project-header">
                    <div class="skeleton skeleton-project-title"></div>
                    <div class="skeleton-project-actions">
                        <div class="skeleton skeleton-button"></div>
                        <div class="skeleton skeleton-button"></div>
                    </div>
                </div>
                <div class="skeleton skeleton-text w-75"></div>
                <div class="skeleton skeleton-text w-50"></div>
                <div class="skeleton-task-meta">
                    <div class="skeleton skeleton-badge"></div>
                    <div class="skeleton skeleton-badge"></div>
                </div>
            </div>
        `;
    }

    /**
     * Generate multiple skeleton projects
     * @param {number} count - Number of skeleton projects to generate
     * @returns {string} HTML string for multiple skeleton projects
     */
    static projects(count = 3) {
        return `
            <div class="skeleton-container">
                ${Array.from({ length: count }, (_, i) => this.project(i)).join('')}
            </div>
        `;
    }

    /**
     * Generate a skeleton category item
     * @param {number} index - Optional index for unique key
     * @returns {string} HTML string for skeleton category
     */
    static category(index = 0) {
        return `
            <li class="category-item" data-skeleton="category-${index}">
                <div class="skeleton skeleton-text w-75"></div>
                <div class="skeleton skeleton-badge" style="width: 32px; height: 20px;"></div>
            </li>
        `;
    }

    /**
     * Generate multiple skeleton categories
     * @param {number} count - Number of skeleton categories to generate
     * @returns {string} HTML string for multiple skeleton categories
     */
    static categories(count = 5) {
        return `
            <ul class="category-list">
                ${Array.from({ length: count }, (_, i) => this.category(i)).join('')}
            </ul>
        `;
    }

    /**
     * Generate a skeleton activity item (for activity log)
     * @param {number} index - Optional index for unique key
     * @returns {string} HTML string for skeleton activity
     */
    static activity(index = 0) {
        return `
            <div class="activity-item" data-skeleton="activity-${index}">
                <div class="skeleton skeleton-icon" style="width: 32px; height: 32px;"></div>
                <div style="flex: 1;">
                    <div class="skeleton skeleton-text w-75"></div>
                    <div class="skeleton skeleton-text w-25" style="margin-top: 4px;"></div>
                </div>
            </div>
        `;
    }

    /**
     * Generate multiple skeleton activities
     * @param {number} count - Number of skeleton activities to generate
     * @returns {string} HTML string for multiple skeleton activities
     */
    static activities(count = 5) {
        return `
            <div class="skeleton-container">
                ${Array.from({ length: count }, (_, i) => this.activity(i)).join('')}
            </div>
        `;
    }

    /**
     * Generate a generic loading skeleton container
     * @param {string} type - Type of skeleton loader
     * @param {number} count - Number of items to generate
     * @returns {string} HTML string for skeleton loader
     */
    static loader(type = 'task', count = 5) {
        switch (type) {
            case 'task':
                return this.tasks(count);
            case 'project':
                return this.projects(count);
            case 'category':
                return this.categories(count);
            case 'stat':
                return this.statsGrid(count);
            case 'featured-stat':
                return this.featuredStat();
            case 'activity':
                return this.activities(count);
            default:
                return this.tasks(count);
        }
    }

    /**
     * Show skeleton loader in a container
     * @param {HTMLElement|string} container - Container element or selector
     * @param {string} type - Type of skeleton loader
     * @param {number} count - Number of items to generate
     */
    static show(container, type = 'task', count = 5) {
        const element = typeof container === 'string' 
            ? document.querySelector(container) 
            : container;
        
        if (!element) {
            logger.warn(`SkeletonLoader: Container not found`);
            return;
        }

        element.innerHTML = this.loader(type, count);
    }

    /**
     * Remove skeleton loaders from a container
     * @param {HTMLElement|string} container - Container element or selector
     */
    static hide(container) {
        const element = typeof container === 'string' 
            ? document.querySelector(container) 
            : container;
        
        if (!element) {
            logger.warn(`SkeletonLoader: Container not found`);
            return;
        }

        // Remove all skeleton elements
        const skeletons = element.querySelectorAll('[data-skeleton]');
        skeletons.forEach(skeleton => skeleton.remove());

        // Remove skeleton containers
        const skeletonContainers = element.querySelectorAll('.skeleton-container');
        skeletonContainers.forEach(container => container.remove());
    }
}
