/**
 * QuickCapture - Floating Quick-Add Button
 * 
 * Provides a floating action button for rapid task creation
 * 
 * @author Jack Ewers / BloodWeb
 * @version 1.0.0
 */

class QuickCapture {
    constructor(app) {
        this.app = app;
        this.isOpen = false;
        this.initialize();
    }

    /**
     * Initialize quick capture FAB
     */
    initialize() {
        this.renderFAB();
        this.setupKeyboardShortcut();
    }

    /**
     * Render floating action button with speed dial
     */
    renderFAB() {
        const existing = document.getElementById('quick-capture-fab');
        if (existing) return;
        
        const fab = document.createElement('div');
        fab.id = 'quick-capture-fab';
        fab.className = 'quick-capture-fab';
        // Check if user is admin to show dev tools
        const isAdmin = this.app?.state?.get('user')?.role === 'admin' || window.DEV_MODE;
        
        fab.innerHTML = `
            <div class="speed-dial-actions">
                <button class="speed-dial-action speed-dial-task" data-action="task" title="New Task">
                    <i class="fas fa-tasks"></i>
                </button>
                <button class="speed-dial-action speed-dial-quick" data-action="quick" title="Quick Task">
                    <i class="fas fa-bolt"></i>
                </button>
                <button class="speed-dial-action speed-dial-category" data-action="category" title="New Category">
                    <i class="fas fa-folder"></i>
                </button>
                <button class="speed-dial-action speed-dial-project" data-action="project" title="New Project">
                    <i class="fas fa-briefcase"></i>
                </button>
                ${isAdmin ? `
                <button class="speed-dial-action speed-dial-dev" data-action="dev" title="Developer Tools">
                    <i class="fas fa-code"></i>
                </button>
                ` : ''}
            </div>
            <button class="fab-button" id="fab-main-btn" title="Quick Actions">
                <i class="fas fa-plus fab-icon"></i>
            </button>
        `;
        
        document.body.appendChild(fab);
        
        const mainBtn = document.getElementById('fab-main-btn');
        const speedDialActions = document.querySelectorAll('.speed-dial-action');
        
        mainBtn.addEventListener('click', () => {
            this.toggleSpeedDial();
        });
        
        speedDialActions.forEach(action => {
            action.addEventListener('click', (e) => {
                const actionType = e.currentTarget.dataset.action;
                this.handleSpeedDialAction(actionType);
            });
        });
    }
    
    /**
     * Toggle speed dial menu
     */
    toggleSpeedDial() {
        const fab = document.getElementById('quick-capture-fab');
        const isOpen = fab.classList.contains('speed-dial-open');
        
        if (isOpen) {
            this.closeSpeedDial();
        } else {
            this.openSpeedDial();
        }
    }
    
    /**
     * Open speed dial menu
     */
    openSpeedDial() {
        const fab = document.getElementById('quick-capture-fab');
        const icon = fab.querySelector('.fab-icon');
        fab.classList.add('speed-dial-open');
        icon.style.transform = 'rotate(45deg)';
    }
    
    /**
     * Close speed dial menu
     */
    closeSpeedDial() {
        const fab = document.getElementById('quick-capture-fab');
        const icon = fab.querySelector('.fab-icon');
        fab.classList.remove('speed-dial-open');
        icon.style.transform = 'rotate(0deg)';
    }
    
    /**
     * Handle speed dial action
     */
    handleSpeedDialAction(actionType) {
        this.closeSpeedDial();
        
        switch(actionType) {
            case 'task':
                this.app.openTaskModal();
                break;
            case 'quick':
                this.open();
                break;
            case 'category':
                this.app.openCategoryModal();
                break;
            case 'project':
                this.app.openProjectModal();
                break;
            case 'dev':
                if (typeof this.app.toggleDevBar === 'function') {
                    this.app.toggleDevBar();
                }
                break;
        }
    }

    /**
     * Setup keyboard shortcut (Shift+N)
     */
    setupKeyboardShortcut() {
        document.addEventListener('keydown', (e) => {
            // Shift+N to toggle quick capture
            if (e.shiftKey && e.key.toLowerCase() === 'n') {
                // Don't trigger if user is typing
                if (e.target.tagName === 'INPUT' || e.target.tagName === 'TEXTAREA') {
                    return;
                }
                
                e.preventDefault();
                this.toggle();
            }
        });
    }

    /**
     * Toggle quick capture panel
     */
    toggle() {
        if (this.isOpen) {
            this.close();
        } else {
            this.open();
        }
    }

    /**
     * Open quick capture panel
     */
    open() {
        this.isOpen = true;
        
        // Hide FAB button, show form
        const fabBtn = document.getElementById('fab-main-btn');
        if (fabBtn) fabBtn.classList.add('hidden');
        
        this.renderQuickForm();
    }

    /**
     * Close quick capture panel
     */
    close() {
        this.isOpen = false;
        
        const form = document.getElementById('quick-capture-form');
        if (form) form.remove();
        
        const fabBtn = document.getElementById('fab-main-btn');
        if (fabBtn) fabBtn.classList.remove('hidden');
    }

    /**
     * Render quick capture form
     */
    renderQuickForm() {
        const existing = document.getElementById('quick-capture-form');
        if (existing) return;
        
        const projects = this.app.state.get('projects') || [];
        const currentProject = this.app.state.get('selectedProject');
        
        const formContainer = document.createElement('div');
        formContainer.id = 'quick-capture-form';
        formContainer.className = 'quick-capture-form';
        formContainer.innerHTML = `
            <div class="quick-capture-content">
                <div class="quick-capture-header">
                    <h4><i class="fas fa-bolt"></i> Quick Add Task</h4>
                    <button class="btn-close" id="quick-capture-close">
                        <i class="fas fa-times"></i>
                    </button>
                </div>
                <form id="quick-task-form">
                    <div class="form-group">
                        <input 
                            type="text" 
                            id="quick-task-title" 
                            class="form-control" 
                            placeholder="What needs to be done?" 
                            required
                            autocomplete="off"
                        />
                    </div>
                    <div class="form-group">
                        <select id="quick-task-project" class="form-control">
                            <option value="">No Project</option>
                            ${projects.map(p => `
                                <option value="${p.id}" ${p.id === currentProject ? 'selected' : ''}>
                                    ${p.name}
                                </option>
                            `).join('')}
                        </select>
                    </div>
                    <div class="form-actions">
                        <button type="button" class="btn btn-text" id="quick-voice-btn" title="Voice input">
                            <i class="fas fa-microphone"></i>
                        </button>
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-check"></i> Add Task
                        </button>
                    </div>
                </form>
            </div>
        `;
        
        document.getElementById('quick-capture-fab').appendChild(formContainer);
        
        // Focus title input
        const titleInput = document.getElementById('quick-task-title');
        setTimeout(() => titleInput.focus(), 100);
        
        this.bindQuickFormEvents();
    }

    /**
     * Bind quick form events
     */
    bindQuickFormEvents() {
        // Close button
        document.getElementById('quick-capture-close')?.addEventListener('click', () => {
            this.close();
        });
        
        // Form submission
        document.getElementById('quick-task-form')?.addEventListener('submit', async (e) => {
            e.preventDefault();
            await this.submitQuickTask();
        });
        
        // Voice input
        document.getElementById('quick-voice-btn')?.addEventListener('click', () => {
            this.startVoiceInput();
        });
        
        // Escape key to close
        document.addEventListener('keydown', (e) => {
            if (e.key === 'Escape' && this.isOpen) {
                this.close();
            }
        });
    }

    /**
     * Submit quick task
     */
    async submitQuickTask() {
        const title = document.getElementById('quick-task-title').value.trim();
        const projectId = document.getElementById('quick-task-project').value;
        
        if (!title) {
            this.app.ui.showToast('Please enter a task title', 'error');
            return;
        }
        
        try {
            const taskData = {
                title,
                project_id: projectId || null,
                status: 'pending',
                priority: 'medium', // Default priority
                due_date: null
            };
            
            const response = await this.app.api.post('tasks', taskData);
            
            if (response.success) {
                this.showSuccessAnimation();
                
                // Reload tasks if on tasks view
                if (this.app.state.get('currentView') === 'tasks') {
                    await this.app.loadTasks();
                }
                
                // Close form after short delay
                setTimeout(() => {
                    this.close();
                }, 1000);
                
                this.app.ui.showToast('Task added!', 'success');
            }
        } catch (error) {
            this.app.ui.showToast('Failed to add task', 'error');
            logger.error('[QuickCapture] Submit failed:', error);
        }
    }

    /**
     * Show success animation
     */
    showSuccessAnimation() {
        const form = document.getElementById('quick-task-form');
        if (!form) return;
        
        form.innerHTML = `
            <div class="quick-capture-success">
                <i class="fas fa-check-circle"></i>
                <p>Task added!</p>
            </div>
        `;
        
        form.classList.add('success');
    }

    /**
     * Start voice input
     */
    startVoiceInput() {
        if (!('webkitSpeechRecognition' in window) && !('SpeechRecognition' in window)) {
            this.app.ui.showToast('Voice input not supported in this browser', 'error');
            return;
        }
        
        const SpeechRecognition = window.SpeechRecognition || window.webkitSpeechRecognition;
        const recognition = new SpeechRecognition();
        
        recognition.lang = 'en-US';
        recognition.interimResults = false;
        recognition.maxAlternatives = 1;
        
        const voiceBtn = document.getElementById('quick-voice-btn');
        voiceBtn.classList.add('listening');
        voiceBtn.innerHTML = '<i class="fas fa-circle"></i>';
        
        recognition.onresult = (event) => {
            const transcript = event.results[0][0].transcript;
            document.getElementById('quick-task-title').value = transcript;
            voiceBtn.classList.remove('listening');
            voiceBtn.innerHTML = '<i class="fas fa-microphone"></i>';
        };
        
        recognition.onerror = () => {
            this.app.ui.showToast('Voice input failed', 'error');
            voiceBtn.classList.remove('listening');
            voiceBtn.innerHTML = '<i class="fas fa-microphone"></i>';
        };
        
        recognition.onend = () => {
            voiceBtn.classList.remove('listening');
            voiceBtn.innerHTML = '<i class="fas fa-microphone"></i>';
        };
        
        recognition.start();
    }
}
