/**
 * KeyboardManager - Global Keyboard Shortcuts Handler
 * 
 * Provides keyboard navigation and command palette functionality
 * Uses window.logger for configurable logging (set in main.js)
 * 
 * @author Jack Ewers / BloodWeb
 * @version 2.0.1
 */

class KeyboardManager {
    constructor(app) {
        this.app = app;
        this.shortcuts = CONFIG.UI.keyboard.shortcuts;
        this.enabled = CONFIG.UI.keyboard.enabled;
        this.commandPaletteOpen = false;
        this.pressedKeys = new Set();
        
        if (this.enabled) {
            this.initialize();
        }
    }
    
    initialize() {
        document.addEventListener('keydown', (e) => this.handleKeyDown(e));
        document.addEventListener('keyup', (e) => this.handleKeyUp(e));
    }
    
    handleKeyDown(e) {
        // Don't trigger shortcuts if user is typing in an input
        if (this.isTyping(e.target)) {
            return;
        }
        
        // Track pressed keys for combinations
        this.pressedKeys.add(e.key);
        
        // Build key combination string
        const combo = this.getKeyCombo(e);
        
        // Check if this combo matches any shortcut
        const action = this.shortcuts[combo];
        
        if (action) {
            e.preventDefault();
            this.executeAction(action);
        }
    }
    
    handleKeyUp(e) {
        this.pressedKeys.delete(e.key);
    }
    
    getKeyCombo(e) {
        const parts = [];
        
        if (e.ctrlKey || e.metaKey) parts.push('ctrl');
        if (e.shiftKey) parts.push('shift');
        if (e.altKey) parts.push('alt');
        
        // Add the main key
        if (!['Control', 'Shift', 'Alt', 'Meta'].includes(e.key)) {
            parts.push(e.key.toLowerCase());
        }
        
        return parts.join('+');
    }
    
    isTyping(element) {
        const tagName = element.tagName.toLowerCase();
        const isInput = ['input', 'textarea', 'select'].includes(tagName);
        const isContentEditable = element.isContentEditable;
        
        return isInput || isContentEditable;
    }
    
    executeAction(action) {
        logger.debug('[KeyboardManager] Executing action:', action);
        
        switch(action) {
            case 'newTask':
                this.app.openTaskModal();
                break;
            case 'viewTasks':
                this.app.navigateToView('tasks');
                break;
            case 'viewDashboard':
                this.app.navigateToView('dashboard');
                break;
            case 'newProject':
                this.app.openProjectModal();
                break;
            case 'newCategory':
                this.app.openCategoryModal();
                break;
            case 'focusSearch':
                this.focusSearch();
                break;
            case 'closeModal':
                if (this.app.modals.activeModal) {
                    this.app.modals.closeModal(this.app.modals.activeModal);
                }
                break;
            case 'save':
                this.saveActiveForm();
                break;
            case 'commandPalette':
                this.toggleCommandPalette();
                break;
            case 'showHelp':
                this.showKeyboardHelp();
                break;
            default:
                logger.warn('[KeyboardManager] Unknown action:', action);
        }
    }
    
    focusSearch() {
        const searchInput = document.getElementById('search-input');
        if (searchInput) {
            searchInput.focus();
            searchInput.select();
        }
    }
    
    saveActiveForm() {
        const activeModal = this.app.modals.activeModal;
        if (!activeModal) return;
        
        const form = document.querySelector(`#${activeModal} form`);
        if (form) {
            form.requestSubmit();
        }
    }
    
    toggleCommandPalette() {
        this.commandPaletteOpen = !this.commandPaletteOpen;
        
        if (this.commandPaletteOpen) {
            this.renderCommandPalette();
        } else {
            this.closeCommandPalette();
        }
    }
    
    renderCommandPalette() {
        const existing = document.getElementById('command-palette');
        if (existing) {
            existing.remove();
        }
        
        const palette = document.createElement('div');
        palette.id = 'command-palette';
        palette.className = 'command-palette';
        palette.innerHTML = `
            <div class="command-palette-backdrop" onclick="app.keyboard.closeCommandPalette()"></div>
            <div class="command-palette-container">
                <div class="command-palette-header">
                    <i class="fas fa-terminal"></i>
                    <input 
                        type="text" 
                        id="command-palette-input" 
                        placeholder="Type a command or search..." 
                        autocomplete="off"
                    >
                </div>
                <div class="command-palette-results" id="command-palette-results">
                    ${this.renderCommandList()}
                </div>
            </div>
        `;
        
        document.body.appendChild(palette);
        
        // Focus input
        const input = document.getElementById('command-palette-input');
        input.focus();
        
        // Add input listener for filtering
        input.addEventListener('input', (e) => this.filterCommands(e.target.value));
        input.addEventListener('keydown', (e) => this.handlePaletteKeydown(e));
    }
    
    renderCommandList(filter = '') {
        const commands = [
            { icon: 'fa-plus', label: 'New Task', action: 'newTask', keys: 'N' },
            { icon: 'fa-folder-plus', label: 'New Project', action: 'newProject', keys: 'P' },
            { icon: 'fa-tag', label: 'New Category', action: 'newCategory', keys: 'C' },
            { icon: 'fa-tachometer-alt', label: 'Go to Dashboard', action: 'viewDashboard', keys: 'D' },
            { icon: 'fa-tasks', label: 'Go to Tasks', action: 'viewTasks', keys: 'T' },
            { icon: 'fa-folder-open', label: 'Go to Projects', action: 'viewProjects', keys: '' },
            { icon: 'fa-tags', label: 'Go to Categories', action: 'viewCategories', keys: '' },
            { icon: 'fa-user-circle', label: 'Go to Account', action: 'viewAccount', keys: '' },
            { icon: 'fa-cog', label: 'Go to Settings', action: 'viewSettings', keys: '' },
            { icon: 'fa-question-circle', label: 'Show Keyboard Shortcuts', action: 'showHelp', keys: '?' }
        ];
        
        const filtered = filter 
            ? commands.filter(cmd => cmd.label.toLowerCase().includes(filter.toLowerCase()))
            : commands;
        
        if (filtered.length === 0) {
            return '<div class="command-palette-empty">No commands found</div>';
        }
        
        return filtered.map((cmd, index) => `
            <div class="command-palette-item ${index === 0 ? 'active' : ''}" 
                 data-action="${cmd.action}"
                 data-index="${index}"
                 onclick="app.keyboard.executeAction('${cmd.action}'); app.keyboard.closeCommandPalette();">
                <i class="fas ${cmd.icon}"></i>
                <span class="command-label">${cmd.label}</span>
                ${cmd.keys ? `<span class="command-keys">${cmd.keys}</span>` : ''}
            </div>
        `).join('');
    }
    
    filterCommands(value) {
        const results = document.getElementById('command-palette-results');
        results.innerHTML = this.renderCommandList(value);
    }
    
    handlePaletteKeydown(e) {
        if (e.key === 'Escape') {
            e.preventDefault();
            this.closeCommandPalette();
            return;
        }
        
        if (e.key === 'Enter') {
            e.preventDefault();
            const activeItem = document.querySelector('.command-palette-item.active');
            if (activeItem) {
                const action = activeItem.dataset.action;
                this.closeCommandPalette();
                if (action.startsWith('view')) {
                    const view = action.replace('view', '').toLowerCase();
                    this.app.navigateToView(view);
                } else {
                    this.executeAction(action);
                }
            }
            return;
        }
        
        if (e.key === 'ArrowDown' || e.key === 'ArrowUp') {
            e.preventDefault();
            this.navigatePaletteItems(e.key === 'ArrowDown' ? 1 : -1);
        }
    }
    
    navigatePaletteItems(direction) {
        const items = document.querySelectorAll('.command-palette-item');
        const activeItem = document.querySelector('.command-palette-item.active');
        
        if (!activeItem || items.length === 0) return;
        
        const currentIndex = parseInt(activeItem.dataset.index);
        let newIndex = currentIndex + direction;
        
        if (newIndex < 0) newIndex = items.length - 1;
        if (newIndex >= items.length) newIndex = 0;
        
        activeItem.classList.remove('active');
        items[newIndex].classList.add('active');
        items[newIndex].scrollIntoView({ block: 'nearest' });
    }
    
    closeCommandPalette() {
        const palette = document.getElementById('command-palette');
        if (palette) {
            palette.remove();
        }
        this.commandPaletteOpen = false;
    }
    
    showKeyboardHelp() {
        const shortcuts = Object.entries(this.shortcuts).map(([key, action]) => ({
            key: key.replace('ctrl+', '⌘ ').replace('shift+', '⇧ '),
            action: action.replace(/([A-Z])/g, ' $1').trim()
        }));
        
        const helpHTML = `
            <div class="keyboard-help">
                <p>Use these keyboard shortcuts to navigate faster:</p>
                <table class="shortcuts-table">
                    <thead>
                        <tr>
                            <th>Shortcut</th>
                            <th>Action</th>
                        </tr>
                    </thead>
                    <tbody>
                        ${shortcuts.map(s => `
                            <tr>
                                <td class="shortcut-key"><kbd>${s.key}</kbd></td>
                                <td class="shortcut-action">${s.action}</td>
                            </tr>
                        `).join('')}
                    </tbody>
                </table>
            </div>
        `;
        
        this.app.modals.showCustomModal('Keyboard Shortcuts', helpHTML);
    }
}
