/**
 * FocusMode - Daily Focus View with Pomodoro Timer
 * 
 * Helps users focus on priority tasks with timer integration
 * 
 * @author Jack Ewers / BloodWeb
 * @version 1.0.0
 */

class FocusMode {
    constructor(app) {
        this.app = app;
        this.timer = {
            minutes: 25,
            seconds: 0,
            isRunning: false,
            isPaused: false,
            interval: null,
            mode: 'work' // 'work', 'short-break', 'long-break'
        };
        this.focusTasks = [];
        this.sessionsToday = 0;
        this.streak = 0;
        this.currentTaskIndex = 0;
        this.isInFocusDialog = false;
        this.focusNotes = [];
        
        // Focus configuration (stored in localStorage)
        this.config = this.loadConfig();
    }
    
    /**
     * Load focus configuration from localStorage
     */
    loadConfig() {
        const defaults = {
            focusSource: 'priority', // 'priority', 'project', 'category', 'oldest', 'due-today'
            selectedProjectId: null,
            selectedCategoryId: null,
            exitOnCompletion: true,
            autoPickNext: true,
            sessionDuration: 25, // minutes
            breakDuration: 5, // minutes for breaks
            shortBreakDuration: 5,
            longBreakDuration: 15
        };
        
        const saved = localStorage.getItem('TL_focusConfig');
        return saved ? { ...defaults, ...JSON.parse(saved) } : defaults;
    }
    
    /**
     * Save focus configuration to localStorage
     */
    saveConfig() {
        localStorage.setItem('TL_focusConfig', JSON.stringify(this.config));
    }

    /**
     * Render focus mode view
     */
    async render() {
        await this.loadFocusTasks();
        await this.loadStreakData();
        
        const projects = this.app.state.get('projects') || [];
        const categories = this.app.state.get('categories') || [];
        
        return `
            <div class="focus-mode" style="margin-top: -2rem;">
                <!-- Focus Header with Duration Clock -->
                <div class="focus-header-card">
                    <h2 class="focus-header-title"><i class="fas fa-fire"></i> Focus</h2>
                    <div class="focus-duration-section">
                        <label class="focus-clock-label">Session Duration</label>
                        <div class="analog-clock-container">
                            <svg class="analog-clock" viewBox="0 0 240 240" id="duration-clock">
                                <circle cx="120" cy="120" r="100" fill="none" stroke="#e5e7eb" stroke-width="3"/>
                                <!-- Hour markers -->
                                ${this.renderClockMarkers()}
                                <!-- Clock hand -->
                                <line id="clock-hand" x1="120" y1="120" x2="120" y2="40" 
                                    stroke="var(--color-primary)" stroke-width="6" stroke-linecap="round"/>
                                <circle cx="120" cy="120" r="10" fill="var(--color-primary)"/>
                            </svg>
                            <div class="clock-duration-display">${this.formatDuration(this.config.sessionDuration)}</div>
                        </div>
                        <input type="hidden" id="session-duration" value="${this.config.sessionDuration}">
                    </div>
                </div>

                <!-- Start Focus Button -->
                <div class="focus-action-card">
                    <button class="btn btn-success btn-lg" id="start-immersive-focus">
                        <i class="fas fa-bullseye"></i> Enter Focus Mode
                    </button>
                    <p class="focus-action-hint">
                        ${this.focusTasks.length} task(s) ready • ${this.formatDuration(this.config.sessionDuration)} sessions
                    </p>
                </div>

                <!-- Configuration Panel -->
                <div class="focus-config-card">
                    <h3><i class="fas fa-sliders-h"></i> Focus Items</h3>
                    <div class="focus-config-grid">
                        <div class="config-group">
                            <label for="focus-source">What to Focus On</label>
                            <select id="focus-source" class="form-control">
                                <option value="priority" ${this.config.focusSource === 'priority' ? 'selected' : ''}>Priority Tasks</option>
                                <option value="project" ${this.config.focusSource === 'project' ? 'selected' : ''}>Specific Project</option>
                                <option value="category" ${this.config.focusSource === 'category' ? 'selected' : ''}>Specific Category</option>
                                <option value="oldest" ${this.config.focusSource === 'oldest' ? 'selected' : ''}>Oldest Tasks</option>
                                <option value="due-today" ${this.config.focusSource === 'due-today' ? 'selected' : ''}>Due Today</option>
                            </select>
                        </div>
                        
                        <div class="config-group" id="project-select-group" style="display: ${this.config.focusSource === 'project' ? 'block' : 'none'}">
                            <label for="focus-project">Select Project</label>
                            <select id="focus-project" class="form-control">
                                <option value="">Choose project...</option>
                                ${projects.map(p => `<option value="${p.id}" ${this.config.selectedProjectId == p.id ? 'selected' : ''}>${this.app.ui.escapeHtml(p.name)}</option>`).join('')}
                            </select>
                        </div>
                        
                        <div class="config-group" id="category-select-group" style="display: ${this.config.focusSource === 'category' ? 'block' : 'none'}">
                            <label for="focus-category">Select Category</label>
                            <select id="focus-category" class="form-control">
                                <option value="">Choose category...</option>
                                ${categories.map(c => `<option value="${c.id}" ${this.config.selectedCategoryId == c.id ? 'selected' : ''}>${this.app.ui.escapeHtml(c.name)}</option>`).join('')}
                            </select>
                        </div>
                        
                        <div class="config-group">
                            <label for="break-duration">Break Duration</label>
                            <select id="break-duration" class="form-control">
                                <option value="5" ${this.config.breakDuration === 5 ? 'selected' : ''}>5 minutes</option>
                                <option value="10" ${this.config.breakDuration === 10 ? 'selected' : ''}>10 minutes</option>
                                <option value="30" ${this.config.breakDuration === 30 ? 'selected' : ''}>30 minutes</option>
                            </select>
                        </div>
                        
                        <div class="config-group">
                            <label class="checkbox-label">
                                <input type="checkbox" id="exit-on-completion" ${this.config.exitOnCompletion ? 'checked' : ''}>
                                <span>Exit focus mode when all tasks complete</span>
                            </label>
                        </div>
                        
                        <div class="config-group">
                            <label class="checkbox-label">
                                <input type="checkbox" id="auto-pick-next" ${this.config.autoPickNext ? 'checked' : ''}>
                                <span>Auto-pick oldest tasks when out of focus items</span>
                            </label>
                        </div>
                    </div>
                    <button class="btn btn-primary" id="save-focus-config">
                        <i class="fas fa-save"></i> Save Configuration
                    </button>
                </div>

                <!-- Stats Overview -->
                <div class="focus-stats-overview">
                    <div class="stat-card">
                        <i class="fas fa-clock stat-icon"></i>
                        <div class="stat-content">
                            <span class="stat-value">${this.sessionsToday}</span>
                            <span class="stat-label">Sessions Today</span>
                        </div>
                    </div>
                    <div class="stat-card">
                        <i class="fas fa-check-circle stat-icon"></i>
                        <div class="stat-content">
                            <span class="stat-value">${this.focusTasks.filter(t => t.status === 'completed').length}/${this.focusTasks.length}</span>
                            <span class="stat-label">Tasks Done</span>
                        </div>
                    </div>
                </div>
                
                <!-- Task Preview List -->
                <div class="focus-preview-card">
                    <h3>Tasks in Queue</h3>
                    <div class="focus-preview-list" id="focus-preview-list">
                        ${this.focusTasks.length > 0 ? this.focusTasks.slice(0, 5).map((task, idx) => `
                            <div class="focus-preview-item ${task.status === 'completed' ? 'completed' : ''}" data-task-id="${task.id}" draggable="true">
                                <button class="btn-drag-handle" title="Drag to reorder">
                                    <i class="fas fa-grip-vertical"></i>
                                </button>
                                <span class="preview-number">${idx + 1}</span>
                                <span class="preview-title">${this.app.ui.escapeHtml(task.title)}</span>
                                <span class="preview-priority priority-${task.priority}">${task.priority}</span>
                                <button class="btn-remove-from-queue" data-task-id="${task.id}" title="Remove from queue">
                                    <i class="fas fa-minus"></i>
                                </button>
                            </div>
                        `).join('') : '<p class="focus-empty">No tasks in queue. Adjust your focus configuration above.</p>'}
                    </div>
                </div>
            </div>
        `;
    }

    /**
     * Load focus tasks based on configuration
     */
    async loadFocusTasks() {
        try {
            // Get tasks based on focus source
            const allTasks = this.app.state.get('tasks') || [];
            let filtered = allTasks.filter(t => t.status !== 'completed' && t.status !== 'cancelled');
            
            switch (this.config.focusSource) {
                case 'project':
                    if (this.config.selectedProjectId) {
                        filtered = filtered.filter(t => t.project_id == this.config.selectedProjectId);
                    }
                    break;
                case 'category':
                    if (this.config.selectedCategoryId) {
                        filtered = filtered.filter(t => t.category_id == this.config.selectedCategoryId);
                    }
                    break;
                case 'oldest':
                    filtered.sort((a, b) => new Date(a.created_at) - new Date(b.created_at));
                    break;
                case 'due-today':
                    const today = new Date().toISOString().split('T')[0];
                    filtered = filtered.filter(t => t.due_date === today);
                    break;
                case 'priority':
                default:
                    filtered.sort((a, b) => {
                        const priorityOrder = { urgent: 4, high: 3, medium: 2, low: 1 };
                        return (priorityOrder[b.priority] || 0) - (priorityOrder[a.priority] || 0);
                    });
                    break;
            }
            
            this.focusTasks = filtered.slice(0, 10); // Limit to 10 tasks
        } catch (error) {
            logger.error('[FocusMode] Failed to load focus tasks:', error);
            this.focusTasks = [];
        }
    }
    
    /**
     * Enter immersive focus mode dialog
     */
    enterImmersiveFocus() {
        if (this.focusTasks.length === 0) {
            this.app.ui.showToast('No tasks available to focus on', 'warning');
            return;
        }
        
        this.isInFocusDialog = true;
        this.currentTaskIndex = 0;
        this.focusNotes = [];
        
        // Set timer to configured duration (get latest from hidden input)
        const durationInput = document.getElementById('session-duration');
        const duration = durationInput ? parseInt(durationInput.value) : this.config.sessionDuration;
        this.timer.minutes = duration;
        this.timer.seconds = 0;
        
        // Create and show dialog
        const dialog = document.createElement('div');
        dialog.id = 'immersive-focus-dialog';
        dialog.className = 'immersive-focus-overlay';
        dialog.innerHTML = this.renderImmersiveFocusDialog();
        
        document.body.appendChild(dialog);
        document.body.style.overflow = 'hidden';
        
        // Bind events
        this.bindImmersiveEvents();
        
        // Auto-start timer
        setTimeout(() => this.startTimer(), 500);
    }
    
    /**
     * Render immersive focus dialog content
     */
    renderImmersiveFocusDialog() {
        const currentTask = this.focusTasks[this.currentTaskIndex];
        const progress = ((this.currentTaskIndex) / this.focusTasks.length) * 100;
        
        if (!currentTask) {
            return this.renderNoTasksView();
        }
        
        return `
            <div class="immersive-focus-container">
                <!-- Top Bar: Progress, Controls, Notes -->
                <div class="immersive-top-bar">
                    <!-- Exit Session -->
                    <button class="btn-exit-session" id="immersive-end">
                        <i class="fas fa-sign-out-alt"></i>
                        <span>Exit</span>
                    </button>
                    
                    <!-- Progress Bar -->
                    <div class="immersive-progress">
                        <div class="immersive-progress-bar" style="width: ${progress}%"></div>
                        <span class="immersive-progress-text">${this.currentTaskIndex} of ${this.focusTasks.length}</span>
                    </div>
                    
                    <!-- Notes Toggle -->
                    <button class="immersive-notes-toggle" id="toggle-notes">
                        <i class="fas fa-sticky-note"></i>
                        <span>Notes</span>
                        ${this.focusNotes.length > 0 ? `<span class="notes-count">${this.focusNotes.length}</span>` : ''}
                    </button>
                </div>
                
                <!-- Timer Display (Prominent & Clickable) -->
                <div class="immersive-timer" id="timer-click-area" title="Click to pause/resume">
                    <div class="immersive-timer-circle">
                        <svg class="immersive-timer-ring" viewBox="0 0 200 200">
                            <circle cx="100" cy="100" r="90" class="timer-ring-bg"/>
                            <circle cx="100" cy="100" r="90" class="timer-ring-progress" 
                                style="stroke-dashoffset: ${this.getTimerDashOffset()}"/>
                        </svg>
                        <div class="immersive-timer-time">
                            <span class="immersive-minutes">${String(this.timer.minutes).padStart(2, '0')}</span>
                            <span class="immersive-colon">:</span>
                            <span class="immersive-seconds">${String(this.timer.seconds).padStart(2, '0')}</span>
                        </div>
                    </div>
                </div>
                
                <!-- Task/Notes Container -->
                <div class="immersive-content-area">
                    <!-- Current Task (Large Display) -->
                    <div class="immersive-task" id="immersive-task-view">
                        <h1 class="immersive-task-title">${this.app.ui.escapeHtml(currentTask.title)}</h1>
                        ${currentTask.description ? `
                            <p class="immersive-task-description">${this.app.ui.escapeHtml(currentTask.description)}</p>
                        ` : ''}
                        <div class="immersive-task-meta">
                            <span class="meta-badge priority-${currentTask.priority}">
                                <i class="fas fa-flag"></i> ${currentTask.priority}
                            </span>
                            ${currentTask.project_name ? `
                                <span class="meta-badge">
                                    <i class="fas fa-folder"></i> ${this.app.ui.escapeHtml(currentTask.project_name)}
                                </span>
                            ` : ''}
                            ${currentTask.due_date ? `
                                <span class="meta-badge">
                                    <i class="fas fa-calendar"></i> ${currentTask.due_date}
                                </span>
                            ` : ''}
                        </div>
                        ${this.renderSubtasksForTask(currentTask)}
                    </div>
                
                    <!-- Notes Section (Replaces Task When Open) -->
                    <!-- Notes View (Hidden by default) -->
                    <div class="immersive-notes-view" id="immersive-notes-view" style="display: none;">
                        <div class="notes-view-header">
                            <h2><i class="fas fa-sticky-note"></i> Quick Notes</h2>
                            <button class="btn-close-notes" id="close-notes">
                                <i class="fas fa-times"></i>
                            </button>
                        </div>
                        <textarea 
                            class="immersive-notes-input" 
                            id="focus-note-input" 
                            placeholder="Capture thoughts, ideas, blockers... (Press Ctrl+Enter to save)"
                            rows="8"
                        ></textarea>
                        <button class="btn btn-primary" id="save-focus-note" style="width: 100%; margin-bottom: 1rem;">
                            <i class="fas fa-plus"></i> Add Note
                        </button>
                        <div class="immersive-notes-list">
                            ${this.focusNotes.map((note, idx) => `
                                <div class="focus-note-item">
                                    <span class="note-text">${this.app.ui.escapeHtml(note.text)}</span>
                                    <button class="btn-note-delete" data-index="${idx}">
                                        <i class="fas fa-times"></i>
                                    </button>
                                </div>
                            `).join('')}
                        </div>
                    </div>
                </div>
                
                <!-- Action Buttons -->
                <div class="immersive-actions">
                    <button class="btn-immersive btn-immersive-info" id="immersive-skip">
                        <i class="fas fa-forward"></i>
                        <span>Skip Task</span>
                    </button>
                    
                    <div class="actions-dropdown-wrapper">
                        <button class="btn-immersive btn-immersive-secondary" id="toggle-actions-menu">
                            <i class="fas fa-ellipsis-v"></i>
                            <span>Actions</span>
                        </button>
                        <div class="actions-dropdown-menu" id="actions-dropdown-menu" style="display: none;">
                            <button class="action-menu-item" id="immersive-break">
                                <i class="fas fa-coffee"></i>
                                <span>Take Break</span>
                            </button>
                            <button class="action-menu-item" id="menu-extend">
                                <i class="fas fa-plus-circle"></i>
                                <span>Add 5 Minutes</span>
                            </button>
                        </div>
                    </div>
                    
                    <button class="btn-immersive btn-immersive-primary" id="immersive-complete">
                        <i class="fas fa-check"></i>
                        <span>Complete Task</span>
                    </button>
                </div>
            </div>
        `;
    }
    
    /**
     * Render subtasks for a task
     */
    renderSubtasksForTask(task) {
        const subtasks = task.subtasks || [];
        
        if (subtasks.length === 0) {
            return '';
        }
        
        const completedCount = subtasks.filter(st => st.completed).length;
        const allComplete = completedCount === subtasks.length;
        
        // Find current (first incomplete) subtask
        const currentIndex = subtasks.findIndex(st => !st.completed);
        const currentSubtask = currentIndex >= 0 ? subtasks[currentIndex] : null;
        const nextSubtask = currentIndex >= 0 && currentIndex + 1 < subtasks.length ? subtasks[currentIndex + 1] : null;
        
        // If all complete, show full list with completion options
        if (allComplete) {
            return `
                <div class="immersive-subtasks-complete">
                    <div class="subtasks-complete-header">
                        <i class="fas fa-check-circle"></i>
                        <span>All Subtasks Complete!</span>
                    </div>
                    <div class="subtasks-list">
                        ${subtasks.map((subtask, idx) => `
                            <div class="subtask-item completed">
                                <i class="fas fa-check"></i>
                                <span class="subtask-title">${this.app.ui.escapeHtml(subtask.title)}</span>
                            </div>
                        `).join('')}
                    </div>
                </div>
            `;
        }
        
        // Show current subtask prominently and next subtask preview
        return `
            <div class="immersive-subtask-focus">
                <div class="subtask-progress-indicator">
                    <span class="progress-text">${completedCount + 1} of ${subtasks.length}</span>
                </div>
                
                ${currentSubtask ? `
                    <div class="current-subtask">
                        <label class="current-subtask-item">
                            <input 
                                type="checkbox" 
                                class="subtask-checkbox-large" 
                                data-task-id="${task.id}"
                                data-subtask-index="${currentIndex}"
                            />
                            <span class="current-subtask-title">${this.app.ui.escapeHtml(currentSubtask.title)}</span>
                        </label>
                    </div>
                ` : ''}
                
                ${nextSubtask ? `
                    <div class="up-next-subtask">
                        <span class="up-next-label"><i class="fas fa-arrow-down"></i> Up Next</span>
                        <span class="up-next-title">${this.app.ui.escapeHtml(nextSubtask.title)}</span>
                    </div>
                ` : ''}
            </div>
        `;
    }
    
    /**
     * Render view when no tasks available
     */
    renderNoTasksView() {
        return `
            <div class="immersive-focus-container immersive-empty">
                <div class="immersive-empty-content">
                    <i class="fas fa-check-circle immersive-empty-icon"></i>
                    <h2>All Done!</h2>
                    <p>You've completed all tasks in your focus queue.</p>
                    ${this.config.autoPickNext ? `
                        <p class="text-muted">Looking for more tasks...</p>
                    ` : ''}
                    <button class="btn btn-primary btn-lg" id="exit-immersive">
                        <i class="fas fa-sign-out-alt"></i> Exit Focus Mode
                    </button>
                </div>
            </div>
        `;
    }
    
    /**
     * Calculate timer ring dash offset for progress circle
     */
    getTimerDashOffset() {
        const totalSeconds = this.config.sessionDuration * 60;
        const currentSeconds = (this.timer.minutes * 60) + this.timer.seconds;
        const progress = currentSeconds / totalSeconds;
        const circumference = 2 * Math.PI * 90; // r=90
        return circumference * (1 - progress);
    }

    /**
     * Load streak data
     */
    async loadStreakData() {
        try {
            const response = await fetch('/app/tasklist/api/focus.php?action=streak')
                .then(r => r.json());
            
            if (response.success) {
                this.streak = response.data.current_streak || 0;
                this.sessionsToday = response.data.sessions_today || 0;
            }
        } catch (error) {
            logger.error('[FocusMode] Failed to load streak:', error);
        }
    }

    /**
     * Bind focus mode events
     */
    bindEvents() {
        // Configuration controls
        document.getElementById('focus-source')?.addEventListener('change', (e) => {
            const source = e.target.value;
            document.getElementById('project-select-group').style.display = source === 'project' ? 'block' : 'none';
            document.getElementById('category-select-group').style.display = source === 'category' ? 'block' : 'none';
        });
        
        document.getElementById('save-focus-config')?.addEventListener('click', () => this.saveFocusConfig());
        
        // Start immersive focus
        document.getElementById('start-immersive-focus')?.addEventListener('click', () => this.enterImmersiveFocus());
        
        // Setup analog clock for duration selection
        this.setupAnalogClock();
        
        // Remove task from queue buttons
        document.querySelectorAll('.btn-remove-from-queue').forEach(btn => {
            btn.addEventListener('click', (e) => {
                e.stopPropagation();
                const taskId = parseInt(e.currentTarget.dataset.taskId);
                this.removeTaskFromQueue(taskId);
            });
        });
        
        // Setup drag and drop for task reordering
        this.setupTaskDragAndDrop();
    }
    
    /**
     * Bind events for immersive focus dialog
     */
    bindImmersiveEvents() {
        // Notes toggle
        document.getElementById('toggle-notes')?.addEventListener('click', () => {
            const taskView = document.getElementById('immersive-task-view');
            const notesView = document.getElementById('immersive-notes-view');
            
            if (taskView && notesView) {
                taskView.style.display = 'none';
                notesView.style.display = 'block';
                // Focus on textarea
                setTimeout(() => document.getElementById('focus-note-input')?.focus(), 100);
            }
        });
        
        // Close notes
        document.getElementById('close-notes')?.addEventListener('click', () => {
            const taskView = document.getElementById('immersive-task-view');
            const notesView = document.getElementById('immersive-notes-view');
            
            if (taskView && notesView) {
                notesView.style.display = 'none';
                taskView.style.display = 'block';
            }
        });
        
        // Save note
        document.getElementById('save-focus-note')?.addEventListener('click', () => this.addFocusNote());
        
        // Note input keyboard shortcut (Ctrl+Enter)
        document.getElementById('focus-note-input')?.addEventListener('keydown', (e) => {
            if (e.ctrlKey && e.key === 'Enter') {
                e.preventDefault();
                this.addFocusNote();
            }
        });
        
        // Delete note
        document.querySelectorAll('.btn-note-delete').forEach(btn => {
            btn.addEventListener('click', (e) => {
                const index = parseInt(e.currentTarget.dataset.index);
                this.deleteFocusNote(index);
            });
        });
        
        // Subtask checkboxes (both regular and large)
        document.querySelectorAll('.subtask-checkbox, .subtask-checkbox-large').forEach(checkbox => {
            checkbox.addEventListener('change', async (e) => {
                const taskId = parseInt(e.target.dataset.taskId);
                const subtaskIndex = parseInt(e.target.dataset.subtaskIndex);
                await this.toggleSubtask(taskId, subtaskIndex, e.target.checked);
            });
        });
        
        // Timer click to pause/resume
        document.getElementById('timer-click-area')?.addEventListener('click', () => {
            if (this.timer.isRunning && !this.timer.isPaused) {
                this.pauseTimer();
            } else if (this.timer.isPaused) {
                this.startTimer();
            }
        });
        
        // Actions dropdown toggle
        document.getElementById('toggle-actions-menu')?.addEventListener('click', (e) => {
            e.stopPropagation();
            const menu = document.getElementById('actions-dropdown-menu');
            if (menu) {
                menu.style.display = menu.style.display === 'none' ? 'block' : 'none';
            }
        });
        
        // Close dropdown when clicking outside
        document.addEventListener('click', (e) => {
            const menu = document.getElementById('actions-dropdown-menu');
            const toggle = document.getElementById('toggle-actions-menu');
            if (menu && !toggle?.contains(e.target) && !menu.contains(e.target)) {
                menu.style.display = 'none';
            }
        });
        
        // Action buttons
        document.getElementById('immersive-complete')?.addEventListener('click', () => this.completeCurrentTask());
        document.getElementById('immersive-skip')?.addEventListener('click', () => this.skipCurrentTask());
        document.getElementById('immersive-break')?.addEventListener('click', () => {
            this.takeBreak();
            // Close dropdown
            const menu = document.getElementById('actions-dropdown-menu');
            if (menu) menu.style.display = 'none';
        });
        document.getElementById('menu-extend')?.addEventListener('click', () => {
            this.extendFocusTime();
            // Close dropdown
            const menu = document.getElementById('actions-dropdown-menu');
            if (menu) menu.style.display = 'none';
        });
        document.getElementById('immersive-end')?.addEventListener('click', () => this.endFocusSession());
        document.getElementById('exit-immersive')?.addEventListener('click', () => this.exitImmersiveFocus());
        
        // ESC key to exit
        document.addEventListener('keydown', this.handleImmersiveKeyPress.bind(this));
    }
    
    /**
     * Handle keyboard shortcuts in immersive mode
     */
    handleImmersiveKeyPress(e) {
        if (!this.isInFocusDialog) return;
        
        if (e.key === 'Escape' && !e.target.matches('textarea')) {
            this.endFocusSession();
        }
    }
    
    /**
     * Save focus configuration
     */
    async saveFocusConfig() {
        this.config.focusSource = document.getElementById('focus-source').value;
        this.config.selectedProjectId = document.getElementById('focus-project')?.value || null;
        this.config.selectedCategoryId = document.getElementById('focus-category')?.value || null;
        this.config.sessionDuration = parseInt(document.getElementById('session-duration').value);
        this.config.breakDuration = parseInt(document.getElementById('break-duration').value);
        this.config.exitOnCompletion = document.getElementById('exit-on-completion').checked;
        this.config.autoPickNext = document.getElementById('auto-pick-next').checked;
        
        this.saveConfig();
        
        // Reload tasks with new config
        await this.loadFocusTasks();
        
        // Re-render
        const contentArea = document.getElementById('content-area');
        if (contentArea) {
            contentArea.innerHTML = await this.render();
            this.bindEvents();
        }
        
        this.app.ui.showToast('Focus configuration saved', 'success');
    }
    
    /**
     * Toggle subtask completion
     */
    async toggleSubtask(taskId, subtaskIndex, completed) {
        try {
            const task = this.focusTasks.find(t => t.id === taskId);
            if (!task || !task.subtasks || !task.subtasks[subtaskIndex]) {
                return;
            }
            
            // Update local state
            task.subtasks[subtaskIndex].completed = completed;
            
            // Update on server
            await this.app.api.put(`tasks/${taskId}`, {
                ...task,
                subtasks: task.subtasks
            });
            
            // Refresh display to show updated progress
            this.refreshImmersiveDialog();
            
        } catch (error) {
            logger.error('[FocusMode] Failed to toggle subtask:', error);
            this.app.ui.showToast('Failed to update subtask', 'error');
        }
    }
    
    /**
     * Add focus note
     */
    addFocusNote() {
        const input = document.getElementById('focus-note-input');
        const text = input.value.trim();
        
        if (!text) return;
        
        this.focusNotes.push({
            text,
            timestamp: Date.now(),
            taskId: this.focusTasks[this.currentTaskIndex]?.id
        });
        
        input.value = '';
        this.refreshImmersiveDialog();
    }
    
    /**
     * Delete focus note
     */
    deleteFocusNote(index) {
        this.focusNotes.splice(index, 1);
        this.refreshImmersiveDialog();
    }
    
    /**
     * Complete current task and move to next
     */
    async completeCurrentTask() {
        const currentTask = this.focusTasks[this.currentTaskIndex];
        
        if (currentTask) {
            // Mark as complete
            await this.app.api.put(`tasks/${currentTask.id}`, {
                ...currentTask,
                status: 'completed'
            });
            
            currentTask.status = 'completed';
            
            this.app.ui.showToast('Task completed!', 'success');
        }
        
        this.moveToNextTask();
    }
    
    /**
     * Skip current task
     */
    skipCurrentTask() {
        // Don't reset timer, just pause and move to next
        if (this.timer.isRunning) {
            this.pauseTimer();
        }
        this.moveToNextTask();
    }
    
    /**
     * Move to next task
     */
    moveToNextTask() {
        this.currentTaskIndex++;
        
        if (this.currentTaskIndex >= this.focusTasks.length) {
            if (this.config.exitOnCompletion) {
                this.exitImmersiveFocus();
            } else if (this.config.autoPickNext) {
                // Load more tasks
                this.loadFocusTasks().then(() => {
                    this.currentTaskIndex = 0;
                    if (this.focusTasks.length > 0) {
                        this.refreshImmersiveDialog();
                    } else {
                        this.refreshImmersiveDialog(); // Show empty view
                    }
                });
            } else {
                this.refreshImmersiveDialog(); // Show empty view
            }
        } else {
            // Only reset timer elements if they exist (not in immersive mode)
            if (document.getElementById('timer-pause')) {
                this.resetTimer();
            } else {
                // In immersive mode, just stop the timer and reset values
                clearInterval(this.timer.interval);
                this.timer.isRunning = false;
                this.timer.isPaused = false;
            }
            this.timer.minutes = this.config.sessionDuration;
            this.timer.seconds = 0;
            this.refreshImmersiveDialog();
            setTimeout(() => this.startTimer(), 500);
        }
    }
    
    /**
     * Extend focus time by 5 minutes
     */
    extendFocusTime() {
        this.timer.minutes += 5;
        this.updateTimerDisplay();
        this.app.ui.showToast('+5 minutes added', 'info');
    }
    
    /**
     * Take a break
     */
    takeBreak() {
        this.pauseTimer();
        this.timer.minutes = this.config.breakDuration || 5;
        this.timer.seconds = 0;
        this.timer.mode = 'short-break';
        this.updateTimerDisplay();
        this.app.ui.showToast(`Break time! Timer set to ${this.config.breakDuration} minutes`, 'info');
        // Auto-start break timer
        setTimeout(() => this.startTimer(), 500);
    }
    
    /**
     * End focus session
     */
    async endFocusSession() {
        const confirmed = confirm('End focus session? Your progress will be saved.');
        
        if (!confirmed) return;
        
        // Log session
        if (this.timer.isRunning || this.currentTaskIndex > 0) {
            await this.logSession();
        }
        
        // Process notes
        if (this.focusNotes.length > 0) {
            await this.processNotes();
        }
        
        this.exitImmersiveFocus();
    }
    
    /**
     * Exit immersive focus
     */
    exitImmersiveFocus() {
        this.isInFocusDialog = false;
        this.pauseTimer();
        
        const dialog = document.getElementById('immersive-focus-dialog');
        if (dialog) {
            dialog.remove();
        }
        
        document.body.style.overflow = '';
        
        // Remove event listener
        document.removeEventListener('keydown', this.handleImmersiveKeyPress);
        
        // Refresh main view
        this.app.navigateToView('focus');
    }
    
    /**
     * Refresh immersive dialog content
     */
    refreshImmersiveDialog() {
        const container = document.querySelector('.immersive-focus-overlay');
        if (container) {
            container.innerHTML = this.renderImmersiveFocusDialog();
            this.bindImmersiveEvents();
        }
    }
    
    /**
     * Process notes and potentially create tasks
     */
    async processNotes() {
        if (this.focusNotes.length === 0) return;
        
        const shouldProcess = confirm(`You have ${this.focusNotes.length} note(s). Would you like to review them for converting to tasks?`);
        
        if (!shouldProcess) {
            // Save notes to localStorage for later
            const savedNotes = JSON.parse(localStorage.getItem('TL_focusNotes') || '[]');
            localStorage.setItem('TL_focusNotes', JSON.stringify([...savedNotes, ...this.focusNotes]));
            this.app.ui.showToast('Notes saved for later review', 'info');
            return;
        }
        
        // Simple algorithm: Notes with keywords become tasks
        const taskKeywords = ['todo', 'task', 'need to', 'must', 'should', 'remember to', 'don\'t forget'];
        const projectKeywords = ['project', 'initiative', 'plan'];
        
        for (const note of this.focusNotes) {
            const lowerText = note.text.toLowerCase();
            
            // Check if it looks like a task
            const isTask = taskKeywords.some(keyword => lowerText.includes(keyword));
            const isProject = projectKeywords.some(keyword => lowerText.includes(keyword));
            
            if (isTask) {
                // Create task
                const taskTitle = note.text.replace(/^(todo|task|need to|must|should|remember to|don't forget):?\s*/i, '').trim();
                
                if (taskTitle.length > 3) {
                    try {
                        await this.app.api.post('tasks', {
                            title: taskTitle,
                            description: `Created from focus note on ${new Date(note.timestamp).toLocaleString()}`,
                            status: 'to-do',
                            priority: 'medium'
                        });
                        
                        this.app.ui.showToast(`Created task: ${taskTitle.substring(0, 30)}...`, 'success');
                    } catch (error) {
                        logger.error('[FocusMode] Failed to create task from note:', error);
                    }
                }
            } else if (isProject && note.text.length > 10) {
                // Suggest as project (just notify user)
                this.app.ui.showToast(`Note might be a project: "${note.text.substring(0, 40)}..."`, 'info');
            }
        }
        
        // Clear processed notes
        this.focusNotes = [];
    }

    /**
     * Start timer
     */
    startTimer() {
        this.timer.isRunning = true;
        this.timer.isPaused = false;
        
        const startBtn = document.getElementById('timer-start');
        const pauseBtn = document.getElementById('timer-pause');
        
        if (startBtn) startBtn.style.display = 'none';
        if (pauseBtn) pauseBtn.style.display = 'inline-block';
        
        this.timer.interval = setInterval(() => {
            if (this.timer.seconds === 0) {
                if (this.timer.minutes === 0) {
                    this.timerComplete();
                    return;
                }
                this.timer.minutes--;
                this.timer.seconds = 59;
            } else {
                this.timer.seconds--;
            }
            
            this.updateTimerDisplay();
        }, 1000);
    }

    /**
     * Pause timer
     */
    pauseTimer() {
        this.timer.isPaused = true;
        this.timer.isRunning = false;
        clearInterval(this.timer.interval);
        
        const pauseBtn = document.getElementById('timer-pause');
        const startBtn = document.getElementById('timer-start');
        
        if (pauseBtn) pauseBtn.style.display = 'none';
        if (startBtn) startBtn.style.display = 'inline-block';
    }

    /**
     * Reset timer
     */
    resetTimer() {
        clearInterval(this.timer.interval);
        this.timer.isRunning = false;
        this.timer.isPaused = false;
        
        this.setTimerDuration(this.timer.mode);
        this.updateTimerDisplay();
        
        document.getElementById('timer-pause').style.display = 'none';
        document.getElementById('timer-start').style.display = 'inline-block';
    }

    /**
     * Update timer display
     */
    updateTimerDisplay() {
        // Update normal view timer
        const minutesEl = document.querySelector('.timer-minutes');
        const secondsEl = document.querySelector('.timer-seconds');
        
        if (minutesEl) minutesEl.textContent = String(this.timer.minutes).padStart(2, '0');
        if (secondsEl) secondsEl.textContent = String(this.timer.seconds).padStart(2, '0');
        
        // Update immersive view timer
        const immersiveMinutes = document.querySelector('.immersive-minutes');
        const immersiveSeconds = document.querySelector('.immersive-seconds');
        
        if (immersiveMinutes) immersiveMinutes.textContent = String(this.timer.minutes).padStart(2, '0');
        if (immersiveSeconds) immersiveSeconds.textContent = String(this.timer.seconds).padStart(2, '0');
        
        // Update progress ring
        const progressRing = document.querySelector('.timer-ring-progress');
        if (progressRing) {
            progressRing.style.strokeDashoffset = this.getTimerDashOffset();
        }
    }

    /**
     * Timer complete
     */
    async timerComplete() {
        clearInterval(this.timer.interval);
        this.timer.isRunning = false;
        
        // Show notification
        if ('Notification' in window && Notification.permission === 'granted') {
            new Notification('Timer Complete!', {
                body: this.timer.mode === 'work' ? 'Time for a break!' : 'Back to work!',
                icon: '/media/images/bloodweb/task_manager/TM_icon.png'
            });
        }
        
        // Play sound (if enabled)
        this.playCompletionSound();
        
        // Log session if work timer
        if (this.timer.mode === 'work') {
            await this.logSession();
            this.sessionsToday++;
            
            // Update sessions display
            const sessionsEl = document.querySelector('.focus-stats-card .stat-value');
            if (sessionsEl) {
                sessionsEl.textContent = `${this.sessionsToday}/8`;
            }
        }
        
        // Auto-switch to break
        const nextMode = this.timer.mode === 'work' 
            ? (this.sessionsToday % 4 === 0 ? 'long-break' : 'short-break')
            : 'work';
            
        this.timer.mode = nextMode;
        this.setTimerDuration(nextMode);
        this.updateTimerDisplay();
        
        this.app.ui.showToast(`${this.getModeName()} complete! Starting ${this.getNextModeName()}`, 'success');
    }

    /**
     * Log focus session
     */
    async logSession() {
        try {
            const completedTaskIds = this.focusTasks
                .filter(t => t.status === 'completed')
                .map(t => t.id);
                
            const response = await fetch('/app/tasklist/api/focus.php?action=log-session', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    duration: 1500, // 25 minutes in seconds
                    completed_tasks: completedTaskIds,
                    xp_earned: 20
                })
            }).then(r => r.json());
            
            // XP is already awarded by the focus API
            if (response.success) {
                this.sessionsToday = response.data.today_sessions;
                // Reload user data to reflect XP changes
                if (this.app.loadUserData) {
                    await this.app.loadUserData();
                }
            }
        } catch (error) {
            logger.error('[FocusMode] Failed to log session:', error);
        }
    }

    /**
     * Switch timer mode
     */
    switchMode() {
        const modes = ['work', 'short-break', 'long-break'];
        const currentIndex = modes.indexOf(this.timer.mode);
        const nextIndex = (currentIndex + 1) % modes.length;
        
        this.timer.mode = modes[nextIndex];
        this.setTimerDuration(this.timer.mode);
        this.resetTimer();
        
        // Update mode badge
        const modeBadge = document.querySelector('.mode-badge');
        if (modeBadge) {
            modeBadge.className = `mode-badge mode-${this.timer.mode}`;
            modeBadge.textContent = this.getModeName();
        }
        
        // Update switch button
        const switchBtn = document.getElementById('timer-mode-switch');
        if (switchBtn) {
            switchBtn.innerHTML = `<i class="fas fa-exchange-alt"></i> ${this.getNextModeName()}`;
        }
    }

    /**
     * Set timer duration
     */
    setTimerDuration(mode) {
        switch(mode) {
            case 'work':
                this.timer.minutes = 25;
                break;
            case 'short-break':
                this.timer.minutes = 5;
                break;
            case 'long-break':
                this.timer.minutes = 15;
                break;
        }
        this.timer.seconds = 0;
    }

    /**
     * Get mode display name
     */
    getModeName() {
        const names = {
            'work': 'Work Time',
            'short-break': 'Short Break',
            'long-break': 'Long Break'
        };
        return names[this.timer.mode] || 'Work Time';
    }

    /**
     * Get next mode name
     */
    getNextModeName() {
        const modes = ['work', 'short-break', 'long-break'];
        const currentIndex = modes.indexOf(this.timer.mode);
        const nextIndex = (currentIndex + 1) % modes.length;
        return this.getModeName.call({ timer: { mode: modes[nextIndex] } });
    }

    /**
     * Toggle task completion
     */
    async toggleTaskComplete(taskId) {
        try {
            await this.app.toggleTaskCompletion(taskId);
            
            // Update local state
            const task = this.focusTasks.find(t => t.id === taskId);
            if (task) {
                task.status = task.status === 'completed' ? 'pending' : 'completed';
            }
            
            // Check if all tasks complete
            if (this.allTasksComplete()) {
                await this.app.ui.renderView('focus');
            }
        } catch (error) {
            logger.error('[FocusMode] Failed to toggle task:', error);
        }
    }

    /**
     * Check if all tasks complete
     */
    allTasksComplete() {
        return this.focusTasks.length > 0 && 
               this.focusTasks.every(t => t.status === 'completed');
    }

    /**
     * Complete day
     */
    async completeDay() {
        // Award bonus XP
        await this.app.api.post('xp/log', {
            source_type: 'day_complete',
            xp_amount: 50,
            description: 'Completed all focus tasks for the day'
        });
        
        // Show confetti animation
        this.showConfetti();
        
        this.app.ui.showToast('🎉 Amazing work! All tasks complete!', 'success');
    }

    /**
     * Show confetti animation
     */
    showConfetti() {
        // Simple confetti effect using CSS animation
        const confetti = document.createElement('div');
        confetti.className = 'confetti-container';
        confetti.innerHTML = '🎉'.repeat(20);
        document.body.appendChild(confetti);
        
        setTimeout(() => confetti.remove(), 3000);
    }

    /**
     * Play completion sound
     */
    playCompletionSound() {
        // Placeholder for sound playback
        // Implement if sound notifications are enabled in settings
    }

    /**
     * Render clock markers for analog duration selector (0-8 hours)
     */
    renderClockMarkers() {
        const markers = [];
        const labels = {
            0: '0',
            15: '15',
            30: '30',
            45: '45'
        };
        
        // Create 12 markers for 5-minute increments (0, 5, 10, 15, 20, 25, 30, 35, 40, 45, 50, 55)
        for (let i = 0; i < 12; i++) {
            const minutes = i * 5;
            const angle = (i / 12) * 360;
            const rad = (angle - 90) * Math.PI / 180;
            const x1 = 120 + 85 * Math.cos(rad);
            const y1 = 120 + 85 * Math.sin(rad);
            const x2 = 120 + 100 * Math.cos(rad);
            const y2 = 120 + 100 * Math.sin(rad);
            
            // Thicker marker for 15-minute intervals
            const strokeWidth = (minutes % 15 === 0) ? '3' : '2';
            markers.push(`<line x1="${x1}" y1="${y1}" x2="${x2}" y2="${y2}" stroke="#9ca3af" stroke-width="${strokeWidth}"/>`);
            
            // Add labels for 0, 15, 30, 45
            if (labels[minutes]) {
                const textX = 120 + 110 * Math.cos(rad);
                const textY = 120 + 110 * Math.sin(rad);
                markers.push(`<text x="${textX}" y="${textY}" text-anchor="middle" dominant-baseline="middle" fill="#6b7280" font-size="14" font-weight="600">${labels[minutes]}</text>`);
            }
        }
        
        return markers.join('');
    }

    /**
     * Format duration for display
     */
    formatDuration(minutes) {
        if (minutes < 60) {
            return `${minutes} min`;
        } else if (minutes % 60 === 0) {
            return `${minutes / 60}h`;
        } else {
            const hours = Math.floor(minutes / 60);
            const mins = minutes % 60;
            return `${hours}h ${mins}m`;
        }
    }

    /**
     * Setup analog clock interaction
     */
    setupAnalogClock() {
        const clock = document.getElementById('duration-clock');
        const hand = document.getElementById('clock-hand');
        const display = document.querySelector('.clock-duration-display');
        const hiddenInput = document.getElementById('session-duration');
        
        if (!clock || !hand) return;
        
        let isDragging = false;
        const maxMinutes = 480; // 8 hours
        
        let rotations = Math.floor(this.config.sessionDuration / 60); // Track full rotations
        let lastAngle = ((this.config.sessionDuration % 60) / 60) * 360;
        
        const updateClock = (clientX, clientY) => {
            const rect = clock.getBoundingClientRect();
            const centerX = rect.left + rect.width / 2;
            const centerY = rect.top + rect.height / 2;
            
            const dx = clientX - centerX;
            const dy = clientY - centerY;
            
            let angle = Math.atan2(dy, dx) * 180 / Math.PI + 90;
            if (angle < 0) angle += 360;
            
            // Detect if we crossed the 0/360 boundary
            if (lastAngle > 270 && angle < 90) {
                rotations++; // Crossed upward
            } else if (lastAngle < 90 && angle > 270) {
                rotations--; // Crossed downward
            }
            
            // Clamp rotations to 0-7 (0-8 hours)
            if (rotations < 0) rotations = 0;
            if (rotations > 7) rotations = 7;
            
            lastAngle = angle;
            
            // Convert to minutes: full rotations * 60 + current angle position
            // Snap to 5-minute increments
            let minutes = (rotations * 60) + Math.round((angle / 360) * 60 / 5) * 5;
            
            // Allow 0 minutes (start at 0)
            if (minutes > maxMinutes) minutes = maxMinutes;
            
            // Prevent going backwards from 5 mins to high value when at rotation 0
            if (rotations === 0 && minutes > 55) {
                minutes = 0;
            }
            
            // Update hand position (hand only shows current angle within rotation)
            const handAngle = angle - 90;
            const rad = handAngle * Math.PI / 180;
            const x2 = 120 + 80 * Math.cos(rad);
            const y2 = 120 + 80 * Math.sin(rad);
            hand.setAttribute('x2', x2);
            hand.setAttribute('y2', y2);
            
            // Update display
            if (display) display.textContent = this.formatDuration(minutes);
            if (hiddenInput) hiddenInput.value = minutes;
        };
        
        const startDrag = (e) => {
            isDragging = true;
            e.preventDefault();
        };
        
        const drag = (e) => {
            if (!isDragging) return;
            
            const clientX = e.clientX || (e.touches && e.touches[0].clientX);
            const clientY = e.clientY || (e.touches && e.touches[0].clientY);
            
            if (clientX && clientY) {
                updateClock(clientX, clientY);
            }
        };
        
        const endDrag = () => {
            isDragging = false;
        };
        
        // Mouse events
        clock.addEventListener('mousedown', startDrag);
        document.addEventListener('mousemove', drag);
        document.addEventListener('mouseup', endDrag);
        
        // Touch events
        clock.addEventListener('touchstart', startDrag);
        document.addEventListener('touchmove', drag);
        document.addEventListener('touchend', endDrag);
        
        // Initialize hand position (show only the angle within current rotation)
        const minutesInRotation = this.config.sessionDuration % 60;
        const initialAngle = ((minutesInRotation / 60) * 360) - 90;
        const rad = initialAngle * Math.PI / 180;
        const x2 = 120 + 80 * Math.cos(rad);
        const y2 = 120 + 80 * Math.sin(rad);
        hand.setAttribute('x2', x2);
        hand.setAttribute('y2', y2);
        
        // Initialize rotations tracker
        rotations = Math.floor(this.config.sessionDuration / 60);
        lastAngle = ((minutesInRotation / 60) * 360);
    }

    /**
     * Remove task from focus queue
     */
    removeTaskFromQueue(taskId) {
        this.focusTasks = this.focusTasks.filter(t => t.id !== taskId);
        this.app.ui.renderView('focus');
        this.app.ui.showToast('Task removed from focus queue', 'success');
    }

    /**
     * Setup drag and drop for task reordering
     */
    setupTaskDragAndDrop() {
        const list = document.getElementById('focus-preview-list');
        if (!list) return;
        
        let draggedElement = null;
        let draggedIndex = -1;
        
        const items = list.querySelectorAll('.focus-preview-item');
        
        items.forEach((item, index) => {
            // Drag start
            item.addEventListener('dragstart', (e) => {
                draggedElement = item;
                draggedIndex = index;
                item.classList.add('dragging');
                e.dataTransfer.effectAllowed = 'move';
                e.dataTransfer.setData('text/html', item.innerHTML);
            });
            
            // Drag end
            item.addEventListener('dragend', (e) => {
                item.classList.remove('dragging');
                items.forEach(i => i.classList.remove('drag-over'));
            });
            
            // Drag over
            item.addEventListener('dragover', (e) => {
                e.preventDefault();
                e.dataTransfer.dropEffect = 'move';
                
                if (draggedElement !== item) {
                    item.classList.add('drag-over');
                }
            });
            
            // Drag leave
            item.addEventListener('dragleave', (e) => {
                item.classList.remove('drag-over');
            });
            
            // Drop
            item.addEventListener('drop', (e) => {
                e.preventDefault();
                item.classList.remove('drag-over');
                
                if (draggedElement !== item) {
                    const dropIndex = Array.from(items).indexOf(item);
                    
                    // Reorder the focusTasks array
                    const [movedTask] = this.focusTasks.splice(draggedIndex, 1);
                    this.focusTasks.splice(dropIndex, 0, movedTask);
                    
                    // Re-render
                    this.app.ui.renderView('focus');
                    this.app.ui.showToast('Task order updated', 'success');
                }
            });
        });
    }
}
