/**
 * CollaborationManager - Team collaboration features
 * Handles project sharing, task assignments, and activity feed
 * @class CollaborationManager
 */
class CollaborationManager {
    constructor(app) {
        this.app = app;
        this.activityFeed = [];
        this.projectMembers = new Map();
    }

    /**
     * Share project with user
     * @param {number} projectId - Project ID
     * @param {string} email - User email to invite
     * @param {string} role - User role (owner, admin, member, viewer)
     * @returns {Promise} API response
     */
    async shareProject(projectId, email, role = 'member') {
        try {
            const response = await this.app.api.post('collaboration/share-project', {
                project_id: projectId,
                email: email,
                role: role
            });

            if (response.success) {
                logger.debug('[Collaboration] Project shared successfully');
                await this.loadProjectMembers(projectId);
                this.logActivity('project_shared', { projectId, email, role });
            }

            return response;
        } catch (error) {
            logger.error('[Collaboration] Error sharing project:', error);
            throw error;
        }
    }

    /**
     * Generate public share link for project
     * @param {number} projectId - Project ID
     * @param {string} accessLevel - Access level (view, comment, edit)
     * @param {Date} expiresAt - Optional expiration date
     * @returns {Promise} Share link data
     */
    async generateShareLink(projectId, accessLevel = 'view', expiresAt = null) {
        try {
            const response = await this.app.api.post('collaboration/generate-link', {
                project_id: projectId,
                access_level: accessLevel,
                expires_at: expiresAt
            });

            if (response.success) {
                logger.debug('[Collaboration] Share link generated:', response.data.share_url);
            }

            return response;
        } catch (error) {
            logger.error('[Collaboration] Error generating share link:', error);
            throw error;
        }
    }

    /**
     * Assign task to user
     * @param {number} taskId - Task ID
     * @param {number} userId - User ID to assign to
     * @param {string} notes - Optional assignment notes
     * @returns {Promise} API response
     */
    async assignTask(taskId, userId, notes = '') {
        try {
            const response = await this.app.api.post('collaboration/assign-task', {
                task_id: taskId,
                assigned_to: userId,
                notes: notes
            });

            if (response.success) {
                logger.debug('[Collaboration] Task assigned successfully');
                this.logActivity('task_assigned', { taskId, userId });
            }

            return response;
        } catch (error) {
            logger.error('[Collaboration] Error assigning task:', error);
            throw error;
        }
    }

    /**
     * Add comment to task
     * @param {number} taskId - Task ID
     * @param {string} comment - Comment text
     * @returns {Promise} API response
     */
    async addComment(taskId, comment) {
        try {
            const response = await this.app.api.post('collaboration/add-comment', {
                task_id: taskId,
                comment: comment
            });

            if (response.success) {
                logger.debug('[Collaboration] Comment added');
                this.logActivity('task_commented', { taskId, comment });
            }

            return response;
        } catch (error) {
            logger.error('[Collaboration] Error adding comment:', error);
            throw error;
        }
    }

    /**
     * Load project members
     * @param {number} projectId - Project ID
     * @returns {Promise} Project members
     */
    async loadProjectMembers(projectId) {
        try {
            const response = await this.app.api.get(`collaboration/project-members/${projectId}`);
            
            if (response.success) {
                this.projectMembers.set(projectId, response.data.members);
                return response.data.members;
            }

            return [];
        } catch (error) {
            logger.error('[Collaboration] Error loading project members:', error);
            return [];
        }
    }

    /**
     * Get activity feed for project
     * @param {number} projectId - Project ID
     * @param {number} limit - Number of activities to fetch
     * @returns {Promise} Activity feed
     */
    async getActivityFeed(projectId, limit = 50) {
        try {
            const response = await this.app.api.get(`collaboration/activity-feed/${projectId}`, {
                limit: limit
            });
            
            if (response.success) {
                this.activityFeed = response.data.activities;
                return this.activityFeed;
            }

            return [];
        } catch (error) {
            logger.error('[Collaboration] Error loading activity feed:', error);
            return [];
        }
    }

    /**
     * Remove member from project
     * @param {number} projectId - Project ID
     * @param {number} userId - User ID to remove
     * @returns {Promise} API response
     */
    async removeMember(projectId, userId) {
        try {
            const response = await this.app.api.delete(`collaboration/remove-member`, {
                body: {
                    project_id: projectId,
                    user_id: userId
                }
            });

            if (response.success) {
                logger.debug('[Collaboration] Member removed');
                await this.loadProjectMembers(projectId);
                this.logActivity('member_removed', { projectId, userId });
            }

            return response;
        } catch (error) {
            logger.error('[Collaboration] Error removing member:', error);
            throw error;
        }
    }

    /**
     * Update member role
     * @param {number} projectId - Project ID
     * @param {number} userId - User ID
     * @param {string} newRole - New role
     * @returns {Promise} API response
     */
    async updateMemberRole(projectId, userId, newRole) {
        try {
            const response = await this.app.api.put('collaboration/update-role', {
                project_id: projectId,
                user_id: userId,
                role: newRole
            });

            if (response.success) {
                logger.debug('[Collaboration] Member role updated');
                await this.loadProjectMembers(projectId);
            }

            return response;
        } catch (error) {
            logger.error('[Collaboration] Error updating member role:', error);
            throw error;
        }
    }

    /**
     * Log activity
     * @param {string} actionType - Action type
     * @param {Object} data - Action data
     */
    logActivity(actionType, data) {
        const activity = {
            action_type: actionType,
            action_data: data,
            timestamp: new Date().toISOString(),
            user_id: CURRENT_USER.id
        };

        this.activityFeed.unshift(activity);
        logger.debug('[Collaboration] Activity logged:', actionType);
    }

    /**
     * Render project sharing modal
     * @param {number} projectId - Project ID
     */
    async renderShareModal(projectId) {
        const project = this.app.state.get('projects').find(p => p.id === projectId);
        if (!project) return;

        const members = await this.loadProjectMembers(projectId);

        const modalHTML = `
            <div class="collaboration-modal">
                <div class="modal-header">
                    <h3><i class="fas fa-users"></i> Share "${this.escapeHtml(project.name)}"</h3>
                </div>
                
                <div class="modal-body">
                    <!-- Invite by email -->
                    <div class="invite-section">
                        <h4>Invite by Email</h4>
                        <div class="invite-form">
                            <input type="email" id="invite-email" placeholder="Enter email address" class="input-field">
                            <select id="invite-role" class="select-field">
                                <option value="viewer">Viewer</option>
                                <option value="member" selected>Member</option>
                                <option value="admin">Admin</option>
                            </select>
                            <button class="btn btn-primary" onclick="app.collaboration.sendInvite(${projectId})">
                                <i class="fas fa-paper-plane"></i> Send Invite
                            </button>
                        </div>
                    </div>

                    <!-- Share link -->
                    <div class="share-link-section">
                        <h4>Share Link</h4>
                        <button class="btn btn-secondary" onclick="app.collaboration.generateAndCopyLink(${projectId})">
                            <i class="fas fa-link"></i> Generate Share Link
                        </button>
                        <div id="share-link-output" style="display: none;">
                            <input type="text" id="share-link-url" readonly class="input-field">
                            <button class="btn btn-icon" onclick="app.collaboration.copyShareLink()">
                                <i class="fas fa-copy"></i>
                            </button>
                        </div>
                    </div>

                    <!-- Current members -->
                    <div class="members-section">
                        <h4>Current Members (${members.length})</h4>
                        <div class="members-list">
                            ${members.map(member => this.renderMemberItem(member, projectId)).join('')}
                        </div>
                    </div>
                </div>
            </div>
        `;

        // TODO: Show modal with HTML
        logger.debug('[Collaboration] Share modal rendered');
    }

    /**
     * Render member list item
     * @param {Object} member - Member object
     * @param {number} projectId - Project ID
     * @returns {string} HTML string
     */
    renderMemberItem(member, projectId) {
        const isCurrentUser = member.user_id === CURRENT_USER.id;
        const canManage = member.role === 'owner' || member.role === 'admin';

        return `
            <div class="member-item">
                <div class="member-info">
                    <div class="member-avatar">${member.username.charAt(0).toUpperCase()}</div>
                    <div class="member-details">
                        <span class="member-name">${this.escapeHtml(member.username)}</span>
                        <span class="member-email">${this.escapeHtml(member.email)}</span>
                    </div>
                </div>
                <div class="member-role">
                    <span class="role-badge role-${member.role}">${member.role}</span>
                </div>
                ${!isCurrentUser && canManage ? `
                    <div class="member-actions">
                        <button class="btn-icon" onclick="app.collaboration.removeMember(${projectId}, ${member.user_id})">
                            <i class="fas fa-times"></i>
                        </button>
                    </div>
                ` : ''}
            </div>
        `;
    }

    /**
     * Escape HTML for safe rendering
     */
    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
}
