/**
 * CalendarManager - Timeline view of upcoming events and tasks
 * 
 * Features:
 * - Timeline view showing events grouped by date
 * - Shows tasks due on each date
 * - Date range filters (7, 14, 30, 90, 180 days)
 * - Lazy loading of events
 * - Event creation, editing, deletion
 * 
 * @author Jack Ewers / BloodWeb
 * @version 2.0.1
 * @date 2025-12-05
 */

class CalendarManager {
    constructor(app) {
        this.app = app;
        this.startDate = new Date();
        this.daysToShow = 30; // Default: 1 month
        this.events = [];
        this.tasks = [];
        this.isLoading = false;
    }

    /**
     * Initialize calendar timeline
     */
    async init() {
        logger.info('[Calendar] Initializing timeline view');
        await this.loadData();
        this.render();
    }

    /**
     * Refresh timeline (reload data and re-render)
     */
    async refresh() {
        await this.loadData();
        this.render();
    }

    /**
     * Load events and tasks for date range
     */
    async loadData() {
        if (this.isLoading) return;
        
        this.isLoading = true;
        try {
            const endDate = new Date(this.startDate);
            endDate.setDate(endDate.getDate() + this.daysToShow);
            
            const startStr = this.formatDateForAPI(this.startDate);
            const endStr = this.formatDateForAPI(endDate);
            
            // Load events
            const eventsResponse = await this.app.api.get(`calendar/events`, {
                start_date: startStr,
                end_date: endStr
            });
            
            if (eventsResponse.success) {
                this.events = eventsResponse.data || [];
                logger.debug('[Calendar] Loaded events:', this.events.length);
            }
            
            // Load tasks due in this range
            await this.loadTasksDue(startStr, endStr);
            
        } catch (error) {
            logger.error('[Calendar] Failed to load data:', error);
            if (this.app.modals) {
                this.app.modals.showToast('Failed to load timeline data', 'error');
            }
        } finally {
            this.isLoading = false;
        }
    }

    /**
     * Load tasks that are due in date range
     */
    async loadTasksDue(startDate, endDate) {
        try {
            // Get all tasks from app state
            const allTasks = this.app.state.get('tasks') || [];
            
            // Filter tasks that have due dates in our range
            this.tasks = allTasks.filter(task => {
                if (!task.due_date) return false;
                const dueDate = new Date(task.due_date);
                const start = new Date(startDate);
                const end = new Date(endDate);
                return dueDate >= start && dueDate <= end;
            });
            
            logger.debug('[Calendar] Loaded tasks due:', this.tasks.length);
        } catch (error) {
            logger.error('[Calendar] Failed to load tasks:', error);
        }
    }

    /**
     * Format date for API (YYYY-MM-DD)
     */
    formatDateForAPI(date) {
        const year = date.getFullYear();
        const month = String(date.getMonth() + 1).padStart(2, '0');
        const day = String(date.getDate()).padStart(2, '0');
        return `${year}-${month}-${day}`;
    }

    /**
     * Render calendar timeline view
     */
    render() {
        const contentBody = document.getElementById('content-body');
        if (!contentBody) return;

        const html = `
            <div class="timeline-container">
                ${this.renderTimeline()}
            </div>
        `;

        contentBody.innerHTML = html;
        
        // Update header actions to show toolbar controls
        if (this.app && this.app.ui) {
            const contentActions = document.getElementById('content-actions');
            if (contentActions) {
                contentActions.innerHTML = this.app.ui.getViewActions('calendar');
            }
        }
        
        // Attach event listeners after DOM is updated
        this.attachEventListeners();
    }

    /**
     * Render timeline of events and tasks
     */
    renderTimeline() {
        if (this.isLoading) {
            return '<div class="timeline-loading"><i class="fas fa-spinner fa-spin"></i> Loading...</div>';
        }

        // Group events and tasks by date
        const timeline = this.buildTimeline();
        
        if (timeline.length === 0) {
            return `
                <div class="timeline-empty">
                    <i class="fas fa-calendar-check"></i>
                    <h3>No Upcoming Events or Tasks</h3>
                    <p>Add events or tasks with due dates to see them here</p>
                </div>
            `;
        }

        return `
            <div class="timeline-content">
                ${timeline.map(day => this.renderDayGroup(day)).join('')}
            </div>
        `;
    }

    /**
     * Build timeline structure grouping items by date
     */
    buildTimeline() {
        const timeline = {};
        
        // Add events to timeline
        this.events.forEach(event => {
            const date = event.event_date;
            if (!timeline[date]) {
                timeline[date] = { date, events: [], tasks: [] };
            }
            timeline[date].events.push(event);
        });
        
        // Add tasks to timeline
        this.tasks.forEach(task => {
            const date = task.due_date;
            if (!timeline[date]) {
                timeline[date] = { date, events: [], tasks: [] };
            }
            timeline[date].tasks.push(task);
        });
        
        // Convert to array and sort by date
        return Object.values(timeline).sort((a, b) => 
            new Date(a.date) - new Date(b.date)
        );
    }

    /**
     * Render a day group with its events and tasks
     */
    renderDayGroup(day) {
        const date = new Date(day.date);
        const today = new Date();
        const isToday = this.isSameDay(date, today);
        const isTomorrow = this.isSameDay(date, new Date(today.getTime() + 86400000));
        
        let dateLabel = this.formatDateLabel(date);
        if (isToday) dateLabel = 'Today';
        else if (isTomorrow) dateLabel = 'Tomorrow';
        
        const totalItems = day.events.length + day.tasks.length;
        
        return `
            <div class="timeline-day ${isToday ? 'is-today' : ''}">
                <div class="day-header">
                    <div class="day-date">
                        <span class="day-label">${dateLabel}</span>
                        <span class="day-full-date">${this.formatFullDate(date)}</span>
                    </div>
                    <div class="day-count">
                        <span class="count-badge">${totalItems}</span>
                        <span class="count-label">${totalItems === 1 ? 'item' : 'items'}</span>
                    </div>
                </div>
                
                <div class="day-items">
                    ${day.events.map(event => this.renderEventItem(event)).join('')}
                    ${day.tasks.map(task => this.renderTaskItem(task)).join('')}
                </div>
            </div>
        `;
    }

    /**
     * Render an event item
     */
    renderEventItem(event) {
        const colorClass = event.color ? `event-${event.color.replace('#', '')}` : 'event-blue';
        const time = event.event_time || 'All day';
        
        return `
            <div class="timeline-item event-item ${colorClass}" data-event-id="${event.id}">
                <div class="item-indicator event-indicator"></div>
                <div class="item-content">
                    <div class="item-header">
                        <div class="item-type">
                            <i class="fas fa-calendar-day"></i>
                            Event
                        </div>
                        <div class="item-time">${this.formatTime(time)}</div>
                    </div>
                    <h4 class="item-title">${this.escapeHtml(event.title)}</h4>
                    ${event.description ? `<p class="item-description">${this.escapeHtml(event.description)}</p>` : ''}
                    ${event.location ? `
                        <div class="item-location">
                            <i class="fas fa-map-marker-alt"></i>
                            ${this.escapeHtml(event.location)}
                        </div>
                    ` : ''}
                    <div class="item-actions">
                        <button class="btn-item-action btn-edit" onclick="window.app.calendar.editEvent(${event.id})">
                            <i class="fas fa-edit"></i> Edit
                        </button>
                        <button class="btn-item-action btn-delete" onclick="window.app.calendar.deleteEvent(${event.id})">
                            <i class="fas fa-trash"></i> Delete
                        </button>
                    </div>
                </div>
            </div>
        `;
    }

    /**
     * Render a task item
     */
    renderTaskItem(task) {
        const priorityClass = `priority-${task.priority || 'medium'}`;
        const statusClass = `status-${task.status || 'pending'}`;
        
        return `
            <div class="timeline-item task-item ${priorityClass} ${statusClass} clickable-task" 
                 data-task-id="${task.id}" 
                 onclick="window.app.editTask(${task.id})">
                <div class="item-indicator task-indicator"></div>
                <div class="item-content">
                    <div class="item-header">
                        <div class="item-type">
                            <i class="fas fa-tasks"></i>
                            Task Due
                        </div>
                        <div class="item-priority priority-badge-${task.priority || 'medium'}">
                            ${(task.priority || 'medium').toUpperCase()}
                        </div>
                    </div>
                    <h4 class="item-title">${this.escapeHtml(task.title)}</h4>
                    ${task.description ? `<p class="item-description">${this.escapeHtml(task.description)}</p>` : ''}
                    ${task.project_name ? `
                        <div class="item-project">
                            <i class="fas fa-folder"></i>
                            ${this.escapeHtml(task.project_name)}
                        </div>
                    ` : ''}
                    <div class="item-actions" onclick="event.stopPropagation()">
                        <button class="btn-item-action btn-view" onclick="window.app.editTask(${task.id})">
                            <i class="fas fa-eye"></i> View
                        </button>
                        <button class="btn-item-action btn-complete" onclick="window.app.calendar.completeTaskAndRefresh(${task.id})">
                            <i class="fas fa-check"></i> Complete
                        </button>
                    </div>
                </div>
            </div>
        `;
    }

    /**
     * Attach event listeners
     */
    attachEventListeners() {
        // Start date picker
        const startDatePicker = document.getElementById('start-date-picker');
        if (startDatePicker) {
            startDatePicker.addEventListener('change', (e) => {
                this.startDate = new Date(e.target.value);
                this.loadData().then(() => this.render());
            });
        }

        // Range buttons
        document.querySelectorAll('.btn-range').forEach(btn => {
            btn.addEventListener('click', (e) => {
                this.daysToShow = parseInt(e.target.dataset.days);
                this.loadData().then(() => this.render());
            });
        });

        // Add event button
        const addEventBtn = document.getElementById('btn-add-event');
        if (addEventBtn) {
            addEventBtn.addEventListener('click', () => this.showEventModal());
        }
    }

    /**
     * Helper: Check if two dates are the same day
     */
    isSameDay(date1, date2) {
        return date1.getDate() === date2.getDate() &&
               date1.getMonth() === date2.getMonth() &&
               date1.getFullYear() === date2.getFullYear();
    }

    /**
     * Helper: Format date label (e.g., "Monday, Dec 18")
     */
    formatDateLabel(date) {
        const options = { weekday: 'long', month: 'short', day: 'numeric' };
        return date.toLocaleDateString('en-US', options);
    }

    /**
     * Helper: Format full date (e.g., "December 18, 2025")
     */
    formatFullDate(date) {
        const options = { year: 'numeric', month: 'long', day: 'numeric' };
        return date.toLocaleDateString('en-US', options);
    }

    /**
     * Helper: Format time (HH:MM or "All day")
     */
    formatTime(time) {
        if (!time || time === 'All day') return 'All day';
        // Convert 24h to 12h format
        const [hours, minutes] = time.split(':');
        const h = parseInt(hours);
        const ampm = h >= 12 ? 'PM' : 'AM';
        const h12 = h % 12 || 12;
        return `${h12}:${minutes} ${ampm}`;
    }

    /**
     * Helper: Escape HTML to prevent XSS
     */
    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    /**
     * Show event modal for creating new event
     */
    showEventModal() {
        if (this.app.modals) {
            this.app.modals.openEventModal();
        }
    }

    /**
     * Edit event - load event data and open modal
     */
    async editEvent(eventId) {
        try {
            const response = await this.app.api.get(`calendar/events/${eventId}`);
            if (response.success && response.data) {
                const event = response.data;
                if (this.app.modals) {
                    this.app.modals.openEventModal(event);
                }
            }
        } catch (error) {
            logger.error('[Calendar] Failed to load event:', error);
            if (this.app.modals) {
                this.app.modals.showToast('Failed to load event', 'error');
            }
        }
    }

    /**
     * Complete a task and refresh timeline
     */
    async completeTaskAndRefresh(taskId) {
        await this.app.toggleTaskStatus(taskId);
        // Wait a moment for the task to be completed, then refresh
        setTimeout(() => this.refresh(), 500);
    }

    /**
     * Delete event
     */
    async deleteEvent(eventId) {
        if (!confirm('Are you sure you want to delete this event?')) return;
        
        try {
            const response = await this.app.api.delete(`calendar/${eventId}`);
            
            if (response.success) {
                if (this.app.modals) {
                    this.app.modals.showToast('Event deleted successfully', 'success');
                }
                await this.loadData();
                this.render();
            }
        } catch (error) {
            logger.error('[Calendar] Failed to delete event:', error);
            if (this.app.modals) {
                this.app.modals.showToast('Failed to delete event', 'error');
            }
        }
    }
}
