/**
 * BulkActions - Bulk Task Operations Manager
 * 
 * Handles selection and batch operations on multiple tasks
 * 
 * @author Jack Ewers / BloodWeb
 * @version 1.0.0
 */

class BulkActions {
    constructor(app) {
        this.app = app;
        this.selectedTasks = new Set();
        this.selectionMode = false;
    }

    /**
     * Toggle bulk selection mode
     */
    toggleSelectionMode() {
        this.selectionMode = !this.selectionMode;
        
        if (this.selectionMode) {
            this.enterSelectionMode();
        } else {
            this.exitSelectionMode();
        }
    }

    /**
     * Enter bulk selection mode
     */
    enterSelectionMode() {
        this.selectedTasks.clear();
        this.renderSelectionUI();
        this.addTaskCheckboxes();
    }

    /**
     * Exit bulk selection mode
     */
    exitSelectionMode() {
        this.selectedTasks.clear();
        this.removeSelectionUI();
        this.removeTaskCheckboxes();
    }

    /**
     * Add checkboxes to task items
     */
    addTaskCheckboxes() {
        const taskItems = document.querySelectorAll('.task-item');
        
        taskItems.forEach(item => {
            if (item.querySelector('.bulk-checkbox')) return; // Already has checkbox
            
            const taskId = parseInt(item.dataset.taskId);
            const checkbox = document.createElement('input');
            checkbox.type = 'checkbox';
            checkbox.className = 'bulk-checkbox';
            checkbox.dataset.taskId = taskId;
            
            checkbox.addEventListener('change', (e) => {
                if (e.target.checked) {
                    this.selectedTasks.add(taskId);
                } else {
                    this.selectedTasks.delete(taskId);
                }
                this.updateSelectionUI();
            });
            
            // Insert at beginning of task item
            item.insertBefore(checkbox, item.firstChild);
            item.classList.add('bulk-selection-mode');
        });
    }

    /**
     * Remove checkboxes from task items
     */
    removeTaskCheckboxes() {
        document.querySelectorAll('.bulk-checkbox').forEach(cb => cb.remove());
        document.querySelectorAll('.bulk-selection-mode').forEach(item => {
            item.classList.remove('bulk-selection-mode');
        });
    }

    /**
     * Render selection action bar
     */
    renderSelectionUI() {
        const existing = document.getElementById('bulk-actions-bar');
        if (existing) existing.remove();
        
        const bar = document.createElement('div');
        bar.id = 'bulk-actions-bar';
        bar.className = 'bulk-actions-bar';
        bar.innerHTML = `
            <div class="bulk-actions-left">
                <button class="btn btn-text" id="bulk-select-all">
                    <i class="fas fa-check-double"></i> Select All
                </button>
                <span class="bulk-count">0 selected</span>
            </div>
            <div class="bulk-actions-right">
                <select id="bulk-status-change" class="bulk-select">
                    <option value="">Change Status...</option>
                    <option value="pending">Pending</option>
                    <option value="in_progress">In Progress</option>
                    <option value="completed">Completed</option>
                </select>
                <select id="bulk-priority-change" class="bulk-select">
                    <option value="">Change Priority...</option>
                    <option value="low">Low</option>
                    <option value="medium">Medium</option>
                    <option value="high">High</option>
                    <option value="urgent">Urgent</option>
                </select>
                <button class="btn btn-danger" id="bulk-delete-btn">
                    <i class="fas fa-trash"></i> Delete
                </button>
                <button class="btn btn-outline" id="bulk-cancel-btn">
                    <i class="fas fa-times"></i> Cancel
                </button>
            </div>
        `;
        
        const content = document.getElementById('content');
        content.insertBefore(bar, content.firstChild);
        
        this.bindSelectionUIEvents();
    }

    /**
     * Bind event handlers to selection UI
     */
    bindSelectionUIEvents() {
        document.getElementById('bulk-select-all')?.addEventListener('click', () => {
            this.selectAll();
        });
        
        document.getElementById('bulk-status-change')?.addEventListener('change', (e) => {
            if (e.target.value) {
                this.bulkUpdateStatus(e.target.value);
                e.target.value = '';
            }
        });
        
        document.getElementById('bulk-priority-change')?.addEventListener('change', (e) => {
            if (e.target.value) {
                this.bulkUpdatePriority(e.target.value);
                e.target.value = '';
            }
        });
        
        document.getElementById('bulk-delete-btn')?.addEventListener('click', () => {
            this.bulkDelete();
        });
        
        document.getElementById('bulk-cancel-btn')?.addEventListener('click', () => {
            this.toggleSelectionMode();
        });
    }

    /**
     * Update selection UI with count
     */
    updateSelectionUI() {
        const countEl = document.querySelector('.bulk-count');
        if (countEl) {
            const count = this.selectedTasks.size;
            countEl.textContent = `${count} selected`;
            
            // Update select all button
            const selectAllBtn = document.getElementById('bulk-select-all');
            if (selectAllBtn) {
                const allTasks = document.querySelectorAll('.task-item').length;
                if (count === allTasks && count > 0) {
                    selectAllBtn.innerHTML = '<i class="fas fa-times"></i> Deselect All';
                } else {
                    selectAllBtn.innerHTML = '<i class="fas fa-check-double"></i> Select All';
                }
            }
        }
    }

    /**
     * Remove selection UI
     */
    removeSelectionUI() {
        document.getElementById('bulk-actions-bar')?.remove();
    }

    /**
     * Select all tasks
     */
    selectAll() {
        const checkboxes = document.querySelectorAll('.bulk-checkbox');
        const allChecked = Array.from(checkboxes).every(cb => cb.checked);
        
        checkboxes.forEach(checkbox => {
            checkbox.checked = !allChecked;
            const taskId = parseInt(checkbox.dataset.taskId);
            
            if (!allChecked) {
                this.selectedTasks.add(taskId);
            } else {
                this.selectedTasks.delete(taskId);
            }
        });
        
        this.updateSelectionUI();
    }

    /**
     * Bulk update task status
     */
    async bulkUpdateStatus(status) {
        if (this.selectedTasks.size === 0) return;
        
        const taskIds = Array.from(this.selectedTasks);
        
        try {
            const response = await this.app.api.post('bulk-tasks?action=bulk-update', {
                task_ids: taskIds,
                updates: { status }
            });
            
            if (response.success) {
                this.app.ui.showToast(`Updated ${taskIds.length} task(s) to ${status}`, 'success');
                await this.app.loadTasks();
                this.exitSelectionMode();
            }
        } catch (error) {
            this.app.ui.showToast('Failed to update tasks', 'error');
            logger.error('[BulkActions] Update failed:', error);
        }
    }

    /**
     * Bulk update task priority
     */
    async bulkUpdatePriority(priority) {
        if (this.selectedTasks.size === 0) return;
        
        const taskIds = Array.from(this.selectedTasks);
        
        try {
            const response = await this.app.api.post('bulk-tasks?action=bulk-update', {
                task_ids: taskIds,
                updates: { priority }
            });
            
            if (response.success) {
                this.app.ui.showToast(`Updated ${taskIds.length} task(s) to ${priority} priority`, 'success');
                await this.app.loadTasks();
                this.exitSelectionMode();
            }
        } catch (error) {
            this.app.ui.showToast('Failed to update tasks', 'error');
            logger.error('[BulkActions] Update failed:', error);
        }
    }

    /**
     * Bulk delete tasks
     */
    async bulkDelete() {
        if (this.selectedTasks.size === 0) return;
        
        const count = this.selectedTasks.size;
        const confirmed = confirm(`Are you sure you want to delete ${count} task(s)? This cannot be undone.`);
        
        if (!confirmed) return;
        
        const taskIds = Array.from(this.selectedTasks);
        
        try {
            const response = await this.app.api.post('bulk-tasks?action=bulk-delete', {
                task_ids: taskIds
            });
            
            if (response.success) {
                this.app.ui.showToast(`Deleted ${count} task(s)`, 'success');
                await this.app.loadTasks();
                this.exitSelectionMode();
            }
        } catch (error) {
            this.app.ui.showToast('Failed to delete tasks', 'error');
            logger.error('[BulkActions] Delete failed:', error);
        }
    }
}
