/**
 * AchievementSystem - Gamification and achievement tracking
 * Handles badges, milestones, streaks, and level progression
 * @class AchievementSystem
 */
class AchievementSystem {
    constructor(app) {
        this.app = app;
        this.achievements = [];
        this.userAchievements = new Map();
        this.currentLevel = 1;
        this.currentXP = 0;
        this.streak = {
            current: 0,
            longest: 0,
            lastActivityDate: null
        };
        this.initialized = false; // Add initialization flag
        
        this.levelThresholds = [
            0, 100, 250, 500, 1000, 2000, 3500, 5500, 8000, 11000,
            15000, 20000, 26000, 33000, 41000, 50000, 60000, 75000, 100000
        ];
    }

    /**
     * Initialize achievement system
     */
    async init() {
        if (this.initialized) {
            logger.debug('[Achievements] Already initialized, skipping');
            return;
        }
        
        await this.loadAchievements();
        await this.loadUserProgress();
        await this.updateStreak();
        // Don't calculate level - use the one from API!
        
        // If no achievements loaded from API, use mock data for testing
        if (this.achievements.length === 0) {
            logger.warn('[Achievements] No achievements loaded from API, using mock data');
            this.loadMockAchievements();
        }
        
        logger.info(`[Achievements] Initialized with ${this.achievements.length} achievements, ${this.currentXP} XP, Level ${this.currentLevel}`);
        
        // Check for any achievements that should be unlocked based on current stats
        await this.checkExistingAchievements();
        
        this.initialized = true;
    }
    
    /**
     * Check for achievements that should already be unlocked based on current stats
     */
    async checkExistingAchievements() {
        try {
            const tasks = this.app.state.get('tasks') || [];
            const completedTasks = tasks.filter(t => t.status === 'completed');
            const projects = this.app.state.get('projects') || [];
            
            logger.debug(`[Achievements] Checking existing stats: ${tasks.length} total tasks, ${completedTasks.length} completed`);
            
            // Check task_created achievements
            const taskCreatedCount = tasks.length;
            await this.checkStatAchievements('task_created', taskCreatedCount);
            
            // Check task_completed achievements
            const taskCompletedCount = completedTasks.length;
            await this.checkStatAchievements('task_completed', taskCompletedCount);
            
            // Check project_created achievements  
            const projectCreatedCount = projects.length;
            await this.checkStatAchievements('project_created', projectCreatedCount);
            
            // Check streak achievements
            if (this.streak && this.streak.current > 0) {
                await this.checkStatAchievements('streak_days', this.streak.current);
            }
            
        } catch (error) {
            logger.error('[Achievements] Error checking existing achievements:', error);
        }
    }
    
    /**
     * Check achievements for a specific stat type
     */
    async checkStatAchievements(statType, currentValue) {
        const relevantAchievements = this.achievements.filter(achievement => {
            const criteria = typeof achievement.criteria === 'string' 
                ? JSON.parse(achievement.criteria) 
                : achievement.criteria;
            return criteria.type === statType && currentValue >= criteria.target;
        });
        
        for (const achievement of relevantAchievements) {
            try {
                const criteria = typeof achievement.criteria === 'string' 
                    ? JSON.parse(achievement.criteria) 
                    : achievement.criteria;
                const target = criteria.target || 1;
                const achievementKey = `achievement_${achievement.id}_${criteria.type}`;
                
                // Check if already unlocked
                const currentProgress = this.userAchievements.get(achievement.id);
                if (currentProgress && currentProgress.is_unlocked) {
                    continue;
                }
                
                // Update progress to target (unlock it)
                const response = await this.app.api.post('achievements/update', {
                    achievement_key: achievementKey,
                    target: target,
                    progress: currentValue,
                    increment: false // Set directly, don't increment
                });
                
                if (response && response.success && response.data && response.data.unlocked) {
                    const progressData = response.data.progress;
                    this.userAchievements.set(achievement.id, {
                        achievement_id: achievement.id,
                        progress: progressData.progress,
                        is_unlocked: progressData.is_unlocked,
                        unlocked_at: progressData.unlocked_at
                    });
                    logger.info(`[Achievements] Retroactively unlocked: ${achievement.name}`);
                    
                    // Show toast and award XP for retroactive unlock
                    this.showAchievementUnlocked(achievement);
                    await this.awardAchievementXP(achievement);
                }
            } catch (error) {
                logger.error(`[Achievements] Error checking stat achievement ${achievement.id}:`, error);
            }
        }
    }
    
    /**
     * Load mock achievement data for testing (only loads achievement definitions, uses real XP)
     */
    loadMockAchievements() {
        this.achievements = [
            // Task Creation Achievements (scaled XP: common 10-100, rare 100-300, epic 300-500, legendary 500-1000)
            { id: 1, name: 'First Steps', description: 'Create your first task', icon: 'fas fa-flag', category: 'tasks', rarity: 'common', xp_reward: 10, criteria: { type: 'task_created', target: 1 } },
            { id: 2, name: 'Task Starter', description: 'Create 10 tasks', icon: 'fas fa-plus-circle', category: 'tasks', rarity: 'common', xp_reward: 50, criteria: { type: 'task_created', target: 10 } },
            { id: 3, name: 'Task Creator', description: 'Create 50 tasks', icon: 'fas fa-layer-group', category: 'tasks', rarity: 'rare', xp_reward: 150, criteria: { type: 'task_created', target: 50 } },
            { id: 4, name: 'Task Architect', description: 'Create 100 tasks', icon: 'fas fa-sitemap', category: 'tasks', rarity: 'epic', xp_reward: 300, criteria: { type: 'task_created', target: 100 } },
            { id: 5, name: 'Task Legend', description: 'Create 500 tasks', icon: 'fas fa-crown', category: 'tasks', rarity: 'legendary', xp_reward: 800, criteria: { type: 'task_created', target: 500 } },
            
            // Task Completion Achievements
            { id: 6, name: 'First Victory', description: 'Complete your first task', icon: 'fas fa-check', category: 'tasks', rarity: 'common', xp_reward: 20, criteria: { type: 'task_completed', target: 1 } },
            { id: 7, name: 'Task Finisher', description: 'Complete 10 tasks', icon: 'fas fa-check-circle', category: 'tasks', rarity: 'common', xp_reward: 75, criteria: { type: 'task_completed', target: 10 } },
            { id: 8, name: 'Task Master', description: 'Complete 50 tasks', icon: 'fas fa-check-double', category: 'tasks', rarity: 'rare', xp_reward: 200, criteria: { type: 'task_completed', target: 50 } },
            { id: 9, name: 'Task Champion', description: 'Complete 100 tasks', icon: 'fas fa-trophy', category: 'tasks', rarity: 'epic', xp_reward: 400, criteria: { type: 'task_completed', target: 100 } },
            { id: 10, name: 'Task Destroyer', description: 'Complete 500 tasks', icon: 'fas fa-medal', category: 'tasks', rarity: 'legendary', xp_reward: 1000, criteria: { type: 'task_completed', target: 500 } },
            
            // Productivity Achievements
            { id: 11, name: 'Quick Starter', description: 'Complete 3 tasks in one day', icon: 'fas fa-rocket', category: 'productivity', rarity: 'common', xp_reward: 50, criteria: { type: 'tasks_in_day', target: 3 } },
            { id: 12, name: 'Speed Demon', description: 'Complete 5 tasks in one day', icon: 'fas fa-bolt', category: 'productivity', rarity: 'rare', xp_reward: 150, criteria: { type: 'tasks_in_day', target: 5 } },
            { id: 13, name: 'Productivity Beast', description: 'Complete 10 tasks in one day', icon: 'fas fa-fire', category: 'productivity', rarity: 'epic', xp_reward: 350, criteria: { type: 'tasks_in_day', target: 10 } },
            { id: 14, name: 'Task Machine', description: 'Complete 20 tasks in one day', icon: 'fas fa-robot', category: 'productivity', rarity: 'legendary', xp_reward: 700, criteria: { type: 'tasks_in_day', target: 20 } },
            
            // Streak Achievements
            { id: 15, name: 'Getting Started', description: 'Log in 3 days in a row', icon: 'fas fa-calendar-check', category: 'streaks', rarity: 'common', xp_reward: 30, criteria: { type: 'streak_days', target: 3 } },
            { id: 16, name: 'Week Warrior', description: 'Maintain a 7-day streak', icon: 'fas fa-calendar-week', category: 'streaks', rarity: 'rare', xp_reward: 150, criteria: { type: 'streak_days', target: 7 } },
            { id: 17, name: 'Two Week Champion', description: 'Maintain a 14-day streak', icon: 'fas fa-calendar-alt', category: 'streaks', rarity: 'epic', xp_reward: 350, criteria: { type: 'streak_days', target: 14 } },
            { id: 18, name: 'Monthly Master', description: 'Maintain a 30-day streak', icon: 'fas fa-calendar', category: 'streaks', rarity: 'legendary', xp_reward: 800, criteria: { type: 'streak_days', target: 30 } },
            { id: 19, name: 'Unstoppable', description: 'Maintain a 100-day streak', icon: 'fas fa-infinity', category: 'streaks', rarity: 'legendary', xp_reward: 1000, criteria: { type: 'streak_days', target: 100 } },
            
            // Project Achievements
            { id: 20, name: 'Project Starter', description: 'Create your first project', icon: 'fas fa-folder-plus', category: 'tasks', rarity: 'common', xp_reward: 20, criteria: { type: 'project_created', target: 1 } },
            { id: 21, name: 'Project Manager', description: 'Create 5 projects', icon: 'fas fa-folders', category: 'tasks', rarity: 'rare', xp_reward: 50, criteria: { type: 'project_created', target: 5 } },
            { id: 22, name: 'Project Architect', description: 'Create 20 projects', icon: 'fas fa-building', category: 'tasks', rarity: 'epic', xp_reward: 150, criteria: { type: 'project_created', target: 20 } },
            { id: 23, name: 'First Milestone', description: 'Complete your first project', icon: 'fas fa-flag-checkered', category: 'tasks', rarity: 'rare', xp_reward: 100, criteria: { type: 'project_completed', target: 1 } },
            { id: 24, name: 'Project Finisher', description: 'Complete 10 projects', icon: 'fas fa-award', category: 'tasks', rarity: 'epic', xp_reward: 300, criteria: { type: 'project_completed', target: 10 } },
            
            // Priority Achievements
            { id: 25, name: 'Urgent Care', description: 'Complete 10 urgent tasks', icon: 'fas fa-exclamation-circle', category: 'productivity', rarity: 'rare', xp_reward: 75, criteria: { type: 'urgent_completed', target: 10 } },
            { id: 26, name: 'Priority Master', description: 'Complete 50 high priority tasks', icon: 'fas fa-star', category: 'productivity', rarity: 'epic', xp_reward: 150, criteria: { type: 'high_priority_completed', target: 50 } },
            
            // Time Management Achievements
            { id: 27, name: 'Early Bird', description: 'Complete 10 tasks before their due date', icon: 'fas fa-clock', category: 'productivity', rarity: 'rare', xp_reward: 100, criteria: { type: 'early_completion', target: 10 } },
            { id: 28, name: 'Just in Time', description: 'Complete a task on its due date', icon: 'fas fa-stopwatch', category: 'productivity', rarity: 'common', xp_reward: 25, criteria: { type: 'on_time_completion', target: 1 } },
            { id: 29, name: 'Marathon Runner', description: 'Work on tasks for 7 days straight', icon: 'fas fa-running', category: 'streaks', rarity: 'epic', xp_reward: 200, criteria: { type: 'work_streak', target: 7 } },
            
            // Category Achievements
            { id: 30, name: 'Category Creator', description: 'Create 5 categories', icon: 'fas fa-tags', category: 'tasks', rarity: 'common', xp_reward: 30, criteria: { type: 'category_created', target: 5 } },
            { id: 31, name: 'Organizer', description: 'Use all priority levels', icon: 'fas fa-sort', category: 'productivity', rarity: 'rare', xp_reward: 50, criteria: { type: 'all_priorities_used', target: 1 } },
            
            // Subtask Achievements
            { id: 32, name: 'Detail Oriented', description: 'Create your first subtask', icon: 'fas fa-list-ul', category: 'tasks', rarity: 'common', xp_reward: 15, criteria: { type: 'subtask_created', target: 1 } },
            { id: 33, name: 'Breakdown Master', description: 'Create 50 subtasks', icon: 'fas fa-stream', category: 'tasks', rarity: 'rare', xp_reward: 75, criteria: { type: 'subtask_created', target: 50 } },
            { id: 34, name: 'Subtask Completionist', description: 'Complete 100 subtasks', icon: 'fas fa-tasks', category: 'tasks', rarity: 'epic', xp_reward: 200, criteria: { type: 'subtask_completed', target: 100 } },
            
            // Collaboration Achievements  
            { id: 35, name: 'Team Player', description: 'Share a project with another user', icon: 'fas fa-users', category: 'social', rarity: 'rare', xp_reward: 50, criteria: { type: 'project_shared', target: 1 } },
            { id: 36, name: 'Collaborator', description: 'Work on 5 shared projects', icon: 'fas fa-handshake', category: 'social', rarity: 'epic', xp_reward: 150, criteria: { type: 'shared_projects_worked', target: 5 } },
            
            // Leveling Achievements (bonus XP for reaching milestones)
            { id: 37, name: 'Level 5', description: 'Reach level 5', icon: 'fas fa-level-up-alt', category: 'special', rarity: 'rare', xp_reward: 100, criteria: { type: 'level_reached', target: 5 } },
            { id: 38, name: 'Level 10', description: 'Reach level 10', icon: 'fas fa-chevron-up', category: 'special', rarity: 'epic', xp_reward: 250, criteria: { type: 'level_reached', target: 10 } },
            { id: 39, name: 'Level 20', description: 'Reach level 20', icon: 'fas fa-arrow-up', category: 'special', rarity: 'legendary', xp_reward: 500, criteria: { type: 'level_reached', target: 20 } },
            
            // XP Milestones (keep at 0 to avoid feedback loops - you can't get XP for getting XP)
            { id: 40, name: '1000 XP', description: 'Earn 1000 total XP', icon: 'fas fa-star', category: 'special', rarity: 'rare', xp_reward: 0, criteria: { type: 'xp_earned', target: 1000 } },
            { id: 41, name: '5000 XP', description: 'Earn 5000 total XP', icon: 'fas fa-gem', category: 'special', rarity: 'epic', xp_reward: 0, criteria: { type: 'xp_earned', target: 5000 } },
            { id: 42, name: '10000 XP', description: 'Earn 10000 total XP', icon: 'fas fa-diamond', category: 'special', rarity: 'legendary', xp_reward: 0, criteria: { type: 'xp_earned', target: 10000 } },
            
            // Special Achievements
            { id: 43, name: 'Night Owl', description: 'Complete a task after midnight', icon: 'fas fa-moon', category: 'special', rarity: 'rare', xp_reward: 50, criteria: { type: 'late_night_completion', target: 1 } },
            { id: 44, name: 'Weekend Warrior', description: 'Complete 10 tasks on weekends', icon: 'fas fa-calendar-weekend', category: 'special', rarity: 'rare', xp_reward: 75, criteria: { type: 'weekend_completions', target: 10 } },
            { id: 45, name: 'Perfect Day', description: 'Complete all tasks due today', icon: 'fas fa-check-square', category: 'productivity', rarity: 'epic', xp_reward: 200, criteria: { type: 'perfect_day', target: 1 } },
            { id: 46, name: 'Overachiever', description: 'Complete 150% of your daily goal', icon: 'fas fa-chart-line', category: 'productivity', rarity: 'epic', xp_reward: 150, criteria: { type: 'overachiever', target: 1 } },
            { id: 47, name: 'Clean Slate', description: 'Have no overdue tasks', icon: 'fas fa-broom', category: 'productivity', rarity: 'rare', xp_reward: 100, criteria: { type: 'no_overdue', target: 1 } },
            { id: 48, name: 'Description Master', description: 'Add descriptions to 50 tasks', icon: 'fas fa-align-left', category: 'tasks', rarity: 'rare', xp_reward: 50, criteria: { type: 'descriptions_added', target: 50 } },
            { id: 49, name: 'Tag Team', description: 'Use all category tags', icon: 'fas fa-tag', category: 'tasks', rarity: 'epic', xp_reward: 100, criteria: { type: 'all_categories_used', target: 1 } },
            { id: 50, name: 'The Beginning', description: 'Welcome to TaskList!', icon: 'fas fa-gift', category: 'special', rarity: 'common', xp_reward: 5, criteria: { type: 'account_created', target: 1 } },
            
            // SECRET ACHIEVEMENTS - Hidden until unlocked (1000-2000 XP)
            { id: 51, name: '???', description: 'This achievement is a secret...', icon: 'fas fa-lock', category: 'special', rarity: 'legendary', xp_reward: 2000, criteria: { type: 'secret_completionist', target: 1 }, secret: true, secretName: 'The Completionist', secretDescription: 'Complete 1000 tasks' },
            { id: 52, name: '???', description: 'This achievement is a secret...', icon: 'fas fa-lock', category: 'special', rarity: 'legendary', xp_reward: 1500, criteria: { type: 'secret_speed_master', target: 1 }, secret: true, secretName: 'Speed Master', secretDescription: 'Complete 50 tasks in one day' },
            { id: 53, name: '???', description: 'This achievement is a secret...', icon: 'fas fa-lock', category: 'special', rarity: 'legendary', xp_reward: 1200, criteria: { type: 'secret_midnight_warrior', target: 1 }, secret: true, secretName: 'Midnight Warrior', secretDescription: 'Complete 100 tasks after midnight' },
            { id: 54, name: '???', description: 'This achievement is a secret...', icon: 'fas fa-lock', category: 'special', rarity: 'legendary', xp_reward: 1800, criteria: { type: 'secret_year_streak', target: 1 }, secret: true, secretName: 'The Dedicated', secretDescription: 'Maintain a 365-day streak' },
            { id: 55, name: '???', description: 'This achievement is a secret...', icon: 'fas fa-lock', category: 'special', rarity: 'legendary', xp_reward: 1400, criteria: { type: 'secret_perfect_week', target: 1 }, secret: true, secretName: 'Perfect Week', secretDescription: 'Complete all tasks on time for 7 days straight' },
            { id: 56, name: '???', description: 'This achievement is a secret...', icon: 'fas fa-lock', category: 'special', rarity: 'legendary', xp_reward: 1100, criteria: { type: 'secret_early_riser', target: 1 }, secret: true, secretName: 'Early Riser', secretDescription: 'Complete 50 tasks before 6 AM' },
            { id: 57, name: '???', description: 'This achievement is a secret...', icon: 'fas fa-lock', category: 'special', rarity: 'legendary', xp_reward: 1700, criteria: { type: 'secret_project_master', target: 1 }, secret: true, secretName: 'Project Master', secretDescription: 'Complete 100 projects' },
            { id: 58, name: '???', description: 'This achievement is a secret...', icon: 'fas fa-lock', category: 'special', rarity: 'legendary', xp_reward: 1300, criteria: { type: 'secret_priority_king', target: 1 }, secret: true, secretName: 'Priority King', secretDescription: 'Complete 200 urgent tasks' },
            { id: 59, name: '???', description: 'This achievement is a secret...', icon: 'fas fa-lock', category: 'special', rarity: 'legendary', xp_reward: 1600, criteria: { type: 'secret_subtask_expert', target: 1 }, secret: true, secretName: 'Subtask Expert', secretDescription: 'Create and complete 500 subtasks' },
            { id: 60, name: '???', description: 'This achievement is a secret...', icon: 'fas fa-lock', category: 'special', rarity: 'legendary', xp_reward: 2000, criteria: { type: 'secret_legend', target: 1 }, secret: true, secretName: 'The Legend', secretDescription: 'Reach level 50' },
            
            // ADMIN-ONLY SECRET ACHIEVEMENT
            { id: 99, name: '???', description: 'Only the chosen ones know...', icon: 'fas fa-user-secret', category: 'special', rarity: 'legendary', xp_reward: 5000, criteria: { type: 'admin_secret', target: 1 }, secret: true, adminOnly: true, secretName: '🔐 System Administrator', secretDescription: 'You have access to the admin panel. With great power comes great responsibility.' }
        ];
        
        logger.info('[Achievements] Loaded 61 mock achievements (50 regular + 10 secret + 1 admin-only, using real XP data from API)');
    }

    /**
     * Load achievement definitions
     */
    async loadAchievements() {
        try {
            const response = await this.app.api.get('achievements/definitions');
            if (response && response.success && response.data) {
                this.achievements = response.data.achievements || response.data || [];
                logger.debug(`[Achievements] Loaded ${this.achievements.length} achievements`);
            } else {
                this.achievements = [];
                logger.warn('[Achievements] No achievements data received, using empty array');
            }
        } catch (error) {
            logger.error('[Achievements] Error loading achievements:', error);
            this.achievements = []; // Fallback to empty array
        }
    }

    /**
     * Load user's achievement progress
     */
    async loadUserProgress() {
        try {
            // Use the existing XP API that's already working
            const xpResponse = await this.app.api.getUserXP(CURRENT_USER.id);
            if (xpResponse && xpResponse.success && xpResponse.data) {
                // Use the EXACT data from the API - don't recalculate!
                this.currentXP = xpResponse.data.total_xp || 0;
                this.currentLevel = xpResponse.data.current_level || 1;
                this.xpToNextLevel = xpResponse.data.xp_to_next_level || 0;
                this.levelProgress = xpResponse.data.level_progress || 0;
                
                logger.debug(`[Achievements] Loaded XP from API: ${this.currentXP} XP, Level ${this.currentLevel}, ${this.xpToNextLevel} to next level`);
            } else {
                logger.warn('[Achievements] No XP data received from API');
                this.currentXP = 0;
                this.currentLevel = 1;
                this.xpToNextLevel = 100;
                this.levelProgress = 0;
            }
            
            // Load achievement-specific progress from new API
            try {
                const response = await this.app.api.get('achievements/progress');
                if (response && response.success && response.data) {
                    const achievements = response.data.achievements || [];
                    
                    // Map progress by extracting achievement ID from achievement_key
                    achievements.forEach(progressRecord => {
                        // Extract achievement ID from key like "achievement_1_task_created"
                        const match = progressRecord.achievement_key.match(/achievement_(\d+)_/);
                        if (match) {
                            const achievementId = parseInt(match[1]);
                            this.userAchievements.set(achievementId, {
                                achievement_id: achievementId,
                                progress: progressRecord.progress,
                                is_unlocked: progressRecord.is_unlocked,
                                unlocked_at: progressRecord.unlocked_at
                            });
                        }
                    });
                    logger.debug(`[Achievements] Loaded ${achievements.length} achievement progress records from API`);
                }
            } catch (achError) {
                logger.error('[Achievements] Error loading achievement progress:', achError);
            }
        } catch (error) {
            logger.error('[Achievements] Error loading user progress:', error);
            this.currentXP = 0;
            this.currentLevel = 1;
            this.xpToNextLevel = 100;
            this.levelProgress = 0;
        }
    }

    /**
     * Update streak tracking
     */
    async updateStreak() {
        try {
            const response = await this.app.api.get('achievements/streak');
            if (response && response.success && response.data && response.data.streak) {
                this.streak = response.data.streak;
                logger.debug(`[Achievements] Current streak: ${this.streak.current} days`);
            } else {
                this.streak = { current: 0, longest: 0 };
                logger.warn('[Achievements] No streak data received, using defaults');
            }
        } catch (error) {
            logger.error('[Achievements] Error updating streak:', error);
            this.streak = { current: 0, longest: 0 };
        }
    }

    /**
     * Calculate current level from XP
     */
    calculateLevel() {
        for (let i = 0; i < this.levelThresholds.length; i++) {
            if (this.currentXP < this.levelThresholds[i]) {
                this.currentLevel = i;
                break;
            }
        }
        
        if (this.currentXP >= this.levelThresholds[this.levelThresholds.length - 1]) {
            this.currentLevel = this.levelThresholds.length;
        }
    }

    /**
     * Get progress to next level - uses API data directly with tier-based scaled XP
     * @returns {Object} Progress info
     */
    getLevelProgress() {
        // Calculate based on the API's SCALED level system with tiers
        // Formula: 100 + n*(10 + tier_multiplier)
        // Tier 0 (L1-9):   multiplier = 0   -> 100+n*10
        // Tier 1 (L10-19): multiplier = 10  -> 100+n*20
        // Tier 2 (L20-29): multiplier = 30  -> 100+n*40
        // Tier 3 (L30-39): multiplier = 70  -> 100+n*80
        
        let xpAccumulator = 0;
        let calculatedLevel = 1;
        
        // Find current level by accumulating XP requirements
        while (true) {
            const tier = Math.floor((calculatedLevel - 1) / 10);
            
            // Calculate tier multiplier (0, 10, 30, 70, 150, 310...)
            let tierMultiplier = 0;
            for (let i = 0; i < tier; i++) {
                tierMultiplier += (10 * Math.pow(2, i));
            }
            
            const xpForNextLevel = 100 + (calculatedLevel * (10 + tierMultiplier));
            
            if (xpAccumulator + xpForNextLevel > this.currentXP) {
                break;
            }
            xpAccumulator += xpForNextLevel;
            calculatedLevel++;
        }
        
        // Calculate XP needed for current level
        const tier = Math.floor((calculatedLevel - 1) / 10);
        let tierMultiplier = 0;
        for (let i = 0; i < tier; i++) {
            tierMultiplier += (10 * Math.pow(2, i));
        }
        const xpNeeded = 100 + (calculatedLevel * (10 + tierMultiplier));
        const xpInLevel = this.currentXP - xpAccumulator;
        const percentage = (xpInLevel / xpNeeded) * 100;

        return {
            level: calculatedLevel,
            currentXP: this.currentXP,
            xpInLevel,
            xpNeeded,
            nextLevelXP: xpAccumulator + xpNeeded,
            percentage: Math.min(percentage, 100)
        };
    }

    /**
     * Check and update achievement progress
     * @param {string} eventType - Event that occurred
     * @param {Object} data - Event data
     */
    async checkAchievements(eventType, data = {}) {
        logger.debug(`[Achievements] Checking achievements for event: ${eventType}`);
        
        const relevantAchievements = this.achievements.filter(achievement => {
            const criteria = typeof achievement.criteria === 'string' 
                ? JSON.parse(achievement.criteria) 
                : achievement.criteria;
            return criteria.type === eventType;
        });

        let unlocked = [];
        for (const achievement of relevantAchievements) {
            try {
                const criteria = typeof achievement.criteria === 'string' 
                    ? JSON.parse(achievement.criteria) 
                    : achievement.criteria;
                const target = criteria.target || 1;
                const achievementKey = `achievement_${achievement.id}_${criteria.type}`;
                
                // Check current progress from map
                const currentProgress = this.userAchievements.get(achievement.id);
                
                // Skip if already unlocked
                if (currentProgress && currentProgress.is_unlocked) {
                    continue;
                }
                
                // Update progress via API
                const response = await this.app.api.post('achievements/update', {
                    achievement_key: achievementKey,
                    target: target,
                    progress: 1,
                    increment: true // Increment by 1
                });
                
                if (response && response.success && response.data) {
                    const progressData = response.data.progress;
                    
                    // Update local cache
                    this.userAchievements.set(achievement.id, {
                        achievement_id: achievement.id,
                        progress: progressData.progress,
                        is_unlocked: progressData.is_unlocked,
                        unlocked_at: progressData.unlocked_at
                    });
                    
                    // Check if newly unlocked
                    if (response.data.unlocked && (!currentProgress || !currentProgress.is_unlocked)) {
                        unlocked.push(achievement);
                        logger.info(`[Achievements] Unlocked: ${achievement.name}`);
                        // Award XP for this achievement
                        await this.awardAchievementXP(achievement);
                    }
                }
            } catch (error) {
                logger.error(`[Achievements] Error updating achievement ${achievement.id}:`, error);
            }
        }
        
        // Show toast for unlocked achievements
        for (const achievement of unlocked) {
            this.showAchievementUnlocked(achievement);
        }
        
        return unlocked;
    }

    /**
     * Check if event is relevant to achievement
     * @param {string} eventType - Event type
     * @param {Object} criteria - Achievement criteria
     * @returns {boolean} Is relevant
     */
    isEventRelevant(eventType, criteria) {
        const eventMap = {
            'task_created': ['task_count', 'tasks_created'],
            'task_completed': ['completed_tasks', 'tasks_per_day', 'high_priority_completed', 'early_completion', 'late_completion'],
            'project_created': ['project_count'],
            'project_shared': ['project_shared'],
            'task_assigned': ['tasks_assigned'],
            'streak_updated': ['streak']
        };

        return eventMap[eventType]?.includes(criteria.type) || false;
    }

    /**
     * Update progress for specific achievement
     * @param {Object} achievement - Achievement definition
     * @param {string} eventType - Event type
     * @param {Object} data - Event data
     */
    async updateAchievementProgress(achievement, eventType, data) {
        const criteria = typeof achievement.criteria === 'string'
            ? JSON.parse(achievement.criteria)
            : achievement.criteria;

        let userAchievement = this.userAchievements.get(achievement.id);
        
        if (!userAchievement) {
            userAchievement = {
                achievement_id: achievement.id,
                progress: 0,
                target: criteria.target || 1,
                is_unlocked: false
            };
        }

        if (userAchievement.is_unlocked) return; // Already unlocked

        // Calculate new progress based on event
        const newProgress = await this.calculateProgress(achievement, userAchievement, eventType, data);
        
        if (newProgress > userAchievement.progress) {
            userAchievement.progress = newProgress;
            
            // Check if unlocked
            if (newProgress >= userAchievement.target) {
                await this.unlockAchievement(achievement, userAchievement);
            } else {
                // Save progress
                await this.saveAchievementProgress(achievement.id, newProgress);
            }
        }
    }

    /**
     * Calculate achievement progress
     * @param {Object} achievement - Achievement definition
     * @param {Object} userAchievement - User's achievement progress
     * @param {string} eventType - Event type
     * @param {Object} data - Event data
     * @returns {number} New progress value
     */
    async calculateProgress(achievement, userAchievement, eventType, data) {
        const criteria = typeof achievement.criteria === 'string'
            ? JSON.parse(achievement.criteria)
            : achievement.criteria;

        switch (criteria.type) {
            case 'completed_tasks':
            case 'task_count':
                const tasks = this.app.state.get('tasks');
                return tasks.filter(t => t.status === 'completed').length;
            
            case 'project_count':
                return this.app.state.get('projects').length;
            
            case 'streak':
                return this.streak.current;
            
            case 'tasks_per_day':
                // Would need to check today's completions
                return userAchievement.progress + (eventType === 'task_completed' ? 1 : 0);
            
            case 'high_priority_completed':
                const highPriorityTasks = this.app.state.get('tasks').filter(
                    t => t.status === 'completed' && t.priority === 'high'
                );
                return highPriorityTasks.length;
            
            case 'project_shared':
            case 'tasks_assigned':
                return userAchievement.progress + 1;
            
            default:
                return userAchievement.progress;
        }
    }

    /**
     * Unlock achievement
     * @param {Object} achievement - Achievement definition
     * @param {Object} userAchievement - User's achievement progress
     */
    async unlockAchievement(achievement, userAchievement) {
        try {
            const response = await this.app.api.post('achievements/unlock', {
                achievement_id: achievement.id
            });

            if (response.success) {
                userAchievement.is_unlocked = true;
                userAchievement.unlocked_at = new Date().toISOString();
                this.userAchievements.set(achievement.id, userAchievement);
                
                // Award XP
                this.currentXP += achievement.xp_reward;
                this.calculateLevel();
                
                // Show notification
                this.showUnlockNotification(achievement);
                
                logger.debug(`[Achievements] Unlocked: ${achievement.name} (+${achievement.xp_reward} XP)`);
            }
        } catch (error) {
            logger.error('[Achievements] Error unlocking achievement:', error);
        }
    }

    /**
     * Save achievement progress
     * @param {number} achievementId - Achievement ID
     * @param {number} progress - New progress value
     */
    async saveAchievementProgress(achievementId, progress) {
        try {
            await this.app.api.post('achievements/update-progress', {
                achievement_id: achievementId,
                progress: progress
            });
        } catch (error) {
            logger.error('[Achievements] Error saving progress:', error);
        }
    }

    /**
     * Show achievement unlock notification
     * @param {Object} achievement - Unlocked achievement
     */
    showUnlockNotification(achievement) {
        const notification = document.createElement('div');
        notification.className = `achievement-notification rarity-${achievement.rarity}`;
        notification.innerHTML = `
            <div class="achievement-notification-content">
                <div class="achievement-icon">
                    <i class="${achievement.icon || 'fas fa-trophy'}"></i>
                </div>
                <div class="achievement-details">
                    <div class="achievement-title">Achievement Unlocked!</div>
                    <div class="achievement-name">${achievement.name}</div>
                    <div class="achievement-reward">+${achievement.xp_reward} XP</div>
                </div>
            </div>
        `;

        document.body.appendChild(notification);

        // Animate in
        setTimeout(() => {
            notification.classList.add('show');
        }, 100);

        // Remove after delay
        setTimeout(() => {
            notification.classList.remove('show');
            setTimeout(() => {
                notification.remove();
            }, 500);
        }, 5000);

        // Play sound if enabled
        this.playUnlockSound();
    }

    /**
     * Play achievement unlock sound
     */
    playUnlockSound() {
        // TODO: Add achievement sound
        logger.debug('[Achievements] 🔊 Achievement sound played');
    }

    /**
     * Get achievement stats
     * @returns {Object} Stats object
     */
    getStats() {
        const totalAchievements = (this.achievements || []).length;
        const unlockedAchievements = Array.from(this.userAchievements.values())
            .filter(a => a.is_unlocked).length;
        const completionRate = totalAchievements > 0 ? (unlockedAchievements / totalAchievements) * 100 : 0;

        logger.debug('[Achievements] Stats calculation:', {
            totalAchievements,
            unlockedAchievements,
            userAchievementsSize: this.userAchievements.size,
            currentXP: this.currentXP,
            currentLevel: this.currentLevel
        });

        return {
            total: totalAchievements,
            unlocked: unlockedAchievements,
            locked: totalAchievements - unlockedAchievements,
            completionRate: completionRate.toFixed(1),
            level: this.currentLevel || 1,
            xp: this.currentXP || 0,
            streak: (this.streak || {}).current || 0,
            longestStreak: (this.streak || {}).longest || 0
        };
    }

    /**
     * Render achievements view
     * @returns {string} HTML string
     */
    renderAchievementsView() {
        const stats = this.getStats();
        const levelProgress = this.getLevelProgress();
        
        // Debug logging
        logger.debug('[Achievements] Rendering view:', {
            achievementsCount: this.achievements?.length || 0,
            currentXP: this.currentXP,
            currentLevel: this.currentLevel,
            stats,
            levelProgress
        });
        
        // Group achievements by category
        const categories = {
            tasks: [],
            productivity: [],
            streaks: [],
            social: [],
            special: []
        };

        (this.achievements || []).forEach(achievement => {
            const userProgress = this.userAchievements.get(achievement.id);
            const category = achievement.category || 'special';
            if (!categories[category]) {
                categories[category] = [];
            }
            categories[category].push({
                ...achievement,
                userProgress: userProgress || { progress: 0, is_unlocked: false }
            });
        });

        // Update header stats if elements exist
        setTimeout(() => {
            const unlockedEl = document.getElementById('header-unlocked-count');
            const totalEl = document.getElementById('header-total-count');
            const completionEl = document.getElementById('header-completion-pct');
            
            if (unlockedEl) unlockedEl.textContent = stats.unlocked || 0;
            if (totalEl) totalEl.textContent = stats.total || 0;
            if (completionEl) completionEl.textContent = Math.round(((stats.unlocked || 0) / (stats.total || 1)) * 100) + '%';
        }, 0);
        
        let html = `
            <div class="achievements-view">
                <!-- Achievements by Category -->
        `;

        for (const [category, achievements] of Object.entries(categories)) {
            if (achievements.length === 0) continue;

            html += `
                <div class="achievement-category">
                    <h3 class="category-title">${this.getCategoryName(category)}</h3>
                    <div class="achievements-grid">
                        ${achievements.map(achievement => this.renderAchievementCard(achievement)).join('')}
                    </div>
                </div>
            `;
        }

        html += `</div>`;
        return html;
    }

    /**
     * Render single achievement card
     * @param {Object} achievement - Achievement with user progress
     * @returns {string} HTML string
     */
    renderAchievementCard(achievement) {
        const isUnlocked = achievement.userProgress.is_unlocked;
        const isSecret = achievement.secret && !isUnlocked;
        const progress = achievement.userProgress.progress || 0;
        const criteria = typeof achievement.criteria === 'string'
            ? JSON.parse(achievement.criteria)
            : achievement.criteria;
        const target = criteria.target || 1;
        const progressPercent = Math.min((progress / target) * 100, 100);
        
        // Use secret name/description if unlocked, otherwise show ???
        const displayName = isSecret ? achievement.name : (isUnlocked && achievement.secretName ? achievement.secretName : achievement.name);
        const displayDescription = isSecret ? achievement.description : (isUnlocked && achievement.secretDescription ? achievement.secretDescription : achievement.description);
        const displayIcon = isSecret ? achievement.icon : (isUnlocked && achievement.secretName ? 'fas fa-star' : achievement.icon);

        return `
            <div class="achievement-card ${isUnlocked ? 'unlocked' : 'locked'} ${isSecret ? 'secret' : ''} rarity-${achievement.rarity}">
                <div class="achievement-icon-wrapper">
                    <i class="${displayIcon || 'fas fa-trophy'}"></i>
                    ${isUnlocked ? '<div class="unlock-badge"><i class="fas fa-check"></i></div>' : ''}
                </div>
                <div class="achievement-info">
                    <div class="achievement-name">${displayName}</div>
                    <div class="achievement-description">${displayDescription}</div>
                    ${!isUnlocked ? `
                        <div class="achievement-progress">
                            ${!isSecret ? `
                                <div class="progress-bar">
                                    <div class="progress-fill" style="width: ${progressPercent}%"></div>
                                </div>
                                <div class="progress-text">${progress} / ${target}</div>
                            ` : `
                                <div class="secret-hint">Complete secret requirements to unlock</div>
                            `}
                        </div>
                    ` : `
                        <div class="achievement-xp">+${achievement.xp_reward} XP</div>
                    `}
                </div>
            </div>
        `;
    }

    /**
     * Get category display name
     * @param {string} category - Category key
     * @returns {string} Display name
     */
    getCategoryName(category) {
        const names = {
            tasks: 'Task Achievements',
            productivity: 'Productivity',
            streaks: 'Streaks',
            social: 'Social',
            special: 'Special'
        };
        return names[category] || category;
    }
    
    /**
     * Show achievement unlocked notification
     * @param {Object} achievement - Unlocked achievement
     */
    showAchievementUnlocked(achievement) {
        const displayName = achievement.secretName || achievement.name;
        const displayDescription = achievement.secretDescription || achievement.description;
        
        // Create a fancy toast notification
        const toast = document.createElement('div');
        toast.className = 'achievement-toast';
        toast.innerHTML = `
            <div class="achievement-toast-icon">
                <i class="${achievement.secretName ? 'fas fa-star' : achievement.icon}"></i>
            </div>
            <div class="achievement-toast-content">
                <div class="achievement-toast-title">🏆 Achievement Unlocked!</div>
                <div class="achievement-toast-name">${displayName}</div>
                <div class="achievement-toast-desc">${displayDescription}</div>
                <div class="achievement-toast-xp">+${achievement.xp_reward} XP</div>
            </div>
        `;
        
        document.body.appendChild(toast);
        
        // Animate in
        setTimeout(() => toast.classList.add('show'), 100);
        
        // Remove after 5 seconds
        setTimeout(() => {
            toast.classList.remove('show');
            setTimeout(() => toast.remove(), 300);
        }, 5000);
    }
    
    /**
     * Award XP for unlocking an achievement
     */
    async awardAchievementXP(achievement) {
        try {
            const xpAmount = achievement.xp_reward || 0;
            if (xpAmount <= 0) {
                return;
            }
            
            logger.info(`[Achievements] Awarding ${xpAmount} XP for achievement: ${achievement.name}`);
            
            // Log the XP to the database
            const response = await this.app.api.post('xp/log', {
                source_type: 'achievement_unlocked',
                source_id: achievement.id,
                xp_amount: xpAmount,
                description: `Achievement unlocked: ${achievement.name}`
            });
            
            if (response && response.success) {
                // Update local user XP
                this.currentXP += xpAmount;
                
                // Update user XP in app
                if (this.app.user && this.app.user.xp_total !== undefined) {
                    this.app.user.xp_total += xpAmount;
                }
                
                // Trigger XP update in UI
                if (this.app.updateUserXP) {
                    await this.app.updateUserXP();
                }
                
                logger.info(`[Achievements] Successfully awarded ${xpAmount} XP`);
            }
        } catch (error) {
            logger.error('[Achievements] Error awarding XP:', error);
        }
    }
}
