/**
 * APICache - Request caching layer with TTL and invalidation
 * Reduces API calls and improves performance
 * @class APICache
 */
class APICache {
    constructor(options = {}) {
        this.ttl = options.ttl || 60000; // Default 60 seconds
        this.maxSize = options.maxSize || 100; // Max entries
        this.storage = options.storage || 'memory'; // 'memory' or 'localStorage'
        this.prefix = options.prefix || 'TL_cache_';
        
        this.cache = new Map();
        this.metrics = {
            hits: 0,
            misses: 0,
            sets: 0,
            invalidations: 0
        };

        // Load cache from localStorage if enabled
        if (this.storage === 'localStorage') {
            this.loadFromStorage();
        }
    }

    /**
     * Generate cache key from endpoint and params
     * @param {string} endpoint - API endpoint
     * @param {Object} params - Request parameters
     * @returns {string} Cache key
     */
    generateKey(endpoint, params = {}) {
        const paramsString = JSON.stringify(params);
        return `${endpoint}::${paramsString}`;
    }

    /**
     * Get cached data
     * @param {string} endpoint - API endpoint
     * @param {Object} params - Request parameters
     * @returns {any|null} Cached data or null
     */
    get(endpoint, params = {}) {
        const key = this.generateKey(endpoint, params);
        const entry = this.cache.get(key);

        if (!entry) {
            this.metrics.misses++;
            return null;
        }

        // Check if expired
        const now = Date.now();
        if (now > entry.expiresAt) {
            this.cache.delete(key);
            this.metrics.misses++;
            return null;
        }

        this.metrics.hits++;
        return entry.data;
    }

    /**
     * Set cached data
     * @param {string} endpoint - API endpoint
     * @param {Object} params - Request parameters
     * @param {any} data - Data to cache
     * @param {number} customTTL - Optional custom TTL in ms
     */
    set(endpoint, params = {}, data, customTTL = null) {
        const key = this.generateKey(endpoint, params);
        const ttl = customTTL || this.ttl;
        const expiresAt = Date.now() + ttl;

        // Enforce max size
        if (this.cache.size >= this.maxSize) {
            this.evictOldest();
        }

        const entry = {
            data,
            expiresAt,
            createdAt: Date.now(),
            endpoint,
            params
        };

        this.cache.set(key, entry);
        this.metrics.sets++;

        // Persist to localStorage if enabled
        if (this.storage === 'localStorage') {
            this.saveToStorage(key, entry);
        }
    }

    /**
     * Invalidate cache for specific endpoint
     * @param {string} endpoint - API endpoint to invalidate
     * @param {Object} params - Optional specific params to invalidate
     */
    invalidate(endpoint, params = null) {
        if (params) {
            // Invalidate specific key
            const key = this.generateKey(endpoint, params);
            this.cache.delete(key);
            
            if (this.storage === 'localStorage') {
                localStorage.removeItem(this.prefix + key);
            }
        } else {
            // Invalidate all keys for this endpoint
            const keysToDelete = [];
            for (const [key, entry] of this.cache.entries()) {
                if (entry.endpoint === endpoint) {
                    keysToDelete.push(key);
                }
            }
            
            keysToDelete.forEach(key => {
                this.cache.delete(key);
                if (this.storage === 'localStorage') {
                    localStorage.removeItem(this.prefix + key);
                }
            });
        }

        this.metrics.invalidations++;
    }

    /**
     * Invalidate multiple related endpoints
     * @param {Array} endpoints - Array of endpoints to invalidate
     */
    invalidateMultiple(endpoints) {
        endpoints.forEach(endpoint => this.invalidate(endpoint));
    }

    /**
     * Clear entire cache
     */
    clear() {
        this.cache.clear();
        
        if (this.storage === 'localStorage') {
            // Remove all cache keys from localStorage
            const keysToRemove = [];
            for (let i = 0; i < localStorage.length; i++) {
                const key = localStorage.key(i);
                if (key.startsWith(this.prefix)) {
                    keysToRemove.push(key);
                }
            }
            keysToRemove.forEach(key => localStorage.removeItem(key));
        }

        logger.debug('[APICache] Cache cleared');
    }

    /**
     * Evict oldest cache entry
     */
    evictOldest() {
        let oldestKey = null;
        let oldestTime = Infinity;

        for (const [key, entry] of this.cache.entries()) {
            if (entry.createdAt < oldestTime) {
                oldestTime = entry.createdAt;
                oldestKey = key;
            }
        }

        if (oldestKey) {
            this.cache.delete(oldestKey);
            if (this.storage === 'localStorage') {
                localStorage.removeItem(this.prefix + oldestKey);
            }
        }
    }

    /**
     * Load cache from localStorage
     */
    loadFromStorage() {
        const now = Date.now();
        let loaded = 0;
        let expired = 0;

        for (let i = 0; i < localStorage.length; i++) {
            const key = localStorage.key(i);
            if (key && key.startsWith(this.prefix)) {
                try {
                    const cacheKey = key.replace(this.prefix, '');
                    const entry = JSON.parse(localStorage.getItem(key));
                    
                    // Check if expired
                    if (now > entry.expiresAt) {
                        localStorage.removeItem(key);
                        expired++;
                    } else {
                        this.cache.set(cacheKey, entry);
                        loaded++;
                    }
                } catch (error) {
                    logger.error('[APICache] Error loading cache entry:', error);
                    localStorage.removeItem(key);
                }
            }
        }

        if (loaded > 0 || expired > 0) {
            logger.debug(`[APICache] Loaded ${loaded} entries, removed ${expired} expired entries`);
        }
    }

    /**
     * Save cache entry to localStorage
     * @param {string} key - Cache key
     * @param {Object} entry - Cache entry
     */
    saveToStorage(key, entry) {
        try {
            localStorage.setItem(this.prefix + key, JSON.stringify(entry));
        } catch (error) {
            // localStorage full or not available
            logger.warn('[APICache] Could not save to localStorage:', error);
        }
    }

    /**
     * Get cache metrics
     * @returns {Object} Cache metrics
     */
    getMetrics() {
        const total = this.metrics.hits + this.metrics.misses;
        const hitRate = total > 0 ? ((this.metrics.hits / total) * 100).toFixed(2) : 0;

        return {
            ...this.metrics,
            size: this.cache.size,
            hitRate: `${hitRate}%`,
            total
        };
    }

    /**
     * Reset metrics
     */
    resetMetrics() {
        this.metrics = {
            hits: 0,
            misses: 0,
            sets: 0,
            invalidations: 0
        };
    }

    /**
     * Get cache size in bytes (approximation)
     * @returns {number} Size in bytes
     */
    getSize() {
        let size = 0;
        for (const [key, entry] of this.cache.entries()) {
            size += key.length;
            size += JSON.stringify(entry).length;
        }
        return size;
    }

    /**
     * Get formatted cache size
     * @returns {string} Formatted size
     */
    getFormattedSize() {
        const bytes = this.getSize();
        if (bytes < 1024) return `${bytes} B`;
        if (bytes < 1024 * 1024) return `${(bytes / 1024).toFixed(2)} KB`;
        return `${(bytes / (1024 * 1024)).toFixed(2)} MB`;
    }
}
